/*******************************************************************************
 * Copyright (C) 2018 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */
 
#ifndef _MAX77734_H_
#define _MAX77734_H_
 
#include "mbed.h"
 
class MAX77734
{
    // https://datasheets.maximintegrated.com/en/ds/MAX77734.pdf
public:
 
    /**
     * @brief   Register Addresses
     * @details Enumerated MAX77734 register addresses
     */
    typedef enum {
        REG_INT_GLBL = 0x00,
        REG_INT_CHG,
        REG_STAT_CHG_A,
        REG_STAT_CHG_B,
        REG_ERCFLAG,
        REG_STAT_GLBL,
        REG_INTM_GLBL,
        REG_INTM_CHG,
        REG_CNFG_GLBL,
        REG_CID,
        REG_CNFG_WDT,
        
        REG_CNFG_CHG_A = 0x20,
        REG_CNFG_CHG_B,
        REG_CNFG_CHG_C,
        REG_CNFG_CHG_D,
        REG_CNFG_CHG_E,
        REG_CNFG_CHG_F,
        REG_CNFG_CHG_G,
        REG_CNFG_CHG_H,
        REG_CNFG_CHG_I,
        
        REG_CNFG_LDO_A = 0x30,
        REG_CNFG_LDO_B,
        
        REG_CNFG_SNK1_A = 0x40,
        REG_CNFG_SNK1_B,
        REG_CNFG_SNK2_A,
        REG_CNFG_SNK2_B,
        REG_CNFG_SNK_TOP,
    } REG_TYPE;
 
    typedef enum {
        I_TERM_5_PERCENT = 0x00,
        I_TERM_7P5_PERCENT,
        I_TERM_10_PERCENT,
        I_TERM_15_PERCENT,        
    } I_TERM_TYPE; 
 
    typedef enum {
        VAL_DISABLE = 0,
        VAL_ENABLE,
        VAL_ENABLE_OPTION1, //OPTION 1: emable when GHGIN is valid
    } ENABLE_TYPE;
 
    typedef enum {
        LDO_LOW_PWR_MODE = 0x00,
        LDO_NORMAL_PWR_MODE,
        LDO_PIN_ACT_HIGH,
        LDO_PIN_ACT_LOW,
    } POWER_MODE_TYPE;
  
    /**
     * MAX77734 constructor.
     *
     * @param sda mbed pin to use for SDA line of I2C interface.
     * @param scl mbed pin to use for SCL line of I2C interface.
     */
    MAX77734(PinName sda, PinName scl);
 
    /**
     * MAX77734 constructor.
     *
     * @param i2c I2C object to use.
     */
    MAX77734(I2C *i2c);
 
    /**
     * MAX77734 destructor.
     */
    ~MAX77734();
 
    /**
     * @brief   Initialize MAX77734
     */
    int32_t init();

    /**
     * @brief   I2C Write 
     * @details Writes data to MAX77734 register
     *
     * @param   reg_addr Register to write
     * @param   reg_data Data to write
     * @returns 0 if no errors, -1 if error.
     */
    int32_t i2c_write_byte(MAX77734::REG_TYPE reg_addr, char reg_data);
    int32_t i2c_write_bytes(MAX77734::REG_TYPE startReg, 
                        MAX77734::REG_TYPE stopReg, 
                        const uint8_t *data);
    /**
     * @brief   I2C Read 
     * @details Reads data from MAX77734 register
     *
     * @param   reg_addr Register to read
     * @returns data if no errors, -1 if error.
     */
    int32_t i2c_read_byte(MAX77734::REG_TYPE reg_addr);
    int32_t i2c_read_bytes(MAX77734::REG_TYPE startReg, 
                            MAX77734::REG_TYPE stopReg, 
                            uint8_t *data);

    int32_t i2c_update_byte(MAX77734::REG_TYPE reg_addr, 
                            char set_data, char mask);
    
    
    /**
     * @brief   read_register
     * @details Reads from the chip.
     *
     * @param   MAX77734 Register Type.
     * @returns register value.
     */
    int32_t read_register(MAX77734::REG_TYPE reg_addr); 
    
    /**
     * @brief   interrupt_isr
     * @details interrupt service routine.
     *
     * @param   none.
     * @returns return interrupt status register values.
     */    
    int32_t interrupt_isr();
    
    /**
     * @brief   fset_i_fastchg_uA
     * @details set fast charging current.
     *
     * @param   current in uA.
     * @returns -1 for error.
     */
    int32_t set_i_fastchg_uA(int32_t current);
    
    /**
     * @brief   set_time_fastchg_hour.
     * @details update REG_CNFG_CHG_E register.
     *
     * @param   time in hours.
     * @returns -1 for error.
     */
    int32_t set_time_fastchg_hour(int32_t hour);
    
    /**
     * @brief   set_v_fastchg_mA.
     * @details update REG_CNFG_CHG_G register.
     *
     * @param   voltage in mV.
     * @returns -1 for error.
     */
    int32_t set_v_fastchg_mA(int32_t voltage);
    
    /**
     * @brief   set_i_term_percent.
     * @details update REG_CNFG_CHG_C register..
     *
     * @param   termination percent.
     * @returns -1 for error.
     */
    int32_t set_i_term_percent(MAX77734::I_TERM_TYPE percent);
        
    /**
     * @brief   set_time_topoff_min.
     * @details update REG_CNFG_CHG_C register..
     *
     * @param   top off time in minute.
     * @returns -1 for error.
     */    
    int32_t set_time_topoff_min(int32_t minute);
        
    /**
     * @brief   set_charger_enable.
     * @details update REG_CNFG_CHG_B register.
     *
     * @param   .
     * @returns -1 for error.
     */    
    int32_t set_charger_enable(MAX77734::ENABLE_TYPE control);
        
    /**
     * @brief   set_ldo_enable.
     * @details update REG_CNFG_LDO_B register.
     *
     * @param   enable/disable.
     * @returns -1 for error.
     */    
    int32_t set_ldo_enable(MAX77734::ENABLE_TYPE control);
    
    /**
     * @brief   set_ldo_pwr_mode.
     * @details update REG_CNFG_LDO_B register.
     *
     * @param   power mode.
     * @returns -1 for error.
     */    
    int32_t set_ldo_pwr_mode(MAX77734::POWER_MODE_TYPE mode);
    
    /**
     * @brief   set_ldo_voltage_mV.
     * @details update REG_CNFG_LDO_A register.
     *
     * @param   voltage in mV.
     * @returns -1 for error.
     */    
     int32_t set_ldo_voltage_mV(int32_t voltage);
    
    
    
private: 

    I2C *i2c_;
    bool i2c_owner;
 
};
 
#endif /* _MAX77734_H_ */