/*******************************************************************************
 * Copyright (C) 2018 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */
#include "max77734.h"
 
/***** Definitions *****/
#define I2C_ADDR            (0x48<<1)

//******************************************************************************
MAX77734::MAX77734(PinName sda, PinName scl)
{
    i2c_ = new I2C(sda, scl);
    i2c_owner = true;
 
    i2c_->frequency(400000);
}
 
//******************************************************************************
MAX77734::MAX77734(I2C *i2c) :
    i2c_(i2c)
{
    i2c_owner = false;
}
 
//******************************************************************************
MAX77734::~MAX77734()
{
    if(i2c_owner) {
        delete i2c_;
    }
}
 
//******************************************************************************
int32_t MAX77734::init()
{     
    // Clear Mask
    interrupt_isr();
    
    // set INTM_GLBL: masked all interrupt(Chip Reset Value)
    i2c_update_byte(REG_INTM_GLBL, 0x7c, 0x7c);
    
    // set INTM_CHG: masked all interrupt(Chip Reset Value)
    i2c_update_byte(REG_INTM_CHG, 0x7f, 0x7f);
    
    return 0;
}

//******************************************************************************
int32_t MAX77734::i2c_write_byte(MAX77734::REG_TYPE reg_addr, char reg_data)
{
    char data[2];
 
    data[0] = reg_addr;
    data[1] = reg_data;
    if (i2c_->write(I2C_ADDR, data, 2) != 0) {
        return -1;
    }
 
    return 0;
}

//*****************************************************************************
int32_t MAX77734::i2c_write_bytes(MAX77734::REG_TYPE startReg, 
                                MAX77734::REG_TYPE stopReg,
                                const uint8_t *data)
{
    int32_t numBytes = ((stopReg - startReg) + 1);
    char packet[numBytes + 1];
    
    packet[0] = static_cast<char>(startReg);    
    memcpy(packet + 1, data, numBytes);
    
    return i2c_->write(I2C_ADDR, packet, sizeof(packet));
}
 
//******************************************************************************
int32_t MAX77734::i2c_read_byte(MAX77734::REG_TYPE reg_addr)
{
    char data;
 
    data = reg_addr;
    if (i2c_->write(I2C_ADDR, &data, 1, true) != 0) {
        return -1;
    }
 
    if (i2c_->read(I2C_ADDR | 0x01, &data, 1) != 0) {
        return -1;
    }
 
    return (0x0 + data);
}

//*****************************************************************************
int32_t MAX77734::i2c_read_bytes(MAX77734::REG_TYPE startReg,
                                MAX77734::REG_TYPE stopReg,
                                uint8_t *data)
{
    int32_t rtnVal = -1;
    int32_t numBytes = ((stopReg - startReg) + 1);
    char packet[] = {static_cast<char>(startReg)};
    
    if(i2c_->write(I2C_ADDR, packet, 1) == 0)
    {
        rtnVal = i2c_->read(I2C_ADDR | 0x01, 
                            reinterpret_cast<char *>(data), numBytes);
    }
    
    return rtnVal;
}

//******************************************************************************
int32_t MAX77734::i2c_update_byte(MAX77734::REG_TYPE reg_addr, 
                                    char set_data, char mask)
{
    int32_t rData;
    
    rData = i2c_read_byte(reg_addr);
    if(rData < 0)
        return rData;
    
    rData &= ~mask;
    rData = (set_data & mask);
 
    rData = i2c_write_byte(reg_addr, rData);    
    if(rData < 0)
        return rData;
        
    return 0;
}

//*****************************************************************************
int32_t MAX77734::read_register(MAX77734::REG_TYPE reg_addr)
{
    int32_t rData, mask;
    
    rData = i2c_read_byte(reg_addr);
    if(rData < 0)
        return rData;
        
    switch(reg_addr) {
        case REG_INT_GLBL:     mask = 0x7F; break;
        case REG_INT_CHG:      mask = 0x7F; break;
        case REG_STAT_CHG_A:   mask = 0x7F; break;
        case REG_INTM_GLBL:    mask = 0x7C; break;
        case REG_INTM_CHG:     mask = 0x7F; break;
        case REG_CID:          mask = 0x0F; break;
        case REG_CNFG_WDT:     mask = 0x3F; break;
        case REG_CNFG_CHG_F:   mask = 0xFE; break;
        case REG_CNFG_CHG_G:   mask = 0xFE; break;
        case REG_CNFG_CHG_H:   mask = 0xFC; break;
        case REG_CNFG_LDO_B:   mask = 0x0F; break;
        case REG_CNFG_SNK_TOP: mask = 0x03; break;
        default: mask = 0xFF; break;
    }
        
    return (rData & mask);
}

//*****************************************************************************
int32_t MAX77734::interrupt_isr()
{
    int32_t rData[2];
    
    rData[0] = i2c_read_byte(REG_INT_GLBL) & 0xff;
    if(rData[0] < 0)
        return rData[0];

    rData[1] = i2c_read_byte(REG_INT_CHG) & 0xff;
    if(rData[1] < 1)
        return rData[1]; 

    return ((rData[1] &0xff) << 8 ) | (rData[0] &0xff);
}

//*****************************************************************************
int32_t MAX77734::set_i_fastchg_uA(int32_t current)
{
    int32_t rData;
    int32_t code = (current - 7500) / 7500;
    
    if( current > 300000)
        code = 0x3f;
    
    rData = i2c_update_byte(REG_CNFG_CHG_E, code <<2, 0xFC);    
    if(rData < 0)
        return rData;
    
    return 0;
}

//*****************************************************************************
int32_t MAX77734::set_time_fastchg_hour(int32_t hour)
{
    int32_t rData, code;
        
    switch(hour) {
      case 0:         code = 0x0; break; //Timer Disabled
      case 1 ... 3:   code = 0x1; break; // 3hours
      case 4 ... 5:   code = 0x2; break; //5 hours
      default:        code = 0x3; break; //7 hours
    };  
    
    rData = i2c_update_byte(REG_CNFG_CHG_E, code, 0x03);    
    if(rData < 0)
        return rData;
    
    return 0;
}

//*****************************************************************************
int32_t MAX77734::set_v_fastchg_mA(int32_t voltage)
{
    int32_t rData;
    int32_t code = (voltage - 3600) / 25;
    
    if( voltage > 4600)
        code = 0x3f;
    
    rData = i2c_update_byte(REG_CNFG_CHG_G, code <<2, 0xFC);    
    if(rData < 0)
        return rData;
    
    return 0;
}

//*****************************************************************************
int32_t MAX77734::set_i_term_percent(MAX77734::I_TERM_TYPE percent)
{
    int32_t rData;
            
    rData = i2c_update_byte(REG_CNFG_CHG_C, percent <<3, 0x18);
    if(rData < 0)
        return rData;
    
    return 0;
}

//*****************************************************************************
int32_t MAX77734::set_time_topoff_min(int32_t minute)
{
    int32_t rData, code;
    
    switch(minute) {
      case 0:         code = 0x0; break;
      case 1 ... 5:   code = 0x1; break;
      case 6 ... 10:  code = 0x2; break;
      case 11 ... 15: code = 0x3; break;
      case 16 ... 20: code = 0x4; break;
      case 21 ... 25: code = 0x5; break;
      case 26 ... 30: code = 0x6; break;
      default:        code = 0x7; break;
    };  
            
    rData = i2c_update_byte(REG_CNFG_CHG_C, code, 0x07);
    if(rData < 0)
        return rData;
    
    return 0;
}

//*****************************************************************************
int32_t MAX77734::set_charger_enable(MAX77734::ENABLE_TYPE control)
{
    int32_t rData, code = 0;
    
    if(control == VAL_ENABLE)
        code = 0x01;
            
    rData = i2c_update_byte(REG_CNFG_CHG_B, code, 0x01);
    if(rData < 0)
        return rData;
    
    return 0;
}

//*****************************************************************************
int32_t MAX77734::set_ldo_enable(MAX77734::ENABLE_TYPE control)
{
    int32_t rData, code;
    
    switch(control) {
        case VAL_ENABLE: code = 0x01; break;
        case VAL_ENABLE_OPTION1: code = 0x02; break;
        default: code = 0x00; break;        
    };
            
    rData = i2c_update_byte(REG_CNFG_LDO_B, code, 0x03);
    if(rData < 0)
        return rData;
    
    return 0;
}

//*****************************************************************************
int32_t MAX77734::set_ldo_pwr_mode(MAX77734::POWER_MODE_TYPE mode)
{
    int32_t rData, code;
    
    switch(mode) {
        case LDO_LOW_PWR_MODE:    code = 0x00; break;
        case LDO_NORMAL_PWR_MODE: code = 0x01; break;
        case LDO_PIN_ACT_HIGH:    code = 0x02; break;
        case LDO_PIN_ACT_LOW:     code = 0x03; break;
        default: return -1;
    };
            
    rData = i2c_update_byte(REG_CNFG_LDO_B, code << 2, 0x0C);
    if(rData < 0)
        return rData;
    
    return 0;
}

//*****************************************************************************
int32_t MAX77734::set_ldo_voltage_mV(int32_t voltage)
{
    int32_t rData;
    int32_t code = (voltage - 800) / 25;
                
    rData = i2c_update_byte(REG_CNFG_LDO_A, code, 0x7F);
    if(rData < 0)
        return rData;
    
    return 0;
}
