/*******************************************************************************
* Copyright (C) 2022 Analog Devices, Inc., All rights Reserved.
*
* This software is protected by copyright laws of the United States and
* of foreign countries. This material may also be protected by patent laws
* and technology transfer regulations of the United States and of foreign
* countries. This software is furnished under a license agreement and/or a
* nondisclosure agreement and may only be used or reproduced in accordance
* with the terms of those agreements. Dissemination of this information to
* any party or parties not specified in the license agreement and/or
* nondisclosure agreement is expressly prohibited.
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Analog Devices, Inc. retains all
* ownership rights.
*******************************************************************************
*/

#ifndef _MAX77659_H_
#define _MAX77659_H_

#include "mbed.h"
#include "MAX77659_regs.h"

#define MAX77659_NO_ERROR                    0
#define MAX77659_VALUE_NULL                 -1
#define MAX77659_WRITE_DATA_FAILED          -2
#define MAX77659_READ_DATA_FAILED           -3
#define MAX77659_INVALID_DATA               -4

#define MAX77659_I2C_ADDRESS              	0x80

/**
 * @brief MAX77659 SIMO PMIC with 300mA Switching Charger.
 *
 * @details The MAX77659 provides highly-efficient integrated battery
 * charging and power supply solutions for low-power applications where size and efficiency are critical.
 * https://www.maximintegrated.com/en/products/power/power-management-ics/MAX77659.html
 *
 * @code
 * @endcode
 */

class MAX77659
{
private:
    I2C *i2c_handler;
    InterruptIn *irq_pin;	// interrupt pin

    /**
     * @brief   Register Addresses
     * @details Enumerated MAX77659 register addresses
     */
    typedef enum {
        /*Global*/
        INT_GLBL0    	= 0x00,    // Interrupt Status 0
        INT_GLBL1    	= 0x04,    // Interrupt Status 1
        ERCFLAG      	= 0x05,    // Flags
        STAT_GLBL    	= 0x06,    // Global Status
        INTM_GLBL1   	= 0x08,    // Interrupt Mask 1
        INTM_GLBL0   	= 0x09,    // Interrupt Mask 0
        CNFG_GLBL    	= 0x10,    // Configuration Global
        CNFG_GPIO0   	= 0x11,    // GPIO0 Configuration
        CNFG_GPIO1   	= 0x12,    // GPIO1 Configuration
        CID          	= 0x14,    // Chip Identification Code
        CNFG_WDT     	= 0x17,    // Configuration WatchDog Timer
        /*Charger*/	
        INT_CHG      	= 0x01,    // Charger Interrupt Status
        STAT_CHG_A   	= 0x02,    // Charger Status A
        STAT_CHG_B   	= 0x03,    // Charger Status B
        INT_M_CHG    	= 0x07,    // Charger Interrupt Mask
        CNFG_CHG_A   	= 0x20,    // Charger Configuration A
        CNFG_CHG_B   	= 0x21,    // Charger Configuration B
        CNFG_CHG_C   	= 0x22,    // Charger Configuration C
        CNFG_CHG_D   	= 0x23,    // Charger Configuration D
        CNFG_CHG_E   	= 0x24,    // Charger Configuration E
        CNFG_CHG_F   	= 0x25,    // Charger Configuration F
        CNFG_CHG_G   	= 0x26,    // Charger Configuration G
        CNFG_CHG_H   	= 0x27,    // Charger Configuration H
        CNFG_CHG_I   	= 0x28,    // Charger Configuration I
        /*SBB*/
        CNFG_SBB0_A 	= 0x29,    // SIMO Buck-Boost 0 Configuration A
        CNFG_SBB0_B  	= 0x2A,    // SIMO Buck-Boost 0 Configuration B
        CNFG_SBB1_A  	= 0x2B,    // SIMO Buck-Boost 1 Configuration A
        CNFG_SBB1_B  	= 0x2C,    // SIMO Buck-Boost 1 Configuration B
        CNFG_SBB2_A  	= 0x2D,    // SIMO Buck-Boost 2 Configuration A
        CNFG_SBB2_B  	= 0x2E,    // SIMO Buck-Boost 2 Configuration B
        CNFG_SBB_TOP 	= 0x2F,    // SIMO Buck-Boost Configuration
        CNFG_SBB_TOP_B 	= 0x30,    // SIMO Buck-Boost Configuration
        /*LDO*/
        CNFG_LDO0_A   	= 0x38,   // LDO Configuration A
        CNFG_LDO0_B   	= 0x39    // LDO Configuration B
    } reg_t;

    /**
    * @brief	Interrupt handler function
    */
    void interrupt_handler();

    void (MAX77659::*funcptr)(void);

    /**
    * @brief	Post interrupt jobs after interrupt is detected.
    */
    void post_interrupt_work();

    Thread *post_intr_work_thread;

    struct handler {
        void (*func)(void *);
        void *cb;
    };

    handler *interrupt_handler_list;

public:
	/**
    * @brief Register Configuration
	*		 All Interrupt Flags combined from INT_GLBL0, INT_GLBL1 and INT_CHG
    *
    * @details
    *  - Register      : ERCFLAG (0x05)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Enumerated interrupt flags
    */
    typedef enum {
        INT_GLBL0_GPI0_F,
        INT_GLBL0_GPI0_R,
        INT_GLBL0_NEN_F,
        INT_GLBL0_NEN_R,
        INT_GLBL0_TJAL1_R,
        INT_GLBL0_TJAL2_R,
        INT_GLBL0_DOD_R,
        INT_GLBL1_GPI1_F,
        INT_GLBL1_GPI1_R,
        INT_GLBL1_SBB_TO,
        INT_GLBL1_LDO_F,
        INT_CHG_THM_I,
        INT_CHG_CGH_I,
        INT_CHG_CHGIN_I,
        INT_CHG_TJ_REG_I,
        INT_CHG_SYS_CTRL_I,
		INT_CHG_END
    } reg_bit_int_glbl_t;

    /**
     * MAX77659 constructor.
     */
    MAX77659(I2C *i2c, PinName IRQPin = NC);

    /**
     * MAX77659 destructor.
     */
    ~MAX77659();

    /**
    * @brief	Function pointer type to interrupt handler function
    */
    typedef void (*interrupt_handler_function)(void *);

    /**
    * @brief Read from a register.
    *
    * @param[in] reg Address of a register to be read.
    * @param[out] value Pointer to save result value.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int read_register(uint8_t reg, uint8_t *value);

    /**
    * @brief Write to a register.
    *
    * @param[in] reg Address of a register to be written.
    * @param[out] value Pointer of value to be written to register.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int write_register(uint8_t reg, const uint8_t *value);

	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : ERCFLAG (0x05)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Event Recorder Flags.
    */
    typedef enum {
		ERCFLAG_TOVLD,
		ERCFLAG_SYSOVLO,
		ERCFLAG_AVLUVLO,
		ERCFLAG_MRST,
		ERCFLAG_SFT_OFF_F,
		ERCFLAG_SFT_CRST_F,
		ERCFLAG_WDT_OFF,
    	ERCFLAG_WDT_RST
    }reg_bit_ercflag_t;

    /**
    * @brief		Get bit field of ERCFLAG (0x05) register.
    *
	* @param[in]	bit_field 	ERCFLAG register bit field to be written.
    * @param[out] 	flag 		Pointer to save result of ercglag bit states.	
	*							For individual bit
	*							0x0: ERCFLAG has not occurred,
    *               			0x1: ERCFLAG has occurred.
    *
    * @return		0 on success, error code on failure.
    */
    int get_ercflag(reg_bit_ercflag_t bit_field, uint8_t *flag);

	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : STAT_GLBL (0x06)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Event Recorder Flags.
    */
    typedef enum {
    	STAT_GLBL_STAT_IRQ,
		STAT_GLBL_STAT_EN,
		STAT_GLBL_TJAL1_S,
		STAT_GLBL_TJAL2_S,
		STAT_GLBL_RSVD,
		STAT_GLBL_DOD_S,
		STAT_GLBL_BOK,
		STAT_GLBL_DIDM
    }reg_bit_stat_glbl_t;
	
    /**
    * @brief		Get bit field of STAT_GLBL (0x06) register.
    *
	* @param[in]	bit_field 	STAT_GLBL register bit field to be written.
    * @param[out] 	status 		Pointer to save result of Status Global bit state.
    *
    * @return		0 on success, error code on failure.
    */
    int get_stat_glbl(reg_bit_stat_glbl_t bit_field, uint8_t *status);

	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : INT_M_CHG (0x07), INTM_GLBL1 (0x08) and INTM_GLBL0 (0x09)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : All interrupt mask bits.
    */
    typedef enum {
		INT_M_CHG_THM_M,
		INT_M_CHG_CHG_M,
		INT_M_CHG_CHGIN_M,
		INT_M_CHG_TJ_REG_M,
		INT_M_CHG_SYS_CTRL_M,
		INTM_GLBL1_GPI1_FM = 8,
		INTM_GLBL1_GPI1_RM,
		INTM_GLBL1_SBB_TO_M,
		INTM_GLBL1_LDO_M,
		INTM_GLBL0_GPI0_FM = 16,
		INTM_GLBL0_GPI0_RM,
		INTM_GLBL0_nEN_FM,
		INTM_GLBL0_nEN_RM,
		INTM_GLBL0_TJAL1_RM,
		INTM_GLBL0_TJAL2_RM,
		INTM_GLBL0_DOD_RM,
		INTM_NUM_OF_BIT
    }reg_bit_int_mask_t;
	
	/**
    * @brief		Set bit field of INT_M_CHG (0x07), INTM_GLBL1 (0x08) or INTM_GLBL0 (0x09) register.
    *
	* @param[in]	bit_field 	Register bit field to be set.
    * @param[out] 	maskBit 	0x0: Interrupt is unmasked,
    *                    		0x1: Interrupt is masked.
    *
    * @return		0 on success, error code on failure.
    */
    int set_interrupt_mask(reg_bit_int_mask_t bit_field, uint8_t maskBit);
	
	/**
    * @brief		Get bit field of INT_M_CHG (0x07), INTM_GLBL0 (0x08) or INTM_GLBL1 (0x09) register.
    *
	* @param[in]	bit_field 	Register bit field to be written.
    * @param[out] 	maskBit 	0x0: Interrupt is unmasked,
    *                    		0x1: Interrupt is masked.
    *
    * @return		0 on success, error code on failure.
    */
    int get_interrupt_mask(reg_bit_int_mask_t bit_field, uint8_t *maskBit);

	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : CNFG_GLBL (0x10)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Event Recorder Flags.
    */
    typedef enum {
    	CNFG_GLBL_SFT_CTRL,
		CNFG_GLBL_DBEN_nEN,
		CNFG_GLBL_nEN_MODE,
		CNFG_GLBL_SBIA_EN,
		CNFG_GLBL_SBIA_LPM,
		CNFG_GLBL_T_MRST,
		CNFG_GLBL_PU_DIS
    }reg_bit_cnfg_glbl_t;	

    /**
  	* @brief		Set CNFG_GLBL (0x10) register.
  	*
	* @param[in]	bit_field 	Register bit field to be written.
  	* @param[in]	config 		Register bit field to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int set_cnfg_glbl(reg_bit_cnfg_glbl_t bit_field, uint8_t config);

    /**
  	* @brief		Get CNFG_GLBL (0x10) register.
  	*
	* @param[in]	bit_field 	Register bit field to be written.
  	* @param[out]	config 		Pointer of value to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int get_cnfg_glbl(reg_bit_cnfg_glbl_t bit_field, uint8_t *config);

	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : CNFG_GPIO0 (0x11) or CNFG_GPIO1 (0x12)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Event Recorder Flags.
    */
    typedef enum {
    	CNFG_GPIO_DIR,
		CNFG_GPIO_DI,
		CNFG_GPIO_DRV,
		CNFG_GPIO_DO,
		CNFG_GPIO_DBEN_GPI,
		CNFG_GPIO_ALT_GPIO,
		CNFG_GPIO_RSVD
    }reg_bit_cnfg_gpio_t;	

    /**
  	* @brief		Set either CNFG_GPIO0 (0x11) or CNFG_GPIO1 (0x12).
  	*
	* @param[in]	bit_field 	Register bit field to be written.
  	* @param[in]	channel 	Channel number: 0 or 1
  	* @param[in]	config 		Register bit field to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int set_cnfg_gpio(reg_bit_cnfg_gpio_t bit_field, uint8_t channel, uint8_t config);

    /**
  	* @brief		Get either CNFG_GPIO0 (0x11) or CNFG_GPIO1 (0x12).
  	*
	* @param[in]	bit_field 	Register bit field to be written.
  	* @param[in]	channel 	Channel number: 0 or 1
  	* @param[out]	config 		Pointer of value to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int get_cnfg_gpio(reg_bit_cnfg_gpio_t bit_field, uint8_t channel, uint8_t *config);
	
    /**
    * @brief		Get bit field of CID (0x14) register.
    *
    * @return		CID on success, error code on failure.
    */
    int get_cid(void);

	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : CNFG_WDT (0x17)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Watchdog Timer Configuration.
    */
    typedef enum {
    	CNFG_WDT_WDT_LOCK,
		CNFG_WDT_WDT_EN,
		CNFG_WDT_WDT_CLR,
		CNFG_WDT_WDT_MODE,
		CNFG_WDT_WDT_PER,
		CNFG_WDT_RSVD
    }reg_bit_cnfg_wdt_t;	
	
    /**
  	* @brief		Set CNFG_WDT (0x17) register.
  	*
	* @param[in]	bit_field 	Register bit field to be written.
  	* @param[in]	config 		Field value to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int set_cnfg_wdt(reg_bit_cnfg_wdt_t bit_field, uint8_t config);

    /**
	* @brief		Get CNFG_WDT (0x17) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	config 		Pointer of value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_cnfg_wdt(reg_bit_cnfg_wdt_t bit_field, uint8_t *config);

	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : STAT_CHG_A (0x02)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Watchdog Timer Configuration.
    */
    typedef enum {
    	STAT_CHG_A_THM_DTLS,
		STAT_CHG_A_TJ_REG_STAT,
		STAT_CHG_A_VSYS_MIN_STAT,
		STAT_CHG_A_RSVD
    }reg_bit_stat_chg_a_t;	
	
	/**
	* @brief		Get STAT_CHG_A (0x02) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	status 		Pointer of value to be read.
	*							For individual bit,
	*							0x0 = It is not engaged,
	*							0x1 = It is engaged.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_stat_chg_a(reg_bit_stat_chg_a_t bit_field, uint8_t *status);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_CHG_A (0x02)
	*  - Bit Fields    : [2:0]
	*  - Default       : 0x0
	*  - Description   : Battery Temperature Details.
	*/
    typedef enum {
        THM_DTLS_THERMISTOR_DISABLED,
        THM_DTLS_BATTERY_COLD,
        THM_DTLS_BATTERY_COOL,
        THM_DTLS_BATTERY_WARM,
        THM_DTLS_BATTERY_HOT,
        THM_DTLS_BATTERY_NORMAL,
        THM_DTLS_RESERVED_0x06,
        THM_DTLS_RESERVED_0x07
    }decode_thm_dtls_t;

    /**
	* @brief		Get Battery Temperature Details.
	* 				Valid only when CHGIN_DTLS[1:0] = 0b11.
	*
	* @param[out]	thm_dtls 	Battery temperature details field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_thm_dtls(decode_thm_dtls_t *thm_dtls);

	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : STAT_CHG_B (0x03)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Watchdog Timer Configuration.
    */
    typedef enum {
    	STAT_CHG_B_TIME_SUS,
		STAT_CHG_B_CHG,
		STAT_CHG_B_CHGIN_DTLS,
		STAT_CHG_B_CHG_DTLS
    }reg_bit_stat_chg_b_t;	

    /**
	* @brief		Get STAT_CHG_B (0x03) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	status 		Pointer of value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_stat_chg_b(reg_bit_stat_chg_b_t bit_field, uint8_t *status);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_CHG_B (0x03)
	*  - Bit Fields    : [7:4]
	*  - Default       : 0x0
	*  - Description   : CHG_DTLS[3:0] Charger Details.
	*/
    typedef enum {
        CHG_DTLS_OFF,
        CHG_DTLS_PREQUALIFICATION_MODE,
        CHG_DTLS_FAST_CHARGE_CC,
        CHG_DTLS_JEITA_FAST_CHARGE_CC,
        CHG_DTLS_FAST_CHARGE_CV,
        CHG_DTLS_JEITA_FAST_CHARGE_CV,
        CHG_DTLS_TOP_OFF_MODE,
        CHG_DTLS_JEITA_MODIFIED_TOP_OFF_MODE,
        CHG_DTLS_DONE,
        CHG_DTLS_JEITA_MODIFIED_DONE,
        CHG_DTLS_PREQUALIFICATION_TIMER_FAULT,
        CHG_DTLS_FAST_CHARGE_TIMER_FAULT,
        CHG_DTLS_BATTERY_TEMPERATURE_FAULT,
        CHG_DTLS_RESERVED_0x0D,
        CHG_DTLS_RESERVED_0x0E,
        CHG_DTLS_RESERVED_0x0F
    }decode_chg_dtls_t;

    /**
	* @brief		Get Charger Details.
	*
	* @param[out]	chg_dtls Charger details field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_chg_dtls(decode_chg_dtls_t *chg_dtls);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_A (0x20)
	*  - Bit Fields    : [7:6]
	*  - Default       : 0x0
	*  - Description   : VHOT JEITA Temperature Threshold.
	*/
    typedef enum {
        THM_HOT_VOLT_0_411V,
        THM_HOT_VOLT_0_367V,
        THM_HOT_VOLT_0_327V,
        THM_HOT_VOLT_0_291V
    }decode_thm_hot_t;

    /**
	* @brief		Set the VHOT JEITA Temperature Threshold.
	*
	* @param[in]	thm_hot The VHOT JEITA temperature threshold field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_thm_hot(decode_thm_hot_t thm_hot);

    /**
	* @brief		Get the VHOT JEITA Temperature Threshold.
	*
	* @param[out]	thm_hot The VHOT JEITA temperature threshold field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_thm_hot(decode_thm_hot_t *thm_hot);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_A (0x20)
	*  - Bit Fields    : [5:4]
	*  - Default       : 0x0
	*  - Description   : VWARM JEITA Temperature Threshold.
	*/
    typedef enum {
        THM_WARM_VOLT_0_511V,
        THM_WARM_VOLT_0_459V,
        THM_WARM_VOLT_0_411V,
        THM_WARM_VOLT_0_367V
    }decode_thm_warm_t;

    /**
	* @brief		Set the VWARM JEITA Temperature Threshold.
	*
	* @param[in]	thm_warm The VWARM JEITA temperature threshold field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_thm_warm(decode_thm_warm_t thm_warm);

    /**
	* @brief		Get the VWARM JEITA Temperature Threshold.
	*
	* @param[out]	thm_warm The VWARM JEITA temperature threshold field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_thm_warm(decode_thm_warm_t *thm_warm);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_A (0x20)
	*  - Bit Fields    : [3:2]
	*  - Default       : 0x0
	*  - Description   : VCOOL JEITA Temperature Threshold.
	*/
    typedef enum {
        THM_COOL_VOLT_0_923V,
        THM_COOL_VOLT_0_867V,
        THM_COOL_VOLT_0_807V,
        THM_COOL_VOLT_0_747V
    }decode_thm_cool_t;

    /**
	* @brief		Set the VCOOL JEITA Temperature Threshold.
	*
	* @param[in]	thm_cool The VCOOL JEITA temperature threshold field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_thm_cool(decode_thm_cool_t thm_cool);

    /**
	* @brief		Get the VCOOL JEITA Temperature Threshold.
	*
	* @param[out]	thm_cool The VCOOL JEITA temperature threshold field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_thm_cool(decode_thm_cool_t *thm_cool);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_A (0x20)
	*  - Bit Fields    : [1:0]
	*  - Default       : 0x0
	*  - Description   : VCOLD JEITA Temperature Threshold.
	*/
    typedef enum {
        THM_COLD_VOLT_1_024V,
        THM_COLD_VOLT_0_976V,
        THM_COLD_VOLT_0_923V,
        THM_COLD_VOLT_0_867V
    }decode_thm_cold_t;

    /**
	* @brief		Set the VCOLD JEITA Temperature Threshold.
	*
	* @param[in]	thm_cold The VCOLD JEITA temperature threshold field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_thm_cold(decode_thm_cold_t thm_cold);

    /**
	* @brief		Get the VCOLD JEITA Temperature Threshold.
	*
	* @param[out]	thm_cold The VCOLD JEITA temperature threshold field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_thm_cold(decode_thm_cold_t *thm_cold);

	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : CNFG_CHG_B (0x21)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Watchdog Timer Configuration.
    */
    typedef enum {
    	CNFG_CHG_B_CHG_EN,
		CNFG_CHG_B_I_PQ,
		CNFG_CHG_B_RSVD
    }reg_bit_cnfg_chg_b_t;	
	
	/**
	* @brief		Set CNFG_CHG_B (0x21) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[in]	config 		Register bit field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_cnfg_chg_b(reg_bit_cnfg_chg_b_t bit_field, uint8_t config);

    /**
	* @brief		Get CNFG_CHG_B (0x21) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	config 		Pointer of value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_cnfg_chg_b(reg_bit_cnfg_chg_b_t bit_field, uint8_t *config);

	/**
	* @brief		Set Battery Prequalification Voltage Threshold (VPQ).
	*				Bit 7:5 of CNFG_CHG_C (0x22) register.
	*
	* @param[in]	voltV 	
	*						2.3V, 2.4V, 2.5V, 2.6V, 
    *                       2.7V, 2.8V, 2.9V, 3.0V.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_chg_pq(float voltV);

    /**
	* @brief		Get Battery Prequalification Voltage Threshold (VPQ).
	*				Bit 7:5 of CNFG_CHG_C (0x22) register.
	*
	* @param[out]	voltV 	Pointer of value to be read.
	*						2.3V, 2.4V, 2.5V, 2.6V, 
    *                       2.7V, 2.8V, 2.9V, 3.0V.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_chg_pq(float *voltV);
	
	/**
	* @brief		Set Charger Termination Current (ITERM).
	* 				I_TERM[1:0] sets the charger termination current
	* 				as a percentage of the fast charge current IFAST-CHG.
	*				Bit 4:3 of CNFG_CHG_C (0x22) register.
	*
	* @param[in]	percent 	
	*							5%, 7.5%, 10%, 15%.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_i_term(float percent);

    /**
	* @brief		Get Charger Termination Current (ITERM).
	* 				I_TERM[1:0] sets the charger termination current
	* 				as a percentage of the fast charge current IFAST-CHG.
	*				Bit 4:3 of CNFG_CHG_C (0x22) register.
	*
	* @param[out]	percent 	Pointer of value to be read.
	*							5%, 7.5%, 10%, 15%.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_i_term(float *percent);
	
	/**
	* @brief		Set Top-off Timer Value.
	*				Bit 2:0 of CNFG_CHG_C (0x22) register.
	*
	* @param[in]	minute 	
	*						0 minutes, 5 minutes, 10 minutes
	*						15 minutes, 20 minutes, 25 minutes, 
    *                       30 minutes, 35 minutes.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_t_topoff(uint8_t minute);

    /**
	* @brief		Get Top-off Timer Value.
	*				Bit 2:0 of CNFG_CHG_C (0x22) register.
	*
	* @param[out]	minute 	Pointer of value to be read.
							0 minutes, 5 minutes, 10 minutes
	*						15 minutes, 20 minutes, 25 minutes, 
    *                       30 minutes, 35 minutes.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_t_topoff(uint8_t *minute);

	/**
	* @brief		Set the Die Junction Temperature Regulation Point, TJ-REG.
	*				Bit 7:5 of CNFG_CHG_D (0x23) register.
	*
	* @param[in]	tempDegC 	60ºC, 70ºC, 80ºC,
	*							90ºC, 100ºC.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_tj_reg(uint8_t tempDegC);

    /**
	* @brief		Get the Die Junction Temperature Regulation Point, TJ-REG.
	*				Bit 7:5 of CNFG_CHG_D (0x23) register.
	*
	* @param[out]	tempDegC	Pointer of value to be read.
	*							60ºC, 70ºC, 80ºC, 90ºC, 100ºC.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_tj_reg(uint8_t *tempDegC);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_D (0x23)
	*  - Bit Fields    : [4]
	*  - Default       : 0x0
	*  - Description   : SYS Headroom Voltage Regulation.
	*/
    typedef enum {
        VSYS_HDRM_VOLT_0_15,
        VSYS_HDRM_VOLT_0_20
    }decode_vsys_hdrm_t;

    /**
	* @brief		Set SYS Headroom Voltage Regulation.
	*
	* @param[in]	vsys_hdrm SYS Headroom Voltage field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_vsys_hdrm(decode_vsys_hdrm_t vsys_hdrm);

    /**
	* @brief		Get SYS Headroom Voltage Regulation.
	*
	* @param[out]	vsys_hdrm SYS Headroom Voltage field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_vsys_hdrm(decode_vsys_hdrm_t *vsys_hdrm);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_D (0x23)
	*  - Bit Fields    : [1:0]
	*  - Default       : 0x0
	*  - Description   : Minimum SYS Voltage.
	*/
    typedef enum {
        VSYS_MIN_VOLT_3_2,
        VSYS_MIN_VOLT_3_3,
        VSYS_MIN_VOLT_3_4,
        VSYS_MIN_VOLT_3_5
    }decode_vsys_min_t;

	/**
	* @brief		Set Minimum SYS Voltage.
	*				Bit 1:0 of CNFG_CHG_D (0x23) register.
	*
	* @param[in]	vsys_min 	Decoded values for 3.2V, 3.3V, 3.4V, 3.5V.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_vsys_min(decode_vsys_min_t vsys_min);

    /**
	* @brief		Get Minimum SYS Voltage.
	*				Bit 1:0 of CNFG_CHG_D (0x23) register.
	*
	* @param[out]	vsys_min 	Pointer of value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_vsys_min(decode_vsys_min_t *vsys_min);

	/**
	* @brief		Set the Fast-Charge Constant Current Value, IFAST-CHG.
	*				Bit 7:2 of CNFG_CHG_E (0x24) register.
	*
	* @param[in]	currentmA 	7.5mA, 15.0mA, 22.5mA, ... 
    *                       	292.5mA, 300.0mA.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_chg_cc(float currentmA);

    /**
	* @brief		Get the Fast-Charge Constant Current Value, IFAST-CHG.
	*				Bit 7:2 of CNFG_CHG_E (0x24) register.
	*
	* @param[out]	currentmA 	Pointer of value to be read.
	*							7.5mA, 15.0mA, 22.5mA, ... 
    *                       	292.5mA, 300.0mA.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_chg_cc(float *currentmA);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_E (0x24)
	*  - Bit Fields    : [1:0]
	*  - Default       : 0x0
	*  - Description   : Fast-charge Safety timer, tFC.
	*/
    typedef enum {
        T_FAST_CHG_TIMER_DISABLED,
        T_FAST_CHG_HOUR_3H,
        T_FAST_CHG_HOUR_5H,
        T_FAST_CHG_HOUR_7H
    }decode_t_fast_chg_t;

    /**
	* @brief		Set the Fast-charge Safety timer, tFC.
	*				Bit 1:0 of CNFG_CHG_E (0x24) register.
	*
	* @param[in]	t_fast_chg Fast-charge safety timer field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_t_fast_chg(decode_t_fast_chg_t t_fast_chg);

    /**
	* @brief		Get the Fast-charge Safety timer, tFC.
	*				Bit 1:0 of CNFG_CHG_E (0x24) register.
	*
	* @param[out]	t_fast_chg Fast-charge safety timer field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_t_fast_chg(decode_t_fast_chg_t *t_fast_chg);

	/**
	* @brief		Set IFAST-CHG-JEITA
	* 				when the battery is either cool or warm as defined by the
	* 				VCOOL and VWARM temperature thresholds.
	*				Bit 7:2 of CNFG_CHG_F (0x25) register.
	*
	* @param[in]	currentmA 	7.5mA, 15.0mA, 22.5mA, ... 
    *                       	292.5mA, 300.0mA.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_chg_cc_jeita(float currentmA);

    /**
	* @brief		Get IFAST-CHG-JEITA
	* 				when the battery is either cool or warm as defined by the
	* 				VCOOL and VWARM temperature thresholds.
	*				Bit 7:2 of CNFG_CHG_F (0x25) register.
	*
	* @param[out]	currentmA 	Pointer of value to be read.
	*							7.5mA, 15.0mA, 22.5mA, ... 
    *                       	292.5mA, 300.0mA.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_chg_cc_jeita(float *currentmA);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_F (0x25)
	*  - Bit Fields    : [1]
	*  - Default       : 0x0
	*  - Description   : Thermistor Enable Bit 
	*/
    typedef enum {
        THM_EN_DISABLED,
        THM_EN_ENABLED
    }decode_thm_en_t;

    /**
	* @brief		Set Thermistor Enable Bit.
	*				Bit 1 of CNFG_CHG_F (0x25) register.
	*
	* @param[in]	thm_en Thermistor Enable Bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_thm_en(decode_thm_en_t thm_en);

    /**
	* @brief		Get Thermistor Enable Bit.
	*				Bit 1:0 of CNFG_CHG_F (0x25) register.
	*
	* @param[out]	thm_en Thermistor Enable Bit field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_thm_en(decode_thm_en_t *thm_en);

	/**
	* @brief		Set Fast-Charge Battery Regulation Voltage, VFAST-CHG.
	*				Bit 7:2 of CNFG_CHG_G (0x26) register.
	*
	* @param[in]	voltV	3.600V, 3.625V, 3.650V, ... 
    *                       4.575V, 4.600V.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_chg_cv(float voltV);

    /**
	* @brief		Get Fast-Charge Battery Regulation Voltage, VFAST-CHG.
	*				Bit 7:2 of CNFG_CHG_G (0x26) register.
	*
	* @param[out]	voltV	Pointer of value to be read.
	*						3.600V, 3.625V, 3.650V, ... 
    *                       4.575V, 4.600V.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_chg_cv(float *voltV);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_G (0x26)
	*  - Bit Fields    : [1]
	*  - Default       : 0x0
	*  - Description   : Setting this bit places CHGIN in USB suspend mode.  
	*/
    typedef enum {
        USBS_CHGIN_NOT_SUSPENDED,
        USBS_CHGIN_SUSPENDED
    }decode_usbs_t;

    /**
	* @brief		Set USB Suspend Mode Bit.
	*				Bit 1 of CNFG_CHG_G (0x26) register.
	*
	* @param[in]	usbs CHGIN in USB suspend mode bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_usbs(decode_usbs_t usbs);

    /**
	* @brief		Get USB Suspend Mode Bit.
	*				Bit 1:0 of CNFG_CHG_G (0x26) register.
	*
	* @param[out]	usbs CHGIN in USB suspend mode bit field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_usbs(decode_usbs_t *usbs);

	/**
	* @brief		Set the modified VFAST-CHG-JEITA for when the battery is either
	* 				cool or warm as defined by the VCOOL and VWARM temperature thresholds.
	*				Bit 7:2 of CNFG_CHG_H (0x27) register.
	*
	* @param[in]	voltV	Pointer of value to be read.
	*						3.600V, 3.625V, 3.650V, ... 
    *                       4.575V, 4.600V.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_chg_cv_jeita(float voltV);

    /**
	* @brief		Get the modified VFAST-CHG-JEITA for when the battery is either
	* 				cool or warm as defined by the VCOOL and VWARM temperature thresholds.
	*				Bit 7:2 of CNFG_CHG_H (0x27) register.
	*
	* @param[out]	voltV 	Pointer of value to be read.
	*						3.600V, 3.625V, 3.650V, ... 
    *                       4.575V, 4.600V.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_chg_cv_jeita(float *voltV);

	/**
	* @brief		Set the Battery Discharge Current Full-Scale Current Value.
	*				Bit 7:4 of CNFG_CHG_I (0x28) register.
	*
	* @param[in]	currentmA 	8.2mA, 40.5mA, 72.3mA, 103.4mA, 
    *                           134.1mA, 164.1mA, 193.7mA, 222.7mA,                       
    *                           251.2mA, 279.3mA, 300.0mA
	*
	* @return		0 on success, error code on failure.
	*/
    int set_imon_dischg_scale(float currentmA);

    /**
	* @brief		Get the Battery Discharge Current Full-Scale Current Value.
	*				Bit 7:4 of CNFG_CHG_I (0x28) register.
	*
	* @param[out]	currentmA 	Pointer of value to be read.
	*							8.2mA, 40.5mA, 72.3mA, 103.4mA, 
    *                           134.1mA, 164.1mA, 193.7mA, 222.7mA,                       
    *                           251.2mA, 279.3mA, 300.0mA
	*
	* @return		0 on success, error code on failure.
	*/
    int get_imon_dischg_scale(float *currentmA);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_I (0x28)
	*  - Bit Fields    : [3:0]
	*  - Default       : 0x0
	*  - Description   : Analog channel to connect to AMUX.
	*/
    typedef enum {
        MUX_SEL_MULTIPLEXER_DISABLED, 
		MUX_SEL_CHGIN_VOLTAGE_MONITOR, 
		MUX_SEL_CHGIN_CURRENT_MONITOR, 
		MUX_SEL_BATTERY_VOLTAGE_MONITOR,
		MUX_SEL_BATTERY_CHARGE_CURRENT_MONITOR, 
		MUX_SEL_BATTERY_DISCHARGE_CURRENT_MONITOR_NORMAL, 
		MUX_SEL_BATTERY_DISCHARGE_CURRENT_MONITOR_NULL, 
		MUX_SEL_RESERVED_0x07,
		MUX_SEL_RESERVED_0x08,
		MUX_SEL_AGND_VOLTAGE_MONITOR,
		MUX_SEL_SYS_VOLTAGE_MONITOR,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0B,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0C,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0D,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0E,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0F
    }decode_mux_sel_t;

    /**
	* @brief		Set the analog channel to connect to AMUX.
	*
	* @param[in]	selection AMUX value field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_mux_sel(decode_mux_sel_t selection);

    /**
	* @brief		Get the analog channel to connect to AMUX.
	*
	* @param[out]	selection AMUX value field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_mux_sel(decode_mux_sel_t *selection);

	/**
	* @brief		Set SIMO Buck-Boost Channel x Target Output Voltage.
	*				CNFG_SBB0_A (0x29), CNFG_SBB1_A (0x2B) and CNFG_SBB2_A (0x2D)
	*
	* @param[in]	channel 	Channel number: 0, 1 or 2.
	* @param[in]	voltV 		SIMO buck-boost channel x target output voltage field to be written.
	*							SBBx = 500mV + 25mV x TV_SBBx[7:0]
	*							0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*							0.650V, 0.675V, 0.700V, ... 
	*							5.425V, 5.450V, 5.475V, 5.500V.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_tv_sbb(uint8_t channel, float voltV);

    /**
	* @brief		Get SIMO Buck-Boost Channel x Target Output Voltage.
	*				CNFG_SBB0_A (0x29), CNFG_SBB1_A (0x2B) and CNFG_SBB2_A (0x2D)
	*
	* @param[in]	channel 	Channel number: 0, 1 or 2.
	* @param[out]	voltV 		SIMO buck-boost channel x target output voltage field to be read.
	*							SBBx = 500mV + 25mV x TV_SBBx[7:0]
	*							0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*							0.650V, 0.675V, 0.700V, ... 
	*							5.425V, 5.450V, 5.475V, 5.500V.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_tv_sbb(uint8_t channel, float *voltV);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x2A), CNFG_SBB1_B (0x2C) and CNFG_SBB2_B (0x2E)
	*  - Bit Fields    : [6]
	*  - Default       : 0x0
	*  - Description   : Operation mode of SBB0, 1 or 2.
	*/
    typedef enum {
		OP_MODE_BUCK_BOOST_MODE,
        OP_MODE_BUCK_MODE
    }decode_op_mode_t;

    /**
	* @brief		Set Operation mode of SBBx.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	mode 	Operation mode of SBBx bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_op_mode(uint8_t channel, decode_op_mode_t mode);

    /**
	* @brief		Get Operation mode of SBBx.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	mode 	Operation mode of SBBx bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_op_mode(uint8_t channel, decode_op_mode_t *mode);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x2A), CNFG_SBB1_B (0x2C) and CNFG_SBB2_B (0x2E)
	*  - Bit Fields    : [3]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost Channel 0, 1 or 2 Active-Discharge Enable.
	*/
    typedef enum {
        ADE_SBB_DISABLED,
        ADE_SBB_ENABLED
    }decode_ade_sbb_t;

    /**
	* @brief		Set SIMO Buck-Boost Channel x Active-Discharge Enable.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	ade_sbb SIMO buck-boost channel 2 active-discharge enable bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_ade_sbb(uint8_t channel, decode_ade_sbb_t ade_sbb);

    /**
	* @brief		Get SIMO Buck-Boost Channel x Active-Discharge Enable.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	ade_sbb SIMO buck-boost channel 2 active-discharge enable bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_ade_sbb(uint8_t channel, decode_ade_sbb_t *ade_sbb);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x2A), CNFG_SBB1_B (0x2C) and CNFG_SBB2_B (0x2E)
	*  - Bit Fields    : [2:0]
	*  - Default       : 0x0
	*  - Description   : Enable Control for SIMO Buck-Boost Channel 0, 1 or 2.
	*/
    typedef enum {
        EN_SBB_FPS_SLOT_0,
        EN_SBB_FPS_SLOT_1,
        EN_SBB_FPS_SLOT_2,
        EN_SBB_FPS_SLOT_3,
        EN_SBB_OFF,
        EN_SBB_SAME_AS_0X04,
        EN_SBB_ON,
        EN_SBB_SAME_AS_0X06
    }decode_en_sbb_t;

    /**
	* @brief		Set Enable Control for SIMO Buck-Boost Channel x.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	en_sbb 	Enable control for SIMO buck-boost channel x field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_en_sbb(uint8_t channel, decode_en_sbb_t en_sbb);

    /**
	* @brief		Get Enable Control for SIMO Buck-Boost Channel x.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	en_sbb 	Enable control for SIMO buck-boost channel x field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_en_sbb(uint8_t channel, decode_en_sbb_t *en_sbb);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB_TOP (0x2F)
	*  - Bit Fields    : [7]
	*  - Default       : 0x0
	*  - Description   : Operation mode of the charging channel of SIMO
	*/
    typedef enum {
        OP_MODE_CHG_BUCK_BOOST,
        OP_MODE_CHG_BUCK
    }decode_op_mode_chg_t;

    /**
	* @brief		Set Operation mode of the charging channel of SIMO.
	*
	* @param[in]	op_mode_chg Operation mode of the charging channel of SIMO bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_op_mode_chg(decode_op_mode_chg_t op_mode_chg);

    /**
	* @brief		Get Operation mode of the charging channel of SIMO.
	*
	* @param[out]	op_mode_chg Operation mode of the charging channel of SIMO bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_op_mode_chg(decode_op_mode_chg_t *op_mode_chg);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB_TOP (0x2F)
	*  - Bit Fields    : [1:0]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost (all channels) Drive Strength Trim.
	*/
    typedef enum {
        DRV_SBB_FASTEST_TRANSITION_TIME,
        DRV_SBB_A_LITTLE_SLOWER_THAN_0X00,
        DRV_SBB_A_LITTLE_SLOWER_THAN_0X01,
        DRV_SBB_A_LITTLE_SLOWER_THAN_0X02
    }decode_drv_sbb_t;

    /**
	* @brief		Set SIMO Buck-Boost (all channels) Drive Strength Trim.
	*
	* @param[in]	drv_sbb SIMO buck-boost drive strength trim field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_drv_sbb(decode_drv_sbb_t drv_sbb);

    /**
	* @brief		Get SIMO Buck-Boost (all channels) Drive Strength Trim.
	*
	* @param[out]	drv_sbb SIMO buck-boost drive strength trim field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_drv_sbb(decode_drv_sbb_t *drv_sbb);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB_TOP_B (0x30)
	*  - Bit Fields    : [7:6]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost Charging Channel Peak Current Limit
	*/
    typedef enum {
        IP_CHG_AMP_2_000,
        IP_CHG_AMP_1_500,
        IP_CHG_AMP_1_000,
        IP_CHG_AMP_0_500
    }decode_ip_chg_t;
	
    /**
	* @brief		Set SIMO Buck-Boost Charging Channel Peak Current Limit.
	*
	* @param[in]	ip_chg SIMO Buck-Boost Charging Channel Peak Current Limit field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_ip_chg(decode_ip_chg_t ip_chg);

    /**
	* @brief		Get SIMO Buck-Boost Charging Channel Peak Current Limit.
	*
	* @param[out]	ip_chg SIMO Buck-Boost Charging Channel Peak Current Limit field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_ip_chg(decode_ip_chg_t *ip_chg);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB_TOP_B (0x30)
	*  - Bit Fields    : [5:4], [3:2] and [1:0]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost Channel 0, 1 or 2 Peak Current Limit
	*/
    typedef enum {
        IP_SBB_AMP_1_000,
        IP_SBB_AMP_0_750,
        IP_SBB_AMP_0_500,
        IP_SBB_AMP_0_333
    }decode_ip_sbb_t;
	
    /**
	* @brief		Set SIMO Buck-Boost Channel 0, 1 or 2 Peak Current Limit.
	*
	* @param[in]	channel 	Channel number: 0, 1 or 2.
	* @param[in]	ip_sbb 		SIMO Buck-Boost Channel x Peak Current Limit field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_ip_sbb(uint8_t channel, decode_ip_sbb_t ip_sbb);

    /**
	* @brief		Get SIMO Buck-Boost Channel 0, 1 or 2 Peak Current Limit.
	*
	* @param[in]	channel 	Channel number: 0, 1 or 2.
	* @param[out]	ip_sbb 		SIMO Buck-Boost Channel x Peak Current Limit field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_ip_sbb(uint8_t channel, decode_ip_sbb_t *ip_sbb);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_A (0x38)
	*  - Bit Fields    : [7]
	*  - Default       : 0x0
	*  - Description   : LDO Output Voltage. This bit applies a 1.325V offset to the output voltage of the LDO. 
	*/
    typedef enum {
        TV_LDO_NO_OFFSET,
        TV_LDO_NO_1_325V
    }decode_tv_ldo_offset_t;
	
	/**
	* @brief		Set LDO Output Channel 0 Target Output Voltage. Bit 7.
	*				CNFG_LDO0_A (0x38)
	*
	* @param[in]	offset 		LDO Output Channel 0 target output voltage offset field to be read.
	*
	*
	* @return		0 on success, error code on failure.
	*/
    int set_tv_ldo_offset(decode_tv_ldo_offset_t offset);

    /**
	* @brief		Get LDO Output Channel 0 Target Output Voltage. Bit 7.
	*				CNFG_LDO0_A (0x38)
	*
	* @param[out]	offset 		LDO Output Channel 0 target output voltage offset field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_tv_ldo_offset(decode_tv_ldo_offset_t *offset);

	/**
	* @brief		Set LDO Output Channel 0 Target Output Voltage. Bit 6:0.
	*				CNFG_LDO0_A (0x38)
	*
	* @param[in]	voltV 		LDO Output Channel 0 target output voltage field to be read.
	*							LDOx = 500mV + 25mV x TV_LDOx[6:0]
	*							0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*							0.650V, 0.675V, 0.700V, ... 
	*							3.650, 3.675.
	*
	*							When TV_LDO[7] = 0, TV_LDO[6:0] sets the
	*							LDO's output voltage range from 0.5V to 3.675V.
	*							When TV_LDO[7] = 1, TV_LDO[6:0] sets the
	*							LDO's output voltage from 1.825V to 5V.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_tv_ldo_volt(float voltV);

    /**
	* @brief		Get LDO Output Channel 0 Target Output Voltage. Bit 6:0.
	*				CNFG_LDO0_A (0x38)
	*
	* @param[out]	voltV 		LDO Output Channel 0 target output voltage field to be read.
	*							LDOx = 500mV + 25mV x TV_LDOx[6:0]
	*							0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*							0.650V, 0.675V, 0.700V, ... 
	*							3.650, 3.675.
	*
	*							When TV_LDO[7] = 0, TV_LDO[6:0] sets the
	*							LDO's output voltage range from 0.5V to 3.675V.
	*							When TV_LDO[7] = 1, TV_LDO[6:0] sets the
	*							LDO's output voltage from 1.825V to 5V.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_tv_ldo_volt(float *voltV);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_B (0x39)
	*  - Bit Fields    : [2:0]
	*  - Default       : 0x0
	*  - Description   : Enable Control for LDO0.
	*/
    typedef enum {
        EN_LDO_FPS_SLOT_0,
        EN_LDO_FPS_SLOT_1,
        EN_LDO_FPS_SLOT_2,
        EN_LDO_FPS_SLOT_3,
        EN_LDO_OFF,
        EN_LDO_SAME_AS_0X04,
        EN_LDO_ON,
        EN_LDO_SAME_AS_0X06
    }decode_en_ldo_t;

    /**
	* @brief		Set Enable Control for LDO Channel0.
	*
	* @param[in]	en_ldo 	Enable control for LDO channel x field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_en_ldo(decode_en_ldo_t en_ldo);

    /**
	* @brief		Get Enable Control for LDO Channel x.
	*
	* @param[out]	en_ldo 	Enable control for LDO channel x field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_en_ldo(decode_en_ldo_t *en_ldo);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_B (0x39)
	*  - Bit Fields    : [3]
	*  - Default       : 0x0
	*  - Description   : LDO0 Active-Discharge Enable.
	*/
    typedef enum {
        ADE_LDO_DISABLED,
        ADE_LDO_ENABLED
    }decode_ade_ldo_t;

    /**
	* @brief		Set LDO0 Active-Discharge Enable.
	*
	* @param[in]	ade_ldo 	LDO0 active-discharge enable bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_ade_ldo(decode_ade_ldo_t ade_ldo);

    /**
	* @brief		Get LDO0 Active-Discharge Enable.
	*
	* @param[out]	ade_ldo 	LDO0 active-discharge enable bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_ade_ldo(decode_ade_ldo_t *ade_ldo);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_B (0x39)
	*  - Bit Fields    : [4]
	*  - Default       : 0x0
	*  - Description   : Operation mode of LDO0.
	*/
    typedef enum {
        LDO_MD_LDO_MODE,
        LDO_MD_LSW_MODE
    }decode_ldo_md_t;

    /**
	* @brief		Set Operation mode of LDOx.
	*
	* @param[in]	mode 	Operation mode of LDOx bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_ldo_md(decode_ldo_md_t mode);

    /**
	* @brief		Get Operation mode of LDOx.
	*
	* @param[out]	mode 	Operation mode of LDOx bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_ldo_md(decode_ldo_md_t *mode);

    /**
    * @brief	Disable all interrupts
    *
    * @return	0 on success, error code on failure
    */
     int irq_disable_all();

    /**
    * @brief		Set Interrupt Handler for a Specific Interrupt ID.
    *
    * @param[in]	id Interrupt id, one of INTR_ID_*.
    * @param[in]	func Interrupt handler function.
    * @param[in]	cb Interrupt handler data.
    */
     void set_interrupt_handler(reg_bit_int_glbl_t id, interrupt_handler_function func, void *cb);
};
#endif /*_MAX77659_H_*/
