/*******************************************************************************
* Copyright (C) 2022 Analog Devices, Inc., All rights Reserved.
*
* This software is protected by copyright laws of the United States and
* of foreign countries. This material may also be protected by patent laws
* and technology transfer regulations of the United States and of foreign
* countries. This software is furnished under a license agreement and/or a
* nondisclosure agreement and may only be used or reproduced in accordance
* with the terms of those agreements. Dissemination of this information to
* any party or parties not specified in the license agreement and/or
* nondisclosure agreement is expressly prohibited.
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Analog Devices, Inc. retains all
* ownership rights.
*******************************************************************************
*/

#include <Thread.h>
#include "MAX77659.h"
#include <math.h>

#define POST_INTR_WORK_SIGNAL_ID			0x1
#define TO_UINT8							0xFF
#define TO_UINT16							0xFFFF

MAX77659::MAX77659(I2C *i2c, PinName IRQPin)
{
    if (i2c == NULL)
        return;

    i2c_handler = i2c;

	interrupt_handler_list = new handler[INT_CHG_END] {};

    if (IRQPin != NC) {
        irq_disable_all();
        post_intr_work_thread = new Thread();
        post_intr_work_thread->start(Callback<void()>(this, &MAX77659::post_interrupt_work));

        this->irq_pin = new InterruptIn(IRQPin);
        this->irq_pin->fall(Callback<void()>(this, &MAX77659::interrupt_handler));
        this->irq_pin->enable_irq();
    } else {
        this->irq_pin = NULL;
    }
}

MAX77659::~MAX77659()
{
    if (post_intr_work_thread)
        delete post_intr_work_thread;

    if (irq_pin)
        delete irq_pin;
	
	if (interrupt_handler_list)
        delete [] interrupt_handler_list;
}

int MAX77659::read_register(uint8_t reg, uint8_t *value)
{
    int rtn_val;

    if (value == NULL)
        return MAX77659_VALUE_NULL;

    rtn_val = i2c_handler->write(MAX77659_I2C_ADDRESS, (const char *)&reg, 1, true);
    if (rtn_val != MAX77659_NO_ERROR)
        return MAX77659_WRITE_DATA_FAILED;

    rtn_val = i2c_handler->read(MAX77659_I2C_ADDRESS, (char *) value, 1, false);
    if (rtn_val != MAX77659_NO_ERROR)
        return MAX77659_READ_DATA_FAILED;

    return MAX77659_NO_ERROR;
}

int MAX77659::write_register(uint8_t reg, const uint8_t *value)
{
    int rtn_val;
    unsigned char local_data[2] = {0};

    if (value == NULL)
        return MAX77659_VALUE_NULL;

    local_data[0] = reg;

    memcpy(&local_data[1], value, 1);

    rtn_val = i2c_handler->write(MAX77659_I2C_ADDRESS, (const char *)local_data, sizeof(local_data));
    if (rtn_val != MAX77659_NO_ERROR)
        return MAX77659_WRITE_DATA_FAILED;

    return MAX77659_NO_ERROR;
}

#define SET_BIT_FIELD(address, reg_name, bit_field_name, value)                         	\
            int ret_val;                                                                    \
            ret_val = read_register(address, (uint8_t *)&(reg_name));                       \
            if (ret_val) {                                                                  \
                return ret_val;                                                             \
            }                                                                           	\
            bit_field_name = value;                                                     	\
            ret_val = write_register(address, (uint8_t *)&(reg_name));                      \
            if (ret_val) {                                                                  \
                return ret_val;                                                             \
            }

int MAX77659::get_ercflag(reg_bit_ercflag_t bit_field, uint8_t *flag)
{
	int ret;
	reg_ercflag_t reg_ercflag = {0};

	ret = read_register(ERCFLAG, (uint8_t *)&(reg_ercflag));
	if (ret != MAX77659_NO_ERROR) return ret;
	
	switch (bit_field)
	{
		case ERCFLAG_TOVLD:
			*flag = (uint8_t)reg_ercflag.bits.tovld;
			break;
		case ERCFLAG_SYSOVLO:
			*flag = (uint8_t)reg_ercflag.bits.sysovlo;
			break;
		case ERCFLAG_AVLUVLO:
			*flag = (uint8_t)reg_ercflag.bits.avluvlo;
			break;
		case ERCFLAG_MRST:
			*flag = (uint8_t)reg_ercflag.bits.mrst;
			break;
		case ERCFLAG_SFT_OFF_F:
			*flag = (uint8_t)reg_ercflag.bits.sft_off_f;
			break;
		case ERCFLAG_SFT_CRST_F:
			*flag = (uint8_t)reg_ercflag.bits.sft_crst_f;
			break;
		case ERCFLAG_WDT_OFF:
			*flag = (uint8_t)reg_ercflag.bits.wdt_off;
			break;
		case ERCFLAG_WDT_RST:
			*flag = (uint8_t)reg_ercflag.bits.wdt_rst;
			break;
		default:
			ret = MAX77659_INVALID_DATA;
			break;		
	}

	return ret;
}

int MAX77659::get_stat_glbl(reg_bit_stat_glbl_t bit_field, uint8_t *status)
{
    int ret;
    reg_stat_glbl_t reg_stat_glbl = {0};

    ret = read_register(STAT_GLBL, (uint8_t *)&(reg_stat_glbl));
    if (ret != MAX77659_NO_ERROR) return ret;

	switch (bit_field)
	{
		case STAT_GLBL_STAT_IRQ:
			*status = (uint8_t)reg_stat_glbl.bits.stat_irq;
			break;
		case STAT_GLBL_STAT_EN:
			*status = (uint8_t)reg_stat_glbl.bits.stat_en;
			break;
		case STAT_GLBL_TJAL1_S:
			*status = (uint8_t)reg_stat_glbl.bits.tjal1_s;
			break;
		case STAT_GLBL_TJAL2_S:
			*status = (uint8_t)reg_stat_glbl.bits.tjal2_s;
			break;
		case STAT_GLBL_RSVD:
			*status = (uint8_t)reg_stat_glbl.bits.rsvd;
			break;
		case STAT_GLBL_DOD_S:
			*status = (uint8_t)reg_stat_glbl.bits.dod_s;
			break;
		case STAT_GLBL_BOK:
			*status = (uint8_t)reg_stat_glbl.bits.bok;
			break;
		case STAT_GLBL_DIDM:
			*status = (uint8_t)reg_stat_glbl.bits.didm;
			break;
		default:
			ret = MAX77659_INVALID_DATA;
			break;		
	}

	return ret;
}

int MAX77659::set_interrupt_mask(reg_bit_int_mask_t bit_field, uint8_t maskBit)
{
	int ret;
	uint8_t reg_addr;
	reg_int_m_chg_t reg_int_m_chg = {0};
	reg_intm_glbl0_t reg_intm_glbl0 = {0};
	reg_intm_glbl1_t reg_intm_glbl1 = {0};
 
	//INT_M_CHG (0x07), INTM_GLBL1 (0x08) and INTM_GLBL0 (0x09)
    reg_addr = (uint8_t)floor((static_cast<uint8_t>(bit_field)) / 8) + 0x07; 
	
	if (reg_addr == INT_M_CHG)
		ret = read_register(INT_M_CHG, (uint8_t *)&(reg_int_m_chg));
	else if (reg_addr == INTM_GLBL0)
		ret = read_register(INTM_GLBL0, (uint8_t *)&(reg_intm_glbl0));
	else if (reg_addr == INTM_GLBL1)
		ret = read_register(INTM_GLBL1, (uint8_t *)&(reg_intm_glbl1));
	else 
		return MAX77659_INVALID_DATA;
	
	if (ret != MAX77659_NO_ERROR) return ret;
	
	switch (bit_field)
	{
		case INT_M_CHG_THM_M:
			reg_int_m_chg.bits.thm_m = maskBit;
			break;
		case INT_M_CHG_CHG_M:
			reg_int_m_chg.bits.chg_m = maskBit;
			break;
		case INT_M_CHG_CHGIN_M:
			reg_int_m_chg.bits.chgin_m = maskBit;
			break;
		case INT_M_CHG_TJ_REG_M:
			reg_int_m_chg.bits.tj_reg_m = maskBit;
			break;
		case INT_M_CHG_SYS_CTRL_M:
			reg_int_m_chg.bits.sys_ctrl_m = maskBit;
			break;	
		case INTM_GLBL0_GPI0_FM:
			reg_intm_glbl0.bits.gpi0_fm = maskBit;
			break;
        case INTM_GLBL1_GPI1_FM:
			reg_intm_glbl1.bits.gpi1_fm = maskBit;
			break;
		case INTM_GLBL1_GPI1_RM:
			reg_intm_glbl1.bits.gpi1_rm = maskBit;
			break;
		case INTM_GLBL1_SBB_TO_M:
			reg_intm_glbl1.bits.sbb_to_m = maskBit;
			break;
		case INTM_GLBL1_LDO_M:
			reg_intm_glbl1.bits.ldo_m = maskBit;
			break;
		case INTM_GLBL0_GPI0_RM:
			reg_intm_glbl0.bits.gpi0_rm = maskBit;
			break;
		case INTM_GLBL0_nEN_FM:
			reg_intm_glbl0.bits.nen_fm = maskBit;
			break;
		case INTM_GLBL0_nEN_RM:
			reg_intm_glbl0.bits.nen_rm = maskBit;
			break;
		case INTM_GLBL0_TJAL1_RM:
			reg_intm_glbl0.bits.tjal1_rm = maskBit;
			break;
		case INTM_GLBL0_TJAL2_RM:
			reg_intm_glbl0.bits.tjal2_rm = maskBit;
			break;
		case INTM_GLBL0_DOD_RM:
			reg_intm_glbl0.bits.dod_rm = maskBit;
			break;	
		default:
			return MAX77659_INVALID_DATA;
			break;		
	}
    
	if (reg_addr == INT_M_CHG)
		return write_register(INT_M_CHG, (uint8_t *)&(reg_int_m_chg));
	else if (reg_addr == INTM_GLBL0)
		return write_register(INTM_GLBL0, (uint8_t *)&(reg_intm_glbl0));
	else if (reg_addr == INTM_GLBL1)
		return write_register(INTM_GLBL1, (uint8_t *)&(reg_intm_glbl1));
	else 
		return MAX77659_INVALID_DATA;
}

int MAX77659::get_interrupt_mask(reg_bit_int_mask_t bit_field, uint8_t *maskBit)
{
    int ret;
	uint8_t reg_addr;
	reg_int_m_chg_t reg_int_m_chg = {0};
	reg_intm_glbl0_t reg_intm_glbl0 = {0};
	reg_intm_glbl1_t reg_intm_glbl1 = {0};
 
	//INT_M_CHG (0x07), INTM_GLBL1 (0x08) and INTM_GLBL0 (0x09)
    reg_addr = (uint8_t)floor((static_cast<uint8_t>(bit_field)) / 8) + 0x07; 
	
	if (reg_addr == INT_M_CHG)
		ret = read_register(INT_M_CHG, (uint8_t *)&(reg_int_m_chg));
	else if (reg_addr == INTM_GLBL0)
		ret = read_register(INTM_GLBL0, (uint8_t *)&(reg_intm_glbl0));
	else if (reg_addr == INTM_GLBL1)
		ret = read_register(INTM_GLBL1, (uint8_t *)&(reg_intm_glbl1));
	else 
		return MAX77659_INVALID_DATA;
	
	if (ret != MAX77659_NO_ERROR) return ret;

    switch (bit_field)
	{
		case INT_M_CHG_THM_M:
			*maskBit = (uint8_t)reg_int_m_chg.bits.thm_m;
			break;
		case INT_M_CHG_CHG_M:
			*maskBit = (uint8_t)reg_int_m_chg.bits.chg_m;
			break;
		case INT_M_CHG_CHGIN_M:
			*maskBit = (uint8_t)reg_int_m_chg.bits.chgin_m;
			break;
		case INT_M_CHG_TJ_REG_M:
			*maskBit = (uint8_t)reg_int_m_chg.bits.tj_reg_m;
			break;
		case INT_M_CHG_SYS_CTRL_M:
			*maskBit = (uint8_t)reg_int_m_chg.bits.sys_ctrl_m;
			break;
        case INTM_GLBL1_GPI1_FM:
			*maskBit = (uint8_t)reg_intm_glbl1.bits.gpi1_fm;
			break;
		case INTM_GLBL1_GPI1_RM:
			*maskBit = (uint8_t)reg_intm_glbl1.bits.gpi1_rm;
			break;
		case INTM_GLBL1_SBB_TO_M:
			*maskBit = (uint8_t)reg_intm_glbl1.bits.sbb_to_m;
			break;
		case INTM_GLBL1_LDO_M:
			*maskBit = (uint8_t)reg_intm_glbl1.bits.ldo_m;
			break;
		case INTM_GLBL0_GPI0_FM:
			*maskBit = (uint8_t)reg_intm_glbl0.bits.gpi0_fm;
			break;
		case INTM_GLBL0_GPI0_RM:
			*maskBit = (uint8_t)reg_intm_glbl0.bits.gpi0_rm;
			break;
		case INTM_GLBL0_nEN_FM:
			*maskBit = (uint8_t)reg_intm_glbl0.bits.nen_fm;
			break;
		case INTM_GLBL0_nEN_RM:
			*maskBit = (uint8_t)reg_intm_glbl0.bits.nen_rm;
			break;
		case INTM_GLBL0_TJAL1_RM:
			*maskBit = (uint8_t)reg_intm_glbl0.bits.tjal1_rm;
			break;
		case INTM_GLBL0_TJAL2_RM:
			*maskBit = (uint8_t)reg_intm_glbl0.bits.tjal2_rm;
			break;
		case INTM_GLBL0_DOD_RM:
			*maskBit = (uint8_t)reg_intm_glbl0.bits.dod_rm;
			break;	
		default:
			return MAX77659_INVALID_DATA;
			break;		
	}

    return MAX77659_NO_ERROR;
}

int MAX77659::set_cnfg_glbl(reg_bit_cnfg_glbl_t bit_field, uint8_t config)
{	
	int ret;
	reg_cnfg_glbl_t reg_cnfg_glbl = {0};
	
	ret = read_register(CNFG_GLBL, (uint8_t *)&(reg_cnfg_glbl));
    if (ret != MAX77659_NO_ERROR) return ret;
	
	switch (bit_field)
	{
		case CNFG_GLBL_SFT_CTRL:
			reg_cnfg_glbl.bits.sft_ctrl = config;
			break;
		case CNFG_GLBL_DBEN_nEN:
			reg_cnfg_glbl.bits.dben_nen = config;
			break;
		case CNFG_GLBL_nEN_MODE:
			reg_cnfg_glbl.bits.nen_mode = config;
			break;
		case CNFG_GLBL_SBIA_EN:
			reg_cnfg_glbl.bits.sbia_en = config;
			break;
		case CNFG_GLBL_SBIA_LPM:
			reg_cnfg_glbl.bits.sbia_lpm = config;
			break;
		case CNFG_GLBL_T_MRST:
			reg_cnfg_glbl.bits.t_mrst = config;
			break;
		case CNFG_GLBL_PU_DIS:
			reg_cnfg_glbl.bits.pu_dis = config;
			break;
		default:
			return MAX77659_INVALID_DATA;
			break;		
	}
                                  	
	return write_register(CNFG_GLBL, (uint8_t *)&(reg_cnfg_glbl));
}

int MAX77659::get_cnfg_glbl(reg_bit_cnfg_glbl_t bit_field, uint8_t *config)
{
    int ret;
    reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    ret = read_register(CNFG_GLBL, (uint8_t *)&(reg_cnfg_glbl));
    if (ret != MAX77659_NO_ERROR) return ret;

    switch (bit_field)
	{
		case CNFG_GLBL_SFT_CTRL:
			*config = (uint8_t)reg_cnfg_glbl.bits.sft_ctrl;
			break;
		case CNFG_GLBL_DBEN_nEN:
			*config = (uint8_t)reg_cnfg_glbl.bits.dben_nen;
			break;
		case CNFG_GLBL_nEN_MODE:
			*config = (uint8_t)reg_cnfg_glbl.bits.nen_mode;
			break;
		case CNFG_GLBL_SBIA_EN:
			*config = (uint8_t)reg_cnfg_glbl.bits.sbia_en;
			break;
		case CNFG_GLBL_SBIA_LPM:
			*config = (uint8_t)reg_cnfg_glbl.bits.sbia_lpm;
			break;
		case CNFG_GLBL_T_MRST:
			*config = (uint8_t)reg_cnfg_glbl.bits.t_mrst;
			break;
		case CNFG_GLBL_PU_DIS:
			*config = (uint8_t)reg_cnfg_glbl.bits.pu_dis;
			break;
		default:
			ret = MAX77659_INVALID_DATA;
			break;		
	}

    return ret;
}

int MAX77659::set_cnfg_gpio(reg_bit_cnfg_gpio_t bit_field, uint8_t channel, uint8_t config)
{
	int ret;
	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
	
	if (channel == 0)
	{
		ret = read_register(CNFG_GPIO0, (uint8_t *)&(reg_cnfg_gpio0));
		if (ret != MAX77659_NO_ERROR) return ret;
		
		switch (bit_field)
		{
			case CNFG_GPIO_DIR:
				reg_cnfg_gpio0.bits.gpo_dir = config;
				break;
			case CNFG_GPIO_DI:
				reg_cnfg_gpio0.bits.gpo_di = config;
				break;
			case CNFG_GPIO_DRV:
				reg_cnfg_gpio0.bits.gpo_drv = config;
				break;
			case CNFG_GPIO_DO:
				reg_cnfg_gpio0.bits.gpo_do = config;
				break;
			case CNFG_GPIO_DBEN_GPI:
				reg_cnfg_gpio0.bits.dben_gpi = config;
				break;
			case CNFG_GPIO_ALT_GPIO:
				reg_cnfg_gpio0.bits.alt_gpio = config;
				break;
			default:
				return MAX77659_INVALID_DATA;
				break;		
		}
		
		return write_register(CNFG_GPIO0, (uint8_t *)&(reg_cnfg_gpio0));
	}
	else if (channel == 1)
	{
		ret = read_register(CNFG_GPIO1, (uint8_t *)&(reg_cnfg_gpio1));
		if (ret != MAX77659_NO_ERROR) return ret;
		
		switch (bit_field)
		{
			case CNFG_GPIO_DIR:
				reg_cnfg_gpio1.bits.gpo_dir = config;
				break;
			case CNFG_GPIO_DI:
				reg_cnfg_gpio1.bits.gpo_di = config;
				break;
			case CNFG_GPIO_DRV:
				reg_cnfg_gpio1.bits.gpo_drv = config;
				break;
			case CNFG_GPIO_DO:
				reg_cnfg_gpio1.bits.gpo_do = config;
				break;
			case CNFG_GPIO_DBEN_GPI:
				reg_cnfg_gpio1.bits.dben_gpi = config;
				break;
			case CNFG_GPIO_ALT_GPIO:
				reg_cnfg_gpio1.bits.alt_gpio = config;
				break;
			default:
				return MAX77659_INVALID_DATA;
				break;		
		}
		
		return write_register(CNFG_GPIO1, (uint8_t *)&(reg_cnfg_gpio1));
	}
	else {
		return MAX77659_INVALID_DATA;
	}
}

int MAX77659::get_cnfg_gpio(reg_bit_cnfg_gpio_t bit_field, uint8_t channel, uint8_t *config)
{
    int ret;
	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
	
	if (channel == 0)
	{
		ret = read_register(CNFG_GPIO0, (uint8_t *)&(reg_cnfg_gpio0));
		if (ret != MAX77659_NO_ERROR) return ret;
	
		switch (bit_field)
		{
			case CNFG_GPIO_DIR:
				*config = (uint8_t)reg_cnfg_gpio0.bits.gpo_dir;
				break;
			case CNFG_GPIO_DI:
				*config = (uint8_t)reg_cnfg_gpio0.bits.gpo_di;
				break;
			case CNFG_GPIO_DRV:
				*config = (uint8_t)reg_cnfg_gpio0.bits.gpo_drv;
				break;
			case CNFG_GPIO_DO:
				*config = (uint8_t)reg_cnfg_gpio0.bits.gpo_do;
				break;
			case CNFG_GPIO_DBEN_GPI:
				*config = (uint8_t)reg_cnfg_gpio0.bits.dben_gpi;
				break;
			case CNFG_GPIO_ALT_GPIO:
				*config = (uint8_t)reg_cnfg_gpio0.bits.alt_gpio;
				break;
			default:
				return MAX77659_INVALID_DATA;
				break;		
		}
	}
	else if (channel == 1)
	{
		ret = read_register(CNFG_GPIO1, (uint8_t *)&(reg_cnfg_gpio1));
		if (ret != MAX77659_NO_ERROR) return ret;
		
		switch (bit_field)
		{
			case CNFG_GPIO_DIR:
				*config = (uint8_t)reg_cnfg_gpio1.bits.gpo_dir;
				break;
			case CNFG_GPIO_DI:
				*config = (uint8_t)reg_cnfg_gpio1.bits.gpo_di;
				break;
			case CNFG_GPIO_DRV:
				*config = (uint8_t)reg_cnfg_gpio1.bits.gpo_drv;
				break;
			case CNFG_GPIO_DO:
				*config = (uint8_t)reg_cnfg_gpio1.bits.gpo_do;
				break;
			case CNFG_GPIO_DBEN_GPI:
				*config = (uint8_t)reg_cnfg_gpio1.bits.dben_gpi;
				break;
			case CNFG_GPIO_ALT_GPIO:
				*config = (uint8_t)reg_cnfg_gpio1.bits.alt_gpio;
				break;
			default:
				return MAX77659_INVALID_DATA;
				break;		
		}
	}
	else {
		return MAX77659_INVALID_DATA;
	}

    return ret;
}

int MAX77659::get_cid(void) {
    char rbuf[1] = {0};
    int ret;

    ret = read_register(CID, (uint8_t *)&(rbuf));
	if (ret != MAX77659_NO_ERROR) return ret;
 
    return *rbuf;
}

int MAX77659::set_cnfg_wdt(reg_bit_cnfg_wdt_t bit_field, uint8_t config)
{
	int ret;
	reg_cnfg_wdt_t reg_cnfg_wdt = {0};
	
	ret = read_register(CNFG_WDT, (uint8_t *)&(reg_cnfg_wdt));
	if (ret != MAX77659_NO_ERROR) return ret;
	
	switch (bit_field)
	{
		case CNFG_WDT_WDT_LOCK:
			reg_cnfg_wdt.bits.wdt_lock = config;
			break;
		case CNFG_WDT_WDT_EN:
			reg_cnfg_wdt.bits.wdt_en = config;
			break;
		case CNFG_WDT_WDT_CLR:
			reg_cnfg_wdt.bits.wdt_clr = config;
			break;
		case CNFG_WDT_WDT_MODE:
			reg_cnfg_wdt.bits.wdt_mode = config;
			break;
		case CNFG_WDT_WDT_PER:
			reg_cnfg_wdt.bits.wdt_per = config;
			break;
		default:
			return MAX77659_INVALID_DATA;
			break;		
	}

	return write_register(CNFG_WDT, (uint8_t *)&(reg_cnfg_wdt));
}

int MAX77659::get_cnfg_wdt(reg_bit_cnfg_wdt_t bit_field, uint8_t *config)
{
    int ret;
    reg_cnfg_wdt_t reg_cnfg_wdt = {0};

    ret = read_register(CNFG_WDT, (uint8_t *)&(reg_cnfg_wdt));
    if (ret != MAX77659_NO_ERROR) return ret;

    switch (bit_field)
	{
		case CNFG_WDT_WDT_LOCK:
			*config = (uint8_t)reg_cnfg_wdt.bits.wdt_lock;
			break;
		case CNFG_WDT_WDT_EN:
			*config = (uint8_t)reg_cnfg_wdt.bits.wdt_en;
			break;
		case CNFG_WDT_WDT_CLR:
			*config = (uint8_t)reg_cnfg_wdt.bits.wdt_clr;
			break;
		case CNFG_WDT_WDT_MODE:
			*config = (uint8_t)reg_cnfg_wdt.bits.wdt_mode;
			break;
		case CNFG_WDT_WDT_PER:
			*config = (uint8_t)reg_cnfg_wdt.bits.wdt_per;
			break;
		default:
			return MAX77659_INVALID_DATA;
			break;		
	}

    return MAX77659_NO_ERROR;
}

int MAX77659::get_stat_chg_a(reg_bit_stat_chg_a_t bit_field, uint8_t *status)
{
    int ret;
    reg_stat_chg_a_t reg_stat_chg_a = {0};

    ret = read_register(STAT_CHG_A, (uint8_t *)&(reg_stat_chg_a));
    if (ret != MAX77659_NO_ERROR) return ret;

    switch (bit_field)
	{
		case STAT_CHG_A_THM_DTLS:
			*status = (uint8_t)reg_stat_chg_a.bits.thm_dtls;
			break;
		case STAT_CHG_A_TJ_REG_STAT:
			*status = (uint8_t)reg_stat_chg_a.bits.tj_reg_stat;
			break;
		case STAT_CHG_A_VSYS_MIN_STAT:
			*status = (uint8_t)reg_stat_chg_a.bits.vsys_min_stat;
			break;
		default:
			return MAX77659_INVALID_DATA;
			break;		
	}

    return MAX77659_NO_ERROR;
}

int MAX77659::get_thm_dtls(decode_thm_dtls_t *thm_dtls)
{
    int ret;
    reg_stat_chg_a_t reg_stat_chg_a = {0};

    ret = read_register(STAT_CHG_A, (uint8_t *)&(reg_stat_chg_a));
    if (ret != MAX77659_NO_ERROR) return ret;

    *thm_dtls = (decode_thm_dtls_t)reg_stat_chg_a.bits.thm_dtls;

    return MAX77659_NO_ERROR;
}

int MAX77659::get_stat_chg_b(reg_bit_stat_chg_b_t bit_field, uint8_t *status)
{
    int ret;
    reg_stat_chg_b_t reg_stat_chg_b = {0};

    ret = read_register(STAT_CHG_B, (uint8_t *)&(reg_stat_chg_b));
    if (ret != MAX77659_NO_ERROR) return ret;

    switch (bit_field)
	{
		case STAT_CHG_B_TIME_SUS:
			*status = (uint8_t)reg_stat_chg_b.bits.time_sus;
			break;
		case STAT_CHG_B_CHG:
			*status = (uint8_t)reg_stat_chg_b.bits.chg;
			break;
		case STAT_CHG_B_CHGIN_DTLS:
			*status = (uint8_t)reg_stat_chg_b.bits.chgin_dtls;
			break;
		case STAT_CHG_B_CHG_DTLS:
			*status = (uint8_t)reg_stat_chg_b.bits.chg_dtls;
			break;
		default:
			return MAX77659_INVALID_DATA;
			break;		
	}

    return MAX77659_NO_ERROR;
}

int MAX77659::get_chg_dtls(decode_chg_dtls_t *chg_dtls)
{
    int ret;
    reg_stat_chg_b_t reg_stat_chg_b = {0};

    ret = read_register(STAT_CHG_B, (uint8_t *)&(reg_stat_chg_b));
    if (ret != MAX77659_NO_ERROR) return ret;

    *chg_dtls = (decode_chg_dtls_t)reg_stat_chg_b.bits.chg_dtls;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_thm_hot(decode_thm_hot_t thm_hot)
{
	reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    SET_BIT_FIELD(CNFG_CHG_A, reg_cnfg_chg_a, reg_cnfg_chg_a.bits.thm_hot, thm_hot);

    return MAX77659_NO_ERROR;
}

int MAX77659::get_thm_hot(decode_thm_hot_t *thm_hot)
{
    int ret;
    reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    ret = read_register(CNFG_CHG_A, (uint8_t *)&(reg_cnfg_chg_a));
    if (ret != MAX77659_NO_ERROR) return ret;

    *thm_hot = (decode_thm_hot_t)reg_cnfg_chg_a.bits.thm_hot;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_thm_warm(decode_thm_warm_t thm_warm)
{
	reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    SET_BIT_FIELD(CNFG_CHG_A, reg_cnfg_chg_a, reg_cnfg_chg_a.bits.thm_warm, thm_warm);

    return MAX77659_NO_ERROR;
}

int MAX77659::get_thm_warm(decode_thm_warm_t *thm_warm)
{
    int ret;
    reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    ret = read_register(CNFG_CHG_A, (uint8_t *)&(reg_cnfg_chg_a));
    if (ret != MAX77659_NO_ERROR) return ret;

    *thm_warm = (decode_thm_warm_t)reg_cnfg_chg_a.bits.thm_warm;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_thm_cool(decode_thm_cool_t thm_cool)
{
	reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    SET_BIT_FIELD(CNFG_CHG_A, reg_cnfg_chg_a, reg_cnfg_chg_a.bits.thm_cool, thm_cool);

    return MAX77659_NO_ERROR;
}

int MAX77659::get_thm_cool(decode_thm_cool_t *thm_cool)
{
    int ret;
    reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    ret = read_register(CNFG_CHG_A, (uint8_t *)&(reg_cnfg_chg_a));
    if (ret != MAX77659_NO_ERROR) return ret;

    *thm_cool = (decode_thm_cool_t)reg_cnfg_chg_a.bits.thm_cool;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_thm_cold(decode_thm_cold_t thm_cold)
{
	reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    SET_BIT_FIELD(CNFG_CHG_A, reg_cnfg_chg_a, reg_cnfg_chg_a.bits.thm_cold, thm_cold);

    return MAX77659_NO_ERROR;
}

int MAX77659:: get_thm_cold(decode_thm_cold_t *thm_cold)
{
    int ret;
    reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    ret = read_register(CNFG_CHG_A, (uint8_t *)&(reg_cnfg_chg_a));
    if (ret != MAX77659_NO_ERROR) return ret;

    *thm_cold = (decode_thm_cold_t)reg_cnfg_chg_a.bits.thm_cold;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_cnfg_chg_b(reg_bit_cnfg_chg_b_t bit_field, uint8_t config)
{
	int ret;
	reg_cnfg_chg_b_t reg_cnfg_chg_b = {0};
	
	ret = read_register(CNFG_CHG_B, (uint8_t *)&(reg_cnfg_chg_b));
	if (ret != MAX77659_NO_ERROR) return ret;
	
	switch (bit_field)
	{
		case CNFG_CHG_B_CHG_EN:
			reg_cnfg_chg_b.bits.chg_en = config;
			break;
		case CNFG_CHG_B_I_PQ:
			reg_cnfg_chg_b.bits.i_pq = config;
			break;
		case CNFG_CHG_B_RSVD:
			reg_cnfg_chg_b.bits.rsvd = config;
			break;
		default:
			return MAX77659_INVALID_DATA;
			break;		
	}

	return write_register(CNFG_CHG_B, (uint8_t *)&(reg_cnfg_chg_b));
}

int MAX77659::get_cnfg_chg_b(reg_bit_cnfg_chg_b_t bit_field, uint8_t *config)
{
    int ret;
    reg_cnfg_chg_b_t reg_cnfg_chg_b = {0};

    ret = read_register(CNFG_CHG_B, (uint8_t *)&(reg_cnfg_chg_b));
    if (ret != MAX77659_NO_ERROR) return ret;

   switch (bit_field)
	{
		case CNFG_CHG_B_CHG_EN:
			*config = (uint8_t)reg_cnfg_chg_b.bits.chg_en;
			break;
		case CNFG_CHG_B_I_PQ:
			*config = (uint8_t)reg_cnfg_chg_b.bits.i_pq;
			break;
		case CNFG_CHG_B_RSVD:
			*config = (uint8_t)reg_cnfg_chg_b.bits.rsvd;
			break;
		default:
			return MAX77659_INVALID_DATA;
			break;		
	}

    return MAX77659_NO_ERROR;
}

int MAX77659::set_chg_pq(float voltV)
{
	uint8_t value;
	reg_cnfg_chg_c_t reg_cnfg_chg_c = {0};
	float voltmV = voltV * 1000;
		
	if (voltmV < 2300) voltmV = 2300;
	else if (voltmV > 3000) voltmV = 3000;

	value = (voltmV - 2300) / 100;

    SET_BIT_FIELD(CNFG_CHG_C, reg_cnfg_chg_c, reg_cnfg_chg_c.bits.chg_pq, value);
	return MAX77659_NO_ERROR;
}

int MAX77659::get_chg_pq(float *voltV)
{
    int ret;
	uint8_t bit_value;
    reg_cnfg_chg_c_t reg_cnfg_chg_c = {0};

    ret = read_register(CNFG_CHG_C, (uint8_t *)&(reg_cnfg_chg_c));
    if (ret != MAX77659_NO_ERROR) return ret;
	
	bit_value = (uint8_t)reg_cnfg_chg_c.bits.chg_pq;	
	*voltV = (bit_value * 0.1f) + 2.3f;
    return MAX77659_NO_ERROR;
}

int MAX77659::set_i_term(float percent)
{
	uint8_t value;
	reg_cnfg_chg_c_t reg_cnfg_chg_c = {0};

	if (percent < 7.5f) value = 0;
	else if ((percent >= 7.5f) && (percent < 10)) value = 1;
	else if ((percent >= 10) && (percent < 15)) value = 2;
	else if (percent >= 15) value = 3;

    SET_BIT_FIELD(CNFG_CHG_C, reg_cnfg_chg_c, reg_cnfg_chg_c.bits.i_term, value);
	return MAX77659_NO_ERROR;
}

int MAX77659::get_i_term(float *percent)
{
    int ret;
	uint8_t bit_value;
    reg_cnfg_chg_c_t reg_cnfg_chg_c = {0};

    ret = read_register(CNFG_CHG_C, (uint8_t *)&(reg_cnfg_chg_c));
    if (ret != MAX77659_NO_ERROR) return ret;
	
	bit_value = (uint8_t)reg_cnfg_chg_c.bits.i_term;
	
	if (bit_value == 0) *percent = 5.0f;
	else if (bit_value == 1) *percent = 7.5f;
	else if (bit_value == 2) *percent = 10.0f;
	else if (bit_value == 3) *percent = 15.0f;
	else return MAX77659_INVALID_DATA;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_t_topoff(uint8_t minute)
{
	uint8_t value;
	reg_cnfg_chg_c_t reg_cnfg_chg_c = {0};

	if (minute > 35) minute = 35;
	
	value = (uint8_t)(minute / 5);

    SET_BIT_FIELD(CNFG_CHG_C, reg_cnfg_chg_c, reg_cnfg_chg_c.bits.t_topoff, value);
	return MAX77659_NO_ERROR;
}

int MAX77659::get_t_topoff(uint8_t *minute)
{
    int ret;
	uint8_t bit_value;
    reg_cnfg_chg_c_t reg_cnfg_chg_c = {0};

    ret = read_register(CNFG_CHG_C, (uint8_t *)&(reg_cnfg_chg_c));
    if (ret != MAX77659_NO_ERROR) return ret;
	
	bit_value = (uint8_t)reg_cnfg_chg_c.bits.t_topoff;	
	*minute = (bit_value * 5);
    return MAX77659_NO_ERROR;
}

int MAX77659::set_tj_reg(uint8_t tempDegC)
{
	uint8_t value;
	reg_cnfg_chg_d_t reg_cnfg_chg_d = {0};

	if (tempDegC < 60) tempDegC = 60;
	else if (tempDegC > 100) tempDegC = 100;

	value = (tempDegC - 60) / 10;

    SET_BIT_FIELD(CNFG_CHG_D, reg_cnfg_chg_d, reg_cnfg_chg_d.bits.tj_reg, value);
	return MAX77659_NO_ERROR;
}

int MAX77659::get_tj_reg(uint8_t *tempDegC)
{
    int ret;
	uint8_t bit_value;
    reg_cnfg_chg_d_t reg_cnfg_chg_d = {0};

    ret = read_register(CNFG_CHG_D, (uint8_t *)&(reg_cnfg_chg_d));
    if (ret != MAX77659_NO_ERROR) return ret;
	
	bit_value = (uint8_t)reg_cnfg_chg_d.bits.tj_reg;
	*tempDegC = (bit_value * 10) + 60;
    return MAX77659_NO_ERROR;
}

int MAX77659::set_vsys_hdrm(decode_vsys_hdrm_t vsys_hdrm)
{
	reg_cnfg_chg_d_t reg_cnfg_chg_d = {0};

    SET_BIT_FIELD(CNFG_CHG_D, reg_cnfg_chg_d, reg_cnfg_chg_d.bits.vsys_hdrm, vsys_hdrm);

    return MAX77659_NO_ERROR;
}

int MAX77659::get_vsys_hdrm(decode_vsys_hdrm_t *vsys_hdrm)
{
    int ret;
    reg_cnfg_chg_d_t reg_cnfg_chg_d = {0};

    ret = read_register(CNFG_CHG_D, (uint8_t *)&(reg_cnfg_chg_d));
    if (ret != MAX77659_NO_ERROR) return ret;

    *vsys_hdrm = (decode_vsys_hdrm_t)reg_cnfg_chg_d.bits.vsys_hdrm;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_vsys_min(decode_vsys_min_t vsys_min)
{
	reg_cnfg_chg_d_t reg_cnfg_chg_d = {0};

    SET_BIT_FIELD(CNFG_CHG_D, reg_cnfg_chg_d, reg_cnfg_chg_d.bits.vsys_min, vsys_min);

    return MAX77659_NO_ERROR;
}

int MAX77659::get_vsys_min(decode_vsys_min_t *vsys_min)
{
    int ret;
    reg_cnfg_chg_d_t reg_cnfg_chg_d = {0};

    ret = read_register(CNFG_CHG_D, (uint8_t *)&(reg_cnfg_chg_d));
    if (ret != MAX77659_NO_ERROR) return ret;

    *vsys_min = (decode_vsys_min_t)reg_cnfg_chg_d.bits.vsys_min;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_chg_cc(float currentmA)
{
	uint8_t value;
	reg_cnfg_chg_e_t reg_cnfg_chg_e = {0};
	float currentuA = currentmA * 1000;

	if (currentuA < 7500) currentuA = 7500;
	if (currentuA > 300000) currentuA = 300000;

	value = (currentuA - 7500) / 7500;

    SET_BIT_FIELD(CNFG_CHG_E, reg_cnfg_chg_e, reg_cnfg_chg_e.bits.chg_cc, value);
	return MAX77659_NO_ERROR;
}

int MAX77659::get_chg_cc(float *currentmA)
{
    int ret;
	uint8_t bit_value;
    reg_cnfg_chg_e_t reg_cnfg_chg_e = {0};

    ret = read_register(CNFG_CHG_E, (uint8_t *)&(reg_cnfg_chg_e));
    if (ret != MAX77659_NO_ERROR) return ret;
	
	bit_value = (uint8_t)reg_cnfg_chg_e.bits.chg_cc;
	if (bit_value >= 39) bit_value = 39; //0x27 to 0x3F = 300.0mA
	
	*currentmA = (bit_value * 7.5f) + 7.5f;
	
    return MAX77659_NO_ERROR;
}

int MAX77659::set_t_fast_chg(decode_t_fast_chg_t t_fast_chg)
{	
	reg_cnfg_chg_e_t reg_cnfg_chg_e = {0};

    SET_BIT_FIELD(CNFG_CHG_E, reg_cnfg_chg_e, reg_cnfg_chg_e.bits.t_fast_chg, t_fast_chg);
	return MAX77659_NO_ERROR;
}

int MAX77659::get_t_fast_chg(decode_t_fast_chg_t *t_fast_chg)
{
    int ret;
    reg_cnfg_chg_e_t reg_cnfg_chg_e = {0};

    ret = read_register(CNFG_CHG_E, (uint8_t *)&(reg_cnfg_chg_e));
    if (ret != MAX77659_NO_ERROR) return ret;

    *t_fast_chg = (decode_t_fast_chg_t)reg_cnfg_chg_e.bits.t_fast_chg;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_chg_cc_jeita(float currentmA)
{
	uint8_t value;
	reg_cnfg_chg_f_t reg_cnfg_chg_f = {0};
	float currentuA = currentmA * 1000;

	if (currentuA < 7500) currentuA = 7500;
	else if (currentuA > 300000) currentuA = 300000;

	value = round(currentuA - 7500) / 7500;

    SET_BIT_FIELD(CNFG_CHG_F, reg_cnfg_chg_f, reg_cnfg_chg_f.bits.chg_cc_jeita, value);
	return MAX77659_NO_ERROR;
}

int MAX77659::get_chg_cc_jeita(float *currentmA)
{
    int ret;
	uint8_t bit_value;
    reg_cnfg_chg_f_t reg_cnfg_chg_f = {0};

    ret = read_register(CNFG_CHG_F, (uint8_t *)&(reg_cnfg_chg_f));
    if (ret != MAX77659_NO_ERROR) return ret;
	
	bit_value = (uint8_t)reg_cnfg_chg_f.bits.chg_cc_jeita;
	if (bit_value >= 39) bit_value = 39; //0x27 to 0x3F = 300.0mA
	
	*currentmA = (bit_value * 7.5f) + 7.5f;
    return MAX77659_NO_ERROR;
}



int MAX77659::set_thm_en(decode_thm_en_t thm_en)
{
	reg_cnfg_chg_f_t reg_cnfg_chg_f = {0};

    SET_BIT_FIELD(CNFG_CHG_F, reg_cnfg_chg_f, reg_cnfg_chg_f.bits.thm_en, thm_en);

    return MAX77659_NO_ERROR;
}

int MAX77659::get_thm_en(decode_thm_en_t *thm_en)
{
    int ret;
    reg_cnfg_chg_f_t reg_cnfg_chg_f = {0};

    ret = read_register(CNFG_CHG_F, (uint8_t *)&(reg_cnfg_chg_f));
    if (ret != MAX77659_NO_ERROR) return ret;

    *thm_en = (decode_thm_en_t)reg_cnfg_chg_f.bits.thm_en;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_chg_cv(float voltV)
{
	uint8_t value;
	reg_cnfg_chg_g_t reg_cnfg_chg_g = {0};
	float voltmV = voltV * 1000;

	if (voltmV < 3600) voltmV = 3600;
	else if (voltmV > 4600) voltmV = 4600;

	value = (voltmV - 3600) / 25;

    SET_BIT_FIELD(CNFG_CHG_G, reg_cnfg_chg_g, reg_cnfg_chg_g.bits.chg_cv, value);
	return MAX77659_NO_ERROR;
}

int MAX77659::get_chg_cv(float *voltV)
{
    int ret;
	uint8_t bit_value;
    reg_cnfg_chg_g_t reg_cnfg_chg_g = {0};

    ret = read_register(CNFG_CHG_G, (uint8_t *)&(reg_cnfg_chg_g));
    if (ret != MAX77659_NO_ERROR) return ret;
	
	bit_value = (uint8_t)reg_cnfg_chg_g.bits.chg_cv;
	*voltV = (bit_value * 0.025f) + 3.6f;
    return MAX77659_NO_ERROR;
}

int MAX77659::set_usbs(decode_usbs_t usbs)
{
	reg_cnfg_chg_g_t reg_cnfg_chg_g = {0};

    SET_BIT_FIELD(CNFG_CHG_G, reg_cnfg_chg_g, reg_cnfg_chg_g.bits.usbs, usbs);

    return MAX77659_NO_ERROR;
}

int MAX77659::get_usbs(decode_usbs_t *usbs)
{
    int ret;
    reg_cnfg_chg_g_t reg_cnfg_chg_g = {0};

    ret = read_register(CNFG_CHG_G, (uint8_t *)&(reg_cnfg_chg_g));
    if (ret != MAX77659_NO_ERROR) return ret;

    *usbs = (decode_usbs_t)reg_cnfg_chg_g.bits.usbs;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_chg_cv_jeita(float voltV)
{
	uint8_t value;
	reg_cnfg_chg_h_t reg_cnfg_chg_h = {0};
	float voltmV = voltV * 1000;

	if (voltmV < 3600) voltmV = 3600;
	else if (voltmV > 4600) voltmV = 4600;

	value = round(voltmV - 3600) / 25;

    SET_BIT_FIELD(CNFG_CHG_H, reg_cnfg_chg_h, reg_cnfg_chg_h.bits.chg_cv_jeita, value);
	return MAX77659_NO_ERROR;
}

int MAX77659::get_chg_cv_jeita(float *voltV)
{
    int ret;
	uint8_t bit_value;
    reg_cnfg_chg_h_t reg_cnfg_chg_h = {0};

    ret = read_register(CNFG_CHG_H, (uint8_t *)&(reg_cnfg_chg_h));
    if (ret != MAX77659_NO_ERROR) return ret;
	
	bit_value = (uint8_t)reg_cnfg_chg_h.bits.chg_cv_jeita;
	*voltV = (bit_value * 0.025f) + 3.6f;
    return MAX77659_NO_ERROR;
}

int MAX77659::set_imon_dischg_scale(float currentmA)
{
	uint8_t value;
	reg_cnfg_chg_i_t reg_cnfg_chg_i = {0};
	
	if (currentmA < 40.5f) value = 0;
	else if ((currentmA >= 40.5f) && (currentmA < 72.3f)) value = 1;
	else if ((currentmA >= 72.3f) && (currentmA < 103.4f)) value = 2;
	else if ((currentmA >= 103.4f) && (currentmA < 134.1f)) value = 3;
	else if ((currentmA >= 134.1f) && (currentmA < 164.1f)) value = 4;
	else if ((currentmA >= 164.1f) && (currentmA < 193.7f)) value = 5;
	else if ((currentmA >= 193.7f) && (currentmA < 222.7f)) value = 6;
	else if ((currentmA >= 222.7f) && (currentmA < 251.2f)) value = 7;
	else if ((currentmA >= 251.2f) && (currentmA < 279.3f)) value = 8;
	else if ((currentmA >= 279.3f) && (currentmA < 300.0f)) value = 9;
	else if (currentmA >= 300.0f) value = 10;

    SET_BIT_FIELD(CNFG_CHG_I, reg_cnfg_chg_i, reg_cnfg_chg_i.bits.imon_dischg_scale, value);
	return MAX77659_NO_ERROR;
}

int MAX77659::get_imon_dischg_scale(float *currentmA)
{
    int ret;
	uint8_t bit_value;
    reg_cnfg_chg_i_t reg_cnfg_chg_i = {0};

    ret = read_register(CNFG_CHG_I, (uint8_t *)&(reg_cnfg_chg_i));
    if (ret != MAX77659_NO_ERROR) return ret;
	
	bit_value = (uint8_t)reg_cnfg_chg_i.bits.imon_dischg_scale;
	
	if (bit_value == 0) *currentmA = 8.2f;
	else if (bit_value == 1) *currentmA = 40.5f;
	else if (bit_value == 2) *currentmA = 72.3f;
	else if (bit_value == 3) *currentmA = 103.4f;
	else if (bit_value == 4) *currentmA = 134.1f;
	else if (bit_value == 5) *currentmA = 164.1f;
	else if (bit_value == 6) *currentmA = 193.7f;
	else if (bit_value == 7) *currentmA = 222.7f;
	else if (bit_value == 8) *currentmA = 251.2f;
	else if (bit_value == 9) *currentmA = 279.3f;
	else *currentmA = 300.0f; //0xA to 0xF = 300.0mA
	
    return MAX77659_NO_ERROR;
}

int MAX77659::set_mux_sel(decode_mux_sel_t selection)
{	
	reg_cnfg_chg_i_t reg_cnfg_chg_i = {0};

    SET_BIT_FIELD(CNFG_CHG_I, reg_cnfg_chg_i, reg_cnfg_chg_i.bits.mux_sel, selection);
	return MAX77659_NO_ERROR;
}

int MAX77659::get_mux_sel(decode_mux_sel_t *selection)
{
    int ret;
    reg_cnfg_chg_i_t reg_cnfg_chg_i = {0};

    ret = read_register(CNFG_CHG_I, (uint8_t *)&(reg_cnfg_chg_i));
    if (ret != MAX77659_NO_ERROR) return ret;

    *selection = (decode_mux_sel_t)reg_cnfg_chg_i.bits.mux_sel;
    return MAX77659_NO_ERROR;
}

int MAX77659::set_tv_sbb(uint8_t channel, float voltV)
{
	uint8_t value;
	reg_cnfg_sbb0_a_t reg_cnfg_sbb0_a = {0};
	reg_cnfg_sbb1_a_t reg_cnfg_sbb1_a = {0};
	reg_cnfg_sbb2_a_t reg_cnfg_sbb2_a = {0};
	float voltmV = voltV * 1000;
	
	if (voltmV < 500) voltmV = 500;
	else if (voltmV > 5500) voltmV = 5500;

	value = (voltmV - 500) / 25;

	if (channel == 0) {
		SET_BIT_FIELD(CNFG_SBB0_A, reg_cnfg_sbb0_a, reg_cnfg_sbb0_a.bits.tv_sbb0, value);
	}		
	else if (channel == 1) {
		SET_BIT_FIELD(CNFG_SBB1_A, reg_cnfg_sbb1_a, reg_cnfg_sbb1_a.bits.tv_sbb1, value);
	}
	else if (channel == 2) {
		SET_BIT_FIELD(CNFG_SBB2_A, reg_cnfg_sbb2_a, reg_cnfg_sbb2_a.bits.tv_sbb2, value);
	}
	else {
		return MAX77659_INVALID_DATA;
	}
	
	return MAX77659_NO_ERROR;
}

int MAX77659::get_tv_sbb(uint8_t channel, float *voltV)
{
    int ret;
	uint8_t bit_value;
    reg_cnfg_sbb0_a_t reg_cnfg_sbb0_a = {0};
    reg_cnfg_sbb1_a_t reg_cnfg_sbb1_a = {0};
    reg_cnfg_sbb2_a_t reg_cnfg_sbb2_a = {0};

	if (channel == 0) { 
		ret = read_register(CNFG_SBB0_A, (uint8_t *)&(reg_cnfg_sbb0_a));
		if (ret != MAX77659_NO_ERROR) return ret;
		
		bit_value = (uint8_t)reg_cnfg_sbb0_a.bits.tv_sbb0;
	}
	else if (channel == 1) {
		ret = read_register(CNFG_SBB1_A, (uint8_t *)&(reg_cnfg_sbb1_a));
		if (ret != MAX77659_NO_ERROR) return ret;
	
		bit_value = (uint8_t)reg_cnfg_sbb1_a.bits.tv_sbb1;
	}
	else if (channel == 2) {
		ret = read_register(CNFG_SBB2_A, (uint8_t *)&(reg_cnfg_sbb2_a));
		if (ret != MAX77659_NO_ERROR) return ret;
	
		bit_value = (uint8_t)reg_cnfg_sbb2_a.bits.tv_sbb2;
	}
	else {
		return MAX77659_INVALID_DATA;
	}
	
	if (bit_value > 200) bit_value = 200;
	*voltV = (bit_value * 0.025f) + 0.5f;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_op_mode(uint8_t channel, decode_op_mode_t mode)
{	
	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
	
	if (channel == 0) {
		SET_BIT_FIELD(CNFG_SBB0_B, reg_cnfg_sbb0_b, reg_cnfg_sbb0_b.bits.op_mode, mode);
	}
	else if (channel == 1) {
		SET_BIT_FIELD(CNFG_SBB1_B, reg_cnfg_sbb1_b, reg_cnfg_sbb1_b.bits.op_mode, mode);
	}
	else if (channel == 2) {
		SET_BIT_FIELD(CNFG_SBB2_B, reg_cnfg_sbb2_b, reg_cnfg_sbb2_b.bits.op_mode, mode);
	}
	else {
		return MAX77659_INVALID_DATA;
	}
	
	return MAX77659_NO_ERROR;
}

int MAX77659::get_op_mode(uint8_t channel, decode_op_mode_t *mode)
{
    int ret;
    reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};

	if (channel == 0) {
		ret = read_register(CNFG_SBB0_B, (uint8_t *)&(reg_cnfg_sbb0_b));
		if (ret != MAX77659_NO_ERROR) return ret;

		*mode = (decode_op_mode_t)reg_cnfg_sbb0_b.bits.op_mode;
	}
	else if (channel == 1) {
		ret = read_register(CNFG_SBB1_B, (uint8_t *)&(reg_cnfg_sbb1_b));
		if (ret != MAX77659_NO_ERROR) return ret;

		*mode = (decode_op_mode_t)reg_cnfg_sbb1_b.bits.op_mode;
	}
	else if (channel == 2) {
		ret = read_register(CNFG_SBB2_B, (uint8_t *)&(reg_cnfg_sbb2_b));
		if (ret != MAX77659_NO_ERROR) return ret;

		*mode = (decode_op_mode_t)reg_cnfg_sbb2_b.bits.op_mode;
	}
	else { 
		return MAX77659_INVALID_DATA;
	}

    return MAX77659_NO_ERROR;
}

int MAX77659::set_ade_sbb(uint8_t channel, decode_ade_sbb_t ade_sbb)
{	
	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
	
	if (channel == 0) {
		SET_BIT_FIELD(CNFG_SBB0_B, reg_cnfg_sbb0_b, reg_cnfg_sbb0_b.bits.ade_sbb0, ade_sbb);
	}
	else if (channel == 1) {
		SET_BIT_FIELD(CNFG_SBB1_B, reg_cnfg_sbb1_b, reg_cnfg_sbb1_b.bits.ade_sbb1, ade_sbb);
	}
	else if (channel == 2) {
		SET_BIT_FIELD(CNFG_SBB2_B, reg_cnfg_sbb2_b, reg_cnfg_sbb2_b.bits.ade_sbb2, ade_sbb);
	}
	else {
		return MAX77659_INVALID_DATA;
	}
	
	return MAX77659_NO_ERROR;
}

int MAX77659::get_ade_sbb(uint8_t channel, decode_ade_sbb_t *ade_sbb)
{
    int ret;
    reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};

	if (channel == 0) {
		ret = read_register(CNFG_SBB0_B, (uint8_t *)&(reg_cnfg_sbb0_b));
		if (ret != MAX77659_NO_ERROR) return ret;

		*ade_sbb = (decode_ade_sbb_t)reg_cnfg_sbb0_b.bits.ade_sbb0;
	}
	else if (channel == 1) {
		ret = read_register(CNFG_SBB1_B, (uint8_t *)&(reg_cnfg_sbb1_b));
		if (ret != MAX77659_NO_ERROR) return ret;

		*ade_sbb = (decode_ade_sbb_t)reg_cnfg_sbb1_b.bits.ade_sbb1;
	}
	else if (channel == 2) {
		ret = read_register(CNFG_SBB2_B, (uint8_t *)&(reg_cnfg_sbb2_b));
		if (ret != MAX77659_NO_ERROR) return ret;

		*ade_sbb = (decode_ade_sbb_t)reg_cnfg_sbb2_b.bits.ade_sbb2;
	}
	else {
		return MAX77659_INVALID_DATA;
	}

    return MAX77659_NO_ERROR;
}

int MAX77659::set_en_sbb(uint8_t channel, decode_en_sbb_t en_sbb)
{	
	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
	
	if (channel == 0) {
		SET_BIT_FIELD(CNFG_SBB0_B, reg_cnfg_sbb0_b, reg_cnfg_sbb0_b.bits.en_sbb0, en_sbb);
	}
	else if (channel == 1) {
		SET_BIT_FIELD(CNFG_SBB1_B, reg_cnfg_sbb1_b, reg_cnfg_sbb1_b.bits.en_sbb1, en_sbb);
	}
	else if (channel == 2) {
		SET_BIT_FIELD(CNFG_SBB2_B, reg_cnfg_sbb2_b, reg_cnfg_sbb2_b.bits.en_sbb2, en_sbb);
	}
	else {
		return MAX77659_INVALID_DATA;
	}
	
	return MAX77659_NO_ERROR;
}

int MAX77659::get_en_sbb(uint8_t channel, decode_en_sbb_t *en_sbb)
{
    int ret;
    reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};

	if (channel == 0) {
		ret = read_register(CNFG_SBB0_B, (uint8_t *)&(reg_cnfg_sbb0_b));
		if (ret != MAX77659_NO_ERROR) return ret;

		*en_sbb = (decode_en_sbb_t)reg_cnfg_sbb0_b.bits.en_sbb0;
	}
	else if (channel == 1) {
		ret = read_register(CNFG_SBB1_B, (uint8_t *)&(reg_cnfg_sbb1_b));
		if (ret != MAX77659_NO_ERROR) return ret;

		*en_sbb = (decode_en_sbb_t)reg_cnfg_sbb1_b.bits.en_sbb1;
	}
	else if (channel == 2) {
		ret = read_register(CNFG_SBB2_B, (uint8_t *)&(reg_cnfg_sbb2_b));
		if (ret != MAX77659_NO_ERROR) return ret;

		*en_sbb = (decode_en_sbb_t)reg_cnfg_sbb2_b.bits.en_sbb2;
	}
	else {
        return MAX77659_INVALID_DATA;
    }

    return MAX77659_NO_ERROR;
}

int MAX77659::set_op_mode_chg(decode_op_mode_chg_t op_mode_chg)
{
	reg_cnfg_sbb_top_t reg_cnfg_sbb_top = {0};

    SET_BIT_FIELD(CNFG_SBB_TOP, reg_cnfg_sbb_top, reg_cnfg_sbb_top.bits.op_mode_chg, op_mode_chg);

    return MAX77659_NO_ERROR;
}

int MAX77659::get_op_mode_chg(decode_op_mode_chg_t *op_mode_chg)
{
    int ret;
    reg_cnfg_sbb_top_t reg_cnfg_sbb_top = {0};

    ret = read_register(CNFG_SBB_TOP, (uint8_t *)&(reg_cnfg_sbb_top));
    if (ret != MAX77659_NO_ERROR) return ret;

    *op_mode_chg = (decode_op_mode_chg_t)reg_cnfg_sbb_top.bits.op_mode_chg;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_drv_sbb(decode_drv_sbb_t drv_sbb)
{
	reg_cnfg_sbb_top_t reg_cnfg_sbb_top = {0};

    SET_BIT_FIELD(CNFG_SBB_TOP, reg_cnfg_sbb_top, reg_cnfg_sbb_top.bits.drv_sbb, drv_sbb);

    return MAX77659_NO_ERROR;
}

int MAX77659::get_drv_sbb(decode_drv_sbb_t *drv_sbb)
{
    int ret;
    reg_cnfg_sbb_top_t reg_cnfg_sbb_top = {0};

    ret = read_register(CNFG_SBB_TOP, (uint8_t *)&(reg_cnfg_sbb_top));
    if (ret != MAX77659_NO_ERROR) return ret;

    *drv_sbb = (decode_drv_sbb_t)reg_cnfg_sbb_top.bits.drv_sbb;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_ip_chg(decode_ip_chg_t ip_chg)
{
	reg_cnfg_sbb_top_b_t reg_cnfg_sbb_top_b = {0};

    SET_BIT_FIELD(CNFG_SBB_TOP_B, reg_cnfg_sbb_top_b, reg_cnfg_sbb_top_b.bits.ip_chg, ip_chg);

    return MAX77659_NO_ERROR;
}

int MAX77659::get_ip_chg(decode_ip_chg_t *ip_chg)
{
    int ret;
    reg_cnfg_sbb_top_b_t reg_cnfg_sbb_top_b = {0};

    ret = read_register(CNFG_SBB_TOP_B, (uint8_t *)&(reg_cnfg_sbb_top_b));
    if (ret != MAX77659_NO_ERROR) return ret;

    *ip_chg = (decode_ip_chg_t)reg_cnfg_sbb_top_b.bits.ip_chg;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_ip_sbb(uint8_t channel, decode_ip_sbb_t ip_sbb)
{
	reg_cnfg_sbb_top_b_t reg_cnfg_sbb_top_b = {0};

    if (channel == 0) {
        SET_BIT_FIELD(CNFG_SBB_TOP_B, reg_cnfg_sbb_top_b, reg_cnfg_sbb_top_b.bits.ip_sbb0, ip_sbb);
    }
    else if (channel == 1) {
        SET_BIT_FIELD(CNFG_SBB_TOP_B, reg_cnfg_sbb_top_b, reg_cnfg_sbb_top_b.bits.ip_sbb1, ip_sbb);
    }
    else if (channel == 2) {
        SET_BIT_FIELD(CNFG_SBB_TOP_B, reg_cnfg_sbb_top_b, reg_cnfg_sbb_top_b.bits.ip_sbb2, ip_sbb);
    }
    else {
        return MAX77659_INVALID_DATA;
    }

    return MAX77659_NO_ERROR;
}

int MAX77659::get_ip_sbb(uint8_t channel, decode_ip_sbb_t *ip_sbb)
{
    int ret;
    reg_cnfg_sbb_top_b_t reg_cnfg_sbb_top_b = {0};

    ret = read_register(CNFG_SBB_TOP_B, (uint8_t *)&(reg_cnfg_sbb_top_b));
    if (ret != MAX77659_NO_ERROR) return ret;

    if (channel == 0)
        *ip_sbb = (decode_ip_sbb_t)reg_cnfg_sbb_top_b.bits.ip_sbb0;
    else if (channel == 1)
        *ip_sbb = (decode_ip_sbb_t)reg_cnfg_sbb_top_b.bits.ip_sbb1;
    else if (channel == 2)
        *ip_sbb = (decode_ip_sbb_t)reg_cnfg_sbb_top_b.bits.ip_sbb2;
    else
        return MAX77659_INVALID_DATA;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_tv_ldo_offset(decode_tv_ldo_offset_t offset)
{
	reg_cnfg_ldo0_a_t reg_cnfg_ldo0_a = {0};

    SET_BIT_FIELD(CNFG_LDO0_A, reg_cnfg_ldo0_a, reg_cnfg_ldo0_a.bits.tv_ldo_offset, offset);

    return MAX77659_NO_ERROR;
}

int MAX77659::get_tv_ldo_offset(decode_tv_ldo_offset_t *offset)
{
    int ret;
    reg_cnfg_ldo0_a_t reg_cnfg_ldo0_a = {0};

    ret = read_register(CNFG_LDO0_A, (uint8_t *)&(reg_cnfg_ldo0_a));
    if (ret != MAX77659_NO_ERROR) return ret;

    *offset = (decode_tv_ldo_offset_t)reg_cnfg_ldo0_a.bits.tv_ldo_offset;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_tv_ldo_volt(float voltV)
{
	int ret;
	uint8_t value;
	reg_cnfg_ldo0_a_t reg_cnfg_ldo0_a = {0};
	float voltmV = voltV * 1000;
	
    ret = read_register(CNFG_LDO0_A, (uint8_t *)&(reg_cnfg_ldo0_a));
    if (ret != MAX77659_NO_ERROR) return ret;

    if (reg_cnfg_ldo0_a.bits.tv_ldo_offset == 0) { //No Offset
        if (voltmV < 500) voltmV = 500;
        else if (voltmV > 3675) voltmV = 3675;

        value = (voltmV - 500) / 25;
    }
    else { //1.325V Offset 
        if (voltmV < 1825) voltmV = 1825;
        else if (voltmV > 5000) voltmV = 5000;

        value = (voltmV - 1825) / 25;
    }
    
    SET_BIT_FIELD(CNFG_LDO0_A, reg_cnfg_ldo0_a, reg_cnfg_ldo0_a.bits.tv_ldo_volt, value);
	
    return MAX77659_NO_ERROR;
}

int MAX77659::get_tv_ldo_volt(float *voltV)
{
    int ret;
	uint8_t bit_value;
    reg_cnfg_ldo0_a_t reg_cnfg_ldo0_a = {0};
	
    ret = read_register(CNFG_LDO0_A, (uint8_t *)&(reg_cnfg_ldo0_a));
    if (ret != MAX77659_NO_ERROR) return ret;

    bit_value = (uint8_t)reg_cnfg_ldo0_a.bits.tv_ldo_volt;
    if (reg_cnfg_ldo0_a.bits.tv_ldo_offset == 0) //No Offset
        *voltV = (bit_value * 0.025f) + 0.5f;
    else //1.325V Offset 
        *voltV = (bit_value * 0.025f) + 1.825f;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_en_ldo(decode_en_ldo_t en_ldo)
{	
	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};
	
	SET_BIT_FIELD(CNFG_LDO0_B, reg_cnfg_ldo0_b, reg_cnfg_ldo0_b.bits.en_ldo, en_ldo);

	return MAX77659_NO_ERROR;
}

int MAX77659::get_en_ldo( decode_en_ldo_t *en_ldo)
{
    int ret;
    reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};

    ret = read_register(CNFG_LDO0_B, (uint8_t *)&(reg_cnfg_ldo0_b));
    if (ret != MAX77659_NO_ERROR) return ret;

    *en_ldo = (decode_en_ldo_t)reg_cnfg_ldo0_b.bits.en_ldo;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_ade_ldo(decode_ade_ldo_t ade_ldo)
{	
	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};
	
	SET_BIT_FIELD(CNFG_LDO0_B, reg_cnfg_ldo0_b, reg_cnfg_ldo0_b.bits.ade_ldo, ade_ldo);
	
	return MAX77659_NO_ERROR;
}

int MAX77659::get_ade_ldo(decode_ade_ldo_t *ade_ldo)
{
    int ret;
    reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};

    ret = read_register(CNFG_LDO0_B, (uint8_t *)&(reg_cnfg_ldo0_b));
    if (ret != MAX77659_NO_ERROR) return ret;

    *ade_ldo = (decode_ade_ldo_t)reg_cnfg_ldo0_b.bits.ade_ldo;

    return MAX77659_NO_ERROR;
}

int MAX77659::set_ldo_md(decode_ldo_md_t mode)
{	
	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};
	
	SET_BIT_FIELD(CNFG_LDO0_B, reg_cnfg_ldo0_b, reg_cnfg_ldo0_b.bits.ldo_md, mode);
	
	return MAX77659_NO_ERROR;
}

int MAX77659::get_ldo_md(decode_ldo_md_t *mode)
{
    int ret;
    reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};

    ret = read_register(CNFG_LDO0_B, (uint8_t *)&(reg_cnfg_ldo0_b));
    if (ret != MAX77659_NO_ERROR) return ret;

    *mode = (decode_ldo_md_t)reg_cnfg_ldo0_b.bits.ldo_md;

    return MAX77659_NO_ERROR;
}

int MAX77659::irq_disable_all()
{
    int ret;
    uint8_t reg = 0;
    uint8_t status = 0;

    //Disable Masks in INTM_GLBL1
    ret = write_register(INTM_GLBL1, &reg);
    if (ret != MAX77659_NO_ERROR) return ret;

    //Disable Masks in INTM_GLBL0
    ret = write_register(INTM_GLBL0, &reg);
    if (ret != MAX77659_NO_ERROR) return ret;

    //Disable Masks in INT_M_CHG
    ret = write_register(INT_M_CHG, &reg);
    if (ret != MAX77659_NO_ERROR) return ret;

    // Clear Interrupt Flags in INT_GLBL1
    ret = read_register(INT_GLBL1, &status);
    if (ret != MAX77659_NO_ERROR) return ret;

    // Clear Interrupt Flags in INT_GLBL0
    ret = read_register(INT_GLBL0, &status);
    if (ret != MAX77659_NO_ERROR) return ret;

    // Clear Interrupt Flags in INT_CHG
    ret = read_register(INT_CHG, &status);
    if (ret != MAX77659_NO_ERROR) return ret;

	return MAX77659_NO_ERROR;
}

void MAX77659::set_interrupt_handler(reg_bit_int_glbl_t id, interrupt_handler_function func, void *cb)
{
    interrupt_handler_list[id].func = func;
    interrupt_handler_list[id].cb = cb;
}

void MAX77659::post_interrupt_work()
{
    int ret;
    uint8_t reg = 0, inten = 0, not_inten = 0, mask = 0;


    while (true) {

        ThisThread::flags_wait_any(POST_INTR_WORK_SIGNAL_ID);

        // Check Interrupt Flags in INT_GLBL0
        ret = read_register(INT_GLBL0, &reg);
        if (ret != MAX77659_NO_ERROR) return;

        ret = read_register(INTM_GLBL0, &inten);
        if (ret != MAX77659_NO_ERROR) return;

        not_inten = ~inten; // 0 means unmasked.

        for (int i = 0; i < INT_GLBL1_GPI1_F; i++) {
            mask = (1 << i);
            if ((reg & mask) && (not_inten & mask)) {
                if (interrupt_handler_list[i].func != NULL) {
                    interrupt_handler_list[i]
                    .func(interrupt_handler_list[i].cb);
                }
            }
        }

        // Check Interrupt Flags in INT_GLBL1
        ret = read_register(INT_GLBL1, &reg);
		if (ret != MAX77659_NO_ERROR) return;

		ret = read_register(INTM_GLBL1, &inten);
		if (ret != MAX77659_NO_ERROR) return;

		not_inten = ~inten; // 0 means unmasked.

		for (int i = INT_GLBL1_GPI1_F; i < INT_CHG_THM_I; i++) {
			mask = (1 << (i - INT_GLBL1_GPI1_F));
			if ((reg & mask) && (not_inten & mask)) {
				if (interrupt_handler_list[i].func != NULL) {
					interrupt_handler_list[i]
					.func(interrupt_handler_list[i].cb);
				}
			}
		}

		// Check Interrupt Flags in INT_CHG
        ret = read_register(INT_CHG, &reg);
		if (ret != MAX77659_NO_ERROR) return;

		ret = read_register(INT_M_CHG, &inten);
		if (ret != MAX77659_NO_ERROR) return;
		not_inten = ~inten; // 0 means unmasked.

		for (int i = INT_CHG_THM_I; i < INT_CHG_END; i++) {
			mask = (1 << (i - INT_CHG_THM_I));
			if ((reg & mask) && (not_inten & mask)) {
				if (interrupt_handler_list[i].func != NULL) {
					interrupt_handler_list[i]
					.func(interrupt_handler_list[i].cb);
				}
			}
		}
    }
}

void MAX77659::interrupt_handler()
{
    post_intr_work_thread->flags_set(POST_INTR_WORK_SIGNAL_ID);
}
