/*******************************************************************************
* Copyright (C) 2022 Analog Devices, Inc., All rights Reserved.
*
* This software is protected by copyright laws of the United States and
* of foreign countries. This material may also be protected by patent laws
* and technology transfer regulations of the United States and of foreign
* countries. This software is furnished under a license agreement and/or a
* nondisclosure agreement and may only be used or reproduced in accordance
* with the terms of those agreements. Dissemination of this information to
* any party or parties not specified in the license agreement and/or
* nondisclosure agreement is expressly prohibited.
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Analog Devices, Inc. retains all
* ownership rights.
*******************************************************************************
*/

#ifndef MAX77659_REGS_H_
#define MAX77659_REGS_H_

/**
 * @brief INT_GLBL0 Register
 *
 * Address : 0x00
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char gpi0_f    : 1;    /**< GPI Falling Interrupt. Bit 0.
                                             Note that "GPI" refers to the GPIO programmed to be an input.
                                             0 = No GPI falling edges have occurred since the last time this bit was read.
                                             1 = A GPI falling edge has occurred since the last time this bit was read. */
        unsigned char gpi0_r    : 1;    /**< GPI Rising Interrupt. Bit 1. 
                                             Note that "GPI" refers to the GPIO programmed to be an input.
                                             0 = No GPI rising edges have occurred since the last time this bit was read. 
                                             1 = A GPI rising edge has occurred since the last time this bit was read. */
        unsigned char nen_f     : 1;    /**< nEN Falling Interrupt.Bit 2.
                                             0 = No nEN falling edges have occurred since the last time this bit was read.
                                             1 = A nEN falling edge as occurred since the last time this bit was read. */
        unsigned char nen_r     : 1;    /**< nEN Rising Interrupt. Bit 3.
                                             0 = No nEN rising edges have occurred since the last time this bit was read.
                                             1 = A nEN rising edge as occurred since the last time this bit was read. */
        unsigned char tjal1_r   : 1;    /**< Thermal Alarm 1 Rising Interrupt. Bit 4.
                                             0 = The junction temperature has not risen above TJAL1 since the last time this bit was read.
                                             1 = The junction temperature has risen above TJAL1 since the last time this bit was read. */
        unsigned char tjal2_r   : 1;    /**< Thermal Alarm 2 Rising Interrupt. Bit 5.
                                             0 = The junction temperature has not risen above TJAL2 since the last time this bit was read.
                                             1 = The junction temperature has risen above TJAL2 since the last time this bit was read. */
        unsigned char rsvd      : 1;    /**< Reserved. Unutilized bit. Write to 0. Reads are don't care. Bit 6. */
        unsigned char dod_r     : 1;    /**< LDO Dropout Detector Rising Interrupt. Bit 7.
                                             0 = The LDO has not detected dropout since the last time this bit was read.
                                             1 = The LDO has detected dropout since the last time this bit was read. */
    } bits;
} reg_int_glbl0_t;

/**
 * @brief INT_GLBL1 Register
 *
 * Address : 0x04
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char gpi1_f    : 1;    /**< GPI Falling Interrupt. Bit 0.
                                             Note that "GPI" refers to the GPIO programmed to be an input.
                                             0 = No GPI falling edges have occurred since the last time this bit was read.
                                             1 = A GPI falling edge has occurred since the last time this bit was read. */
        unsigned char gpi1_r    : 1;    /**< GPI Rising Interrupt. Bit 1.
                                             Note that "GPI" refers to the GPIO programmed to be an input.
                                             0 = No GPI rising edges have occurred since the last time this bit was read. 
                                             1 = A GPI rising edge has occurred since the last time this bit was read. */
        unsigned char rsvd1     : 2;    /**< Reserved. Unutilized bit. Write to 0. Reads are don't care. Bit 3:2. */
        unsigned char sbb_to    : 1;    /**< SBB Timeout. Bit 4.
                                             0 = NO SBB timeout occurred since the last time this bit was read
                                             1 = SBB timeout occurred since the last time this bit was read */
        unsigned char ldo_f     : 1;    /**< LDO Fault Interrupt. Bit 5.
                                             0 = No fault has occurred on LDO since the last time this bit was read.
                                             1 = LDO has fallen out of regulation since the last time this bit was read. */
        unsigned char ldo1_f    : 1;    /**< LDO1 Fault Interrupt. Bit 6.
                                             0 = No fault has occurred on LDO1 since the last time this bit was read.
                                             1 = LDO1 has fallen out of regulation since the last time this bit was read.  */
        unsigned char rsvd2     : 2;    /**< Reserved. Unutilized bit. Write to 0. Reads are don't care. Bit 7:6. */
    } bits;
} reg_int_glbl1_t;

/**
 * @brief ERCFLAG Register
 *
 * Address : 0x05
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char tovld     : 1;    /**< Thermal Overload. Bit 0.
                                             0 = Thermal overload has not occurred since the last read of this register.
                                             1 = Thermal overload has occurred since the list read of this register.
                                             This indicates that the junction temperature has exceeded 165ºC. */
        unsigned char sysovlo   : 1;    /**< SYS Domain Overvoltage Lockout. Bit 1.
                                             0 = The SYS domain overvoltage lockout has not occurred since this last read of this register.
                                             1 = The SYS domain overvoltage lockout has occurred since the last read of this register. */
        unsigned char avluvlo   : 1;    /**< AVL Domain Undervoltage Lockout. Bit 2.
                                             0 = The AVL domain undervoltage lockout has not occurred since this last read of this register.
                                             1 = The AVL domain undervoltage lockout has occurred since the last read of this register. */
        unsigned char mrst      : 1;    /**< Manual Reset Timer. Bit 3.
                                             0 = A Manual Reset has not occurred since this last read of this register.
                                             1 = A Manual Reset has occurred since this last read of this register. */
        unsigned char sft_off_f : 1;    /**< Software Off Flag. Bit 4.
                                             0 = The SFT_OFF function has not occurred since the last read of this register.
                                             1 = The SFT_OFF function has occurred since the last read of this register. */
        unsigned char sft_crst_f: 1;    /**< Software Cold Reset Flag. Bit 5.
                                             0 = The software cold reset has not occurred since the last read of this register.
                                             1 = The software cold reset has occurred since the last read of this register. */
        unsigned char wdt_off   : 1;    /**< Watchdog Timer OFF Flag. Bit 6.
                                             This bit sets when the watchdog timer expires and causes a power-off.
                                             0 = Watchdog timer has not caused a power-off since the last time this bit was read.
                                             1 = Watchdog timer has expired and caused a power-off since the last time this bit was read.  */
        unsigned char wdt_rst   : 1;    /**< Watchdog Timer Reset Flag. Bit 7.
                                             This bit sets when the watchdog timer expires and causes a power-reset. 
                                             0 = Watchdog timer has not caused a power-reset since the last time this bit was read. 
                                             1 = Watchdog timer has expired and caused a power-reset since the last time this bit was read.*/
    } bits;
} reg_ercflag_t;

/**
 * @brief STAT_GLBL Register
 *
 * Address : 0x06
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char stat_irq  : 1;    /**< Software Version of the nIRQ MOSFET gate drive. Bit 0.
                                             0 = unmasked gate drive is logic low 
                                             1 = unmasked gate drive is logic high */
        unsigned char stat_en   : 1;    /**< Debounced Status for the nEN input. Bit 1.
                                             0 = nEN is not active (logic high) 
                                             1 = nEN is active (logic low) */
        unsigned char tjal1_s   : 1;    /**< Thermal Alarm 1 Status. Bit 2.
                                             0 = The junction temperature is less than TJAL1 
                                             1 = The junction temperature is greater than TJAL1 */
        unsigned char tjal2_s   : 1;    /**< Thermal Alarm 2 Status. Bit 3.
                                             0 = The junction temperature is less than TJAL2 
                                             1 = The junction temperature is greater than TJAL2 */
        unsigned char rsvd      : 1;    /**< Reserved. Unutilized bit. Write to 0. Reads are don't care. */
        unsigned char dod_s     : 1;    /**< LDO0 Dropout Detector Rising Status. Bit 5.
                                             0 = LDO0 is not in dropout 
                                             1 = LDO0 is in dropout */
        unsigned char bok       : 1;    /**< BOK Interrupt Status. Bit 6.
                                             0 = Main Bias is not ready. 
                                             1 = Main Bias enabled and ready.  */
        unsigned char didm      : 1;    /**< Device Identification Bits for Metal Options. Bit 7.
                                             0 = MAX77659 
                                             1 = Reserved */
    } bits;
} reg_stat_glbl_t;

/**
 * @brief INTM_GLBL1 Register
 *
 * Address : 0x08
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char gpi1_fm   : 1;    /**< GPI Falling Interrupt Mask. Bit 0. 
                                             0 = Unmasked. If GPI_F goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to GPI_F. */
        unsigned char gpi1_rm   : 1;    /**< GPI Rising Interrupt Mask. Bit 1. 
                                             0 = Unmasked. If GPI_R goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to GPI_R. */
        unsigned char rsvd1     : 2;       /**< Reserved. Unutilized bit. Write to 0. Reads are don't care. Bit 3:2. */
        unsigned char sbb_to_m  : 1;    /**< SBB Timeout Mask. Bit 4.
                                             0 = Unmasked. If SBB_TO goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to SBB_TO */
        unsigned char ldo_m     : 1;    /**< LDO0 Fault Interrupt. Bit 5.
                                             0 = Unmasked. If LDO0_F goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared.
                                             1 = Masked. nIRQ does not go low due to LDO0_F. */
        unsigned char rsvd2     : 2;    /**< Reserved. Unutilized bit. Write to 0. Reads are don't care. Bit 7:6. */
    } bits;
} reg_intm_glbl1_t;

/**
 * @brief INTM_GLBL0 Register
 *
 * Address : 0x09
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char gpi0_fm   : 1;    /**< GPI Falling Interrupt Mask. Bit 0. 
                                             0 = Unmasked. If GPI_F goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to GPI_F. */
        unsigned char gpi0_rm   : 1;    /**< GPI Rising Interrupt Mask. Bit 1. 
                                             0 = Unmasked. If GPI_R goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to GPI_R. */
        unsigned char nen_fm    : 1;    /**< nEN Falling Interrupt Mask. Bit 2.
                                             0 = Unmasked. If nEN_F goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to nEN_F. */
        unsigned char nen_rm    : 1;    /**< nEN Rising Interrupt Mask. Bit 3.
                                             0 = Unmasked. If nEN_R goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to nEN_R. */
        unsigned char tjal1_rm  : 1;    /**< Thermal Alarm 1 Rising Interrupt Mask. Bit 4.
                                             0 = Unmasked. If TJAL1_R goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to TJAL1_R. */
        unsigned char tjal2_rm  : 1;    /**< Thermal Alarm 2 Rising Interrupt Mask. Bit 5.
                                             0 = Unmasked. If TJAL2_R goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to TJAL2_R. */
        unsigned char rsvd      : 1;    /**< Reserved. Unutilized bit. Write to 0. Reads are don't care. Bit 6. */
        unsigned char dod_rm    : 1;    /**< LDO Dropout Detector Rising Interrupt Mask. Bit 7.
                                             0 = Unmasked. If DOD0_R goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to DOD0_R. */
    } bits;
} reg_intm_glbl0_t;

/**
 * @brief CNFG_GLBL Register
 *
 * Address : 0x10
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char sft_ctrl  : 2;    /**< Software Reset Functions. Bit 1:0. 
                                             0b00 = No Action 
                                             0b01 = Software Cold Reset (SFT_CRST). The device powers down, resets, and the powers up again. 
                                             0b10 = Software Off (SFT_OFF). The device powers down, resets, and then remains off and waiting for a wake-up event. 
                                             0b11 = Factory-Ship Mode Enter (FSM). */
        unsigned char dben_nen  : 1;    /**< Debounce Timer Enable for the nEN Pin. Bit 2.
                                             0 = 500μs Debounce 
                                             1 = 30ms Debounce */
        unsigned char nen_mode  : 1;    /**< nEN Input (ON-KEY) Default Configuration Mode. Bit 3.
                                             0 = Push-Button Mode 
                                             1 = Slide-Switch Mode */
        unsigned char sbia_en   : 1;    /**< Main Bias Enable Software Request. Bit 4.
                                             0 = Main Bias not enabled by software. 
                                             Note that the main bias may be on via the on/off controller. 
                                             1 = Main Bias force enabled by software. */
        unsigned char sbia_lpm  : 1;    /**< Main Bias Low-Power Mode Software Request. Bit 5.
                                             0 = Main Bias requested to be in Normal-Power Mode by software. 
                                             1 = Main Bias request to be in Low-Power Mode by software. */
        unsigned char t_mrst    : 1;    /**< Sets the Manual Reset Time (tMRST). Bit 6.
                                             0 = 8s 
                                             1 = 3.3s  */
        unsigned char pu_dis    : 1;    /**< nEN Internal Pullup Resistor. Bit 7.
                                             0 = Strong internal nEN pullup (200kΩ) 
                                             1 = Weak internal nEN pullup (10MΩ) */
    } bits;
} reg_cnfg_glbl_t;

/**
 * @brief CNFG_GPIO0 Register
 *
 * Address : 0x11
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char gpo_dir   : 1;    /**< GPIO Direction. Bit 0. 
                                             0 = General purpose output (GPO) 
                                             1 = General purpose input (GPI) */
        unsigned char gpo_di    : 1;    /**< GPIO Digital Input Value. Bit 1.
                                             0 = Input logic low 
                                             1 = Input logic high */
        unsigned char gpo_drv   : 1;    /**< General Purpose Output Driver Type. Bit 2.
                                             This bit is a don't care when DIR = 1 (configured as input) When set for GPO (DIR = 0): 
                                             0 = Open-Drain 
                                             1 = Push-Pull */
        unsigned char gpo_do    : 1;    /**< General Purpose Output Data Output. Bit 3.
                                             This bit is a don't care when DIR = 1 (configured as input). When set for GPO (DIR = 0): 
                                             0 = GPIO is output is logic low 
                                             1 = GPIO is output logic high when set as push-pull output (DRV = 1). */
        unsigned char dben_gpi  : 1;    /**< General Purpose Input Debounce Timer Enable. Bit 4.
                                             0 = no debounce 
                                             1 = 30ms debounce */
        unsigned char alt_gpio  : 1;    /**< Alternate Mode Enable for GPIO0. Bit 5.
                                             0 = Standard GPIO. 
                                             1 = Active-high input, enable control for low-power mode. */
        unsigned char           : 1;    /**< Bit 6. */
        unsigned char rsvd      : 1;    /**< Reserved. Bit 7. Unutilized bit. Write to 0. Reads are don't care. */
    } bits;
} reg_cnfg_gpio0_t;

/**
 * @brief CNFG_GPIO1 Register
 *
 * Address : 0x12
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char gpo_dir   : 1;    /**< GPIO Direction. Bit 0. 
                                             0 = General purpose output (GPO) 
                                             1 = General purpose input (GPI) */
        unsigned char gpo_di    : 1;    /**< GPIO Digital Input Value. Bit 1.
                                             0 = Input logic low 
                                             1 = Input logic high */
        unsigned char gpo_drv   : 1;    /**< General Purpose Output Driver Type. Bit 2.
                                             This bit is a don't care when DIR = 1 (configured as input) When set for GPO (DIR = 0): 
                                             0 = Open-Drain 
                                             1 = Push-Pull */
        unsigned char gpo_do    : 1;    /**< General Purpose Output Data Output. Bit 3.
                                             This bit is a don't care when DIR = 1 (configured as input). When set for GPO (DIR = 0): 
                                             0 = GPIO is output is logic low 
                                             1 = GPIO is output logic high when set as push-pull output (DRV = 1). */
        unsigned char dben_gpi  : 1;    /**< General Purpose Input Debounce Timer Enable. Bit 4.
                                             0 = no debounce 
                                             1 = 30ms debounce */
        unsigned char alt_gpio  : 1;    /**< Alternate Mode Enable for GPIO1. Bit 5.
                                             0 = Standard GPIO. 
                                             1 = Active-high input, enable control for the DVS feature for SBB0. */
        unsigned char rsvd      : 2;    /**< Reserved. Bit 7:6. Unutilized bit. Write to 0. Reads are don't care. */
    } bits;
} reg_cnfg_gpio1_t;

/**
 * @brief CID Register
 *
 * Address : 0x14
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char cid_3_0   : 4;    /**< Bits 0 to 3 of the Chip Identification Code. Bit 3:0.
                                             The Chip Identification Code refers to a set of reset values in the register map, or the "OTP configuration.". */
        unsigned char           : 3;    /**< Bit 6:4. */
        unsigned char cid_7     : 1;    /**< Bit 4 of the Chip Identification Code. Bit 7.
                                             The Chip Identification Code refers to a set of reset values in the register map, or the "OTP configuration.". */
    } bits;
} reg_cid_t;

/**
 * @brief CNFG_WDT Register
 *
 * Address : 0x17
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char wdt_lock  : 1;    /**< Factory-Set Safety Bit for the Watchdog Timer. Bit 0. 
                                             0 = Watchdog timer can be enabled and disabled with WDT_EN. 
                                             1 = Watchdog timer can not be disabled with WDT_EN. 
                                             However, WDT_EN can still be used to enable the watchdog timer. */
        unsigned char wdt_en    : 1;    /**< Watchdog Timer Enable. Bit 1.
                                             0 = Watchdog timer is not enabled. 
                                             1 = Watchdog timer is enabled. The timer will expire if not reset by setting WDT_CLR. */
        unsigned char wdt_clr   : 1;    /**< Watchdog Timer Clear Control. Bit 2.
                                             0 = Watchdog timer period is not reset. 
                                             1 = Watchdog timer is reset back to tWD. */
        unsigned char wdt_mode  : 1;    /**< Watchdog Timer Expired Action. Bit 3.
                                             0 = Watchdog timer expire causes power-off. 
                                             1 = Watchdog timer expire causes power-reset. */
        unsigned char wdt_per   : 2;    /**< Watchdog Timer Period. Bit 5:4.
                                             0b00 = 16 seconds      0b01 = 32 seconds 
                                             0b10 = 64 seconds      0b11 = 128 seconds. */
        unsigned char rsvd      : 2;    /**< Reserved. Bit 7:6.
                                             Unutilized bit. Write to 0. Reads are don't care. */
    } bits;
} reg_cnfg_wdt_t;

/**
 * @brief INT_CHG Register
 *
 * Address : 0x01
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char thm_i         : 1;    /**< Thermistor related interrupt. Bit 0. 
                                             	 0 = The bits in THM_DTLS[2:0] have not changed since the last time this bit was read
                                             	 1 = The bits in THM_DTLS[2:0] have changed since the last time this bit was read */
        unsigned char chg_i         : 1;    /**< Charger related interrupt. Bit 1. 
                                             	 0 = The bits in CHG_DTLS[3:0] have not changed since the last time this bit was read
                                             	 1 = The bits in CHG_DTLS[3:0] have changed since the last time this bit was read */
        unsigned char chgin_i       : 1;    /**< CHGIN related interrupt. Bit 2.
                                             	 0 = The bits in CHGIN_DTLS[1:0] have not changed since the last time this bit was read
                                             	 1 = The bits in CHGIN_DTLS[1:0] have changed since the last time this bit was read */
        unsigned char tj_reg_i      : 1;    /**< Die junction temperature regulation interrupt. Bit 3.
                                             	 0 = The die temperature has not exceeded TJ-REG since the last time this bit was read
                                             	 1 = The die temperature has exceeded TJ-REG since the last time this bit was read */
        unsigned char sys_ctrl_i    : 1;    /**< Minimum System Voltage Regulation-loop related interrupt. Bit 4.
                                             	 0 = The minimum system voltage regulation loop has not engaged since the last time this bit was read
                                             	 1 = The minimum system voltage regulation loop has engaged since the last time this bit was read */
        unsigned char rsvd          : 1;    /**< Reserved. Bit 7:5. Unutilized bit. Write to 0. Reads are don't care. */
    } bits;
} reg_int_chg_t;

/**
 * @brief STAT_CHG_A
 * 
 * Address : 0x02
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char thm_dtls          : 3;    /**< Battery Temperature Details. Bit 2:0.
                                                    0b000 = Thermistor is disabled (THM_EN = 0) 
                                                    0b001 = Battery is cold as programmed by THM_COLD[1:0] If thermistor and charger are enabled while the battery is cold, a battery temperature fault will occur. 
                                                    0b010 = Battery is cool as programmed by THM_COOL[1:0] 
                                                    0b011 = Battery is warm as programmed by THM_WARM[1:0] 
                                                    0b100 = Battery is hot as programmed by THM_HOT[1:0]. If thermistor and charger are enabled while the battery is hot, a battery temperature fault will occur. 
                                                    0b101 = Battery is in the normal temperature region 
                                                    0b110 - 0b111 = reserved */
        unsigned char tj_reg_stat       : 1;    /**< Maximum Junction Temperature Regulation Loop Status. Bit 3.
                                                    0 = The maximum junction temperature regulation loop is not engaged 
                                                    1 = The maximum junction temperature regulation loop has engaged to regulate the junction temperature to less than TJ-REG */
        unsigned char vsys_min_stat     : 1;    /**< Minimum System Voltage Regulation Loop Status. Bit 4.
                                                    0 = The minimum system voltage regulation loop is not enganged 
                                                    1 = The minimum system voltage regulation loop is engaged to regulate VSYS ≥ VSYS-MIN */
        unsigned char rsvd              : 3;    /**< Reserved. Bit 7:5. Unutilized bit. Write to 0. Reads are don't care. */
    } bits;
} reg_stat_chg_a_t;

/**
 * @brief STAT_CHG_B
 * 
 * Address : 0x03
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char time_sus      : 1;    /**< Time Suspend Indicator. Bit 0.
                                                0 = The charger's timers are either not active, or not suspended 
                                                1 = The charger's active timer is suspended due to one of three reasons: 
                                                charge current dropped below 20% of IFAST-CHG while the charger state machine is in FAST CHARGE CC mode, 
                                                the charger is in SUPPLEMENT mode, or the charger state machine is in BATTERY TEMPERATURE FAULT mode. */
        unsigned char chg           : 1;    /**< Quick Charger Status. Bit 1.
                                                0 = Charging is not happening 
                                                1 = Charging is happening */
        unsigned char chgin_dtls    : 2;    /**< CHGIN Status Detail. Bit 3:2.
                                                0b00 = The CHGIN input voltage is below the UVLO threshold (VCHGIN < VUVLO) 
                                                0b01 = The CHGIN input voltage is above the OVP threshold (VCHGIN > VOVP) 
                                                0b10 = The CHGIN input is being debounced (no power accepted from CHGIN during debounce) 
                                                0b11 = The CHGIN input is okay and debounced  */
        unsigned char chg_dtls      : 4;    /**< Charger Details. Bit 7:4.
                                                0b0000 = Off 
                                                0b0001 = Prequalification mode 
                                                0b0010 = Fast-charge constant-current (CC) mode 
                                                0b0011 = JEITA modified fast-charge constant-current mode 
                                                0b0100 = Fast-charge constant-voltage (CV) mode 
                                                0b0101 = JEITA modified fast-charge constant-voltage mode 
                                                0b0110 = Top-off mode 
                                                0b0111 = JEITA modified top-off mode 
                                                0b1000 = Done 
                                                0b1001 = JEITA modified done (done was entered through the JEITA-modified fast-charge states) 
                                                0b1010 = Prequalification timer fault 
                                                0b1011 = Fast-charge timer fault 
                                                0b1100 = Battery temperature fault 
                                                0b1101 - 0b1111 = reserved */
    } bits;
} reg_stat_chg_b_t;

/**
 * @brief INT_M_CHG Register
 *
 * Address : 0x07
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char thm_m         : 1;    /**< Setting this bit prevents the THM_I bit from causing hardware IRQs. Bit 0. 
                                             	 0 = THM_I is not masked
                                             	 1 = THM_I is masked */
        unsigned char chg_m         : 1;    /**< Setting this bit prevents the CHG_I bit from causing hardware IRQs. Bit 1. 
                                             	 0 = CHG_I is not masked
                                             	 1 = CHG_I is masked */
        unsigned char chgin_m       : 1;    /**< Setting this bit prevents the CHGIN_I bit from causing hardware IRQs. Bit 2.
                                             	 0 = CHGIN_I is not masked
                                             	 1 = CHGIN_I is masked */
        unsigned char tj_reg_m      : 1;    /**< Setting this bit prevents the TJREG_I bit from causing hardware IRQs. Bit 3.
                                             	 0 = TJREG_I is not masked
                                             	 1 = TJREG_I is masked */
        unsigned char sys_ctrl_m    : 1;    /**< Setting this bit prevents the SYS_CTRL_I bit from causing hardware IRQs. Bit 4.
                                             	 0 = SYS_CTRL_I is not masked
                                             	 1 = SYS_CTRL_I is masked */
        unsigned char rsvd          : 3;    /**< Reserved. Bit 7:5. Unutilized bit. Write to 0. Reads are don't care. */
    } bits;
} reg_int_m_chg_t;

/**
 * @brief CNFG_CHG_A
 * 
 * Address : 0x20
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char thm_cold  : 2;    /**< Sets the VCOLD JEITA Temperature Threshold. Bit 1:0.
                                            0b00 = VCOLD = 1.024V (-10ºC for β = 3380K) 
                                            0b01 = VCOLD = 0.976V (-5ºC for β = 3380K) 
                                            0b10 = VCOLD = 0.923V (0ºC for β = 3380K) 
                                            0b11 = VCOLD = 0.867V (5ºC for β = 3380K) */
        unsigned char thm_cool  : 2;    /**< Sets the VCOOL JEITA Temperature Threshold. Bit 3:2.
                                            0b00 = VCOOL = 0.923V (0ºC for β = 3380K) 
                                            0b01 = VCOOL = 0.867V (5ºC for β = 3380K) 
                                            0b10 = VCOOL = 0.807V (10ºC for β = 3380K) 
                                            0b11 = VCOOL = 0.747V (15ºC for β = 3380K) */
        unsigned char thm_warm  : 2;    /**< Sets the VWARM JEITA Temperature Threshold. Bit 5:4.
                                            0b00 = VWARM = 0.511V (35ºC for β = 3380K) 
                                            0b01 = VWARM = 0.459V (40ºC for β = 3380K) 
                                            0b10 = VWARM = 0.411V (45ºC for β = 3380K) 
                                            0b11 = VWARM = 0.367V (50ºC for β = 3380K)  */
        unsigned char thm_hot   : 2;    /**< Sets the VHOT JEITA Temperature Threshold. Bit 7:6.
                                            0b00 = VHOT = 0.411V (45ºC for β = 3380K) 
                                            0b01 = VHOT = 0.367V (50ºC for β = 3380K) 
                                            0b10 = VHOT = 0.327V (55ºC for β = 3380K) 
                                            0b11 = VHOT = 0.291V (60ºC for β = 3380K) */
    } bits;
} reg_cnfg_chg_a_t;

/**
 * @brief CNFG_CHG_B
 * 
 * Address : 0x21
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char chg_en        : 1;    /**< Charger Enable. Bit 0.
                                                0 = the battery charger is disabled 
                                                1 = the battery charger is enabled */
        unsigned char i_pq          : 1;    /**< Sets the prequalification charge current (IPQ) as a percentage of IFAST-CHG. Bit 1.
                                                0 = 10%     1 = 20% */
        unsigned char rsvd          : 6;    /**< Reserved. Bit 7:2. Unutilized bit. Write to 0. Reads are don't care. */
    } bits;
} reg_cnfg_chg_b_t;

/**
 * @brief CNFG_CHG_C
 * 
 * Address : 0x22
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char t_topoff  : 3;    /**< Top-off timer value (tTO). Bit 2:0.
                                            0b000 = 0 minutes       0b001 = 5 minutes 
                                            0b010 = 10 minutes      0b011 = 15 minutes 
                                            0b100 = 20 minutes      0b101 = 25 minutes 
                                            0b110 = 30 minutes      0b111 = 35 minutes */
        unsigned char i_term    : 2;    /**< Charger Termination Current (ITERM). Bit 4:3.
                                            00 = 5%     01 = 7.5% 
                                            10 = 10%    11 = 15%  */
        unsigned char chg_pq    : 3;    /**< Battery prequalification voltage threshold (VPQ). Bit 7:5.
                                            0b000 = 2.3V    0b001 = 2.4V 
                                            0b010 = 2.5V    0b011 = 2.6V 
                                            0b100 = 2.7V    0b101 = 2.8V 
                                            0b110 = 2.9V    0b111 = 3.0V */
    } bits;
} reg_cnfg_chg_c_t;

/**
 * @brief CNFG_CHG_D
 * 
 * Address : 0x23
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char vsys_min      : 2;    /**< Minimum SYS Voltage . Bit 1:0.
                                                0x0 = 3.2V    0x1 = 3.3V 
                                                0x2 = 3.4V  0x3 = 3.5V  */
        unsigned char rsvd          : 2;    /**< Reserved. Bit 3:2. Unutilized bit. Write to 0. Reads are don't care. */
        unsigned char vsys_hdrm     : 1;    /**< SYS Headroom Voltage Regulation. Bit 4.
                                                0b0 = 0.15V 0b1 = 0.20V  */
        unsigned char tj_reg        : 3;    /**< Sets the die junction temperature regulation point, TJ-REG. Bit 7:5.
                                                0b000 = 60ºC        0b001 = 70ºC 
                                                0b010 = 80ºC        0b011 = 90ºC 
                                                0b100 - 0b111 = 100ºC */
    } bits;
} reg_cnfg_chg_d_t;

/**
 * @brief CNFG_CHG_E
 * 
 * Address : 0x25
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char t_fast_chg    : 2;    /**< System voltage regulation (VSYS-REG). Bit 1:0.
                                                0b00 = timer disabled   0b01 = 3 hours 
                                                0b10 = 5 hours          0b11 = 7 hours  */
        unsigned char chg_cc        : 6;    /**< Sets the fast-charge constant current value, IFAST-CHG. Bit 7:2.
                                                0x0 = 7.5mA         0x1 = 15.0mA 
                                                0x2 = 22.5mA        ... 
                                                0x26 = 292.5mA      0x27 - 0x3F = 300.0mA */
    } bits;
} reg_cnfg_chg_e_t;

/**
 * @brief CNFG_CHG_F
 * 
 * Address : 0x25
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char               : 1;    /**< Bit 0*/
        unsigned char thm_en        : 1;    /**< Thermistor enable bit. Bit 1.
                                                0 = Thermistor is disabled 
                                                1 = Thermistor is enabled  */
        unsigned char chg_cc_jeita  : 6;    /**< Sets IFAST-CHG-JEITA for when the battery is either cool or warm as defined 
                                                by the VCOOL and VWARM temperature thresholds. Bit 7:2.
                                                0x0 = 7.5mA         0x1 = 15.0mA 
                                                0x2 = 22.5mA        ... 
                                                0x26 = 292.5mA      0x27 - 0x3F = 300.0mA */
    } bits;
} reg_cnfg_chg_f_t;

/**
 * @brief CNFG_CHG_G
 * 
 * Address : 0x26
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char rsvd      : 1;    /**< Reserved. Bit 0. Unutilized bit. Write to 0. Reads are don't care.*/
        unsigned char usbs      : 1;    /**< Setting this bit places CHGIN in USB suspend mode. Bit 1.
											0 = CHGIN is not suspended and may draw current from an adapter source 
											1 = CHGIN is suspended and may draw no current from an adapter source  */
        unsigned char chg_cv    : 6;    /**< Sets fast-charge battery regulation voltage, VFAST-CHG. Bit 7:2.
										   0x0 = 3.600V         0x1 = 3.625V 
										   0x2 = 3.650V         ... 
										   0x27 = 4.575V        0x28 - 0x3F = 4.600V */
} bits;
} reg_cnfg_chg_g_t;

/**
 * @brief CNFG_CHG_H
 * 
 * Address : 0x27
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char rsvd          : 2;    /**< Reserved. Bit 1:0. Unutilized bit. Write to 0. Reads are don't care.*/
        unsigned char chg_cv_jeita  : 6;    /**< Sets fast-charge battery regulation voltage, VFAST-CHG. Bit 7:2.
                                               0x0 = 3.600V         0x1 = 3.625V 
                                               0x2 = 3.650V         ... 
                                               0x27 = 4.575V        0x28 - 0x3F = 4.600V */
    } bits;
} reg_cnfg_chg_h_t;

/**
 * @brief CNFG_CHG_I
 * 
 * Address : 0x28
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char mux_sel               : 4;    /**< Selects the analog channel to connect to AMUX. Bit 3:0.
                                                        0b0000 = Multiplexer is disabled and AMUX is high-impedance. 
                                                        0b0001 = CHGIN voltage monitor. 
                                                        0b0010 = CHGIN current monitor. 
                                                        0b0011 = BATT voltage monitor. 
                                                        0b0100 = BATT charge current monitor. Valid only while battery charging is happening (CHG = 1). 
                                                        0b0101 = BATT discharge current monitor normal measurement. 
                                                        0b0110 = BATT discharge current monitor nulling measurement. 
                                                        0b0111 = THM voltage monitor 0b1000 = TBIAS voltage monitor 
                                                        0b1001 = AGND voltage monitor (through 100Ω pull-down resistor) 
                                                        0b1010 - 0b1111 = SYS voltage monitor */
        unsigned char imon_dischg_scale    : 4;    /**< Selects the battery discharge current full-scale current value. Bit 7:4.
                                                        0x0 = 8.2mA     0x1 = 40.5mA 0x2 = 72.3mA 
                                                        0x3 = 103.4mA   0x4 = 134.1mA 
                                                        0x5 = 164.1mA   0x6 = 193.7mA 
                                                        0x7 = 222.7mA   0x8 = 251.2mA
                                                        0x9 = 279.3mA   0xA - 0xF = 300.0mA */
    } bits;
} reg_cnfg_chg_i_t;

/**
 * @brief CNFG_SBB0_A
 * 
 * Address : 0x29
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char tv_sbb0   : 7;    /**< SIMO Buck-Boost Channel 0 Target Output Voltage. Bit 6:0.
                                            0x00 = 0.500V 0x01 = 0.525V 0x02 = 0.550V
                                            0x03 = 0.575V 0x04 = 0.600V
                                            ...
                                            0x30 = 1.700V 0x31 = 1.750V 0x32 = 1.800V
                                            ...
                                            0x7B = 5.450V 0x7C = 5.500V
                                            0x7D to 0x7F = Reserved */
        unsigned char           : 1;    /**< Bit 7. */
    } bits;
} reg_cnfg_sbb0_a_t;

/**
 * @brief CNFG_SBB0_B
 * 
 * Address : 0x2A
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char en_sbb0       : 3;    /**< Enable Control for SIMO Buck-Boost Channel 0, 
                                                selecting either an FPS slot the channel powers-up and powers-down in 
                                                or whether the channel is forced on or off. Bit 2:0.
                                                0b000 = FPS slot 0      0b001 = FPS slot 1 
                                                0b010 = FPS slot 2      0b011 = FPS slot 3      
                                                0b100 = Off irrespective of FPS 
                                                0b101 = same as 0b100   0b110 = On irrespective of FPS 
                                                0b111 = same as 0b110 */
        unsigned char ade_sbb0      : 1;    /**< SIMO Buck-Boost Channel 0 Active-Discharge Enable. Bit 3.
                                                0 = The active discharge function is disabled. 
                                                When SBB0 is disabled, its discharge rate is a function of the output capacitance and the external load. 
                                                1 = The active discharge function is enabled. 
                                                When SBB0 is disabled, an internal resistor (RAD_SBB0) is activated from SBB0 to PGND to help the output voltage discharge. */
        unsigned char               : 2;    /**< Bit 5:4*/
        unsigned char op_mode       : 1;    /**< Operation mode of SBB0. Bit 6.
                                                0 = Buck-Boost Mode 
                                                1 = Buck Mode*/
        unsigned char rsvd          : 1;    /**< Reserved. Bit 7. Unutilized bit. Write to 0. Reads are don't care.*/
    } bits;
} reg_cnfg_sbb0_b_t;

/**
 * @brief CNFG_SBB1_A
 * 
 * Address : 0x2B
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char tv_sbb1   : 7;    /**< SIMO Buck-Boost Channel 1 Target Output Voltage. Bit 6:0.
                                            0x00 = 0.500V 0x01 = 0.525V 0x02 = 0.550V
                                            0x03 = 0.575V 0x04 = 0.600V
                                            ...
                                            0x30 = 1.700V 0x31 = 1.750V 0x32 = 1.800V
                                            ...
                                            0x7B = 5.450V 0x7C = 5.500V
                                            0x7D to 0x7F = Reserved */
        unsigned char           : 1;    /**< Bit 7. */
    } bits;
} reg_cnfg_sbb1_a_t;

/**
 * @brief CNFG_SBB1_B
 * 
 * Address : 0x2C
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char en_sbb1       : 3;    /**< Enable Control for SIMO Buck-Boost Channel 1, 
                                                selecting either an FPS slot the channel powers-up and powers-down in 
                                                or whether the channel is forced on or off. Bit 2:0.
                                                0b000 = FPS slot 0      0b001 = FPS slot 1 
                                                0b010 = FPS slot 2      0b011 = FPS slot 3      
                                                0b100 = Off irrespective of FPS 
                                                0b101 = same as 0b100   0b110 = On irrespective of FPS 
                                                0b111 = same as 0b110 */
        unsigned char ade_sbb1      : 1;    /**< SIMO Buck-Boost Channel 1 Active-Discharge Enable. Bit 3.
                                                0 = The active discharge function is disabled. 
                                                When SBB0 is disabled, its discharge rate is a function of the output capacitance and the external load. 
                                                1 = The active discharge function is enabled. 
                                                When SBB0 is disabled, an internal resistor (RAD_SBB0) is activated from SBB0 to PGND to help the output voltage discharge. */
        unsigned char               : 2;    /**< Bit 5:4.*/
        unsigned char op_mode       : 1;    /**< Operation mode of SBB1. Bit 6.
                                                0 = Buck-Boost Mode 
                                                1 = Buck Mode*/
        unsigned char rsvd          : 1;    /**< Reserved. Bit 7. Unutilized bit. Write to 0. Reads are don't care.*/
    } bits;
} reg_cnfg_sbb1_b_t;

/**
 * @brief CNFG_SBB2_A
 * 
 * Address : 0x2D
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char tv_sbb2   : 7;    /**< SIMO Buck-Boost Channel 2 Target Output Voltage. Bit 6:0.
                                            0x00 = 0.500V 0x01 = 0.525V 0x02 = 0.550V
                                            0x03 = 0.575V 0x04 = 0.600V
                                            ...
                                            0x30 = 1.700V 0x31 = 1.750V 0x32 = 1.800V
                                            ...
                                            0x7B = 5.450V 0x7C = 5.500V
                                            0x7D to 0x7F = Reserved */
        unsigned char           : 1;    /**< Bit 7. */
    } bits;
} reg_cnfg_sbb2_a_t;

/**
 * @brief CNFG_SBB2_B
 * 
 * Address : 0x2E
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char en_sbb2       : 3;    /**< Enable Control for SIMO Buck-Boost Channel 2, 
                                                selecting either an FPS slot the channel powers-up and powers-down in 
                                                or whether the channel is forced on or off. Bit 2:0.
                                                0b000 = FPS slot 0      0b001 = FPS slot 1 
                                                0b010 = FPS slot 2      0b011 = FPS slot 3      
                                                0b100 = Off irrespective of FPS 
                                                0b101 = same as 0b100   0b110 = On irrespective of FPS 
                                                0b111 = same as 0b110 */
        unsigned char ade_sbb2      : 1;    /**< SIMO Buck-Boost Channel 2 Active-Discharge Enable Bit 3.
                                                0 = The active discharge function is disabled. 
                                                When SBB0 is disabled, its discharge rate is a function of the output capacitance and the external load. 
                                                1 = The active discharge function is enabled. 
                                                When SBB0 is disabled, an internal resistor (RAD_SBB0) is activated from SBB0 to PGND to help the output voltage discharge. */
        unsigned char               : 2;    /**< Bit 5:4. */
        unsigned char op_mode       : 1;    /**< Operation mode of SBB2. Bit 6.
                                                0 = Buck-Boost Mode 
                                                1 = Buck Mode*/
        unsigned char rsvd          : 1;    /**< Reserved. Bit 7. Unutilized bit. Write to 0. Reads are don't care.*/
    } bits;
} reg_cnfg_sbb2_b_t;

/**
 * @brief CNFG_SBB_TOP
 * 
 * Address : 0x2F
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char drv_sbb       : 2;    /**< SIMO Buck-Boost (all channels) Drive Strength Trim. Bit 1:0.
										        0b00 = fastest transition time 
												0b01 = a little slower than 0b00 
												0b10 = a little slower than 0b01 
												0b11 = a little slower than 0b10 */
        unsigned char               : 5;    /**< Bit 6:2. */
        unsigned char op_mode_chg   : 1;    /**< Operation mode of the charging channel of SIMO. Bit 7.
                                                0 = Buck-boost mode
                                                1 = Buck mode */
    } bits;
} reg_cnfg_sbb_top_t;

/**
 * @brief CNFG_SBB_TOP_B
 * 
 * Address : 0x30
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char ip_sbb0       : 2;    /**< SIMO Buck-Boost Channel 0 Peak Current Limit. Bit 1:0.
										        0b00 = 1.000A
                                                0b01 = 0.750A
                                                0b10 = 0.500A
                                                0b11 = 0.333A  */
        unsigned char ip_sbb1       : 2;    /**< SIMO Buck-Boost Channel 1 Peak Current Limit. Bit 3:2.
										        0b00 = 1.000A
                                                0b01 = 0.750A
                                                0b10 = 0.500A
                                                0b11 = 0.333A  */
        unsigned char ip_sbb2       : 2;    /**< SIMO Buck-Boost Channel 2 Peak Current Limit. Bit 5:4.
										        0b00 = 1.000A
                                                0b01 = 0.750A
                                                0b10 = 0.500A
                                                0b11 = 0.333A  */
        unsigned char ip_chg        : 2;    /**< SIMO Buck-Boost Charging Channel Peak Current Limit. Bit 7:6.
										        0b00 = 2.000A
                                                0b01 = 1.500A
                                                0b10 = 1.000A
                                                0b11 = 0.500A  */
    } bits;
} reg_cnfg_sbb_top_b_t;

/**
 * @brief CNFG_LDO0_A
 * 
 * Address : 0x38
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char tv_ldo_volt       : 7;    /**< LDO Target Output Voltage This 7-bit configuration is a linear transfer function 
                                                    that starts at 0.5V and ends at 3.675V, with 25mV increments. Bit 6:0.
                                                    0x00 = 0.500V       0x01 = 0.525V 
                                                    0x02 = 0.550V       0x03 = 0.575V
                                                    0x04 = 0.600V       0x05 = 0.625V 
                                                    0x06 = 0.650V       ... 
                                                    0x7D = 3.625V       0x7E = 3.650V 
                                                    0x7F = 3.675V */
        unsigned char tv_ldo_offset     : 1;    /**< LDO Output Voltage. This bit applies a 1.325V offset to the output voltage of the LDO. Bit 7.
                                                    0b0 = No offset 0b1 = 1.325V offset*/
    } bits;
} reg_cnfg_ldo0_a_t;

/**
 * @brief CNFG_LDO0_B
 * 
 * Address : 0x39
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char en_ldo    : 3;    /**< Enable Control for LDO0, selecting either an FPS slot the channel powers-up and 
                                            powers-down in or whether the channel is forced on or off. Bit 2:0.
                                            0b000 = FPS slot 0      0b001 = FPS slot 1 
                                            0b010 = FPS slot 2      0b011 = FPS slot 3 
                                            0b100 = Off irrespective of FPS 
                                            0b101 = same as 0b100 
                                            0b110 = On irrespective of FPS
                                            0b111 = same as 0b110 */
        unsigned char ade_ldo   : 1;    /**< LDO0 Active-Discharge Enable. Bit 3.
                                            0 = The active discharge function is disabled. 
                                            1 = The active discharge function is enabled. */
        unsigned char ldo_md    : 1;    /**< Operation mode of LDO0. Bit 4.
                                            0 = Low Dropout Linear Regulator (LDO) Mode 
                                            1 = Load Switch (LSW) Mode */
        unsigned char rsvd      : 3;    /**< Bit 7:5. Reserved. Unutilized bit. Write to 0. Reads are don't care. */
    } bits;
} reg_cnfg_ldo0_b_t;

#endif /* MAX77659_REGS_H_ */
