/*******************************************************************************
 * Copyright(C) Analog Devices Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files(the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Analog Devices Inc.
 * shall not be used except as stated in the Analog Devices Inc.
 * Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Analog Devices Inc.retains all ownership rights.
 *******************************************************************************
 */

#ifndef MAX77658_REGS_H_
#define MAX77658_REGS_H_

/**
 * @brief INT_GLBL0 Register
 *
 * Address : 0x00
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char gpi0_f    : 1;    /**< GPI Falling Interrupt. Bit 0.
                                             Note that "GPI" refers to the GPIO programmed to be an input.
                                             0 = No GPI falling edges have occurred since the last time this bit was read.
                                             1 = A GPI falling edge has occurred since the last time this bit was read. */
        unsigned char gpi0_r    : 1;    /**< GPI Rising Interrupt. Bit 1. 
                                             Note that "GPI" refers to the GPIO programmed to be an input.
                                             0 = No GPI rising edges have occurred since the last time this bit was read. 
                                             1 = A GPI rising edge has occurred since the last time this bit was read. */
        unsigned char nen_f     : 1;    /**< nEN Falling Interrupt.Bit 2.
                                             0 = No nEN falling edges have occurred since the last time this bit was read.
                                             1 = A nEN falling edge as occurred since the last time this bit was read. */
        unsigned char nen_r     : 1;    /**< nEN Rising Interrupt. Bit 3.
                                             0 = No nEN rising edges have occurred since the last time this bit was read.
                                             1 = A nEN rising edge as occurred since the last time this bit was read. */
        unsigned char tjal1_r   : 1;    /**< Thermal Alarm 1 Rising Interrupt. Bit 4.
                                             0 = The junction temperature has not risen above TJAL1 since the last time this bit was read.
                                             1 = The junction temperature has risen above TJAL1 since the last time this bit was read. */
        unsigned char tjal2_r   : 1;    /**< Thermal Alarm 2 Rising Interrupt. Bit 5.
                                             0 = The junction temperature has not risen above TJAL2 since the last time this bit was read.
                                             1 = The junction temperature has risen above TJAL2 since the last time this bit was read. */
        unsigned char d0d1_r    : 1;    /**< LDO Dropout Detector Rising Interrupt. Bit 6.
                                             0 = The LDO has not detected dropout since the last time this bit was read.
                                             1 = The LDO has detected dropout since the last time this bit was read.  */
        unsigned char dod0_r    : 1;    /**< LDO Dropout Detector Rising Interrupt. Bit 7.
                                             0 = The LDO has not detected dropout since the last time this bit was read.
                                             1 = The LDO has detected dropout since the last time this bit was read. */
    } bits;
} reg_int_glbl0_t;

/**
 * @brief INT_GLBL1 Register
 *
 * Address : 0x04
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char gpi1_f    : 1;    /**< GPI Falling Interrupt. Bit 0.
                                             Note that "GPI" refers to the GPIO programmed to be an input.
                                             0 = No GPI falling edges have occurred since the last time this bit was read.
                                             1 = A GPI falling edge has occurred since the last time this bit was read. */
        unsigned char gpi1_r    : 1;    /**< GPI Rising Interrupt. Bit 1.
                                             Note that "GPI" refers to the GPIO programmed to be an input.
                                             0 = No GPI rising edges have occurred since the last time this bit was read. 
                                             1 = A GPI rising edge has occurred since the last time this bit was read. */
        unsigned char sbb0_f    : 1;    /**< SBB0 Fault Indicator. Bit 2.
                                             0 = No fault has occurred on SBB0 since the last time this bit was read.
                                             1 = SBB0 has fallen out of regulation since the last time this bit was read. */
		unsigned char sbb1_f    : 1;    /**< SBB1 Fault Indicator. Bit 3.
                                             0 = No fault has occurred on SBB1 since the last time this bit was read.
                                             1 = SBB1 has fallen out of regulation since the last time this bit was read. */
		unsigned char sbb2_f    : 1;    /**< SBB2 Fault Indicator. Bit 4.
                                             0 = No fault has occurred on SBB2 since the last time this bit was read.
                                             1 = SBB2 has fallen out of regulation since the last time this bit was read. */
        unsigned char ldo0_f    : 1;    /**< LDO0 Fault Interrupt. Bit 5.
                                             0 = No fault has occurred on LDO0 since the last time this bit was read.
                                             1 = LDO0 has fallen out of regulation since the last time this bit was read. */
        unsigned char ldo1_f    : 1;    /**< LDO1 Fault Interrupt. Bit 6.
                                             0 = No fault has occurred on LDO1 since the last time this bit was read.
                                             1 = LDO1 has fallen out of regulation since the last time this bit was read.  */
        unsigned char rsvd      : 1;    /**< Reserved.
                                             Unutilized bit. Write to 0. Reads are don't care. Bit 7. */
    } bits;
} reg_int_glbl1_t;

/**
 * @brief ERCFLAG Register
 *
 * Address : 0x05
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char tovld     : 1;    /**< Thermal Overload. Bit 0.
                                             0 = Thermal overload has not occurred since the last read of this register.
                                             1 = Thermal overload has occurred since the list read of this register.
                                             This indicates that the junction temperature has exceeded 165ºC. */
        unsigned char sysovlo   : 1;    /**< SYS Domain Overvoltage Lockout. Bit 1.
                                             0 = The SYS domain overvoltage lockout has not occurred since this last read of this register.
                                             1 = The SYS domain overvoltage lockout has occurred since the last read of this register. */
        unsigned char sysuvlo   : 1;    /**< SYS Domain Undervoltage Lockout. Bit 2.
                                             0 = The SYS domain undervoltage lockout has not occurred since this last read of this register.
                                             1 = The SYS domain undervoltage lockout has occurred since the last read of this register. */
        unsigned char mrst_f    : 1;    /**< Manual Reset Timer. Bit 3.
                                             0 = A Manual Reset has not occurred since this last read of this register.
                                             1 = A Manual Reset has occurred since this last read of this register. */
        unsigned char sft_off_f : 1;    /**< Software Off Flag. Bit 4.
                                             0 = The SFT_OFF function has not occurred since the last read of this register.
                                             1 = The SFT_OFF function has occurred since the last read of this register. */
        unsigned char sft_crst_f: 1;    /**< Software Cold Reset Flag. Bit 5.
                                             0 = The software cold reset has not occurred since the last read of this register.
                                             1 = The software cold reset has occurred since the last read of this register. */
        unsigned char wdt_off   : 1;    /**< Watchdog Timer OFF Flag. Bit 6.
                                             This bit sets when the watchdog timer expires and causes a power-off.
                                             0 = Watchdog timer has not caused a power-off since the last time this bit was read.
                                             1 = Watchdog timer has expired and caused a power-off since the last time this bit was read.  */
        unsigned char wdt_rst   : 1;    /**< Watchdog Timer Reset Flag. Bit 7.
                                             This bit sets when the watchdog timer expires and causes a power-reset. 
                                             0 = Watchdog timer has not caused a power-reset since the last time this bit was read. 
                                             1 = Watchdog timer has expired and caused a power-reset since the last time this bit was read.*/
    } bits;
} reg_ercflag_t;

/**
 * @brief STAT_GLBL Register
 *
 * Address : 0x06
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char stat_irq  : 1;    /**< Software Version of the nIRQ MOSFET gate drive. Bit 0.
                                             0 = unmasked gate drive is logic low 
                                             1 = unmasked gate drive is logic high */
        unsigned char stat_en   : 1;    /**< Debounced Status for the nEN input. Bit 1.
                                             0 = nEN is not active (logic high) 
                                             1 = nEN is active (logic low) */
        unsigned char tjal1_s   : 1;    /**< Thermal Alarm 1 Status. Bit 2.
                                             0 = The junction temperature is less than TJAL1 
                                             1 = The junction temperature is greater than TJAL1 */
        unsigned char tjal2_s   : 1;    /**< Thermal Alarm 2 Status. Bit 3.
                                             0 = The junction temperature is less than TJAL2 
                                             1 = The junction temperature is greater than TJAL2 */
        unsigned char dod1_s    : 1;    /**< LDO1 Dropout Detector Rising Status. Bit 4.
                                             0 = LDO1 is not in dropout 
                                             1 = LDO1 is in dropout */
        unsigned char dod0_s    : 1;    /**< LDO0 Dropout Detector Rising Status. Bit 5.
                                             0 = LDO0 is not in dropout 
                                             1 = LDO0 is in dropout */
        unsigned char bok       : 1;    /**< BOK Interrupt Status. Bit 6.
                                             0 = Main Bias is not ready. 
                                             1 = Main Bias enabled and ready.  */
        unsigned char didm      : 1;    /**< Device Identification Bits for Metal Options. Bit 7.
                                             0 = MAX77658 
                                             1 = Reserved */
    } bits;
} reg_stat_glbl_t;

/**
 * @brief INTM_GLBL0 Register
 *
 * Address : 0x08
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char gpi0_fm   : 1;    /**< GPI Falling Interrupt Mask. Bit 0. 
                                             0 = Unmasked. If GPI_F goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to GPI_F. */
        unsigned char gpi0_rm   : 1;    /**< GPI Rising Interrupt Mask. Bit 1. 
                                             0 = Unmasked. If GPI_R goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to GPI_R. */
        unsigned char nen_fm    : 1;    /**< nEN Falling Interrupt Mask. Bit 2.
                                             0 = Unmasked. If nEN_F goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to nEN_F. */
        unsigned char nen_rm    : 1;    /**< nEN Rising Interrupt Mask. Bit 3.
                                             0 = Unmasked. If nEN_R goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to nEN_R. */
        unsigned char tjal1_rm  : 1;    /**< Thermal Alarm 1 Rising Interrupt Mask. Bit 4.
                                             0 = Unmasked. If TJAL1_R goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to TJAL1_R. */
        unsigned char tjal2_rm  : 1;    /**< Thermal Alarm 2 Rising Interrupt Mask. Bit 5.
                                             0 = Unmasked. If TJAL2_R goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to TJAL2_R. */
        unsigned char dod1_rm   : 1;    /**< LDO Dropout Detector Rising Interrupt Mask. Bit 6.
                                             0 = Unmasked. If DOD1_R goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to DOD1_R.  */
        unsigned char dod0_rm   : 1;    /**< LDO Dropout Detector Rising Interrupt Mask. Bit 7.
                                             0 = Unmasked. If DOD0_R goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to DOD0_R. */
    } bits;
} reg_intm_glbl0_t;

/**
 * @brief INTM_GLBL1 Register
 *
 * Address : 0x09
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char gpi1_fm   : 1;    /**< GPI Falling Interrupt Mask. Bit 0. 
                                             0 = Unmasked. If GPI_F goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to GPI_F. */
        unsigned char gpi1_rm   : 1;    /**< GPI Rising Interrupt Mask. Bit 1. 
                                             0 = Unmasked. If GPI_R goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to GPI_R. */
        unsigned char sbb0_fm  : 1;    /**< SBB0 Fault Interrupt Mask. Bit 2.
                                             0 = Unmasked. If SBB0_F goes from 0 to 1, then nIRQ goes low. 
											 nIRQ goes high when all interrupt bits are cleared..
                                             1 = Masked. nIRQ does not go low due to SBB0_F. */
        unsigned char sbb1_fm  : 1;    /**< SBB1 Fault Interrupt Mask. Bit 3.
                                             0 = Unmasked. If SBB1_F goes from 0 to 1, then nIRQ goes low. 
											 nIRQ goes high when all interrupt bits are cleared..
                                             1 = Masked. nIRQ does not go low due to SBB1_F. */
        unsigned char sbb2_fm  : 1;    /**< SBB2 Fault Interrupt Mask. Bit 4.
                                             0 = Unmasked. If SBB2_F goes from 0 to 1, then nIRQ goes low. 
											 nIRQ goes high when all interrupt bits are cleared..
                                             1 = Masked. nIRQ does not go low due to SBB2_F. */
        unsigned char ldo0_m    : 1;    /**< LDO0 Fault Interrupt. Bit 5.
                                             0 = Unmasked. If LDO0_F goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared.
                                             1 = Masked. nIRQ does not go low due to LDO0_F. */
        unsigned char ldo1_m    : 1;    /**< LDO1 Fault Interrupt Mask. Bit 6.
                                             0 = Unmasked. If LDO1_F goes from 0 to 1, then nIRQ goes low. 
                                             nIRQ goes high when all interrupt bits are cleared. 
                                             1 = Masked. nIRQ does not go low due to LDO1_F.  */
        unsigned char rsvd      : 1;    /**< Reserved. 
                                             Unutilized bit. Write to 0. Reads are don't care. Bit 7. */
    } bits;
} reg_intm_glbl1_t;

/**
 * @brief CNFG_GLBL Register
 *
 * Address : 0x10
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char sft_ctrl  : 2;    /**< Software Reset Functions. Bit 1:0. 
                                             0b00 = No Action 
                                             0b01 = Software Cold Reset (SFT_CRST). The device powers down, resets, and the powers up again. 
                                             0b10 = Software Off (SFT_OFF). The device powers down, resets, and then remains off and waiting for a wake-up event. 
                                             0b11 = Factory-Ship Mode Enter (FSM). */
        unsigned char dben_nen  : 1;    /**< Debounce Timer Enable for the nEN Pin. Bit 2.
                                             0 = 500μs Debounce 
                                             1 = 30ms Debounce */
        unsigned char nen_mode  : 2;    /**< nEN Input (ON-KEY) Default Configuration Mode. Bit 4:3.
                                             0b00 = Push-button mode 
											 0b01 = Slide-switch mode 
											 0b10 = Logic mode 
											 0b11 = Reserved */
        unsigned char sbia_lpm  : 1;    /**< Main Bias Low-Power Mode Software Request. Bit 5.
                                             0 = Main Bias requested to be in Normal-Power Mode by software. 
                                             1 = Main Bias request to be in Low-Power Mode by software. */
        unsigned char t_mrst    : 1;    /**< Sets the Manual Reset Time (tMRST). Bit 6.
                                             0 = 8s 
                                             1 = 4s  */
        unsigned char pu_dis    : 1;    /**< nEN Internal Pullup Resistor. Bit 7.
                                             0 = Strong internal nEN pullup (200kΩ) 
                                             1 = Weak internal nEN pullup (10MΩ) */
    } bits;
} reg_cnfg_glbl_t;

/**
 * @brief CNFG_GPIO0 Register
 *
 * Address : 0x11
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char gpo_dir   : 1;    /**< GPIO Direction. Bit 0. 
                                             0 = General purpose output (GPO) 
                                             1 = General purpose input (GPI) */
        unsigned char gpo_di    : 1;    /**< GPIO Digital Input Value. Bit 1.
                                             0 = Input logic low 
                                             1 = Input logic high */
        unsigned char gpo_drv   : 1;    /**< General Purpose Output Driver Type. Bit 2.
                                             This bit is a don't care when DIR = 1 (configured as input) When set for GPO (DIR = 0): 
                                             0 = Open-Drain 
                                             1 = Push-Pull */
        unsigned char gpo_do    : 1;    /**< General Purpose Output Data Output. Bit 3.
                                             This bit is a don't care when DIR = 1 (configured as input). When set for GPO (DIR = 0): 
                                             0 = GPIO is output is logic low 
                                             1 = GPIO is output logic high when set as push-pull output (DRV = 1). */
        unsigned char dben_gpi  : 1;    /**< General Purpose Input Debounce Timer Enable. Bit 4.
                                             0 = no debounce 
                                             1 = 30ms debounce */
        unsigned char alt_gpio  : 1;    /**< Alternate Mode Enable for GPIO0. Bit 5.
                                             0 = Standard GPIO. 
                                             1 = Active-high input, Force USB Suspend (FUS). FUS is only active if the FUS_M bit is set to 0.  */
        unsigned char           : 1;    /**< Bit 6. */
        unsigned char rsvd      : 1;    /**< Reserved. Bit 7.
                                             Unutilized bit. Write to 0. Reads are don't care. */
    } bits;
} reg_cnfg_gpio0_t;

/**
 * @brief CNFG_GPIO1 Register
 *
 * Address : 0x12
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char gpo_dir   : 1;    /**< GPIO Direction. Bit 0. 
                                             0 = General purpose output (GPO) 
                                             1 = General purpose input (GPI) */
        unsigned char gpo_di    : 1;    /**< GPIO Digital Input Value. Bit 1.
                                             0 = Input logic low 
                                             1 = Input logic high */
        unsigned char gpo_drv   : 1;    /**< General Purpose Output Driver Type. Bit 2.
                                             This bit is a don't care when DIR = 1 (configured as input) When set for GPO (DIR = 0): 
                                             0 = Open-Drain 
                                             1 = Push-Pull */
        unsigned char gpo_do    : 1;    /**< General Purpose Output Data Output. Bit 3.
                                             This bit is a don't care when DIR = 1 (configured as input). When set for GPO (DIR = 0): 
                                             0 = GPIO is output is logic low 
                                             1 = GPIO is output logic high when set as push-pull output (DRV = 1). */
        unsigned char dben_gpi  : 1;    /**< General Purpose Input Debounce Timer Enable. Bit 4.
                                             0 = no debounce 
                                             1 = 30ms debounce */
        unsigned char alt_gpio  : 1;    /**< Alternate Mode Enable for GPIO1. Bit 5.
                                             0 = Standard GPIO. 
                                             1 = Active-high output of SBB2's Flexible Power Sequencer (FPS) slot. */
        unsigned char rsvd      : 2;    /**< Reserved. Bit 7:6.
                                             Unutilized bit. Write to 0. Reads are don't care. */
    } bits;
} reg_cnfg_gpio1_t;

/**
 * @brief CNFG_GPIO2 Register
 *
 * Address : 0x13
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char gpo_dir   : 1;    /**< GPIO Direction. Bit 0. 
                                             0 = General purpose output (GPO) 
                                             1 = General purpose input (GPI) */
        unsigned char gpo_di    : 1;    /**< GPIO Digital Input Value. Bit 1.
                                             0 = Input logic low 
                                             1 = Input logic high */
        unsigned char gpo_drv   : 1;    /**< General Purpose Output Driver Type. Bit 2.
                                             This bit is a don't care when DIR = 1 (configured as input) When set for GPO (DIR = 0): 
                                             0 = Open-Drain 
                                             1 = Push-Pull */
        unsigned char gpo_do    : 1;    /**< General Purpose Output Data Output. Bit 3.
                                             This bit is a don't care when DIR = 1 (configured as input). When set for GPO (DIR = 0): 
                                             0 = GPIO is output is logic low 
                                             1 = GPIO is output logic high when set as push-pull output (DRV = 1). */
        unsigned char dben_gpi  : 1;    /**< General Purpose Input Debounce Timer Enable. Bit 4.
                                             0 = no debounce 
                                             1 = 30ms debounce */
        unsigned char alt_gpio  : 1;    /**< Alternate Mode Enable for GPIO2. Bit 5.
                                             0 = Standard GPIO. 
                                             1 = Active-high input, Enable DISQBAT. */
        unsigned char rsvd      : 2;    /**< Reserved. Bit 7:6.
                                             Unutilized bit. Write to 0. Reads are don't care. */
    } bits;
} reg_cnfg_gpio2_t;

/**
 * @brief CID Register
 *
 * Address : 0x14
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char cid   : 4;    /**< Chip Identification Code. Bit 4:0.
                                    The Chip Identification Code refers to a set of reset values in the register map, or the "OTP configuration.". */
        unsigned char       : 4;    /**< Bit 7:4. */
    } bits;
} reg_cid_t;

/**
 * @brief CNFG_WDT Register
 *
 * Address : 0x17
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char wdt_lock  : 1;    /**< Factory-Set Safety Bit for the Watchdog Timer. Bit 0. 
                                             0 = Watchdog timer can be enabled and disabled with WDT_EN. 
                                             1 = Watchdog timer can not be disabled with WDT_EN. 
                                             However, WDT_EN can still be used to enable the watchdog timer. */
        unsigned char wdt_en    : 1;    /**< Watchdog Timer Enable. Bit 1.
                                             0 = Watchdog timer is not enabled. 
                                             1 = Watchdog timer is enabled. The timer will expire if not reset by setting WDT_CLR. */
        unsigned char wdt_clr   : 1;    /**< Watchdog Timer Clear Control. Bit 2.
                                             0 = Watchdog timer period is not reset. 
                                             1 = Watchdog timer is reset back to tWD. */
        unsigned char wdt_mode  : 1;    /**< Watchdog Timer Expired Action. Bit 3.
                                             0 = Watchdog timer expire causes power-off. 
                                             1 = Watchdog timer expire causes power-reset. */
        unsigned char wdt_per   : 2;    /**< Watchdog Timer Period. Bit 5:4.
                                             0b00 = 16 seconds      0b01 = 32 seconds 
                                             0b10 = 64 seconds      0b11 = 128 seconds. */
        unsigned char rsvd      : 2;    /**< Reserved. Bit 7:6.
                                             Unutilized bit. Write to 0. Reads are don't care. */
    } bits;
} reg_cnfg_wdt_t;

/**
 * @brief INT_CHG Register
 *
 * Address : 0x01
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char thm_i         : 1;    /**< Thermistor related interrupt. Bit 0. 
                                             	 0 = The bits in THM_DTLS[2:0] have not changed since the last time this bit was read
                                             	 1 = The bits in THM_DTLS[2:0] have changed since the last time this bit was read */
        unsigned char chg_i         : 1;    /**< Charger related interrupt. Bit 1. 
                                             	 0 = The bits in CHG_DTLS[3:0] have not changed since the last time this bit was read
                                             	 1 = The bits in CHG_DTLS[3:0] have changed since the last time this bit was read */
        unsigned char chgin_i       : 1;    /**< CHGIN related interrupt. Bit 2.
                                             	 0 = The bits in CHGIN_DTLS[1:0] have not changed since the last time this bit was read
                                             	 1 = The bits in CHGIN_DTLS[1:0] have changed since the last time this bit was read */
        unsigned char tj_reg_i      : 1;    /**< Die junction temperature regulation interrupt. Bit 3.
                                             	 0 = The die temperature has not exceeded TJ-REG since the last time this bit was read
                                             	 1 = The die temperature has exceeded TJ-REG since the last time this bit was read */
        unsigned char chgin_ctrl_i  : 1;    /**< CHGIN control-loop related interrupt. Bit 4.
                                             	 0 = Neither the VCHGIN_MIN_STAT nor the ICHGIN_LIM_STAT bits have changed since the last time this bit was read
                                             	 1 = The VCHGIN_MIN_STAT or ICHGIN_LIM_STAT bits have changed since the last time this bit was read */
        unsigned char sys_ctrl_i    : 1;    /**< Minimum System Voltage Regulation-loop related interrupt. Bit 5.
                                             	 0 = The minimum system voltage regulation loop has not engaged since the last time this bit was read
                                             	 1 = The minimum system voltage regulation loop has engaged since the last time this bit was read */
        unsigned char sys_cnfg_i    : 1;    /**< System voltage configuration error interrupt. Bit 6.
                                             	 0 = The bit combination in CHG_CV has not been forced to change since the last time this bit was read
                                             	 1 = The bit combination in CHG_CV has been forced to change to ensure VSYS-REG = VFAST-CHG + 200mV
                                             	 since the last time this bit was read  */
        unsigned char rsvd          : 1;    /**< Reserved. Bit 7.
                                             	 Unutilized bit. Write to 0. Reads are don't care. */
    } bits;
} reg_int_chg_t;

/**
 * @brief STAT_CHG_A
 * 
 * Address : 0x02
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char thm_dtls          : 3;    /**< Battery Temperature Details. Bit 2:0.
                                                    0b000 = Thermistor is disabled (THM_EN = 0) 
                                                    0b001 = Battery is cold as programmed by THM_COLD[1:0] If thermistor and charger are enabled while the battery is cold, a battery temperature fault will occur. 
                                                    0b010 = Battery is cool as programmed by THM_COOL[1:0] 
                                                    0b011 = Battery is warm as programmed by THM_WARM[1:0] 
                                                    0b100 = Battery is hot as programmed by THM_HOT[1:0]. If thermistor and charger are enabled while the battery is hot, a battery temperature fault will occur. 
                                                    0b101 = Battery is in the normal temperature region 
                                                    0b110 - 0b111 = reserved */
        unsigned char tj_reg_stat       : 1;    /**< Maximum Junction Temperature Regulation Loop Status. Bit 3.
                                                    0 = The maximum junction temperature regulation loop is not engaged 
                                                    1 = The maximum junction temperature regulation loop has engaged to regulate the junction temperature to less than TJ-REG */
        unsigned char vsys_min_stat     : 1;    /**< Minimum System Voltage Regulation Loop Status. Bit 4.
                                                    0 = The minimum system voltage regulation loop is not enganged 
                                                    1 = The minimum system voltage regulation loop is engaged to regulate VSYS ≥ VSYS-MIN */
        unsigned char ichgin_lim_stat   : 1;    /**< Input Current Limit Loop Status. Bit 5.
                                                    0 = The CHGIN current limit loop is not engaged 
                                                    1 = The CHGIN current limit loop has engaged to regulate ICHGIN ≤ ICHGIN-LIM */
        unsigned char vchgin_min_stat   : 1;    /**< Minimum Input Voltage Regulation Loop Status. Bit 6.
                                                    0 = The minimum CHGIN voltage regulation loop is not engaged 
                                                    1 = The minimum CHGIN voltage regulation loop has engaged to regulate VCHGIN ≥ VCHGIN-MIN  */
        unsigned char rsvd              : 1;    /**< Reserved. Bit 7.
                                                    Unutilized bit. Write to 0. Reads are don't care. */
    } bits;
} reg_stat_chg_a_t;

/**
 * @brief STAT_CHG_B
 * 
 * Address : 0x03
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char time_sus      : 1;    /**< Time Suspend Indicator. Bit 0.
                                                0 = The charger's timers are either not active, or not suspended 
                                                1 = The charger's active timer is suspended due to one of three reasons: 
                                                charge current dropped below 20% of IFAST-CHG while the charger state machine is in FAST CHARGE CC mode, 
                                                the charger is in SUPPLEMENT mode, or the charger state machine is in BATTERY TEMPERATURE FAULT mode. */
        unsigned char chg           : 1;    /**< Quick Charger Status. Bit 1.
                                                0 = Charging is not happening 
                                                1 = Charging is happening */
        unsigned char chgin_dtls    : 2;    /**< CHGIN Status Detail. Bit 3:2.
                                                0b00 = The CHGIN input voltage is below the UVLO threshold (VCHGIN < VUVLO) 
                                                0b01 = The CHGIN input voltage is above the OVP threshold (VCHGIN > VOVP) 
                                                0b10 = The CHGIN input is being debounced (no power accepted from CHGIN during debounce) 
                                                0b11 = The CHGIN input is okay and debounced  */
        unsigned char chg_dtls      : 4;    /**< Charger Details. Bit 7:4.
                                                0b0000 = Off 
                                                0b0001 = Prequalification mode 
                                                0b0010 = Fast-charge constant-current (CC) mode 
                                                0b0011 = JEITA modified fast-charge constant-current mode 
                                                0b0100 = Fast-charge constant-voltage (CV) mode 
                                                0b0101 = JEITA modified fast-charge constant-voltage mode 
                                                0b0110 = Top-off mode 
                                                0b0111 = JEITA modified top-off mode 
                                                0b1000 = Done 
                                                0b1001 = JEITA modified done (done was entered through the JEITA-modified fast-charge states) 
                                                0b1010 = Prequalification timer fault 
                                                0b1011 = Fast-charge timer fault 
                                                0b1100 = Battery temperature fault 
                                                0b1101 - 0b1111 = reserved */
    } bits;
} reg_stat_chg_b_t;

/**
 * @brief INT_M_CHG Register
 *
 * Address : 0x07
 */
typedef union {
    unsigned char raw;
    struct {
        unsigned char thm_m         : 1;    /**< Setting this bit prevents the THM_I bit from causing hardware IRQs. Bit 0. 
                                             	 0 = THM_I is not masked
                                             	 1 = THM_I is masked */
        unsigned char chg_m         : 1;    /**< Setting this bit prevents the CHG_I bit from causing hardware IRQs. Bit 1. 
                                             	 0 = CHG_I is not masked
                                             	 1 = CHG_I is masked */
        unsigned char chgin_m       : 1;    /**< Setting this bit prevents the CHGIN_I bit from causing hardware IRQs. Bit 2.
                                             	 0 = CHGIN_I is not masked
                                             	 1 = CHGIN_I is masked */
        unsigned char tj_reg_m      : 1;    /**< Setting this bit prevents the TJREG_I bit from causing hardware IRQs. Bit 3.
                                             	 0 = TJREG_I is not masked
                                             	 1 = TJREG_I is masked */
        unsigned char chgin_ctrl_m  : 1;    /**< Setting this bit prevents the CHGIN_CTRL_I bit from causing hardware IRQs. Bit 4.
                                             	 0 = CHGIN_CTRL_I is not masked
                                             	 1 = CHGIN_CTRL_I is masked */
        unsigned char sys_ctrl_m    : 1;    /**< Setting this bit prevents the SYS_CTRL_I bit from causing hardware IRQs. Bit 5.
                                             	 0 = SYS_CTRL_I is not masked
                                             	 1 = SYS_CTRL_I is masked */
        unsigned char sys_cnfg_m    : 1;    /**< Setting this bit prevents the SYS_CNFG_I bit from causing hardware IRQs. Bit 6.
                                             	 0 = SYS_CNFG_I is not masked
                                             	 1 = SYS_CNFG_I is masked
                                             	 since the last time this bit was read  */
        unsigned char dis_aicl          : 1;    /**< Active input current loop. Bit 7.
                                             	 0 = Active Input Current Loop Active.
												 1 = Active Input Current Loop Disabled. */
    } bits;
} reg_int_m_chg_t;

/**
 * @brief CNFG_CHG_A
 * 
 * Address : 0x20
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char thm_cold  : 2;    /**< Sets the TCOLD JEITA Temperature Threshold. Bit 1:0.
                                            0b00 = TCOLD = -10ºC 
											0b01 = TCOLD = -5ºC 
											0b10 = TCOLD = 0ºC 
											0b11 = TCOLD = 5ºC */
        unsigned char thm_cool  : 2;    /**< Sets the TCOOL JEITA Temperature Threshold. Bit 3:2.
                                            0b00 = TCOOL = 0ºC 
											0b01 = TCOOL = 5ºC 
											0b10 = TCOOL = 10ºC 
											0b11 = TCOOL = 15ºC */
        unsigned char thm_warm  : 2;    /**< Sets the TWARM JEITA Temperature Threshold. Bit 5:4.
                                            0b00 = TWARM = 35ºC 
											0b01 = TWARM = 40ºC 
											0b10 = TWARM = 45ºC 
											0b11 = TWARM = 50ºC  */
        unsigned char thm_hot   : 2;    /**< Sets the THOT JEITA Temperature Threshold. Bit 7:6.
                                            0b00 = THOT = 45ºC 
											0b01 = THOT = 50ºC 
											0b10 = THOT = 55ºC 
											0b11 = THOT = 60ºC */
    } bits;
} reg_cnfg_chg_a_t;

/**
 * @brief CNFG_CHG_B
 * 
 * Address : 0x21
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char chg_en        : 1;    /**< Charger Enable. Bit 0. Default value defined by OTP bit OTP_CHG_EN:
                                                0 = the battery charger is disabled 
                                                1 = the battery charger is enabled */
        unsigned char i_pq          : 1;    /**< Sets the prequalification charge current (IPQ) as a percentage of IFAST-CHG. Bit 1.
                                                0 = 10%     1 = 20% */
        unsigned char ichgin_lim    : 3;    /**< CHGIN Input Current Limit (ICHGIN-LIM). Bit 4:2. When CNFG_SBB_TOP.ICHGIN_LIM_DEF = 0:
                                                0b000 = 95mA    0b001 = 190mA 
                                                0b010 = 285mA   0b011 = 380mA 
                                                0b100 = 475mA   0b101 0b111 = Reserved.  */
        unsigned char vchgin_min    : 3;    /**< Minimum CHGIN regulation voltage (VCHGIN-MIN). Bit 7:5.
                                                0b000 = 4.0V    0b001 = 4.1V 
                                                0b010 = 4.2V    0b011 = 4.3V 
                                                0b100 = 4.4V    0b101 = 4.5V 
                                                0b110 = 4.6V    0b111 = 4.7V */
    } bits;
} reg_cnfg_chg_b_t;

/**
 * @brief CNFG_CHG_C
 * 
 * Address : 0x22
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char t_topoff  : 3;    /**< Top-off timer value (tTO). Bit 2:0.
                                            0b000 = 0 minutes       0b001 = 5 minutes 
                                            0b010 = 10 minutes      0b011 = 15 minutes 
                                            0b100 = 20 minutes      0b101 = 25 minutes 
                                            0b110 = 30 minutes      0b111 = 35 minutes */
        unsigned char i_term    : 2;    /**< Charger Termination Current (ITERM). Bit 4:3.
                                            00 = 5%     01 = 7.5% 
                                            10 = 10%    11 = 15%  */
        unsigned char chg_pq    : 3;    /**< Battery prequalification voltage threshold (VPQ). Bit 7:5.
                                            0b000 = 2.3V    0b001 = 2.4V 
                                            0b010 = 2.5V    0b011 = 2.6V 
                                            0b100 = 2.7V    0b101 = 2.8V 
                                            0b110 = 2.9V    0b111 = 3.0V */
    } bits;
} reg_cnfg_chg_c_t;

/**
 * @brief CNFG_CHG_D
 * 
 * Address : 0x23
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char vsys_reg  : 5;    /**< System voltage regulation (VSYS-REG). Bit 4:0.
                                            0x0 = 3.300V 		0x1 = 3.350V 
											0x2 = 3.400V 		... 
											0x1D = 4.750V 		0x1E - 0x1F = 4.800V  */
        unsigned char tj_reg    : 3;    /**< Sets the die junction temperature regulation point, TJ-REG. Bit 7:5.
                                            0b000 = 60ºC        0b001 = 70ºC 
                                            0b010 = 80ºC        0b011 = 90ºC 
                                            0b100 - 0b111 = 100ºC */
    } bits;
} reg_cnfg_chg_d_t;

/**
 * @brief CNFG_CHG_E
 * 
 * Address : 0x24
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char t_fast_chg    : 2;    /**< Sets the fast-charge safety timer, tFC. Bit 1:0.
                                                0b00 = Timer disabled   0b01 = 3 hours 
                                                0b10 = 5 hours          0b11 = 7 hours  */
        unsigned char chg_cc        : 6;    /**< Sets the fast-charge constant current value, IFAST-CHG. Bit 7:2.
                                                0x0 = 7.5mA         0x1 = 15.0mA 
                                                0x2 = 22.5mA        ... 
                                                0x26 = 292.5mA      0x27 - 0x3F = 300.0mA */
    } bits;
} reg_cnfg_chg_e_t;

/**
 * @brief CNFG_CHG_F
 * 
 * Address : 0x25
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char               : 2;    /**< Bit 1:0*/
        unsigned char chg_cc_jeita  : 6;    /**< Sets IFAST-CHG-JEITA for when the battery is either cool or warm as defined 
                                                 by the VCOOL and VWARM temperature thresholds. Bit 7:2.
                                                0x0 = 7.5mA         0x1 = 15.0mA 
                                                0x2 = 22.5mA        ... 
                                                0x26 = 292.5mA      0x27 - 0x3F = 300.0mA */
    } bits;
} reg_cnfg_chg_f_t;

/**
 * @brief CNFG_CHG_G
 * 
 * Address : 0x26
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char fus_m     : 1;    /**< Forced USB Suspend Mask. Bit 0.
												FUS is only active if the FUS_M bit is set to 0.*/
        unsigned char usbs      : 1;    /**< Setting this bit places CHGIN in USB suspend mode. Bit 1.
                                                0 = CHGIN is not suspended and may draw current from an adapter source 
                                                1 = CHGIN is suspended and may draw no current from an adapter source  */
        unsigned char chg_cv    : 6;    /**< Sets fast-charge battery regulation voltage, VFAST-CHG. Bit 7:2.
                                               0x0 = 3.600V         0x1 = 3.625V 
                                               0x2 = 3.650V         ... 
                                               0x27 = 4.575V        0x28 - 0x3F = 4.600V */
    } bits;
} reg_cnfg_chg_g_t;

/**
 * @brief CNFG_CHG_H
 * 
 * Address : 0x27
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char chr_th_dis    : 1;    /**< Charger restart threshold disable. Bit 0. */
        unsigned char sys_bat_prt   : 1;    /**< VSYS_REG - CHG_CV clamp. Bit 1.
												By default, the VSYS_REG has to be at least 200mV higher than the programmed CHG_CV.
                                                If this bit is set (hardware protection is turned off), the software has to provide 
												the protection (the SYS voltage has to be 200mV higher than the BATT voltage). 
												If the VSYS_REG is lower than CHG_CV+200mV, the charger will reduce. */
        unsigned char chg_cv_jeita  : 6;    /**< Sets the modified VFAST-CHG-JEITA for when the battery is either cool or 
												warm as defined by the VCOOL and VWARM temperature thresholds. Bit 7:2.
												0x0 = 3.600V         0x1 = 3.625V 
												0x2 = 3.650V         ... 
												0x27 = 4.575V        0x28 - 0x3F = 4.600V */
    } bits;
} reg_cnfg_chg_h_t;

/**
 * @brief CNFG_CHG_I
 * 
 * Address : 0x28
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char mux_sel               : 4;    /**< Selects the analog channel to connect to AMUX. Bit 3:0.
                                                        0b0000 = Multiplexer is disabled and AMUX is high-impedance. 
                                                        0b0001 = CHGIN voltage monitor. 
                                                        0b0010 = CHGIN current monitor. 
                                                        0b0011 = BATT voltage monitor. 
                                                        0b0100 = BATT charge current monitor. Valid only while battery charging is happening (CHG = 1). 
                                                        0b0101 = BATT discharge current monitor normal measurement. 
                                                        0b0110 = BATT discharge current monitor nulling measurement. 
                                                        0b0111 = Reserved. 
														0b1000 = Reserved.
                                                        0b1001 = AGND voltage monitor (through 100Ω pull-down resistor) 
                                                        0b1010 - 0b1111 = SYS voltage monitor */
        unsigned char imon_dischg_scale    : 4;    /**< Selects the battery discharge current full-scale current value. Bit 7:4.
                                                        0x0 = 8.2mA     0x1 = 40.5mA 0x2 = 72.3mA 
                                                        0x3 = 103.4mA   0x4 = 134.1mA 
                                                        0x5 = 164.1mA   0x6 = 193.7mA 
                                                        0x7 = 222.7mA   0x8 = 251.2mA
                                                        0x9 = 279.3mA   0xA - 0xF = 300.0mA */
    } bits;
} reg_cnfg_chg_i_t;

/**
 * @brief CNFG_SBB_TOP
 * 
 * Address : 0x38
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char drv_sbb   : 2;    /**< SIMO Buck-Boost (all channels) Drive Strength Trim. Bit 1:0.
                                            0b00 = Fastest transition time 
											0b01 = A little slower than 0b00 
											0b10 = A little slower than 0b01 
											0b11 = A little slower than 0b10 */
		unsigned char 		    : 5;    /**< Bit 6:2.*/
        unsigned char dis_lpm   : 1;    /**< Disables the automatic Low Power Mode for Each SIMO Channel. Bit 7.
											0b0 = Automatic Low Power Mode for each SIMO channel 
											0b1 = Disable LPM feature for each SIMO channel */
    } bits;
} reg_cnfg_sbb_top_t;

/**
 * @brief CNFG_SBB0_A
 * 
 * Address : 0x39
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char tv_sbb0   : 8;    /**< SIMO Buck-Boost Channel 0 Target Output Voltage. Bit 7:0.
                                            0x00 = 0.500V 0x01 = 0.525V 0x02 = 0.550V 
											0x03 = 0.575V 0x04 = 0.600V 0x05 = 0.625V 
											0x06 = 0.650V 0x07 = 0.675V 0x08 = 0.700V 
											... 
											0xC5 = 5.425V 0xC6 = 5.450V 0xC7 = 5.475V 
											0xC8 to 0xFF = 5.500V */
    } bits;
} reg_cnfg_sbb0_a_t;

/**
 * @brief CNFG_SBB0_B
 * 
 * Address : 0x3A
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char en_sbb0       : 3;    /**< Enable Control for SIMO Buck-Boost Channel 0, 
                                                selecting either an FPS slot the channel powers-up and powers-down in 
                                                or whether the channel is forced on or off. Bit 2:0.
                                                0b000 = FPS slot 0      0b001 = FPS slot 1 
                                                0b010 = FPS slot 2      0b011 = FPS slot 3      
                                                0b100 = Off irrespective of FPS 
                                                0b101 = same as 0b100   0b110 = On irrespective of FPS 
                                                0b111 = same as 0b110 */
        unsigned char ade_sbb0      : 1;    /**< SIMO Buck-Boost Channel 0 Active-Discharge Enable. Bit 3.
                                                0 = The active discharge function is disabled. 
                                                When SBB0 is disabled, its discharge rate is a function of the output capacitance and the external load. 
                                                1 = The active discharge function is enabled. 
                                                When SBB0 is disabled, an internal resistor (RAD_SBB0) is activated from SBB0 to PGND to help the output voltage discharge. */
        unsigned char ip_sbb0       : 2;    /**< SIMO Buck-Boost Channel 0 Peak Current Limit. Bit 5:4
                                                0b00 = 1.000A       0b01 = 0.750A 
                                                0b10 = 0.500A       0b11 = 0.333A*/
        unsigned char op_mode0      : 2;    /**<  Operation mode of SBB0. Bit 6.
                                                0b00 = Automatic 
												0b01 = Buck mode 
												0b10 = Boost mode 
												0b11 = Buck-boost mode*/
    } bits;
} reg_cnfg_sbb0_b_t;

/**
 * @brief CNFG_SBB1_A
 * 
 * Address : 0x3B
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char tv_sbb1   : 8;    /**< SIMO Buck-Boost Channel 1 Target Output Voltage. Bit 7:0.
                                            0x00 = 0.500V 0x01 = 0.525V 0x02 = 0.550V 
											0x03 = 0.575V 0x04 = 0.600V 0x05 = 0.625V 
											0x06 = 0.650V 0x07 = 0.675V 0x08 = 0.700V 
											... 
											0xC5 = 5.425V 0xC6 = 5.450V 0xC7 = 5.475V 
											0xC8 to 0xFF = 5.500V */
    } bits;
} reg_cnfg_sbb1_a_t;

/**
 * @brief CNFG_SBB1_B
 * 
 * Address : 0x3C
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char en_sbb1       : 3;    /**< Enable Control for SIMO Buck-Boost Channel 1, 
                                                selecting either an FPS slot the channel powers-up and powers-down in 
                                                or whether the channel is forced on or off. Bit 2:0.
                                                0b000 = FPS slot 0      0b001 = FPS slot 1 
                                                0b010 = FPS slot 2      0b011 = FPS slot 3      
                                                0b100 = Off irrespective of FPS 
                                                0b101 = same as 0b100   0b110 = On irrespective of FPS 
                                                0b111 = same as 0b110 */
        unsigned char ade_sbb1      : 1;    /**< SIMO Buck-Boost Channel 1 Active-Discharge Enable. Bit 3.
                                                0 = The active discharge function is disabled. 
                                                When SBB0 is disabled, its discharge rate is a function of the output capacitance and the external load. 
                                                1 = The active discharge function is enabled. 
                                                When SBB0 is disabled, an internal resistor (RAD_SBB0) is activated from SBB0 to PGND to help the output voltage discharge. */
        unsigned char ip_sbb1       : 2;    /**< SIMO Buck-Boost Channel 1 Peak Current Limit. Bit 5:4.
                                                0b00 = 1.000A       0b01 = 0.750A 
                                                0b10 = 0.500A       0b11 = 0.333A*/
        unsigned char op_mode1      : 2;    /**<  Operation mode of SBB1. Bit 7:6.
                                                0b00 = Automatic 
												0b01 = Buck mode 
												0b10 = Boost mode 
												0b11 = Buck-boost mode*/
    } bits;
} reg_cnfg_sbb1_b_t;

/**
 * @brief CNFG_SBB2_A
 * 
 * Address : 0x3D
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char tv_sbb2   : 8;    /**< SIMO Buck-Boost Channel 2 Target Output Voltage. Bit 7:0.
                                            0x00 = 0.500V 0x01 = 0.525V 0x02 = 0.550V 
											0x03 = 0.575V 0x04 = 0.600V 0x05 = 0.625V 
											0x06 = 0.650V 0x07 = 0.675V 0x08 = 0.700V 
											... 
											0xC5 = 5.425V 0xC6 = 5.450V 0xC7 = 5.475V 
											0xC8 to 0xFF = 5.500V */
    } bits;
} reg_cnfg_sbb2_a_t;

/**
 * @brief CNFG_SBB2_B
 * 
 * Address : 0x3E
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char en_sbb2       : 3;    /**< Enable Control for SIMO Buck-Boost Channel 2, 
                                                selecting either an FPS slot the channel powers-up and powers-down in 
                                                or whether the channel is forced on or off. Bit 2:0.
                                                0b000 = FPS slot 0      0b001 = FPS slot 1 
                                                0b010 = FPS slot 2      0b011 = FPS slot 3      
                                                0b100 = Off irrespective of FPS 
                                                0b101 = same as 0b100   0b110 = On irrespective of FPS 
                                                0b111 = same as 0b110 */
        unsigned char ade_sbb2      : 1;    /**< SIMO Buck-Boost Channel 2 Active-Discharge Enable Bit 3.
                                                0 = The active discharge function is disabled. 
                                                When SBB0 is disabled, its discharge rate is a function of the output capacitance and the external load. 
                                                1 = The active discharge function is enabled. 
                                                When SBB0 is disabled, an internal resistor (RAD_SBB0) is activated from SBB0 to PGND to help the output voltage discharge. */
        unsigned char ip_sbb2       : 2;    /**< SIMO Buck-Boost Channel 2 Peak Current Limit. Bit 5:4.
                                                0b00 = 1.000A       0b01 = 0.750A 
                                                0b10 = 0.500A       0b11 = 0.333A*/
        unsigned char op_mode2      : 2;    /**< Operation mode of SBB2. Bit 7:6.
                                                0b00 = Automatic 
												0b01 = Buck mode 
												0b10 = Boost mode 
												0b11 = Buck-boost mode*/
    } bits;
} reg_cnfg_sbb2_b_t;

/**
 * @brief CNFG_DVS_SBB0_A
 * 
 * Address : 0x3F
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char tv_sbb0_dvs   : 8;    /**<  SIMO Buck-Boost Channel 0 Target Output Voltage. Bit 7:0.
                                                0x00 = 0.500V 0x01 = 0.525V 0x02 = 0.550V 
												0x03 = 0.575V 0x04 = 0.600V 0x05 = 0.625V 
												0x06 = 0.650V 0x07 = 0.675V 0x08 = 0.700V 
												... 
												0xC5 = 5.425V 0xC6 = 5.450V 0xC7 = 5.475V 
												0xC8 to 0xFF = 5.500V */
    } bits;
} reg_cnfg_dvs_sbb0_a_t;

/**
 * @brief CNFG_LDO0_A
 * 
 * Address : 0x48
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char tv_ldo0_6_0   : 7;    /**<  LDO0 Target Output Voltage. Bit 6:0.
                                                0x00 = 0.500V 0x01 = 0.525V 0x02 = 0.550V 
												0x03 = 0.575V 0x04 = 0.600V 0x05 = 0.625V 
												0x06 = 0.650V 0x07 = 0.675V 0x08 = 0.700V 
												... 
												0x7E = 3.650V
												0x7F = 3.675V 
												When TV_LDO[7] = 0, TV_LDO[6:0] sets the
												LDO's output voltage range from 0.5V to 3.675V.
												When TV_LDO[7] = 1, TV_LDO[6:0] sets the
												LDO's output voltage from 1.825V to 5V. */
		unsigned char tv_ldo0_7   	: 1; 	/**< LDO0 Output Voltage. Bit7. 
												This bit applies a 1.325V offset to the output voltage of the LDO0. 
												0b0 = No Offset, 0b1 = 1.325V Offset
												*/
		
    } bits;
} reg_cnfg_ldo0_a_t;

/**
 * @brief CNFG_LDO0_B
 * 
 * Address : 0x49
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char en_ldo0       : 3;    /**< Enable Control for LDO0, 
                                                selecting either an FPS slot the channel powers-up and 
												powersdown in or whether the channel is forced on or off.  Bit 2:0.
                                                0b000 = FPS slot 0      0b001 = FPS slot 1 
                                                0b010 = FPS slot 2      0b011 = FPS slot 3      
                                                0b100 = Off irrespective of FPS 
                                                0b101 = same as 0b100   0b110 = On irrespective of FPS 
                                                0b111 = same as 0b110 */
        unsigned char ade_ldo0      : 1;    /**< LDO0 Active-Discharge Enable. Bit 3.
                                                0 = The active discharge function is disabled.                                            
                                                1 = The active discharge function is enabled.*/
        unsigned char ldo0_md       : 1;    /**< Operation Mode of LDO0. Bit 4.
                                                0b0 = Low dropout linear regulator (LDO) mode
												0b1 = Load switch (LSW) mode*/
        unsigned char       		: 3;    /**< */
    } bits;
} reg_cnfg_ldo0_b_t;

/**
 * @brief CNFG_LDO1_A
 * 
 * Address : 0x4A
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char tv_ldo1_6_0   : 7;    /**<  LDO1 Target Output Voltage. Bit 6:0.
                                                0x00 = 0.500V 0x01 = 0.525V 0x02 = 0.550V 
												0x03 = 0.575V 0x04 = 0.600V 0x05 = 0.625V 
												0x06 = 0.650V 0x07 = 0.675V 0x08 = 0.700V 
												... 
												0x7E = 3.650V
												0x7F = 3.675V 
												When TV_LDO[7] = 0, TV_LDO[6:0] sets the
												LDO's output voltage range from 0.5V to 3.675V.
												When TV_LDO[7] = 1, TV_LDO[6:0] sets the
												LDO's output voltage from 1.825V to 5V. */
		unsigned char tv_ldo1_7   	: 1; 	/**< LDO1 Output Voltage. Bit7. 
												This bit applies a 1.325V offset to the output voltage of the LDO0. 
												0b0 = No Offset, 0b1 = 1.325V Offset
												*/
		
    } bits;
} reg_cnfg_ldo1_a_t;

/**
 * @brief CNFG_LDO1_B
 * 
 * Address : 0x4B
 */
typedef union {
    unsigned char raw;
    struct 
    {
        unsigned char en_ldo1       : 3;    /**< Enable Control for LDO1, 
                                                selecting either an FPS slot the channel powers-up and 
												powersdown in or whether the channel is forced on or off.  Bit 2:0.
                                                0b000 = FPS slot 0      0b001 = FPS slot 1 
                                                0b010 = FPS slot 2      0b011 = FPS slot 3      
                                                0b100 = Off irrespective of FPS 
                                                0b101 = same as 0b100   0b110 = On irrespective of FPS 
                                                0b111 = same as 0b110 */
        unsigned char ade_ldo1      : 1;    /**< LDO1 Active-Discharge Enable. Bit 3.
                                                0 = The active discharge function is disabled.                                            
                                                1 = The active discharge function is enabled.*/
        unsigned char ldo1_md       : 1;    /**< Operation Mode of LDO1. Bit 4.
                                                0b0 = Low dropout linear regulator (LDO) mode
												0b1 = Load switch (LSW) mode*/
        unsigned char       		: 3;    /**< */
    } bits;
} reg_cnfg_ldo1_b_t;

/*FUEL GAUGE REGISTERS*/

/**
 * @brief Status
 * 
 * Address : 0x00
 */
typedef union {
    uint16_t raw;
    struct 
    {
		unsigned char imn      	: 1;    /**< Minimum Isys Threshold Exceeded. Bit 0. */
		unsigned char por      	: 1;    /**< Power-On Reset. Bit 1.
                                            This bit is set to 1 when the device detects that a software or hardware POR event has occurred. */
        unsigned char spr_2   	: 1;    /**< Bit 2. */
		unsigned char bst      	: 1;    /**< Battery Status. Bit 3.
                                            Unutilized bit. Write to 0. Reads are don't care. */
		unsigned char isysmx  	: 1;    /**< SYS current is over OCP limit. Bit 4.
                                            Maximum SYS current threshold exceeded. */
		unsigned char spr_5     : 1;    /**< Bit 5. */
		unsigned char thmhot 	: 1;    /**< FG Control Charger Input Current Limit. Bit 6.
                                            Set to 1 to indicate a Thermistor Hot to allow FG control charger input current limit. 
											ThmHot is set to 0 at power-up. */
		unsigned char dsoci     : 1;    /**< 1% SOC Change Alert. Bit 7.
                                            This bit is set to 1 to indicate a 1% SOC change alert. dSOCi is set to 0 at power-up. */
		unsigned char vmn      	: 1;    /**< Minimum VALRT Threshold Exceeded. Bit 8.
                                            This bit is set to 1 whenever a VCELL register reading is below the minimum VALRT value. 
											This bit may or may not need to be cleared by system software to detect the next event. 
											See VS in the CONFIG register. Vmn is set to 0 at power-up. */
		unsigned char tmn      	: 1;    /**< Minimum TALRT Threshold Exceeded. Bit 9.
                                            This bit is set to 1 whenever a Temperature register reading is below the minimum TALRT value.
											This bit may or may not need to be cleared by system software to detect the next event. 
											See TS in the CONFIG register. Tmn is set to 0 at power-up. */
		unsigned char smn      	: 1;    /**< Minimum SOCALRT Threshold Exceeded. Bit 10.
                                            This bit is set to 1 whenever SOC falls below the minimum SOCALRT value. 
											This bit may or may not need to be cleared by system software to detect the next event. 
											See SS in the CONFIG register and SACFG in the MiscCFG register. Smn is set to 0 at power-up.*/
		unsigned char bi      	: 1;    /**< Battery Insertion. Bit 11.
                                            This bit is set to 1 when the device detects that a battery has been inserted into the system by monitoring the AIN pin. 
											This bit must be cleared by system software to detect the next insertion event. 
											Bi is set to 0 at power-up. */									
        unsigned char vmx      	: 1;    /**< Maximum VALRT Threshold Exceeded. Bit 12.
                                            This bit is set to 1 whenever a VCELL register reading is above the maximum VALRT value. 
											This bit may or may not need to be cleared by system software to detect the next event. 
											See VS in the CONFIG register. Vmx is set to 0 at power-up. */
		unsigned char tmx      	: 1;    /**< Maximum TALRT Threshold Exceeded Bit 13.
                                            This bit is set to 1 whenever a Temperature register reading is above the maximum TALRT value. 
											This bit may or may not need to be cleared by system software to detect the next event. 
											See TS in the CONFIG register. Tmx is set to 0 at power-up. */
		unsigned char smx      	: 1;    /**< Maximum SOCALRT Threshold Exceeded. Bit 14.
                                            This bit is set to 1 whenever SOC rises above the maximum SOCALRT value. 
											This bit may or may not need to be cleared by system software to detect the next event. 
											See SS in the CONFIG register and SACFG in the MiscCFG register. Smx is set to 0 at power-up. */
		unsigned char br      	: 1;    /**< Battery Removal. Bit 15.
                                            This bit is set to 1 when the device detects that a battery has been removed from the system. 
											This bit must be cleared by system software to detect the next removal event. 
											Br is set to 0 at power-up. */
    } bits;
} reg_status_t;

/**
 * @brief VAlrtTh
 * 
 * Address : 0x01
 */
typedef union {
    uint16_t raw;
    struct 
    {
		unsigned char min_voltage_alrt  : 8;    /**< Minimum voltage reading. Bit 7:0,
													An alert is generated if the VCell register reading falls below this value. 
													Register type: special Set Min=0x00 to disable. Units of LSB are 20mV. */
		unsigned char max_voltage_alrt  : 8;    /**< Maximum voltage reading. Bit 15:8.
													An alert is generated if the VCell register reading exceeds this value.
													Register type: special Set Max=0xFF to disable. Units of LSB are 20mV. */
    } bits;
} reg_valrt_th_t;

/**
 * @brief TAlrtTh
 * 
 * Address : 0x02
 */
typedef union {
    uint16_t raw;
    struct 
    {
		unsigned char min_temp_alrt  	: 8;    /**< Sets an alert threshold for minimum temperature. Bit 7:0.  
													Register type: special Set Min=0x80 to disable. Units of LSB are 1˚C. */
		unsigned char max_temp_alrt  	: 8;    /**< Sets an alert threshold for maximum temperature. Bit 15:8. 
													Register type: special Set Max=0x7F to disable. Units of LSB are 1˚C. */
    } bits;
} reg_talrt_th_t;

/**
 * @brief SAlrtTh
 * 
 * Address : 0x03
 */
typedef union {
    uint16_t raw;
    struct 
    {
		unsigned char min_soc_alrt  	: 8;    /**< Sets an alert for minimum SOC. Bit 7:0.  
													Register type: special This may be used for charge/discharge termination, or 
													for power-management near empty. Set Min=0x00 to disable. Units of LSB are 1%. */
		unsigned char max_soc_alrt  	: 8;    /**< Sets an alert for maximum SOC. Bit 15:8. 
													Register type: special This may be used for charge/discharge termination, or 
													for power-management near empty. Set Max=0xFF to disable. Units of LSB are 1%. */
    } bits;
} reg_salrt_th_t;

/**
 * @brief FullSocThr
 * 
 * Address : 0x13
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t full_soc_thr  : 16;    /**< FullSOCThr comes from OTP if the OTP register is enabled. 
											Otherwise it POR’s to a default of 95%. Bit 15:0.  
											LSB unit is 1/256%. */
    } bits;
} reg_full_soc_thr_t;

/**
 * @brief DesignCap
 * 
 * Address : 0x18
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t design_cap  	: 16;    /**< Bit 15:0.  
											Register type: capacity. */
    } bits;
} reg_design_cap_t;

/**
 * @brief Config
 * 
 * Address : 0x1D
 */
typedef union {
    uint16_t raw;
    struct 
    {
		unsigned char ber      	: 1;    /**< Enable alert on battery removal. Bit 0. 
											When Ber = 1, a battery-removal condition, as detected by the AIN pin voltage, 
											triggers an alert. Set to 0 at power-up. */
		unsigned char bei      	: 1;    /**< Enable alert on battery insertion. Bit 1.
                                            When Bei = 1, a battery-insertion condition, as detected by the AIN pin voltage, 
											triggers an alert. Set to 0 at power-up. */
        unsigned char aen   	: 1;    /**< Enable alert on fuel-gauge outputs. Bit 2. 
											When Aen = 1, violation of any of the alert threshold register values by temperature, voltage, 
											or SOC triggers an alert. This bit affects the INTB pin (FG_INT) operation only. 
											The Smx, Smn, Tmx, Tmn, Vmx, and Vmn bits are not disabled. This bit is set to 0 at power-up.*/
		unsigned char fthrm     : 1;    /**< Force Thermistor Bias Switch. Bit 3.
                                            Set FTHRM = 1 to always enable the thermistor bias switch. With a standard 10kOhm thermistor, 
											this adds an additional ~200μA to the current drain of the circuit. 
											This bit is set to 0 at power-up. */
		unsigned char ethrm  	: 1;    /**< Enable Thermistor. Bit 4.
                                            Set to logic 1 to enable the automatic THRM output bias and AIN measurement every 1.4s. 
											This bit is set to 1 at power-up. */
		unsigned char spr_5     : 1;    /**< I2C Charge Fail Enable. Bit 5. 
											Set to 1 to enable I2CChgFail interrupt to drive the INTB pin. 
											This bit is not accessible by the fuel gauge firmware. */
		unsigned char i2csh 	: 1;    /**< I2C Shutdown. Bit 6.
                                            Set to logic 1 to force the device to enter shutdown mode if both SDA and SCL are held low for 
											more than timeout of the SHDNTIMER register. This also configures the device to wake up on a 
											rising edge of either SDA or SCL. Set to 1 at power-up. */
		unsigned char shdn     	: 1;    /**< Shutdown. Bit 7.
                                            Write this bit to logic 1 to force a shutdown of the device after timeout of the SHDNTIMER register. 
											SHDN is reset to 0 at power-up and upon exiting shutdown mode. */
		unsigned char tex      	: 1;    /**< Temperature External. Bit 8.
                                            When set to 1, the fuel gauge requires external temperature measurements to be written from the host. 
											When set to 0, measurements on the AIN pin are converted to a temperature value and stored 
											in the Temperature register instead. */
		unsigned char ten      	: 1;    /**< Enable Temperature Channel. Bit 9.
                                            Set to 1 and set ETHRM or FTHRM to 1 to enable measurements on the AIN pin. 
											Ten is set to 1 at power-up. */
		unsigned char ainsh     : 1;    /**< AIN Pin Shutdown. Bit 10.
                                            Set to 1 to enable device shutdown when the battery is removed. 
											The IC enters shutdown if the AIN pin remains high (AIN reading > VTHRM - VDETR) for longer than 
											the timeout of the SHDNTIMER register. This also configures the device to wake up when AIN is pulled 
											low on cell insertion. AINSH is set to 0 at power-up. */
		unsigned char spr_11    : 1;    /**< Fuel Gauge Charger Control. Bit 11.
                                            Set to 1 to enable MaxCharge (fuel-gauge controlled charging). */									
        unsigned char vs      	: 1;    /**< Voltage ALRT Sticky. Bit 12.
                                            When VS = 1, voltage alerts can only be cleared through software. 
											When VS = 0, voltage alerts are cleared automatically when the threshold is no longer exceeded. 
											VS is set to 0 at power-up. */
		unsigned char ts      	: 1;    /**< Temperature ALRT Sticky. Bit 13.
                                            When TS = 1, temperature alerts can only be cleared through software. 
											When TS = 0, temperature alerts are cleared automatically when the threshold is no longer exceeded. 
											TS is set to 1 at power-up. */
		unsigned char ss      	: 1;    /**< SOC ALRT Sticky. Bit 14.
                                            When SS = 1, SOC alerts can only be cleared through software. 
											When SS = 0, SOC alerts are cleared automatically when the threshold is no longer exceeded. 
											SS is set to 0 at power-up. */
		unsigned char spr_15    : 1;    /**< Fuel gauge Charger Fail Enable. Bit 15.
                                            Set to 1 to enable FGCHGFAIL interrupt to drive the INTB pin. 
											This bit is not accessible by the fuel gauge firmware. */
    } bits;
} reg_config_t;

/**
 * @brief IChgTerm
 * 
 * Address : 0x1E
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t ichg_term 	: 16;    /**< Program IChgTerm to the exact charge termination current used in the application. Bit 15:0.  
										Register type: current. */														
    } bits;
} reg_ichg_term_t;

/**
 * @brief DevName
 * 
 * Address : 0x21
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t dev_name  : 16;    /**< holds revision information. Bit 15:0. */														
    } bits;
} reg_dev_name_t;

/**
 * @brief FilterCfg
 * Use of unsigned char resulted in error in register bit field values.
 * Thus, uint16_t is here used to define the size of bit fields.
 * Address : 0x29
 */
typedef union {
    uint16_t raw;
    struct 
    {
    	uint16_t ncurr  	: 4;    /**< Sets the time constant for the AverageCurrent register. Bit 3:0.
											The default POR value of 4h gives a time constant of 11.25 seconds.
											The equation setting the period is:AverageCurrent time constant = 175.8ms × 2^(2+NCURR).*/
    	uint16_t navgcell 	: 3;    /**< Sets the time constant for the AverageVCELL register. Bit 6:4.
											The default POR value of 2h gives a time constant of 45.0s.
											The equation setting the period is:AverageVCELL time constant = 175.8ms × 2(6+NAVGVCELL). */
    	uint16_t nmix  		: 4;    /**< Sets the time constant for the mixing algorithm. Bit 10:7.
											The default POR value of Dh gives a time constant of 12.8 hours = 46080s.
											The equation setting the period is:Mixing Period = 175.8ms × 2(5+NMIX). */
    	uint16_t rsvd		: 3;    /**< Bit 13:11. */
    	uint16_t nempty  	: 2;    /**< Sets the filtering for empty learning for both the I_Avgempty and QRTable registers. Bit 15:14.
											Iavg_empty is learned with(NEMPTY+3) right shifts. 
											QRTable is learned with (NEMPTY + sizeof(Iavgempty) – sizeof (AvgCurrent)) right shifts. */										
    } bits;
} reg_filter_cfg_t;

/**
 * @brief IAvgEmpty
 * 
 * Address : 0x36
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t rsvd 	: 16;    /**< Register type: current. Bit 15:0. */														
    } bits;
} reg_iavg_empty_t;

/**
 * @brief VEmpty
 * 
 * Address : 0x3A
 */
typedef union {
    uint16_t raw;
    struct 
    {
		unsigned char v_recover  	: 7;    /**< Recovery Voltage. Sets the voltage level for clearing empty detection. Bit 6:0. 
												Once the cell voltage rises above this point, empty voltage detection is reenabled.
												A 40mV resolution gives a 0 to 5.08V range. This value is written to 3.68V at power-up. */														
		uint16_t v_empty  	: 9;    /**< Empty Voltage. Sets the voltage level for detecting empty. Bit 15:7. 
												A 10mV resolution gives a 0 to 5.11V range. This value is written to 3.12V at power-up.*/														
    } bits;
} reg_v_empty_t;

/**
 * @brief Config2
 * 
 * Address : 0xBB
 */
typedef union {
    uint16_t raw;
    struct 
    {
		unsigned char isys_ncurr  		: 4;    /**< Sets the time constant for the AvgISys register. Bit 3:0. 
													The default POR value of 0100b gives a time constant of 5.625. 
													The equation setting the period is: AvgISys time constant = 45s x 2(ISysNCurr-7).*/
		unsigned char qcvqen 			: 1;    /**< Set OCVQen=1 to enable automatic empty compensation based on VFOCV information. Bit 4. */
		unsigned char ldmdl  			: 1;    /**< Host sets this bit to 1 in order to initiate firmware to finish processing a newly loaded model. Bit 5. 
													Firmware clears this bit to zero to indicate that model loading is finished. */
		unsigned char tairt_en  		: 1;    /**< Set to 1 to enable temperature alert. Bit 6. 
													If TAlrtEn=0, the alert on is disabled. */			
		unsigned char dsocen 			: 1;    /**< Set to 1 to enable SOC 1% change alert. Bit 7. 
													If dSCOCen=0, the alert on is disabled.*/	
		unsigned char thm_hotairt_en	: 1;    /**< Set to 1 to enable ThmHotAlrtEn alert. Bit 8. 
													If ThmHotAlrtEn = 0, the alert on is disabled.*/	
		unsigned char thmhot_en			: 1;    /**< Set to 1 to enable ThmHot function by comparing Voltage/Temp condition with THMHOT (40h). Bit 9. 
													ThmHot function is FG Charger input current limit control. */
		unsigned char fc_thmhot			: 1;    /**< Set to 1 to enable thermistor hot forcedly, regardless of actually sense thermistor temperature. Bit 10. */	
		unsigned char spr  				: 5;    /**< Bit 15:11. */										
    } bits;
} reg_config2_t;

/**
 * @brief Temp
 * 
 * Address : 0x08
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t temp  : 16;    /**< This is the most recent trimmed temperature measurement. Bit 15:0. 
									Temperature is measured every 1.4 seconds. Register type: temperature.*/														
    } bits;
} reg_temp_t;

/**
 * @brief Vcell
 * 
 * Address : 0x09
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t vcell  : 16;    /**< This is the most recent trimmed cell voltage result. Bit 15:0. 
									It represents an FIR average of raw results. 
									The VOLT_Raw is sampled every 175.8ms and gain and offset trim are applied to calculate VCELL. 
									Register type: voltage.*/														
    } bits;
} reg_vcell_t;

/**
 * @brief Current
 * 
 * Address : 0x0A
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t current  	: 16;    /**< Register type: current. Bit 15:0. */														
    } bits;
} reg_current_t;

/**
 * @brief AvgCurrent
 * 
 * Address : 0x0B
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t avg_current  	: 16;    /**< This is the 0.7s to 6.4hr (configurable) IIR average of the current. Bit 15:0. 
													This register represents the upper 16 bits of the 32-bit shift register that filters current. 
													The average should be set equal to Current upon startup.
													Register type: current. */														
    } bits;
} reg_avg_current_t;

/**
 * @brief AvgTA
 * 
 * Address : 0x16
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t avg_ta  	: 16;    /**< This is the 6min to 12hr (configurable) IIR average of the Temperature. Bit 15:0. 
												The average is set equal to Temp upon startup. Register type: temperature. */														
    } bits;
} reg_avg_ta_t;

/**
 * @brief AvgVCell
 * 
 * Address : 0x19
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t avg_vcell  : 16;    /**< This reports the 12s to 24min (configurable) IIR average of VCELL. Bit 15:0. 
												The average is set equal to VCELL at startup. Register type: voltage. */														
    } bits;
} reg_avg_vcell_t;

/**
 * @brief MaxMinTemp
 * 
 * Address : 0x1A
 */
typedef union {
    uint16_t raw;
    struct 
    {
		unsigned char min_temp  : 8;    /**< Records the minimum Temperature. Bit 7:0. 
											Register type: special Two's complement 8-bit value with 1°C resolution. */
		unsigned char max_temp  : 8;    /**< Records the maximum Temperature. Bit 15:8. 
											Register type: special Two's complement 8-bit value with 1°C resolution. */												
    } bits;
} reg_max_min_temp_t;

/**
 * @brief MaxMinVolt
 * 
 * Address : 0x1B
 */
typedef union {
    uint16_t raw;
    struct 
    {
		unsigned char min_volt  : 8;    /**< Records the VCELL minimum voltage. Bit 7:0. 
											Register type: The maximum and minimum voltages are each stored as 8-bit values with a 20mV resolution. */
		unsigned char max_volt  : 8;    /**< Records the VCELL maximum voltage. Bit 15:8. 
											Register type: The maximum and minimum voltages are each stored as 8-bit values with a 20mV resolution. */												
    } bits;
} reg_max_min_volt_t;

/**
 * @brief MaxMinCurr
 * 
 * Address : 0x1C
 */
typedef union {
    uint16_t raw;
    struct 
    {
		unsigned char min_charge_curr  	: 8;    /**< Records the minimum charge current. Bit 7:0. 
													Register type: Two's complement 8-bit values with 0.4mV/RSENSE resolution. */
		unsigned char max_charge_curr  	: 8;    /**< Records the maximum charge current. Bit 15:8. 
													Register type: Two's complement 8-bit values with 0.4mV/RSENSE resolution. */												
    } bits;
} reg_max_min_curr_t;

/**
 * @brief AIN0
 * 
 * Address : 0x27
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t ain0  : 16;    /**< The TGain, TOff, and Curve register values are then applied to this ratio-metric reading to 
									convert the result to temperature. Bit 15:0. 
									Register type: special The MAX77658 stores the result as a ratio-metric value 
									from 0% to 100% in the AIN register with an LSB of 0.0122%. */														
    } bits;
} reg_ain0_t;

/**
 * @brief Timer
 * 
 * Address : 0x3E
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t timer : 16;    /**< Timer increments once every task period. Bit 15:0. 
									With default TaskPeriod, timer has units 0.1758 seconds. 
									Register type: special The Timer register LSB is 175.8ms, giving a full-scale range of 0 to 3.2 hours. */														
    } bits;
} reg_timer_t;

/**
 * @brief ShdnTimer
 * 
 * Address : 0x3F
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t shdnctr  		: 13;    /**< Shutdown Counter. Bit 12:0. 
											This register counts the total amount of elapsed time since the shutdown trigger event. 
											Register type: special. This counter value stops and resets to 0 when the shutdown timeout completes. 
											The counter LSB is 1.4s. */	
		unsigned char shdn_thr  : 3;    /**< Sets the shutdown timeout period from a minimum of 45s to a maximum of 1.6h. Bit 15:13. 
											The default POR value of 7h gives a shutdown delay of 1.6h. 
											Register type: special The equation setting the period is: Shutdown Timeout Period = 175.8ms × 2(8+THR). */											
    } bits;
} reg_shdn_timer_t;

/**
 * @brief TimerH
 * 
 * Address : 0xBE
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t timerh	: 16;    /**< TIMERH is a 16-bit high-word extension to the TIMER register. Bit 15:0. 
												This extension allows time counting up to 24 years. This register can be enabled in the save & restore registers. 
												Register type: special A 3.2-hour LSb gives a full-scale range for the register of up to 23.94 years. */														
    } bits;
} reg_timerh_t;

/**
 * @brief RepCap
 * 
 * Address : 0x05
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t rep_cap	: 16;    /**< RepCap or reported capacity. Bit 15:0. 
												A filtered version of the AvCap register that prevents large jumps in the reported 
												value caused by changes in the application such as abrupt changes in tempreature or 
												load current.  Register type: capacity. */														
    } bits;
} reg_rep_cap_t;

/**
 * @brief RepSOC
 * 
 * Address : 0x06
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t rep_soc	: 16;    /**< RepSOC is the complete calculation for State of Charge. Bit 15:0. 
												This includes all processing, including: ModelGauge Mixing, Empty Compensation. 
												Register type: percentage. */														
    } bits;
} reg_rep_soc_t;

/**
 * @brief AvSOC
 * 
 * Address : 0x0E
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t av_soc	: 16;    /**< This register provides unfiltered results. Bit 15:0. 
												Jumps in the reported values can be caused by abrupt changes in load current or temperature. 
												See the Empty Compensation section for details. 
												This includes all processing, including: ModelGauge Mixing, Empty Compensation. 
												Register type: percentage. */														
    } bits;
} reg_av_soc_t;

/**
 * @brief FullCapRep
 * 
 * Address : 0x10
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t full_cap_rep	: 16;    /**< Most applications should only monitor FullCapRep, instead of FullCap or FullCapNom. Bit 15:0. 
													A new full-capacity value is calculated at the end of every charge cycle in the application. 
													Register type: capacity. */														
    } bits;
} reg_full_cap_rep_t;

/**
 * @brief TTE
 * 
 * Address : 0x11
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t tte	: 16;    /**< The TTE value is determined by relating AvCap with AvgCurrent. Bit 15:0. 
											The corresponding AvgCurrent filtering gives a delay in TTE but provides more stable results. 
											Register type: time. */														
    } bits;
} reg_tte_t;

/**
 * @brief RCell
 * 
 * Address : 0x14
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t rcell	: 16;    /**< This register provides the calculated internal resistance of the cell. Bit 15:0. 
											RCell is determined by comparing open-circuit voltage (VFOCV) against measured voltage (VCell) 
											over a long time period while under load or charge current. 
											Register type: resistance. */														
    } bits;
} reg_rcell_t;

/**
 * @brief Cycles
 * 
 * Address : 0x17
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t cycles	: 16;    /**< This register provides the calculated internal resistance of the cell. Bit 15:0. 
												RCell is determined by comparing open-circuit voltage (VFOCV) against measured voltage (VCell) 
												over a long time period while under load or charge current. 
												Register type: resistance. */														
    } bits;
} reg_cycles_t;

/**
 * @brief AvCap
 * 
 * Address : 0x1F
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t av_cap	: 16;    /**< This is the remaining capacity with coulomb-counter + Voltage-Fuel-Gauge mixing, 
												after accounting for capacity that is unavailable due to the discharge rate. Bit 15:0.
												Register type: capacity. */														
    } bits;
} reg_av_cap_t;

/**
 * @brief TTF
 * 
 * Address : 0x20
 */
typedef union {
    uint16_t raw;
    struct 
    {
		uint16_t ttf	: 16;    /**< The TTF value is determined by learning the constant current and 
											constant voltage portions of the charge cycle based on experience of prior charge cycles. Bit 15:0.
											Register type: time. */														
    } bits;
} reg_ttf_t;

#endif /* MAX77658_REGS_H_ */
