/*******************************************************************************
 * Copyright(C) Analog Devices Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files(the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Analog Devices Inc.
 * shall not be used except as stated in the Analog Devices Inc.
 * Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Analog Devices Inc.retains all ownership rights.
 *******************************************************************************
 */

#ifndef _MAX77658_H_
#define _MAX77658_H_

#include "mbed.h"
#include "MAX77658_regs.h"

#define MAX77658_NO_ERROR                    0
#define MAX77658_VALUE_NULL                 -1
#define MAX77658_WRITE_DATA_FAILED          -2
#define MAX77658_READ_DATA_FAILED           -3
#define MAX77658_INVALID_DATA               -4

#define MAX77658_I2C_ADDRESS_PMIC_0        	0x80
#define MAX77658_I2C_ADDRESS_PMIC_1         0x90
#define MAX77658_I2C_ADDRESS_FG          	0x6C

/**
 * @brief MAX77658 Ultra-Low Power PMIC Featuring Single-Inductor, 3-Output Buck-Boost, 2-LDOs, 
 * Power-Path Charger for Small Li+, Fuel Gauge M5.
 *
 * @details The MAX77658 provides highly-integrated battery charging and power supply solutions 
 * for low-power applications where size and efficiency are critical.
 *
 * @code
 * @endcode
 */

class MAX77658
{
private:
    I2C *i2c_handler;
    InterruptIn *irq_pin;	// interrupt pin

    /**
     * @brief   	Register Addresses
     * @details 	Enumerated MAX77658 register addresses
     */
    typedef enum {
        /*Global*/
        INT_GLBL0    	= 0x00,    // Interrupt Status 0
        INT_GLBL1   	= 0x04,    // Interrupt Status 1
        ERCFLAG      	= 0x05,    // Flags
        STAT_GLBL    	= 0x06,    // Global Status
        INTM_GLBL1   	= 0x08,    // Interrupt Mask 1
        INTM_GLBL0   	= 0x09,    // Interrupt Mask 0
        CNFG_GLBL    	= 0x10,    // Configuration Global
        CNFG_GPIO0   	= 0x11,    // GPIO0 Configuration
        CNFG_GPIO1   	= 0x12,    // GPIO1 Configuration
        CNFG_GPIO2   	= 0x13,    // GPIO2 Configuration
        CID          	= 0x14,    // Chip Identification Code
        CNFG_WDT     	= 0x17,    // Configuration WatchDog Timer
        /*Charger*/
        INT_CHG      	= 0x01,    // Charger Interrupt Status
        STAT_CHG_A   	= 0x02,    // Charger Status A
        STAT_CHG_B   	= 0x03,    // Charger Status B
        INT_M_CHG    	= 0x07,    // Charger Interrupt Mask
        CNFG_CHG_A   	= 0x20,    // Charger Configuration A
        CNFG_CHG_B   	= 0x21,    // Charger Configuration B
        CNFG_CHG_C   	= 0x22,    // Charger Configuration C
        CNFG_CHG_D   	= 0x23,    // Charger Configuration D
        CNFG_CHG_E   	= 0x24,    // Charger Configuration E
        CNFG_CHG_F   	= 0x25,    // Charger Configuration F
        CNFG_CHG_G   	= 0x26,    // Charger Configuration G
        CNFG_CHG_H   	= 0x27,    // Charger Configuration H
        CNFG_CHG_I   	= 0x28,    // Charger Configuration I
        /*SBB*/
        CNFG_SBB_TOP 	= 0x38,    // SIMO Buck-Boost Configuration
        CNFG_SBB0_A  	= 0x39,    // SIMO Buck-Boost 0 Configuration A
        CNFG_SBB0_B  	= 0x3A,    // SIMO Buck-Boost 0 Configuration B
        CNFG_SBB1_A  	= 0x3B,    // SIMO Buck-Boost 1 Configuration A
        CNFG_SBB1_B  	= 0x3C,    // SIMO Buck-Boost 1 Configuration B
        CNFG_SBB2_A  	= 0x3D,    // SIMO Buck-Boost 2 Configuration A
        CNFG_SBB2_B  	= 0x3E,    // SIMO Buck-Boost 2 Configuration B
		CNFG_DVS_SBB0_A = 0x3F,    // SIMO Buck-Boost 0 DVS Configuration A
        /*LDO*/
		CNFG_LDO0_A		= 0x48,	   // LDO0 Output Voltage
		CNFG_LDO0_B		= 0x49,	   // LDO0 Output Voltage Configuration
		CNFG_LDO1_A		= 0x4A,	   // LDO1 Output Voltage
		CNFG_LDO1_B		= 0x4B,	   // LDO2 Output Voltage Configuration
        /*FuelGauge*/
        Status   		= 0x00,    // Interrupt status register for the FG block
        VAlrtTh   		= 0x01,    // Voltage AIrt
        TAlrtTh   		= 0x02,    // Temperature AIrt
        SAlrtTh   		= 0x03,    // Soc AIrt
        FullSocThr  	= 0x13,    // Full Soc Thr
        DesignCap  		= 0x18,    // Design Cap
        Config  		= 0x1D,    // Configuration
        IChgTerm  		= 0x1E,    // IChg Term
        DevName  		= 0x21,    // Dev Name
        FilterCfg  		= 0x29,    // Filter Configuration
        IAvgEmpty  		= 0x36,    // IAvgEmpty
        VEmpty  		= 0x3A,    // VEmpty
        Config2  		= 0xBB,    // Configuration 2
        Temp  			= 0x08,    // Temp
        Vcell  			= 0x09,    // Vcell
        Current			= 0x0A,    // Current
        AvgCurrent		= 0x0B,    // AvgCurrent
        AvgTA			= 0x16,    // AvgTA
        AvgVCell		= 0x19,    // AvgVCell
        MaxMinTemp		= 0x1A,    // MaxMinTemp
        MaxMinVolt		= 0x1B,    // MaxMinVolt
        MaxMinCurr		= 0x1C,    // MaxMinCurr
        AIN0			= 0x27,    // AIN0
        Timer			= 0x3E,    // Timer
        ShdnTimer		= 0x3F,    // ShdnTimer
        TimerH			= 0xBE,    // TimerH
        RepCap			= 0x05,    // RepCap
        RepSOC			= 0x06,    // RepSOC
        AvSOC			= 0x0E,    // AvSOC
        FullCapRep		= 0x10,    // FullCapRep
        TTE				= 0x11,    // TTE
        RCell			= 0x14,    // RCell
        Cycles			= 0x17,    // Cycles
        AvCap			= 0x1F,    // AvCap
        TTF				= 0x20     // TTF
    } reg_t;

    void interrupt_handler();

    void (MAX77658::*funcptr)(void);

    void post_interrupt_work();

    Thread *post_intr_work_thread;

    struct handler {
        void (*func)(void *);
        void *cb;
    };

    handler *interrupt_handler_list;

public:
    /**
    * @brief		MAX77658 constructor.
    */
    MAX77658(I2C *i2c, PinName IRQPin = NC);

    /**
    * @brief		MAX77658 destructor.
    */
    ~MAX77658();

    /**
    * @brief		Function pointer type to interrupt handler function
    */
    typedef void (*interrupt_handler_function)(void *);

    /**
    * @brief 		Read from a register.
    *
    * @param[in] 	reg Address of a register to be written.
    * @param[out] 	value Pointer to save result value.
    *
    * @returns 	0 on success, negative error code on failure.
    */
    int read_register(uint8_t reg, uint8_t *value);

    /**
    * @brief 		Write to a register.
    *
    * @param[in] 	reg Address of a register to be written.
    * @param[out] 	value Pointer of value to be written to register.
    *
    * @returns 	0 on success, negative error code on failure.
    */
    int write_register(uint8_t reg, const uint8_t *value);	
	
	/**
    * @brief 		Read from a fuel-gauge register.
    *
    * @param[in] 	reg Address of a fuel-gauge register to be written.
    * @param[out] 	value Pointer to save result value.
    *
    * @returns 	0 on success, negative error code on failure.
    */
    int read_fg_register(uint8_t reg, uint8_t *value);

    /**
    * @brief 		Write to a fuel-gauge register.
    *
    * @param[in] 	reg Address of a fuel-gauge register to be written.
    * @param[out] 	value Pointer of value to be written to fuel-gauge register.
    *
    * @returns 	0 on success, negative error code on failure.
    */
    int write_fg_register(uint8_t reg, const uint8_t *value);
	
	/**
    * @brief 		Register Configuration.
	*		 		All Interrupt Flags combined from INT_GLBL0, INT_GLBL1 and INT_CHG.
    *
    * @details
    *  - Register      : INT_GLBL0 (0x00), INT_GLBL1(0x04) and INT_CHG (0x01)
    *  - Bit Fields    :
    *  - Default       : 0x0
    *  - Description   : Enumerated interrupts.
    */
    typedef enum {
        INT_GLBL0_GPI0_F,
        INT_GLBL0_GPI0_R,
        INT_GLBL0_NEN_F,
        INT_GLBL0_NEN_R,
        INT_GLBL0_TJAL1_R,
        INT_GLBL0_TJAL2_R,
        INT_GLBL0_DOD1_R,
        INT_GLBL0_DOD0_R,
        INT_GLBL1_GPI1_F,
        INT_GLBL1_GPI1_R,
        INT_GLBL1_SBB0_F,
        INT_GLBL1_SBB1_F,
        INT_GLBL1_SBB2_F,
        INT_GLBL1_LDO0_F,
        INT_GLBL1_LDO1_F,
        INT_GLBL1_RSVD,
        INT_CHG_THM_I,
        INT_CHG_CHG_I,
        INT_CHG_CHGIN_I,
        INT_CHG_TJ_REG_I,
        INT_CHG_CHGIN_CTRL_I,
        INT_CHG_SYS_CTRL_I,
        INT_CHG_SYS_CNFG_I,
        INT_CHG_RSVD,
		INT_CHG_END
    } reg_bit_int_glbl_t;
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : ERCFLAG (0x05)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Event Recorder Flags.
    */
    typedef enum {
		ERCFLAG_TOVLD,
		ERCFLAG_SYSOVLO,
		ERCFLAG_SYSUVLO,
		ERCFLAG_MRST_F,
		ERCFLAG_SFT_OFF_F,
		ERCFLAG_SFT_CRST_F,
		ERCFLAG_WDT_OFF,
    	ERCFLAG_WDT_RST
    }reg_bit_ercflag_t;

    /**
    * @brief		Get bit field of ERCFLAG (0x05) register.
    *
	* @param[in]	bit_field 	ERCFLAG register bit field to be written.
    * @param[out] 	flag 		Pointer to save result of ercglag bit states.	
	*							For individual bit
	*							0x0: ERCFLAG has not occurred,
    *               			0x1: ERCFLAG has occurred.
    *
    * @return		0 on success, error code on failure.
    */
    int get_ercflag(reg_bit_ercflag_t bit_field, uint8_t *flag);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : STAT_GLBL (0x06)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Event Recorder Flags.
    */
    typedef enum {
    	STAT_GLBL_STAT_IRQ,
		STAT_GLBL_STAT_EN,
		STAT_GLBL_TJAL1_S,
		STAT_GLBL_TJAL2_S,
		STAT_GLBL_DOD1_S,
		STAT_GLBL_DOD0_S,
		STAT_GLBL_BOK,
		STAT_GLBL_DIDM
    }reg_bit_stat_glbl_t;
	
    /**
    * @brief		Get bit field of STAT_GLBL (0x06) register.
    *
	* @param[in]	bit_field 	STAT_GLBL register bit field to be written.
    * @param[out] 	status 		Pointer to save result of Status Global bit state.
    *
    * @return		0 on success, error code on failure.
    */
    int get_stat_glbl(reg_bit_stat_glbl_t bit_field, uint8_t *status);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : INT_M_CHG (0x07), INTM_GLBL0 (0x08) and INTM_GLBL1 (0x09)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : All interrupt mask bits.
    */
    typedef enum {
		INT_M_CHG_THM_M,
		INT_M_CHG_CHG_M,
		INT_M_CHG_CHGIN_M,
		INT_M_CHG_TJ_REG_M,
		INT_M_CHG_CHGIN_CTRL_M,
		INT_M_CHG_SYS_CTRL_M,
		INT_M_CHG_SYS_CNFG_M,
		INT_M_CHG_DIS_AICL,
    	INTM_GLBL0_GPI0_FM,
		INTM_GLBL0_GPI0_RM,
		INTM_GLBL0_nEN_FM,
		INTM_GLBL0_nEN_RM,
		INTM_GLBL0_TJAL1_RM,
		INTM_GLBL0_TJAL2_RM,
		INTM_GLBL0_DOD1_RM,
		INTM_GLBL0_DOD0_RM,
		INTM_GLBL1_GPI1_FM,
		INTM_GLBL1_GPI1_RM,
		INTM_GLBL1_SBB0_FM,
		INTM_GLBL1_SBB1_FM,
		INTM_GLBL1_SBB2_FM,
		INTM_GLBL1_LDO0_M,
		INTM_GLBL1_LDO1_M,
		INTM_GLBL1_RSVD,
		INTM_NUM_OF_BIT
    }reg_bit_int_mask_t;
	
	/**
    * @brief		Set bit field of INT_M_CHG (0x07), INTM_GLBL0 (0x08) or INTM_GLBL1 (0x09) register.
    *
	* @param[in]	bit_field 	Register bit field to be set.
    * @param[out] 	maskBit 	0x0: Interrupt is unmasked,
    *                    		0x1: Interrupt is masked.
    *
    * @return		0 on success, error code on failure.
    */
    int set_interrupt_mask(reg_bit_int_mask_t bit_field, uint8_t maskBit);
	
	/**
    * @brief		Get bit field of INT_M_CHG (0x07), INTM_GLBL0 (0x08) or INTM_GLBL1 (0x09) register.
    *
	* @param[in]	bit_field 	Register bit field to be written.
    * @param[out] 	maskBit 	0x0: Interrupt is unmasked,
    *                    		0x1: Interrupt is masked.
    *
    * @return		0 on success, error code on failure.
    */
    int get_interrupt_mask(reg_bit_int_mask_t bit_field, uint8_t *maskBit);

	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : CNFG_GLBL (0x10)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Event Recorder Flags.
    */
    typedef enum {
    	CNFG_GLBL_SFT_CTRL,
		CNFG_GLBL_DBEN_nEN,
		CNFG_GLBL_nEN_MODE,
		CNFG_GLBL_SBIA_LPM,
		CNFG_GLBL_T_MRST,
		CNFG_GLBL_PU_DIS
    }reg_bit_cnfg_glbl_t;	

    /**
  	* @brief		Set CNFG_GLBL (0x10) register.
  	*
	* @param[in]	bit_field 	Register bit field to be written.
  	* @param[in]	config 		Register bit field to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int set_cnfg_glbl(reg_bit_cnfg_glbl_t bit_field, uint8_t config);

    /**
  	* @brief		Get CNFG_GLBL (0x10) register.
  	*
	* @param[in]	bit_field 	Register bit field to be written.
  	* @param[out]	config 		Pointer of value to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int get_cnfg_glbl(reg_bit_cnfg_glbl_t bit_field, uint8_t *config);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : CNFG_GPIO0 (0x11), CNFG_GPIO1 (0x12) or CNFG_GPIO2 (0x13)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Event Recorder Flags.
    */
    typedef enum {
    	CNFG_GPIO_DIR,
		CNFG_GPIO_DI,
		CNFG_GPIO_DRV,
		CNFG_GPIO_DO,
		CNFG_GPIO_DBEN_GPI,
		CNFG_GPIO_ALT_GPIO,
		CNFG_GPIO_RSVD
    }reg_bit_cnfg_gpio_t;	

    /**
  	* @brief		Set either CNFG_GPIO0 (0x11), CNFG_GPIO1 (0x12) or CNFG_GPIO2 (0x13).
  	*
	* @param[in]	bit_field 	Register bit field to be written.
  	* @param[in]	channel 	Channel number: 0, 1 or 2
  	* @param[in]	config 		Register bit field to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int set_cnfg_gpio(reg_bit_cnfg_gpio_t bit_field, uint8_t channel, uint8_t config);

    /**
  	* @brief		Get either CNFG_GPIO0 (0x11), CNFG_GPIO1 (0x12) or CNFG_GPIO2 (0x13).
  	*
	* @param[in]	bit_field 	Register bit field to be written.
  	* @param[in]	channel 	Channel number: 0, 1 or 2
  	* @param[out]	config 		Pointer of value to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int get_cnfg_gpio(reg_bit_cnfg_gpio_t bit_field, uint8_t channel, uint8_t *config);
	
    /**
    * @brief		Get bit field of CID (0x14) register.
    *
    * @return		CID on success, error code on failure.
    */
    int get_cid(void);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : CNFG_WDT (0x17)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Watchdog Timer Configuration.
    */
    typedef enum {
    	CNFG_WDT_WDT_LOCK,
		CNFG_WDT_WDT_EN,
		CNFG_WDT_WDT_CLR,
		CNFG_WDT_WDT_MODE,
		CNFG_WDT_WDT_PER,
		CNFG_WDT_RSVD
    }reg_bit_cnfg_wdt_t;	
	
    /**
  	* @brief		Set CNFG_WDT (0x17) register.
  	*
	* @param[in]	bit_field 	Register bit field to be written.
  	* @param[in]	config 		Field value to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int set_cnfg_wdt(reg_bit_cnfg_wdt_t bit_field, uint8_t config);

    /**
	* @brief		Get CNFG_WDT (0x17) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	config 		Pointer of value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_cnfg_wdt(reg_bit_cnfg_wdt_t bit_field, uint8_t *config);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : STAT_CHG_A (0x02)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Watchdog Timer Configuration.
    */
    typedef enum {
    	STAT_CHG_A_THM_DTLS,
		STAT_CHG_A_TJ_REG_STAT,
		STAT_CHG_A_VSYS_MIN_STAT,
		STAT_CHG_A_ICHGIN_LIM_STAT,
		STAT_CHG_A_VCHGIN_MIN_STAT,
		STAT_CHG_A_RSVD
    }reg_bit_stat_chg_a_t;	
	
	/**
	* @brief		Get STAT_CHG_A (0x02) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	status 		Pointer of value to be read.
	*							For individual bit,
	*							0x0 = It is not engaged,
	*							0x1 = It is engaged.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_stat_chg_a(reg_bit_stat_chg_a_t bit_field, uint8_t *status);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_CHG_A (0x02)
	*  - Bit Fields    : [2:0]
	*  - Default       : 0x0
	*  - Description   : Battery Temperature Details.
	*/
    typedef enum {
        THM_DTLS_THERMISTOR_DISABLED,
        THM_DTLS_BATTERY_COLD,
        THM_DTLS_BATTERY_COOL,
        THM_DTLS_BATTERY_WARM,
        THM_DTLS_BATTERY_HOT,
        THM_DTLS_BATTERY_NORMAL,
        THM_DTLS_RESERVED_0x06,
        THM_DTLS_RESERVED_0x07
    }decode_thm_dtls_t;

    /**
	* @brief		Get Battery Temperature Details.
	* 				Valid only when CHGIN_DTLS[1:0] = 0b11.
	*
	* @param[out]	thm_dtls 	Battery temperature details field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_thm_dtls(decode_thm_dtls_t *thm_dtls);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : STAT_CHG_B (0x03)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Watchdog Timer Configuration.
    */
    typedef enum {
    	STAT_CHG_B_TIME_SUS,
		STAT_CHG_B_CHG,
		STAT_CHG_B_CHGIN_DTLS,
		STAT_CHG_B_CHG_DTLS
    }reg_bit_stat_chg_b_t;	

    /**
	* @brief		Get STAT_CHG_B (0x03) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	status 		Pointer of value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_stat_chg_b(reg_bit_stat_chg_b_t bit_field, uint8_t *status);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_CHG_B (0x03)
	*  - Bit Fields    : [7:4]
	*  - Default       : 0x0
	*  - Description   : Charger Details.
	*/
    typedef enum {
        CHG_DTLS_OFF,
        CHG_DTLS_PREQUALIFICATION_MODE,
        CHG_DTLS_FAST_CHARGE_CC,
        CHG_DTLS_JEITA_FAST_CHARGE_CC,
        CHG_DTLS_FAST_CHARGE_CV,
        CHG_DTLS_JEITA_FAST_CHARGE_CV,
        CHG_DTLS_TOP_OFF_MODE,
        CHG_DTLS_JEITA_MODIFIED_TOP_OFF_MODE,
        CHG_DTLS_DONE,
        CHG_DTLS_JEITA_MODIFIED_DONE,
        CHG_DTLS_PREQUALIFICATION_TIMER_FAULT,
        CHG_DTLS_FAST_CHARGE_TIMER_FAULT,
        CHG_DTLS_BATTERY_TEMPERATURE_FAULT,
        CHG_DTLS_RESERVED_0x0D,
        CHG_DTLS_RESERVED_0x0E,
        CHG_DTLS_RESERVED_0x0F
    }decode_chg_dtls_t;

    /**
	* @brief		Get Charger Details.
	*
	* @param[out]	chg_dtls Charger details field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_chg_dtls(decode_chg_dtls_t *chg_dtls);
	
	/**
	* @brief		Set the VHOT JEITA Temperature Threshold.
	*				Bit 7:6 of CNFG_CHG_A (0x20) register.
	*
	* @param[in]	tempDegC 	Register bit field to be read.
	*							45ºC, 50ºC, 55ºC or 60ºC.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_thm_hot(int tempDegC);

    /**
	* @brief		Get the VHOT JEITA Temperature Threshold.
	*				Bit 7:6 of CNFG_CHG_A (0x20) register.
	*
	* @param[out]	tempDegC 	Pointer of value to be read.
	*							45ºC, 50ºC, 55ºC or 60ºC.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_thm_hot(int *tempDegC);
	
	/**
	* @brief		Set the VWARM JEITA Temperature Threshold.
	*				Bit 5:4 of CNFG_CHG_A (0x20) register.
	*
	* @param[in]	tempDegC 	Register bit field to be read.
	*							35ºC, 40ºC, 45ºC or 50ºC.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_thm_warm(int tempDegC);

    /**
	* @brief		Get the VWARM JEITA Temperature Threshold.
	*				Bit 5:4 of CNFG_CHG_A (0x20) register.
	*
	* @param[out]	tempDegC 	Pointer of value to be read.
	*							35ºC, 40ºC, 45ºC or 50ºC.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_thm_warm(int *tempDegC);
	
	 /**
	* @brief		Set the VCOOL JEITA Temperature Threshold.
	*				Bit 3:2 of CNFG_CHG_A (0x20) register.
	*
	* @param[in]	tempDegC 	Register bit field to be read.
	*							0ºC, 5ºC, 10ºC or 15ºC.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_thm_cool(int tempDegC);

    /**
	* @brief		Get the VCOOL JEITA Temperature Threshold.
	*				Bit 3:2 of CNFG_CHG_A (0x20) register.
	*
	* @param[out]	tempDegC 	Pointer of value to be read.
	*							0ºC, 5ºC, 10ºC or 15ºC.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_thm_cool(int *tempDegC);
	
	/**
	* @brief		Set the VCOLD JEITA Temperature Threshold.
	*				Bit 1:0 of CNFG_CHG_A (0x20) register.
	*
	* @param[in]	tempDegC 	Register bit field to be read.
	*							-10ºC, -5ºC, 0ºC or 5ºC.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_thm_cold(int tempDegC);

    /**
	* @brief		Get the VCOLD JEITA Temperature Threshold.
	*				Bit 1:0 of CNFG_CHG_A (0x20) register.
	*
	* @param[out]	tempDegC	Pointer of value to be read.
	*							-10ºC, -5ºC, 0ºC or 5ºC.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_thm_cold(int *tempDegC);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : CNFG_CHG_B (0x21)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Watchdog Timer Configuration.
    */
    typedef enum {
    	CNFG_CHG_B_CHG_EN,
		CNFG_CHG_B_I_PQ,
		CNFG_CHG_B_ICHGIN_LIM,
		CNFG_CHG_B_VCHGIN_MIN
    }reg_bit_cnfg_chg_b_t;	
	
	/**
	* @brief		Set CNFG_CHG_B (0x21) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[in]	config 		Register bit field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_cnfg_chg_b(reg_bit_cnfg_chg_b_t bit_field, uint8_t config);

    /**
	* @brief		Get CNFG_CHG_B (0x21) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	config 		Pointer of value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_cnfg_chg_b(reg_bit_cnfg_chg_b_t bit_field, uint8_t *config);
		
    /**
	* @brief		Set Minimum CHGIN Regulation Voltage.
	*				Bit 7:5 of CNFG_CHG_B (0x21) register.
	*
	* @param[in]	voltV 	
	*						4.0V, 4.1V, 4.2V, 4.3V,
	*						4.4V, 4.5V, 4.6V, 4.7V.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_vchgin_min(float voltV);

    /**
	* @brief		Get Minimum CHGIN Regulation Voltage.
	*				Bit 7:5 of CNFG_CHG_B (0x21) register.
	*
	* @param[in]	voltV 	Pointer of value to be read.
	*						4.0V, 4.1V, 4.2V, 4.3V,
	*						4.4V, 4.5V, 4.6V, 4.7V.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_vchgin_min(float *voltV);
	
	/**
	* @brief		Set CHGIN Input Current Limit.
	*				Bit 4:2 of CNFG_CHG_B (0x21) register.
	*
	* @param[in]	currentmA 	
	*							95mA, 190mA, 285mA, 380mA, 475mA.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_ichgin_lim(int currentmA);

    /**
	* @brief		Get CHGIN Input Current Limit.
	*				Bit 4:2 of CNFG_CHG_B (0x21) register.
	*
	* @param[out]	currentmA 	Pointer of value to be read.
	*							95mA, 190mA, 285mA, 380mA, 475mA.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_ichgin_lim(int *currentmA);

	/**
	* @brief		Set Battery Prequalification Voltage Threshold (VPQ).
	*				Bit 7:5 of CNFG_CHG_C (0x22) register.
	*
	* @param[in]	voltV 	
	*						2.3V, 2.4V, 2.5V, 2.6V, 
    *                       2.7V, 2.8V, 2.9V, 3.0V.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_chg_pq(float voltV);

    /**
	* @brief		Get Battery Prequalification Voltage Threshold (VPQ).
	*				Bit 7:5 of CNFG_CHG_C (0x22) register.
	*
	* @param[out]	voltV 	Pointer of value to be read.
	*						2.3V, 2.4V, 2.5V, 2.6V, 
    *                       2.7V, 2.8V, 2.9V, 3.0V.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_chg_pq(float *voltV);
	
	/**
	* @brief		Set Charger Termination Current (ITERM).
	* 				I_TERM[1:0] sets the charger termination current
	* 				as a percentage of the fast charge current IFAST-CHG.
	*				Bit 4:3 of CNFG_CHG_C (0x22) register.
	*
	* @param[in]	percent 	
	*							5%, 7.5%, 10%, 15%.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_i_term(float percent);

    /**
	* @brief		Get Charger Termination Current (ITERM).
	* 				I_TERM[1:0] sets the charger termination current
	* 				as a percentage of the fast charge current IFAST-CHG.
	*				Bit 4:3 of CNFG_CHG_C (0x22) register.
	*
	* @param[out]	percent 	Pointer of value to be read.
	*							5%, 7.5%, 10%, 15%.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_i_term(float *percent);
	
	/**
	* @brief		Set Top-off Timer Value.
	*				Bit 2:0 of CNFG_CHG_C (0x22) register.
	*
	* @param[in]	minute 	
	*						0 minutes, 5 minutes, 10 minutes
	*						15 minutes, 20 minutes, 25 minutes, 
    *                       30 minutes, 35 minutes.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_t_topoff(uint8_t minute);

    /**
	* @brief		Get Top-off Timer Value.
	*				Bit 2:0 of CNFG_CHG_C (0x22) register.
	*
	* @param[out]	minute 	Pointer of value to be read.
							0 minutes, 5 minutes, 10 minutes
	*						15 minutes, 20 minutes, 25 minutes, 
    *                       30 minutes, 35 minutes.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_t_topoff(uint8_t *minute);
	
	/**
	* @brief		Set the Die Junction Temperature Regulation Point, TJ-REG.
	*				Bit 7:5 of CNFG_CHG_D (0x23) register.
	*
	* @param[in]	tempDegC 	60ºC, 70ºC, 80ºC,
	*							90ºC, 100ºC.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_tj_reg(uint8_t tempDegC);

    /**
	* @brief		Get the Die Junction Temperature Regulation Point, TJ-REG.
	*				Bit 7:5 of CNFG_CHG_D (0x23) register.
	*
	* @param[out]	tempDegC	Pointer of value to be read.
	*							60ºC, 70ºC, 80ºC, 90ºC, 100ºC.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_tj_reg(uint8_t *tempDegC);
	
	/**
	* @brief		Set System Voltage Regulation (VSYS-REG).
	*				Bit 4:0 of CNFG_CHG_D (0x23) register.
	*
	* @param[in]	voltV 	3.300V, 3.350V, 3.400V, ... 
	*						4.750V, 4.800V.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_vsys_reg(float voltV);

    /**
	* @brief		Get System Voltage Regulation (VSYS-REG).
	*				Bit 4:0 of CNFG_CHG_D (0x23) register.
	*
	* @param[out]	voltV 	Pointer of value to be read.
	*						3.300V, 3.350V, 3.400V, ... 
	*						4.750V, 4.800V.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_vsys_reg(float *voltV);
	
	/**
	* @brief		Set the Fast-Charge Constant Current Value, IFAST-CHG.
	*				Bit 7:2 of CNFG_CHG_E (0x24) register.
	*
	* @param[in]	currentmA 	7.5mA, 15.0mA, 22.5mA, ... 
    *                       	292.5mA, 300.0mA.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_chg_cc(float currentmA);

    /**
	* @brief		Get the Fast-Charge Constant Current Value, IFAST-CHG.
	*				Bit 7:2 of CNFG_CHG_E (0x24) register.
	*
	* @param[out]	currentmA 	Pointer of value to be read.
	*							7.5mA, 15.0mA, 22.5mA, ... 
    *                       	292.5mA, 300.0mA.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_chg_cc(float *currentmA);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_E (0x24)
	*  - Bit Fields    : [1:0]
	*  - Default       : 0x0
	*  - Description   : Fast-charge Safety timer, tFC.
	*/
    typedef enum {
        T_FAST_CHG_TIMER_DISABLED,
        T_FAST_CHG_HOUR_3H,
        T_FAST_CHG_HOUR_5H,
        T_FAST_CHG_HOUR_7H
    }decode_t_fast_chg_t;

    /**
	* @brief		Set the Fast-charge Safety timer, tFC.
	*				Bit 1:0 of CNFG_CHG_E (0x24) register.
	*
	* @param[in]	t_fast_chg Fast-charge safety timer field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_t_fast_chg(decode_t_fast_chg_t t_fast_chg);

    /**
	* @brief		Get the Fast-charge Safety timer, tFC.
	*				Bit 1:0 of CNFG_CHG_E (0x24) register.
	*
	* @param[out]	t_fast_chg Fast-charge safety timer field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_t_fast_chg(decode_t_fast_chg_t *t_fast_chg);
	
	/**
	* @brief		Set IFAST-CHG-JEITA
	* 				when the battery is either cool or warm as defined by the
	* 				VCOOL and VWARM temperature thresholds.
	*				Bit 7:2 of CNFG_CHG_F (0x25) register.
	*
	* @param[in]	currentmA 	7.5mA, 15.0mA, 22.5mA, ... 
    *                       	292.5mA, 300.0mA.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_chg_cc_jeita(float currentmA);

    /**
	* @brief		Get IFAST-CHG-JEITA
	* 				when the battery is either cool or warm as defined by the
	* 				VCOOL and VWARM temperature thresholds.
	*				Bit 7:2 of CNFG_CHG_F (0x25) register.
	*
	* @param[out]	currentmA 	Pointer of value to be read.
	*							7.5mA, 15.0mA, 22.5mA, ... 
    *                       	292.5mA, 300.0mA.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_chg_cc_jeita(float *currentmA);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : CNFG_CHG_G (0x26)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Watchdog Timer Configuration.
    */
    typedef enum {
    	CNFG_CHG_G_FUS_M,
		CNFG_CHG_G_USBS,
		CNFG_CHG_G_CHG_CV
    }reg_bit_cnfg_chg_g_t;	
	
	/**
	* @brief		Set CNFG_CHG_G (0x26) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[in]	config 		Register bit field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_cnfg_chg_g(reg_bit_cnfg_chg_g_t bit_field, uint8_t config);

    /**
	* @brief		Get CNFG_CHG_G (0x26) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	config 		Pointer of value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_cnfg_chg_g(reg_bit_cnfg_chg_g_t bit_field, uint8_t *config);
	
	/**
	* @brief		Set Fast-Charge Battery Regulation Voltage, VFAST-CHG.
	*				Bit 7:2 of CNFG_CHG_G (0x26) register.
	*
	* @param[in]	voltV	3.600V, 3.625V, 3.650V, ... 
    *                       4.575V, 4.600V.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_chg_cv(float voltV);

    /**
	* @brief		Get Fast-Charge Battery Regulation Voltage, VFAST-CHG.
	*				Bit 7:2 of CNFG_CHG_G (0x26) register.
	*
	* @param[out]	voltV	Pointer of value to be read.
	*						3.600V, 3.625V, 3.650V, ... 
    *                       4.575V, 4.600V.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_chg_cv(float *voltV);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : CNFG_CHG_H (0x27)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Watchdog Timer Configuration.
    */
    typedef enum {
    	CNFG_CHG_H_CHR_TH_DIS,
		CNFG_CHG_H_SYS_BAT_PRT,
		CNFG_CHG_H_CHG_CV_JEITA
    }reg_bit_cnfg_chg_h_t;	
	
	/**
	* @brief		Set CNFG_CHG_H (0x27) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[in]	config 		Register bit field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_cnfg_chg_h(reg_bit_cnfg_chg_h_t bit_field, uint8_t config);

    /**
	* @brief		Get CNFG_CHG_H (0x27) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	config 		Pointer of value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_cnfg_chg_h(reg_bit_cnfg_chg_h_t bit_field, uint8_t *config);

	/**
	* @brief		Set the modified VFAST-CHG-JEITA for when the battery is either
	* 				cool or warm as defined by the VCOOL and VWARM temperature thresholds.
	*				Bit 7:2 of CNFG_CHG_H (0x27) register.
	*
	* @param[in]	voltV	Pointer of value to be read.
	*						3.600V, 3.625V, 3.650V, ... 
    *                       4.575V, 4.600V.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_chg_cv_jeita(float voltV);

    /**
	* @brief		Get the modified VFAST-CHG-JEITA for when the battery is either
	* 				cool or warm as defined by the VCOOL and VWARM temperature thresholds.
	*				Bit 7:2 of CNFG_CHG_H (0x27) register.
	*
	* @param[out]	voltV 	Pointer of value to be read.
	*						3.600V, 3.625V, 3.650V, ... 
    *                       4.575V, 4.600V.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_chg_cv_jeita(float *voltV);
	
	/**
	* @brief		Set the Battery Discharge Current Full-Scale Current Value.
	*				Bit 7:4 of CNFG_CHG_I (0x28) register.
	*
	* @param[in]	currentmA 	8.2mA, 40.5mA, 72.3mA, 103.4mA, 
    *                           134.1mA, 164.1mA, 193.7mA, 222.7mA,                       
    *                           251.2mA, 279.3mA, 300.0mA
	*
	* @return		0 on success, error code on failure.
	*/
    int set_imon_dischg_scale(float currentmA);

    /**
	* @brief		Get the Battery Discharge Current Full-Scale Current Value.
	*				Bit 7:4 of CNFG_CHG_I (0x28) register.
	*
	* @param[out]	currentmA 	Pointer of value to be read.
	*							8.2mA, 40.5mA, 72.3mA, 103.4mA, 
    *                           134.1mA, 164.1mA, 193.7mA, 222.7mA,                       
    *                           251.2mA, 279.3mA, 300.0mA
	*
	* @return		0 on success, error code on failure.
	*/
    int get_imon_dischg_scale(float *currentmA);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_I (0x28)
	*  - Bit Fields    : [3:0]
	*  - Default       : 0x0
	*  - Description   : Analog channel to connect to AMUX.
	*/
    typedef enum {
        MUX_SEL_MULTIPLEXER_DISABLED, 
		MUX_SEL_CHGIN_VOLTAGE_MONITOR, 
		MUX_SEL_CHGIN_CURRENT_MONITOR, 
		MUX_SEL_BATTERY_VOLTAGE_MONITOR,
		MUX_SEL_BATTERY_CHARGE_CURRENT_MONITOR, 
		MUX_SEL_BATTERY_DISCHARGE_CURRENT_MONITOR_NORMAL, 
		MUX_SEL_BATTERY_DISCHARGE_CURRENT_MONITOR_NULL, 
		MUX_SEL_RESERVED_0x07,
		MUX_SEL_RESERVED_0x08,
		MUX_SEL_AGND_VOLTAGE_MONITOR,
		MUX_SEL_SYS_VOLTAGE_MONITOR,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0B,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0C,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0D,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0E,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0F
    }decode_mux_sel_t;

    /**
	* @brief		Set the analog channel to connect to AMUX.
	*
	* @param[in]	selection AMUX value field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_mux_sel(decode_mux_sel_t selection);

    /**
	* @brief		Get the analog channel to connect to AMUX.
	*
	* @param[out]	selection AMUX value field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_mux_sel(decode_mux_sel_t *selection);
	
	/*SBB*/
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : CNFG_SBB_TOP (0x38)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Watchdog Timer Configuration.
    */
    typedef enum {
    	CNFG_SBB_TOP_DRV_SBB,
		CNFG_SBB_TOP_DIS_LPM
    }reg_bit_cnfg_sbb_top_t;	
	
	/**
	* @brief		Set CNFG_SBB_TOP (0x38) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[in]	config 		Configuration value to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_cnfg_sbb_top(reg_bit_cnfg_sbb_top_t bit_field, uint8_t config);

    /**
	* @brief		Get CNFG_SBB_TOP (0x38) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	config 		Configuration value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_cnfg_sbb_top(reg_bit_cnfg_sbb_top_t bit_field, uint8_t *config);
	
	/**
	* @brief		Set SIMO Buck-Boost Channel x Target Output Voltage.
	*				CNFG_SBB0_A (0x39), CNFG_SBB1_A (0x3B) and CNFG_SBB2_A (0x3D)
	*
	* @param[in]	channel 	Channel number: 0, 1 or 2.
	* @param[in]	voltV 		SIMO buck-boost channel x target output voltage field to be written.
	*							SBBx = 500mV + 25mV x TV_SBBx[7:0]
	*							0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*							0.650V, 0.675V, 0.700V, ... 
	*							5.425V, 5.450V, 5.475V, 5.500V.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_tv_sbb_a(uint8_t channel, float voltV);

    /**
	* @brief		Get SIMO Buck-Boost Channel x Target Output Voltage.
	*				CNFG_SBB0_A (0x39), CNFG_SBB1_A (0x3B) and CNFG_SBB2_A (0x3D)
	*
	* @param[in]	channel 	Channel number: 0, 1 or 2.
	* @param[out]	voltV 		SIMO buck-boost channel x target output voltage field to be read.
	*							SBBx = 500mV + 25mV x TV_SBBx[7:0]
	*							0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*							0.650V, 0.675V, 0.700V, ... 
	*							5.425V, 5.450V, 5.475V, 5.500V.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_tv_sbb_a(uint8_t channel, float *voltV);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x3A), CNFG_SBB1_B (0x3C) and CNFG_SBB2_B (0x3E)
	*  - Bit Fields    : [7:6]
	*  - Default       : 0x0
	*  - Description   : Operation mode of SBB0, 1 or 2.
	*/
    typedef enum {
        OP_MODE_AUTOMATIC,
        OP_MODE_BUCK_MODE,
        OP_MODE_BOOST_MODE,
        OP_MODE_BUCK_BOOST_MODE
    }decode_op_mode_t;

    /**
	* @brief		Set Operation mode of SBBx.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	mode 	Operation mode of SBBx bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_op_mode(uint8_t channel, decode_op_mode_t mode);

    /**
	* @brief		Get Operation mode of SBBx.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	mode 	Operation mode of SBBx bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_op_mode(uint8_t channel, decode_op_mode_t *mode);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x3A), CNFG_SBB1_B (0x3C) and CNFG_SBB2_B (0x3E)
	*  - Bit Fields    : [5:4]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost Channel 0, 1 or 2 Peak Current Limit.
	*/
    typedef enum {
        IP_SBB_AMP_1_000A,
        IP_SBB_AMP_0_750A,
        IP_SBB_AMP_0_500A,
        IP_SBB_AMP_0_333A
    }decode_ip_sbb_t;

    /**
	* @brief		Set SIMO Buck-Boost Channel x Peak Current Limit.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	ip_sbb 	SIMO buck-boost channel 2 peak current limit field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_ip_sbb(uint8_t channel, decode_ip_sbb_t ip_sbb);

    /**
	* @brief		Get SIMO Buck-Boost Channel x Peak Current Limit.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	ip_sbb 	SIMO buck-boost channel 2 peak current limit field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_ip_sbb(uint8_t channel, decode_ip_sbb_t *ip_sbb);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x3A), CNFG_SBB1_B (0x3C) and CNFG_SBB2_B (0x3E)
	*  - Bit Fields    : [3]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost Channel 0, 1 or 2 Active-Discharge Enable.
	*/
    typedef enum {
        ADE_SBB_DISABLED,
        ADE_SBB_ENABLED
    }decode_ade_sbb_t;

    /**
	* @brief		Set SIMO Buck-Boost Channel x Active-Discharge Enable.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	ade_sbb SIMO buck-boost channel 2 active-discharge enable bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_ade_sbb(uint8_t channel, decode_ade_sbb_t ade_sbb);

    /**
	* @brief		Get SIMO Buck-Boost Channel x Active-Discharge Enable.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	ade_sbb SIMO buck-boost channel 2 active-discharge enable bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_ade_sbb(uint8_t channel, decode_ade_sbb_t *ade_sbb);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x3A), CNFG_SBB1_B (0x3C) and CNFG_SBB2_B (0x3E)
	*  - Bit Fields    : [2:0]
	*  - Default       : 0x0
	*  - Description   : Enable Control for SIMO Buck-Boost Channel 0, 1 or 2.
	*/
    typedef enum {
        EN_SBB_FPS_SLOT_0,
        EN_SBB_FPS_SLOT_1,
        EN_SBB_FPS_SLOT_2,
        EN_SBB_FPS_SLOT_3,
        EN_SBB_OFF,
        EN_SBB_SAME_AS_0X04,
        EN_SBB_ON,
        EN_SBB_SAME_AS_0X06
    }decode_en_sbb_t;

    /**
	* @brief		Set Enable Control for SIMO Buck-Boost Channel x.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	en_sbb 	Enable control for SIMO buck-boost channel x field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_en_sbb(uint8_t channel, decode_en_sbb_t en_sbb);

    /**
	* @brief		Get Enable Control for SIMO Buck-Boost Channel x.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	en_sbb 	Enable control for SIMO buck-boost channel x field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_en_sbb(uint8_t channel, decode_en_sbb_t *en_sbb);
	
	/**
	* @brief		Set SIMO Buck-Boost Channel 0 Target Output Voltage.
	*				Bit 7:0 of CNFG_DVS_SBB0_A (0x3F).
	*
	* @param[in]	voltV 	SIMO buck-boost channel 0 target output voltage field to be written.
	*						SBBx = 500mV + 25mV x TV_SBBx[7:0]
	*						0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*						0.650V, 0.675V, 0.700V, ... 
	*						5.425V, 5.450V, 5.475V, 5.500V.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_tv_sbb_dvs(float voltV);

    /**
	* @brief		Get SIMO Buck-Boost Channel 0 Target Output Voltage.
	*				Bit 7:0 of CNFG_DVS_SBB0_A (0x3F).
	*
	* @param[out]	voltV 	SIMO buck-boost channel 0 target output voltage field to be read.
	*						SBBx = 500mV + 25mV x TV_SBBx[7:0]
	*						0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*						0.650V, 0.675V, 0.700V, ... 
	*						5.425V, 5.450V, 5.475V, 5.500V.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_tv_sbb_dvs(float *voltV);
	
	/*LDO*/
	
	/**
	* @brief		Set LDO Output Channel x Target Output Voltage. Bit 6:0.
	*				CNFG_LDO0_A (0x48) and CNFG_LDO1_A (0x4A)
	*
	* @param[in]	channel 	Channel number: 0 or 1.
	* @param[in]	voltV 		LDO Output Channel x target output voltage field to be read.
	*							LDOx = 500mV + 25mV x TV_LDOx[6:0]
	*							0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*							0.650V, 0.675V, 0.700V, ... 
	*							3.650, 3.675.
	*
	*							When TV_LDO[7] = 0, TV_LDO[6:0] sets the
	*							LDO's output voltage range from 0.5V to 3.675V.
	*							When TV_LDO[7] = 1, TV_LDO[6:0] sets the
	*							LDO's output voltage from 1.825V to 5V.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_tv_ldo_volt_a(uint8_t channel, float voltV);

    /**
	* @brief		Get LDO Output Channel x Target Output Voltage. Bit 6:0.
	*				CNFG_LDO0_A (0x48) and CNFG_LDO1_A (0x4A)
	*
	* @param[in]	channel 	Channel number: 0 or 1.
	* @param[out]	voltV 		LDO Output Channel x target output voltage field to be read.
	*							LDOx = 500mV + 25mV x TV_LDOx[6:0]
	*							0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*							0.650V, 0.675V, 0.700V, ... 
	*							3.650, 3.675.
	*
	*							When TV_LDO[7] = 0, TV_LDO[6:0] sets the
	*							LDO's output voltage range from 0.5V to 3.675V.
	*							When TV_LDO[7] = 1, TV_LDO[6:0] sets the
	*							LDO's output voltage from 1.825V to 5V.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_tv_ldo_volt_a(uint8_t channel, float *voltV);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_A (0x48) and CNFG_LDO1_A (0x4A)
	*  - Bit Fields    : [7]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost Channel 0, 1 or 2 Peak Current Limit.
	*/
    typedef enum {
        TV_LDO0_A_NO_OFFSET,
        TV_LDO0_A_NO_1_325V
    }decode_tv_ldo_offset_a_t;
	
	/**
	* @brief		Set LDO Output Channel x Target Output Voltage. Bit 7.
	*				CNFG_LDO0_A (0x48) and CNFG_LDO1_A (0x4A)
	*
	* @param[in]	channel 	Channel number: 0 or 1.
	* @param[in]	offset 		LDO Output Channel x target output voltage offset field to be read.
	*							0b0 = No Offset
	*							0b1 = 1.325V Offset
	*
	*
	* @return		0 on success, error code on failure.
	*/
    int set_tv_ldo_offset_a(uint8_t channel, decode_tv_ldo_offset_a_t offset);

    /**
	* @brief		Get LDO Output Channel x Target Output Voltage. Bit 7.
	*				CNFG_LDO0_A (0x48) and CNFG_LDO1_A (0x4A)
	*
	* @param[in]	channel 	Channel number: 0 or 1.
	* @param[out]	offset 		LDO Output Channel x target output voltage offset field to be read.
	*							0b0 = No Offset
	*							0b1 = 1.325V Offset
	*
	* @return		0 on success, error code on failure.
	*/
    int get_tv_ldo_offset_a(uint8_t channel, decode_tv_ldo_offset_a_t *offset);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_B (0x49) and CNFG_LDO1_B (0x4B)
	*  - Bit Fields    : [2:0]
	*  - Default       : 0x0
	*  - Description   : Enable Control for LDO 0 or 1.
	*/
    typedef enum {
        EN_LDO_FPS_SLOT_0,
        EN_LDO_FPS_SLOT_1,
        EN_LDO_FPS_SLOT_2,
        EN_LDO_FPS_SLOT_3,
        EN_LDO_OFF,
        EN_LDO_SAME_AS_0X04,
        EN_LDO_ON,
        EN_LDO_SAME_AS_0X06
    }decode_en_ldo_t;

    /**
	* @brief		Set Enable Control for LDO Channel x.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	en_ldo 	Enable control for LDO channel x field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_en_ldo(uint8_t channel, decode_en_ldo_t en_ldo);

    /**
	* @brief		Get Enable Control for LDO Channel x.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	en_ldo 	Enable control for LDO channel x field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_en_ldo(uint8_t channel, decode_en_ldo_t *en_ldo);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_B (0x49) and CNFG_LDO1_B (0x4B)
	*  - Bit Fields    : [3]
	*  - Default       : 0x0
	*  - Description   : LDO Channel 0 or 1 Active-Discharge Enable.
	*/
    typedef enum {
        ADE_LDO_DISABLED,
        ADE_LDO_ENABLED
    }decode_ade_ldo_t;

    /**
	* @brief		Set LDO Channel x Active-Discharge Enable.
	*
	* @param[in]	channel Channel number: 0 or 1.
	* @param[in]	ade_ldo LDO channel x active-discharge enable bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_ade_ldo(uint8_t channel, decode_ade_ldo_t ade_ldo);

    /**
	* @brief		Get LDO Channel x Active-Discharge Enable.
	*
	* @param[in]	channel Channel number: 0 or 1.
	* @param[out]	ade_ldo LDO channel x active-discharge enable bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_ade_ldo(uint8_t channel, decode_ade_ldo_t *ade_ldo);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_B (0x49) and CNFG_LDO1_B (0x4B)
	*  - Bit Fields    : [4]
	*  - Default       : 0x0
	*  - Description   : Operation mode of LDO 0 or 1.
	*/
    typedef enum {
        LDO_MD_LDO_MODE,
        LDO_MD_LSW_MODE
    }decode_ldo_md_t;

    /**
	* @brief		Set Operation mode of LDOx.
	*
	* @param[in]	channel Channel number: 0 or 1.
	* @param[in]	mode 	Operation mode of LDOx bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_ldo_md(uint8_t channel, decode_ldo_md_t mode);

    /**
	* @brief		Get Operation mode of LDOx.
	*
	* @param[in]	channel Channel number: 0 or 1.
	* @param[out]	mode 	Operation mode of LDOx bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_ldo_md(uint8_t channel, decode_ldo_md_t *mode);
	
	/*FuelGauge*/
	/*Status and Configuration Registers*/
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : Fuel-gauge Status (0x00)
    *  - Bit Fields    : [15:0]
    *  - Default       : 0x0000
    *  - Description   : All Status register bit fields.
    */
    typedef enum {
		Status_Imn,
		Status_POR,
		Status_SPR_2,
		Status_Bst,
		Status_Isysmx,
		Status_SPR_5,
		Status_ThmHot,
		Status_dSOCi,
		Status_Vmn,
		Status_Tmn,
		Status_Smn,
		Status_Bi,
		Status_Vmx,
		Status_Tmx,
		Status_Smx,
		Status_Br
    }reg_bit_status_t;
	
	/**
    * @brief		Set bit field of Fuel-gauge Status[15:0] (0x00)  register.
    *
    * @param[in]	bit_field Status bit field to be written to register.
    * @param[in]	status 	to be written to register.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_status(reg_bit_status_t bit_field, uint8_t status);

    /**
    * @brief		Get bit field of Fuel-gauge Status[15:0] (0x00) register.
    *
	* @param[in]	bit_field Status bit field to be written to register.
    * @param[out] 	status 	Pointer to save result value.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_status(reg_bit_status_t bit_field, uint8_t *status);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : Fuel-gauge VAlrtTh (0x01)
    *  - Bit Fields    : [15:0]
    *  - Default       : 0xFF00 (Disabled) 
    *  - Description   : The VAlrtTh register sets upper and lower limits that generate an alert if exceeded by the VCell register value.
    */
    typedef enum {
		VAlrtTh_MinVoltageAlrt,
		VAlrtTh_MaxVoltageAlrt
    }reg_bit_valrt_th_t;
	
	/**
    * @brief		Set bit field of fuel-gauge VAlrtTh (0x01) register.
    *
    * @param[in]	bit_field 	VAlrtTh register bit field to be written.
    * @param[in]	voltV 		to be written to register.
	*							Interrupt threshold limits are selectable with 20mV resolution 
	*							over the full operating range of the VCell register. 
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_valrt_th(reg_bit_valrt_th_t bit_field, float voltV);

    /**
    * @brief		Get bit field of fuel-gauge VAlrtTh (0x01) register.
    *
	* @param[in]	bit_field 	VAlrtTh register bit field to be written.
    * @param[out] 	voltV 		Pointer to save result value.
	*							Interrupt threshold limits are selectable with 20mV resolution 
	*							over the full operating range of the VCell register. 
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_valrt_th(reg_bit_valrt_th_t bit_field, float *voltV);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : Fuel-gauge TAlrtTh (0x02)
    *  - Bit Fields    : [15:0]
    *  - Default       : 0x7F80 (Disabled)
    *  - Description   : All TAlrtTh register bit fields.
    */
    typedef enum {
		TAlrtTh_MinTempAlrt,
		TAlrtTh_MaxTempAlrt
    }reg_bit_talrt_th_t;
	
	/**
    * @brief		Set bit field of fuel-gauge TAlrtTh (0x02) register.
    *
    * @param[in]	bit_field 	TAlrtTh register bit field to be written.
    * @param[in]	tempDegC 	to be written to register.
	*							Interrupt threshold limits are stored in two’s-complement format 
	*							with 1°C resolution over the full operating range of the Temp register. 
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_talrt_th(reg_bit_talrt_th_t bit_field, int tempDegC);

    /**
    * @brief		Get bit field of fuel-gauge TAlrtTh (0x02) register.
    *
	* @param[in]	bit_field 	TAlrtTh register bit field to be written.
    * @param[out] 	tempDegC 	Pointer to save result value.
	*							Interrupt threshold limits are stored in two’s-complement format 
	*							with 1°C resolution over the full operating range of the Temp register. 
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_talrt_th(reg_bit_talrt_th_t bit_field, int *tempDegC);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : Fuel-gauge SAlrtTh (0x03)
    *  - Bit Fields    : [15:0]
    *  - Default       : 0xFF00
    *  - Description   : The SAlrtTh register sets upper and lower limits that generate an alert if exceeded by RepSOC.
    */
    typedef enum {
		SAlrtTh_MinSocAlrt,
		SAlrtTh_MaxSocAlrt
    }reg_bit_salrt_th_t;
	
	/**
    * @brief		Set bit field of fuel-gauge SAlrtTh (0x03) register.
    *
    * @param[in]	bit_field 	SAlrtTh register bit field to be written.
    * @param[in]	soc 		to be written to register.
	*							Interrupt threshold limits are configurable with 1% resolution 
	*							over the full operating range of the RepSOC register. 
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_salrt_th(reg_bit_salrt_th_t bit_field, uint8_t soc);

    /**
    * @brief		Get bit field of fuel-gauge SAlrtTh (0x03) register.
    *
	* @param[in]	bit_field 	SAlrtTh register bit field to be written.
    * @param[out] 	soc			Pointer to save result value.
	*							Interrupt threshold limits are configurable with 1% resolution 
	*							over the full operating range of the RepSOC register. 
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_salrt_th(reg_bit_salrt_th_t bit_field, uint8_t *soc);	
	
	/**
    * @brief		Set bit field of fuel-gauge FullSocThr (0x13) register.
    *
    * @param[in]	soc_thr to be written to register.
	*						FullSOCThr comes from OTP if the OTP register is enabled. 
	*						Otherwise it POR’s to a default of 95%.
	*						LSB unit is 1/256%.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_full_soc_thr(float soc_thr);

    /**
    * @brief		Get bit field of fuel-gauge FullSocThr (0x13) register.
    *
    * @param[out] 	soc_thr Pointer to save result value.
	*						FullSOCThr comes from OTP if the OTP register is enabled. 
	*						Otherwise it POR’s to a default of 95%.
	*						LSB unit is 1/256%.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_full_soc_thr(float *soc_thr);
	
	/**
    * @brief		Set bit field of fuel-gauge DesignCap (0x18) register.
    *
    * @param[in]	capacitymAh to be written to register.
	*				LSB unit is 0.1mAh. Min is 0.0mAh and Max is 6553.5mAh.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_design_cap(float capacitymAh);

    /**
    * @brief		Get bit field of fuel-gauge DesignCap (0x18) register.
    *
    * @param[out] 	capacitymAh Pointer to save result value.
	*				LSB unit is 0.1mAh. Min is 0.0mAh and Max is 6553.5mAh.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_design_cap(float *capacitymAh);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : Fuel-gauge Config (0x1D)
    *  - Bit Fields    : [15:0]
    *  - Default       : 0x0000
    *  - Description   : The Config registers hold all shutdown enable, alert enable, 
	*					 and temperature enable control bits.
    */
    typedef enum {
		Config_Ber,
		Config_Bei,
		Config_Aen,
		Config_FTHRM,
		Config_ETHRM,
		Config_SPR_5,
		Config_I2CSH,
		Config_SHDN,
		Config_Tex,
		Config_Ten,
		Config_AINSH,
		Config_SPR_11,
		Config_Vs,
		Config_Ts,
		Config_Ss,
		Config_SPR_15
    }reg_bit_config_t;
	
	/**
    * @brief		Set bit field of fuel-gauge Config (0x1D) register.
    *
	* @param[in]	bit_field 	Config bit field to be written to register.
    * @param[in]	config 		to be written to register.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_config(reg_bit_config_t bit_field, uint8_t config);

    /**
    * @brief		Get bit field of fuel-gauge Config (0x1D) register.
    *
	* @param[in]	bit_field 	Config bit field to be written to register.
    * @param[out] 	config 		Pointer to save result value.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_config(reg_bit_config_t bit_field, uint8_t *config);
	
	/**
    * @brief		Set bit field of fuel-gauge IChgTerm (0x1E) register.
    *
    * @param[in]	currentA to be written to register.
	*				The current register has a LSB value of 156.25μA, 
	*				a register scale range of ± 5.12 A.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_ichg_term(float currentA);

    /**
    * @brief		Get bit field of fuel-gauge IChgTerm (0x1E) register.
    *
    * @param[out] 	currentA Pointer to save result value.
	*				The current register has a LSB value of 156.25μA, 
	*				a register scale range of ± 5.12 A.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_ichg_term(float *currentA);
	
	/**
     * @brief		Get bit field of fuel-gauge DevName (0x21) register.
     *
     * @param[out] 	value Pointer to save result value.
     *
     * @return		0 on success, error code on failure.
     */
    int get_fg_dev_name(uint16_t *value);
	
	/**
	* @brief		Set the filtering for empty learning for both the I_Avgempty and QRTable registers.
	*				Bit 15:14 of FilterCfg (0x29) register.
	*
	* @param[in]	nempty 		Iavg_empty is learned with(NEMPTY+3) right shifts. QRTable is learned with 
	*				(NEMPTY + sizeof(Iavgempty) – sizeof (AvgCurrent)) right shifts.
	* @return		0 on success, error code on failure.
	*/
    int set_fg_nempty(uint8_t nempty);

    /**
	* @brief		Get the filtering for empty learning for both the I_Avgempty and QRTable registers.
	*				Bit 15:14 of FilterCfg (0x29) register.
	*
	* @param[out]	nempty 		Iavg_empty is learned with(NEMPTY+3) right shifts. QRTable is learned with 
	*				(NEMPTY + sizeof(Iavgempty) – sizeof (AvgCurrent)) right shifts.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_fg_nempty(uint8_t *nempty);
	
	/**
	* @brief		Set the time constant for the mixing algorithm.
	*				Bit 10:7 of FilterCfg (0x29) register.
	*
	* @param[in]	second 	The equation setting the period is:
	*				Mixing Period = 175.8ms × 2^(5+NMIX) to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_fg_nmix(float second);

    /**
	* @brief		Get the time constant for the mixing algorithm.
	*				Bit 10:7 of FilterCfg (0x29) register.
	*
	* @param[out]	second 	The equation setting the period is:
	*				Mixing Period = 175.8ms × 2^(5+NMIX) to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_fg_nmix(float *second);
	
	/**
	* @brief		Set the time constant for the AverageVCELL register.
	*				Bit 6:4 of FilterCfg (0x29) register.
	*
	* @param[in]	second 	The equation setting the period is:
	*				AverageVCELL time constant = 175.8ms × 2^(6+NAVGVCELL) to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_fg_navgcell(float second);

    /**
	* @brief		Get the time constant for the AverageVCELL register.
	*				Bit 6:4 of FilterCfg (0x29) register.
	*
	* @param[out]	second 	The equation setting the period is:
	*				AverageVCELL time constant = 175.8ms × 2^(6+NAVGVCELL) to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_fg_navgcell(float *second);
	
	/**
	* @brief		Set the time constant for the AverageCurrent register.
	*				Bit 3:0 of FilterCfg (0x29) register.
	*
	* @param[in]	second 	The equation setting the period is:
	*				AverageCurrent time constant = 175.8ms × 2^(2+NCURR) to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_fg_ncurr(float second);

    /**
	* @brief		Get the time constant for the AverageCurrent register.
	*				Bit 3:0 of FilterCfg (0x29) register.
	*
	* @param[out]	second 	The equation setting the period is:
	*				AverageCurrent time constant = 175.8ms × 2^(2+NCURR) to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_fg_ncurr(float *second);
	
	/**
    * @brief		Set bit field of fuel-gauge IAvgEmpty (0x36) register.
    *
    * @param[in]	currentA 	to be written to register.
	*				The current register has a LSB value of 156.25μA, 
	*				a register scale range of ± 5.12 A.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_iavg_empty(float currentA);

    /**
    * @brief		Get bit field of fuel-gauge IAvgEmpty (0x36) register.
    *
    * @param[out] 	currentA 	Pointer to save result value.
	*				The current register has a LSB value of 156.25μA, 
	*				a register scale range of ± 5.12 A.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_iavg_empty(float *currentA);
	
	/**
	* @brief		Set the voltage level for detecting empty.
	*				Bit 15:7 of VEmpty(0x3A) register.
	*
	* @param[in]	voltV 	Empty voltage to be written.
	*				A 10mV resolution gives a 0 to 5.11V range.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_fg_v_empty(float voltV);

    /**
	* @brief		Get the voltage level for detecting empty.
	*				Bit 15:7 of VEmpty(0x3A) register.
	*
	* @param[out]	voltV 	Empty voltage to be read.
	*				A 10mV resolution gives a 0 to 5.11V range.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_fg_v_empty(float *voltV);
	
	/**
	* @brief		Set the voltage level for clearing empty detection.
	*				Bit 6:0 of VEmpty(0x3A) register.
	*
	* @param[in]	voltV 	Recovery voltage to be written.
	*				A 40mV resolution gives a 0 to 5.08V range.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_fg_v_recover(float voltV);

    /**
	* @brief		Get the voltage level for clearing empty detection.
	*				Bit 6:0 of VEmpty(0x3A) register.
	*
	* @param[out]	voltV 	Recovery voltage to be read.
	*				A 40mV resolution gives a 0 to 5.08V range.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_fg_v_recover(float *voltV);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : Fuel-gauge Config2 (0xBB)
    *  - Bit Fields    : [15:0]
    *  - Default       : 0x0000
    *  - Description   : The Config registers hold all shutdown enable, alert enable, 
	*					 and temperature enable control bits.
    */
    typedef enum {
		Config2_ISysNCurr,
		Config2_OCVQen,
		Config2_LdMdl,
		Config2_TAlrtEn,
		Config2_dSOCen,
		Config2_ThmHotAlrtEn,
		Config2_ThmHotEn,
		Config2_FCThmHot,
		Config2_SPR
    }reg_bit_config2_t;
		
	/**
	* @brief		Set Config2 (0xBB) register holding all shutdown enable, alert enable, 
	*				and temperature enable control bits.
	*
	* @param[in]	bit_field 	Config bit field to be written to register.
	* @param[in]	config2 	value to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_fg_config2(reg_bit_config2_t bit_field, uint8_t config2);

    /**
	* @brief		Get Config2 (0xBB) register holding all shutdown enable, alert enable, 
	*				and temperature enable control bits.
	*
	* @param[in]	bit_field 	Config bit field to be written to register.
	* @param[out]	config2 	value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_fg_config2(reg_bit_config2_t bit_field, uint8_t *config2);
	
	/**
	* @brief		Set the time constant for the AvgISys register.
	*				Bit 3:0 of Config2(0xBB) register.
	*
	* @param[in]	second 		Time constant to be written.
	*				The default POR value of 0100b gives a time constant of 5.625. 
	*				The equation setting the period is: AvgISys time constant = 45s x 2^(ISysNCurr-7).
	*
	* @return		0 on success, error code on failure.
	*/
    int set_fg_isys_ncurr(float second);

    /**
	* @brief		Get the time constant for the AvgISys register.
	*				Bit 3:0 of Config2(0xBB) register.
	*
	* @param[out]	second 		Time constant to be read.
	*				The default POR value of 0100b gives a time constant of 5.625. 
	*				The equation setting the period is: AvgISys time constant = 45s x 2^(ISysNCurr-7).
	*
	* @return		0 on success, error code on failure.
	*/
    int get_fg_isys_ncurr(float *second);
	
	/*Measurement Registers*/
	
	/**
    * @brief		Set bit field of fuel-gauge Temp (0x08) register.
    *
    * @param[in]	tempDegC 	to be written to register.
	*				The LSB is 0.0039˚C.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_temp(float tempDegC);

    /**
    * @brief		Get bit field of fuel-gauge Temp (0x08) register.
    *
    * @param[out] 	tempDegC 	Pointer to save result value.
	*				The LSB is 0.0039˚C.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_temp(float *tempDegC);
	
	/**
    * @brief		Set bit field of fuel-gauge Vcell (0x09) register.
	*				VCell reports the voltage measured between BATT and GND.
    *
    * @param[in]	voltV 	to be written to register.
	*						VCell register reports the 2.5X the voltage measured at the CELLX pin.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_vcell(float voltV);

    /**
    * @brief		Get bit field of fuel-gauge Vcell (0x09) register.
	*				VCell reports the voltage measured between BATT and GND.
    *
    * @param[out] 	voltV 	Pointer to save result value.
	*						VCell register reports the 2.5X the voltage measured at the CELLX pin.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_vcell(float *voltV);
	
	/**
    * @brief		Set bit field of fuel-gauge Current (0x0A) register.			
    *
    * @param[in]	currentA 	to be written to register.
	*				The current register has a LSB value of 31.25uA, a register scale of 1.024A
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_current(float currentA);

    /**
    * @brief		Get bit field of fuel-gauge Current (0x0A) register.
    *
    * @param[out] 	currentA 	Pointer to save result value.
	*				The current register has a LSB value of 31.25uA, a register scale of 1.024A.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_current(float *currentA);
	
	/**
    * @brief		Set bit field of fuel-gauge AvgCurrent (0x0B) register.
    *
    * @param[in]	currentA 	to be written to register.
	*				The current register has a LSB value of 31.25uA, a register scale of 1.024A
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_avg_current(float currentA);

    /**
    * @brief		Get bit field of fuel-gauge AvgCurrent (0x0B) register.
    *
    * @param[out] 	currentA 	Pointer to save result value.
	*				The current register has a LSB value of 31.25uA, a register scale of 1.024A
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_avg_current(float *currentA);
	
	/**
    * @brief		Set bit field of fuel-gauge AvgTA (0x16) register.
    *
    * @param[in]	tempDegC 	to be written to register.
	*				LSB is 1/256°C. Min value is -128.0°C and Max value is 127.996°C. 
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_avgta(float tempDegC);

    /**
    * @brief		Get bit field of fuel-gauge AvgTA (0x16) register.
    *
    * @param[out] 	tempDegC 	Pointer to save result value.
	*				LSB is 1/256°C. Min value is -128.0°C and Max value is 127.996°C. 
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_avgta(float *tempDegC);
	
	/**
    * @brief		Set bit field of fuel-gauge AvgVCell (0x19) register.
    *
    * @param[in]	voltV to be written to register.
	*				LSB is 1.25mV/16. Min value is 0.0V and Max value is 5.11992V. 
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_avgvcell(float voltV);

    /**
    * @brief		Get bit field of fuel-gauge AvgVCell (0x19) register.
    *
    * @param[out] 	voltV Pointer to save result value.
	*				LSB is 1.25mV/16. Min value is 0.0V and Max value is 5.11992V. 
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_avgvcell(float *voltV);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : Fuel-gauge MaxMinTemp (0x1A)
    *  - Bit Fields    : [15:0]
    *  - Default       : 0x7F80
    *  - Description   : All MaxMinTemp register bit fields.
    */
    typedef enum {
		MaxMinTemp_MinTemperature,
		MaxMinTemp_MaxTemperature
    }reg_bit_max_min_temp_t;
	
	/**
    * @brief		Set bit field of fuel-gauge MaxMinTemp (0x1A) register.
    *
    * @param[in]	bit_field	to be written to register.
    * @param[in]	tempDegC 	to be written to register.
	*				The maximum and minimum temperatures are each stored as two’s complement 
	*				8-bit values with 1°C resolution. 
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_max_min_temp(reg_bit_max_min_temp_t bit_field, int tempDegC);

    /**
    * @brief		Get bit field of fuel-gauge MaxMinTemp (0x1A) register.
    *
	* @param[in]	bit_field	to be written to register.
    * @param[out] 	tempDegC 	Pointer to save result value.
	*				The maximum and minimum temperatures are each stored as two’s complement 
	*				8-bit values with 1°C resolution.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_max_min_temp(reg_bit_max_min_temp_t bit_field, int *tempDegC);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : Fuel-gauge MaxMinVolt (0x1B)
    *  - Bit Fields    : [15:0]
    *  - Default       : 0x00FF
    *  - Description   : All MaxMinVolt register bit fields.
    */
    typedef enum {
		MaxMinVolt_MinVoltage,
		MaxMinVolt_MaxVoltage
    }reg_bit_max_min_volt_t;
	
	/**
    * @brief		Set bit field of fuel-gauge MaxMinVolt (0x1B) register.
    *
    * @param[in]	bit_field 	MaxMinVolt register bit field to be written.
    * @param[in]	voltV 		to be written to register.
	*				The maximum and minimum voltages are each stored as 8-bit
	*				values with a 20mV resolution.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_max_min_volt(reg_bit_max_min_volt_t bit_field, float voltV);

    /**
    * @brief		Get bit field of fuel-gauge MaxMinVolt (0x1B) register.
    *
	* @param[in]	bit_field 	MaxMinVolt register bit field to be written.
    * @param[out] 	voltV 		Pointer to save result value.
	*				The maximum and minimum voltages are each stored as 8-bit
	*				values with a 20mV resolution.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_max_min_volt(reg_bit_max_min_volt_t bit_field, float *voltV);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : Fuel-gauge MaxMinCurr (0x1C)
    *  - Bit Fields    : [15:0]
    *  - Default       : 0x807F
    *  - Description   : All MaxMinCurr register bit fields.
    */
    typedef enum {
		MaxMinCurr_MaxDisCurrent,
		MaxMinCurr_MaxChargeCurrent
    }reg_bit_max_min_curr_t;
	
	/**
    * @brief		Set bit field of fuel-gauge MaxMinCurr (0x1C) register.
    *
    * @param[in]	bit_field 	MaxMinCurr register bit field to be written.
    * @param[in]	currentA 	to be written to register.
	*				The maximum and minimum currents are each stored 
	*				as two’s complement 8-bit values with 8mA resolution.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_max_min_curr(reg_bit_max_min_curr_t bit_field, float currentA);

    /**
    * @brief		Get bit field of fuel-gauge MaxMinCurr (0x1C) register.
    *
	* @param[in]	bit_field 	MaxMinCurr register bit field to be written.
    * @param[out] 	currentA 	Pointer to save result value.
	*				The maximum and minimum currents are each stored 
	*				as two’s complement 8-bit values with 8mA resolution.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_max_min_curr(reg_bit_max_min_curr_t bit_field, float *currentA);
	
	/**
    * @brief		Set bit field of fuel-gauge AIN0 (0x27) register.
    *
    * @param[in]	percent 	to be written to register.
	*				AIN register with an LSB of 0.0122%. The value from 0% to 100%.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_ain0(float percent);

    /**
    * @brief		Get bit field of fuel-gauge AIN0 (0x27) register.
    *
    * @param[out] 	percent 	Pointer to save result value.
	*				AIN register with an LSB of 0.0122%. The value from 0% to 100%.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_ain0(float *percent);
	
	/**
    * @brief		Set bit field of fuel-gauge Timer (0x3E) register.
    *
    * @param[in]	second 		to be written to register.
	*				The Timer register LSB is 175.8ms, giving a full-scale range of 0 to 3.2 hours.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_timer(float second);

    /**
    * @brief		Get bit field of fuel-gauge Timer (0x3E) register.
    *
    * @param[out] 	second 		Pointer to save result value.
	*				The Timer register LSB is 175.8ms, giving a full-scale range of 0 to 3.2 hours.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_timer(float *second);
	
	/**
    * @brief		Set bit field of fuel-gauge ShdnTimer (0x3F) register.
	*				Shutdown Counter. Bit 12:0.
    *
    * @param[in]	second 		to be written to register.
	*				The counter LSB is 1.4s.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_shdnctr(float second);

    /**
    * @brief		Get bit field of fuel-gauge ShdnTimer (0x3F) register.
	*				Shutdown Counter. Bit 12:0.
    *
    * @param[out] 	second 		Pointer to save result value.
	*				The counter LSB is 1.4s.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_shdnctr(float *second);
	
	/**
    * @brief		Set bit field of fuel-gauge ShdnTimer (0x3F) register.
	*				Shutdown timeout period from a minimum of 45s to a maximum of 1.6h. Bit 15:13.
    *
    * @param[in]	second 		to be written to register.
	*				Shutdown Timeout Period = 175.8ms × 2^(8+THR).
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_shdn_thr(float second);

    /**
    * @brief		Get bit field of fuel-gauge ShdnTimer (0x3F) register.
	*				Shutdown timeout period from a minimum of 45s to a maximum of 1.6h. Bit 15:13.
    *
    * @param[out] 	second 	Pointer to save result value.
	*				Shutdown Timeout Period = 175.8ms × 2^(8+THR).
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_shdn_thr(float *second);
	
	/**
    * @brief		Set bit field of fuel-gauge TimerH (0xBE) register.
    *
    * @param[in]	hour 	to be written to register.
	*				A 3.2-hour LSB gives a full-scale range for the register of up to 23.94 years.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_timerh(float hour);

    /**
    * @brief		Get bit field of fuel-gauge TimerH (0xBE) register.
    *
    * @param[out] 	hour 	Pointer to save result value.
	*				A 3.2-hour LSB gives a full-scale range for the register of up to 23.94 years.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_timerh(float *hour);
	
	/*ModelGauge m5 Output Registers*/
	
	/**
    * @brief		Set bit field of fuel-gauge RepCap (0x05) register.
	*				RepCap is the reported remaining capacity in mAh.
    *
    * @param[in]	repCapmAh 	to be written to register.
	*				LSB is 0.1mAh. Min value is 0.0mAh and Max value is 6553.5mAh.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_rep_cap(float repCapmAh);

    /**
    * @brief		Get bit field of fuel-gauge RepCap (0x05) register.
	*				RepCap is the reported remaining capacity in mAh.
    *
    * @param[out] 	repCapmAh 	Pointer to save result value.
	*				LSB is 0.1mAh. Min value is 0.0mAh and Max value is 6553.5mAh.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_rep_cap(float *repCapmAh);
	
	/**
    * @brief		Set bit field of fuel-gauge RepSOC (0x06) register.
    *
    * @param[in]	percent 	to be written to register.
	*				LSB is 1/256%. Min value is 0.0% and Max value is 255.9961%.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_rep_soc(float percent);

    /**
    * @brief		Get bit field of fuel-gauge RepSOC (0x06) register.
    *
    * @param[out] 	percent 	Pointer to save result value.
	*				LSB is 1/256%. Min value is 0.0% and Max value is 255.9961%.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_rep_soc(float *percent);
	
	/**
    * @brief		Set bit field of fuel-gauge AvSOC (0x0E) register.
    *
    * @param[in]	percent 	to be written to register.
	*				LSB is 1/256%. Min value is 0.0% and Max value is 255.9961%.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_av_soc(float percent);

    /**
    * @brief		Get bit field of fuel-gauge AvSOC (0x0E) register.
    *
    * @param[out] 	percent 	Pointer to save result value.
	*				LSB is 1/256%. Min value is 0.0% and Max value is 255.9961%.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_av_soc(float *percent);
	
	/**
    * @brief		Set bit field of fuel-gauge FullCapRep (0x10) register.
    *
    * @param[in]	repCapmAh 	to be written to register.
	*				LSB is 0.1mAh. Min value is 0.0mAh and Max value is 6553.5mAh.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_full_cap_reg(float repCapmAh);

    /**
    * @brief		Get bit field of fuel-gauge FullCapRep (0x10) register.
    *
    * @param[out] 	repCapmAh 	Pointer to save result value.
	*				LSB is 0.1mAh. Min value is 0.0mAh and Max value is 6553.5mAh.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_full_cap_reg(float *repCapmAh);
	
	/**
    * @brief		Set bit field of fuel-gauge TTE (0x11) register.
    *
    * @param[in]	minute 	to be written to register.
	*				LSB is 5.625s. Min value is 0.0s and Max value is 102.3984h.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_tte(float minute);

    /**
    * @brief		Get bit field of fuel-gauge TTE (0x11) register.
    *
    * @param[out] 	minute 	Pointer to save result value.
	*				LSB is 5.625s. Min value is 0.0s and Max value is 102.3984h.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_tte(float *minute);
	
	/**
    * @brief		Set bit field of fuel-gauge RCell (0x14) register.
    *
    * @param[in]	resOhm 		to be written to register.
	*				LSB is 1/4096Ohm. Min value is 0.0Ohm and Max value is 15.99976Ohm.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_rcell(float resOhm);

    /**
    * @brief		Get bit field of fuel-gauge RCell (0x14) register.
    *
    * @param[out] 	resOhm 		Pointer to save result value.
	*				LSB is 1/4096Ohm. Min value is 0.0Ohm and Max value is 15.99976Ohm.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_rcell(float *resOhm);
	
	/**
    * @brief		Set bit field of fuel-gauge Cycles (0x17) register.
    *
    * @param[in]	percent 	to be written to register.
	*				The LSB indicates 1% of a battery cycle (1% charge + 1% discharge).
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_cycles(uint16_t percent);

    /**
    * @brief		Get bit field of fuel-gauge Cycles (0x17) register.
    *
    * @param[out] 	percent 	Pointer to save result value.
	*				The LSB indicates 1% of a battery cycle (1% charge + 1% discharge).
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_cycles(uint16_t *percent);
	
	/**
    * @brief		Set bit field of fuel-gauge AvCap (0x1F) register.
    *
    * @param[in]	avCapmAh 	to be written to register.
	*				LSB is 0.1mAh. Min value is 0.0mAh and Max value is 6553.5mAh.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_av_cap(float avCapmAh);

    /**
    * @brief		Get bit field of fuel-gauge AvCap (0x1F) register.
    *
    * @param[out] 	avCapmAh 	Pointer to save result value.
	*				LSB is 0.1mAh. Min value is 0.0mAh and Max value is 6553.5mAh.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_av_cap(float *avCapmAh);
	
	/**
    * @brief		Set bit field of fuel-gauge TTF (0x20) register.
    *
    * @param[in]	second 		to be written to register.
	*				LSB is 5.625s. Min value is 0.0s and Max value is 102.3984h.
    *
    * @return		0 on success, error code on failure.
    */
    int set_fg_ttf(float second);

    /**
    * @brief		Get bit field of fuel-gauge TTF (0x20) register.
    *
    * @param[out] 	second 		Pointer to save result value.
	*				LSB is 5.625s. Min value is 0.0s and Max value is 102.3984h.
    *
    * @return		0 on success, error code on failure.
    */
    int get_fg_ttf(float *second);

    /**
     * @brief	Disable all interrupts
     *
     * @return	0 on success, error code on failure
     */
     int irq_disable_all();

     /**
     * @brief		Set Interrupt Handler for a Specific Interrupt ID.
     *
     * @param[in]	id reg_bit_reg_bit_int_glbl_t 	id, one of INTR_ID_*.
     * @param[in]	func 					Interrupt handler function.
     * @param[in]	cb 						Interrupt handler data.
     */
     void set_interrupt_handler(reg_bit_int_glbl_t id, interrupt_handler_function func, void *cb);
};
#endif
