/*******************************************************************************
 * Copyright(C) Analog Devices Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files(the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Analog Devices Inc.
 * shall not be used except as stated in the Analog Devices Inc.
 * Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Analog Devices Inc.retains all ownership rights.
 *******************************************************************************
 */

#include <Thread.h>
#include "MAX77655.h"
#include <math.h>

#define POST_INTR_WORK_SIGNAL_ID			0x1
#define TO_UINT8							0xFF
#define TO_UINT16							0xFFFF

MAX77655::MAX77655(I2C *i2c, PinName IRQPin):interrupt_handler_list{NULL}
{
	if (i2c == NULL)
		return;

	i2c_handler = i2c;

	if (IRQPin != NC) {
        	irq_disable_all();
        	post_intr_work_thread = new Thread();
        	post_intr_work_thread->start(Callback<void()>(this, &MAX77655::post_interrupt_work));

        	this->irq_pin = new InterruptIn(IRQPin);
        	this->irq_pin->fall(Callback<void()>(this, &MAX77655::interrupt_handler));
        	this->irq_pin->enable_irq();
    	} else {
        	this->irq_pin = NULL;
    	}
}

MAX77655::~MAX77655()
{
	if (post_intr_work_thread)
        	delete post_intr_work_thread;

    	if (irq_pin)
        	delete irq_pin;
}

int MAX77655::read_register(uint8_t reg, uint8_t *value)
{
	int rtn_val;

    	if (value == NULL)
        	return MAX77655_VALUE_NULL;

    	rtn_val = i2c_handler->write(MAX77655_I2C_ADDRESS, (const char *)&reg, 1, true);
    	if (rtn_val != 0)
        	return MAX77655_WRITE_DATA_FAILED;

    	rtn_val = i2c_handler->read(MAX77655_I2C_ADDRESS, (char *) value, 1, false);
    	if (rtn_val < 0)
        	return MAX77655_READ_DATA_FAILED;

    	return MAX77655_NO_ERROR;
}

int MAX77655::write_register(uint8_t reg, const uint8_t *value)
{
    	int rtn_val;
    	unsigned char local_data[2];

    	if (value == NULL)
        	return MAX77655_VALUE_NULL;

    	local_data[0] = reg;

    	memcpy(&local_data[1], value, 1);

    	rtn_val = i2c_handler->write(MAX77655_I2C_ADDRESS, (const char *)local_data, sizeof(local_data));
    	if (rtn_val != MAX77655_NO_ERROR)
        	return MAX77655_WRITE_DATA_FAILED;

    	return MAX77655_NO_ERROR;
}

#define SET_BIT_FIELD(address, reg_name, bit_field_name, value)                         	\
            int ret_val;                                                                    \
            ret_val = read_register(address, (uint8_t *)&(reg_name));                       \
            if (ret_val) {                                                                  \
                return ret_val;                                                             \
            }                                                                           	\
            bit_field_name = value;                                                     	\
            ret_val = write_register(address, (uint8_t *)&(reg_name));                      \
            if (ret_val) {                                                                  \
        	return ret_val;                                                             \
            }

int MAX77655::set_cnfg_glbl_a(reg_bit_cnfg_glbl_a_t bit_field, uint8_t config)
{
	int ret;
	reg_cnfg_glbl_a_t reg_cnfg_glbl_a;
	
	ret = read_register(CNFG_GLBL_A, (uint8_t *)&(reg_cnfg_glbl_a));
	if (ret != MAX77655_NO_ERROR) return ret;
	
	switch (bit_field)
	{
	case CNFG_GLBL_A_DBEN_nEN:
		reg_cnfg_glbl_a.bits.dben_nen = config;
		break;
	case CNFG_GLBL_A_nEN_MODE:
		reg_cnfg_glbl_a.bits.nen_mode = config;
		break;
	case CNFG_GLBL_A_MRT:
		reg_cnfg_glbl_a.bits.mrt = config;
		break;
	case CNFG_GLBL_A_BIAS_LPM:
		reg_cnfg_glbl_a.bits.bias_lpm = config;
		break;
	case CNFG_GLBL_A_PU_DIS:
		reg_cnfg_glbl_a.bits.pu_dis = config;
		break;
	default:
		return MAX77655_INVALID_DATA;
		break;		
	}
    
	return write_register(CNFG_GLBL_A, (uint8_t *)&(reg_cnfg_glbl_a));
}

int MAX77655::get_cnfg_glbl_a(reg_bit_cnfg_glbl_a_t bit_field, uint8_t *config)
{
    	int ret;
	reg_cnfg_glbl_a_t reg_cnfg_glbl_a = {0};
	
	ret = read_register(CNFG_GLBL_A, (uint8_t *)&(reg_cnfg_glbl_a));
	if (ret != MAX77655_NO_ERROR) return ret;

    	switch (bit_field)
	{	
	case CNFG_GLBL_A_DBEN_nEN:
		*config = (uint8_t)reg_cnfg_glbl_a.bits.dben_nen;
		break;
	case CNFG_GLBL_A_nEN_MODE:
		*config = (uint8_t)reg_cnfg_glbl_a.bits.nen_mode;
		break;
	case CNFG_GLBL_A_MRT:
		*config = (uint8_t)reg_cnfg_glbl_a.bits.mrt;
		break;
	case CNFG_GLBL_A_BIAS_LPM:
		*config = (uint8_t)reg_cnfg_glbl_a.bits.bias_lpm;
		break;
	case CNFG_GLBL_A_PU_DIS:
		*config = (uint8_t)reg_cnfg_glbl_a.bits.pu_dis;
		break;
	default:
		return MAX77655_INVALID_DATA;
		break;		
	}

    	return MAX77655_NO_ERROR;
}

int MAX77655::set_sft_ctrl(decode_sft_ctrl_t config)
{	
	reg_cnfg_glbl_b_t reg_cnfg_glbl_b;
	
	SET_BIT_FIELD(CNFG_GLBL_B, reg_cnfg_glbl_b, reg_cnfg_glbl_b.bits.sft_ctrl, config);
	
	return MAX77655_NO_ERROR;
}

int MAX77655::get_sft_ctrl(decode_sft_ctrl_t *config)
{
	int ret;
	reg_cnfg_glbl_b_t reg_cnfg_glbl_b = {0};

	ret = read_register(CNFG_GLBL_B, (uint8_t *)&(reg_cnfg_glbl_b));
	if (ret != MAX77655_NO_ERROR) return ret;

	*config = (decode_sft_ctrl_t)reg_cnfg_glbl_b.bits.sft_ctrl;

    	return MAX77655_NO_ERROR;
}

int MAX77655::set_interrupt_mask(reg_bit_int_mask_t bit_field, uint8_t maskBit)
{
	int ret;
	uint8_t reg_addr;
	reg_intm_glbl_t reg_intm_glbl;
	
	ret = read_register(INTM_GLBL, (uint8_t *)&(reg_intm_glbl));
	
	if (ret != MAX77655_NO_ERROR) return ret;
	
	switch (bit_field)
	{
	case INTM_GLBL_nEN_FM:
		reg_intm_glbl.bits.nen_fm = maskBit;
		break;
	case INTM_GLBL_nEN_RM:
		reg_intm_glbl.bits.nen_rm = maskBit;
		break;
	case INTM_GLBL_TJAL1_RM:
		reg_intm_glbl.bits.tjal1_rm = maskBit;
		break;
	case INTM_GLBL_TJAL2_RM:
		reg_intm_glbl.bits.tjal2_rm = maskBit;
		break;
	case INTM_GLBL_SBB0_FM:
		reg_intm_glbl.bits.sbb0_fm = maskBit;
		break;
	case INTM_GLBL_SBB1_FM:
		reg_intm_glbl.bits.sbb1_fm = maskBit;
		break;
	case INTM_GLBL_SBB2_FM:
		reg_intm_glbl.bits.sbb2_fm = maskBit;
		break;
	case INTM_GLBL_SBB3_FM:
		reg_intm_glbl.bits.sbb3_fm = maskBit;
		break;
	default:
		return MAX77655_INVALID_DATA;
		break;		
	}
    
	return write_register(INTM_GLBL, (uint8_t *)&(reg_intm_glbl));
}

int MAX77655::get_interrupt_mask(reg_bit_int_mask_t bit_field, uint8_t *maskBit)
{
    	int ret;
	uint8_t reg_addr;
	reg_intm_glbl_t reg_intm_glbl = {0};
	
	ret = read_register(INTM_GLBL, (uint8_t *)&(reg_intm_glbl));
	
	if (ret != MAX77655_NO_ERROR) return ret;

	switch (bit_field)
	{	
	case INTM_GLBL_nEN_FM:
		*maskBit = (uint8_t)reg_intm_glbl.bits.nen_fm;
		break;
	case INTM_GLBL_nEN_RM:
		*maskBit = (uint8_t)reg_intm_glbl.bits.nen_rm;
		break;
	case INTM_GLBL_TJAL1_RM:
		*maskBit = (uint8_t)reg_intm_glbl.bits.tjal1_rm;
		break;
	case INTM_GLBL_TJAL2_RM:
		*maskBit = (uint8_t)reg_intm_glbl.bits.tjal2_rm;
		break;
	case INTM_GLBL_SBB0_FM:
		*maskBit = (uint8_t)reg_intm_glbl.bits.sbb0_fm;
		break;
	case INTM_GLBL_SBB1_FM:
		*maskBit = (uint8_t)reg_intm_glbl.bits.sbb1_fm;
		break;
	case INTM_GLBL_SBB2_FM:
		*maskBit = (uint8_t)reg_intm_glbl.bits.sbb2_fm;
		break;
	case INTM_GLBL_SBB3_FM:
		*maskBit = (uint8_t)reg_intm_glbl.bits.sbb3_fm;
		break;
	default:
		return MAX77655_INVALID_DATA;
		break;		
	}

    	return MAX77655_NO_ERROR;
}

int MAX77655::get_stat_glbl(reg_bit_stat_glbl_t bit_field, uint8_t *status)
{
	int ret;
    	reg_stat_glbl_t reg_stat_glbl = {0};

    	ret = read_register(STAT_GLBL, (uint8_t *)&(reg_stat_glbl));
    	if (ret != MAX77655_NO_ERROR) return ret;

	switch (bit_field)
	{
	case STAT_GLBL_STAT_EN:
		*status = (uint8_t)reg_stat_glbl.bits.stat_en;
		break;
	case STAT_GLBL_TJAL1_S:
		*status = (uint8_t)reg_stat_glbl.bits.tjal1_s;
		break;
	case STAT_GLBL_TJAL2_S:
		*status = (uint8_t)reg_stat_glbl.bits.tjal2_s;
		break;
	case STAT_GLBL_SBB0_S:
		*status = (uint8_t)reg_stat_glbl.bits.sbb0_s;
		break;
	case STAT_GLBL_SBB1_S:
		*status = (uint8_t)reg_stat_glbl.bits.sbb1_s;
		break;
	case STAT_GLBL_SBB2_S:
		*status = (uint8_t)reg_stat_glbl.bits.sbb2_s;
		break;
	case STAT_GLBL_SBB3_S:
		*status = (uint8_t)reg_stat_glbl.bits.sbb3_s;
		break;
	default:
		ret = MAX77655_INVALID_DATA;
		break;		
	}

	return ret;
}

int MAX77655::get_ercflag(reg_bit_ercflag_t bit_field, uint8_t *flag)
{
	int ret;
	reg_ercflag_t reg_ercflag = {0};

	ret = read_register(ERCFLAG, (uint8_t *)&(reg_ercflag));
	if (ret != MAX77655_NO_ERROR) return ret;
	
	switch (bit_field)
	{
	case ERCFLAG_TOVLD:
		*flag = (uint8_t)reg_ercflag.bits.tovld;
		break;
	case ERCFLAG_OVLO:
		*flag = (uint8_t)reg_ercflag.bits.ovlo;
		break;
	case ERCFLAG_UVLO:
		*flag = (uint8_t)reg_ercflag.bits.uvlo;
		break;
	case ERCFLAG_MRST:
		*flag = (uint8_t)reg_ercflag.bits.mrst;
		break;
	case ERCFLAG_SFT_OFF_F:
		*flag = (uint8_t)reg_ercflag.bits.sft_off_f;
		break;
	case ERCFLAG_SFT_CRST_F:
		*flag = (uint8_t)reg_ercflag.bits.sft_crst_f;
		break;
	default:
		ret = MAX77655_INVALID_DATA;
		break;		
	}

	return ret;
}

int MAX77655::get_cid(void) {
    	char rbuf[1] = {0};
    	int ret;

    	ret = read_register(CID, (uint8_t *)&(rbuf));
	if (ret != MAX77655_NO_ERROR) return ret;
 
    	return *rbuf;
}

int MAX77655::set_drv_sbb(decode_drv_sbb_t config)
{	
	reg_config_sbb_top_t reg_config_sbb_top;
	
	SET_BIT_FIELD(CONFIG_SBB_TOP, reg_config_sbb_top, reg_config_sbb_top.bits.drv_sbb, config);
	
	return MAX77655_NO_ERROR;
}

int MAX77655::get_drv_sbb(decode_drv_sbb_t *config)
{
    	int ret;
    	reg_config_sbb_top_t reg_config_sbb_top = {0};

	ret = read_register(CONFIG_SBB_TOP, (uint8_t *)&(reg_config_sbb_top));
	if (ret != MAX77655_NO_ERROR) return ret;

	*config = (decode_drv_sbb_t)reg_config_sbb_top.bits.drv_sbb;

    	return MAX77655_NO_ERROR;
}

int MAX77655::set_tv_sbb(uint8_t channel, float voltV)
{
	uint8_t value;
	reg_cnfg_sbb0_a_t reg_cnfg_sbb0_a;
	reg_cnfg_sbb1_a_t reg_cnfg_sbb1_a;
	reg_cnfg_sbb2_a_t reg_cnfg_sbb2_a;
	reg_cnfg_sbb3_a_t reg_cnfg_sbb3_a;
	float voltmV = voltV * 1000;
	
	if (voltmV < 500) voltmV = 500;
	else if (voltmV > 4000) voltmV = 4000;

	value = (voltmV - 500) / 25;

	if (channel == 0) {
		SET_BIT_FIELD(CNFG_SBB0_A, reg_cnfg_sbb0_a, reg_cnfg_sbb0_a.bits.tv_sbb0, value);
	}		
	else if (channel == 1) {
		SET_BIT_FIELD(CNFG_SBB1_A, reg_cnfg_sbb1_a, reg_cnfg_sbb1_a.bits.tv_sbb1, value);
	}
	else if (channel == 2) {
		SET_BIT_FIELD(CNFG_SBB2_A, reg_cnfg_sbb2_a, reg_cnfg_sbb2_a.bits.tv_sbb2, value);
	}
	else if (channel == 3) {
		SET_BIT_FIELD(CNFG_SBB3_A, reg_cnfg_sbb3_a, reg_cnfg_sbb3_a.bits.tv_sbb3, value);
	}
	else {
		return MAX77655_INVALID_DATA;
	}
	
	return MAX77655_NO_ERROR;
}

int MAX77655::get_tv_sbb(uint8_t channel, float *voltV)
{
	int ret;
	uint8_t bit_value;
	reg_cnfg_sbb0_a_t reg_cnfg_sbb0_a = {0};
	reg_cnfg_sbb1_a_t reg_cnfg_sbb1_a = {0};
	reg_cnfg_sbb2_a_t reg_cnfg_sbb2_a = {0};
	reg_cnfg_sbb3_a_t reg_cnfg_sbb3_a = {0};

	if (channel == 0) { 
		ret = read_register(CNFG_SBB0_A, (uint8_t *)&(reg_cnfg_sbb0_a));
		if (ret != MAX77655_NO_ERROR) return ret;
		
		bit_value = (uint8_t)reg_cnfg_sbb0_a.bits.tv_sbb0;
	}
	else if (channel == 1) {
		ret = read_register(CNFG_SBB1_A, (uint8_t *)&(reg_cnfg_sbb1_a));
		if (ret != MAX77655_NO_ERROR) return ret;
	
		bit_value = (uint8_t)reg_cnfg_sbb1_a.bits.tv_sbb1;
	}
	else if (channel == 2) {
		ret = read_register(CNFG_SBB2_A, (uint8_t *)&(reg_cnfg_sbb2_a));
		if (ret != MAX77655_NO_ERROR) return ret;
	
		bit_value = (uint8_t)reg_cnfg_sbb2_a.bits.tv_sbb2;
	}
	else if (channel == 3) {
		ret = read_register(CNFG_SBB3_A, (uint8_t *)&(reg_cnfg_sbb3_a));
		if (ret != MAX77655_NO_ERROR) return ret;
	
		bit_value = (uint8_t)reg_cnfg_sbb3_a.bits.tv_sbb3;
	}
	else { 
		return MAX77655_INVALID_DATA;
	}
	
	if (bit_value > 141) bit_value = 141;
	*voltV = (bit_value * 0.025f) + 0.5f;

    	return MAX77655_NO_ERROR;
}

int MAX77655::set_ade_sbb(uint8_t channel, decode_ade_sbb_t ade_sbb)
{	
	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b;
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b;
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b;
	reg_cnfg_sbb3_b_t reg_cnfg_sbb3_b;
	
	if (channel == 0) {
		SET_BIT_FIELD(CNFG_SBB0_B, reg_cnfg_sbb0_b, reg_cnfg_sbb0_b.bits.ade_sbb0, ade_sbb);
	}
	else if (channel == 1) {
		SET_BIT_FIELD(CNFG_SBB1_B, reg_cnfg_sbb1_b, reg_cnfg_sbb1_b.bits.ade_sbb1, ade_sbb);
	}
	else if (channel == 2) {
		SET_BIT_FIELD(CNFG_SBB2_B, reg_cnfg_sbb2_b, reg_cnfg_sbb2_b.bits.ade_sbb2, ade_sbb);
	}
	else if (channel == 3) {
		SET_BIT_FIELD(CNFG_SBB3_B, reg_cnfg_sbb3_b, reg_cnfg_sbb3_b.bits.ade_sbb3, ade_sbb);
	}
	else {
		return MAX77655_INVALID_DATA;
	}
	
	return MAX77655_NO_ERROR;
}

int MAX77655::get_ade_sbb(uint8_t channel, decode_ade_sbb_t *ade_sbb)
{
    	int ret;
    	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
	reg_cnfg_sbb3_b_t reg_cnfg_sbb3_b = {0};

	if (channel == 0) {
		ret = read_register(CNFG_SBB0_B, (uint8_t *)&(reg_cnfg_sbb0_b));
		if (ret != MAX77655_NO_ERROR) return ret;

		*ade_sbb = (decode_ade_sbb_t)reg_cnfg_sbb0_b.bits.ade_sbb0;
	}
	else if (channel == 1) {
		ret = read_register(CNFG_SBB1_B, (uint8_t *)&(reg_cnfg_sbb1_b));
		if (ret != MAX77655_NO_ERROR) return ret;

		*ade_sbb = (decode_ade_sbb_t)reg_cnfg_sbb1_b.bits.ade_sbb1;
	}
	else if (channel == 2) {
		ret = read_register(CNFG_SBB2_B, (uint8_t *)&(reg_cnfg_sbb2_b));
		if (ret != MAX77655_NO_ERROR) return ret;

		*ade_sbb = (decode_ade_sbb_t)reg_cnfg_sbb2_b.bits.ade_sbb2;
	}
	else if (channel == 3) {
		ret = read_register(CNFG_SBB3_B, (uint8_t *)&(reg_cnfg_sbb3_b));
		if (ret != MAX77655_NO_ERROR) return ret;

		*ade_sbb = (decode_ade_sbb_t)reg_cnfg_sbb3_b.bits.ade_sbb3;
	}
	else {
		return MAX77655_INVALID_DATA;
	}

    	return MAX77655_NO_ERROR;
}

int MAX77655::set_en_sbb(uint8_t channel, decode_en_sbb_t en_sbb)
{	
	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b;
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b;
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b;
	reg_cnfg_sbb3_b_t reg_cnfg_sbb3_b;
	
	if (channel == 0) {
		SET_BIT_FIELD(CNFG_SBB0_B, reg_cnfg_sbb0_b, reg_cnfg_sbb0_b.bits.en_sbb0, en_sbb);
	}
	else if (channel == 1) {
		SET_BIT_FIELD(CNFG_SBB1_B, reg_cnfg_sbb1_b, reg_cnfg_sbb1_b.bits.en_sbb1, en_sbb);
	}
	else if (channel == 2) {
		SET_BIT_FIELD(CNFG_SBB2_B, reg_cnfg_sbb2_b, reg_cnfg_sbb2_b.bits.en_sbb2, en_sbb);
	}
	else if (channel == 3) {
		SET_BIT_FIELD(CNFG_SBB3_B, reg_cnfg_sbb3_b, reg_cnfg_sbb3_b.bits.en_sbb3, en_sbb);
	}
	else {
		return MAX77655_INVALID_DATA;
	}
	
	return MAX77655_NO_ERROR;
}

int MAX77655::get_en_sbb(uint8_t channel, decode_en_sbb_t *en_sbb)
{
    	int ret;
    	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
	reg_cnfg_sbb3_b_t reg_cnfg_sbb3_b = {0};

	if (channel == 0) {
		ret = read_register(CNFG_SBB0_B, (uint8_t *)&(reg_cnfg_sbb0_b));
		if (ret != MAX77655_NO_ERROR) return ret;

		*en_sbb = (decode_en_sbb_t)reg_cnfg_sbb0_b.bits.en_sbb0;
	}
	else if (channel == 1) {
		ret = read_register(CNFG_SBB1_B, (uint8_t *)&(reg_cnfg_sbb1_b));
		if (ret != MAX77655_NO_ERROR) return ret;

		*en_sbb = (decode_en_sbb_t)reg_cnfg_sbb1_b.bits.en_sbb1;
	}
	else if (channel == 2) {
		ret = read_register(CNFG_SBB2_B, (uint8_t *)&(reg_cnfg_sbb2_b));
		if (ret != MAX77655_NO_ERROR) return ret;

		*en_sbb = (decode_en_sbb_t)reg_cnfg_sbb2_b.bits.en_sbb2;
	}
	else if (channel == 3) {
		ret = read_register(CNFG_SBB3_B, (uint8_t *)&(reg_cnfg_sbb3_b));
		if (ret != MAX77655_NO_ERROR) return ret;

		*en_sbb = (decode_en_sbb_t)reg_cnfg_sbb3_b.bits.en_sbb3;
	}
	else return MAX77655_INVALID_DATA;

    	return MAX77655_NO_ERROR;
}

int MAX77655::irq_disable_all()
{
    	int ret;
    	uint8_t reg = 0;
    	uint8_t status = 0;

    	//Disable Masks in INTM_GLBL
    	ret = write_register(INTM_GLBL, &reg);
    	if (ret != MAX77655_NO_ERROR) return ret;

    	// Clear Interrupt Flags in INT_GLBL
    	return read_register(INT_GLBL, &status);
}

void MAX77655::set_interrupt_handler(reg_bit_int_glbl_t id, interrupt_handler_function func, void *cb)
{
    	interrupt_handler_list[id].func = func;
    	interrupt_handler_list[id].cb = cb;
}

void MAX77655::post_interrupt_work()
{
    	int ret;
    	uint8_t reg = 0, inten = 0, not_inten = 0, mask = 0;

    	while (true) {

        	ThisThread::flags_wait_any(POST_INTR_WORK_SIGNAL_ID);
        	// Check Interrupt Flags in INT_GLBL0
        	ret = read_register(INT_GLBL, &reg);
        	if (ret != MAX77655_NO_ERROR) return;

        	ret = read_register(INTM_GLBL, &inten);
        	if (ret != MAX77655_NO_ERROR) return;

        	not_inten = ~inten; // 0 means unmasked.

        	for (int i = 0; i <= INT_GLBL_SBB3_FM; i++) {
            		mask = (1 << i);
			/*
			***************************************************
			This is special case.  when nEN is pulled high, the
			on/off controller initiates a power-down sequence and
			goes to shutdown mode.
			***************************************************
			*/
            		if (((reg & mask) && (not_inten & mask))
				|| ((reg & (1 << INT_GLBL_nEN_F)) && (i == INT_GLBL_nEN_F))
				|| ((reg & (1 << INT_GLBL_nEN_R)) && (i == INT_GLBL_nEN_R))) {
					if (interrupt_handler_list[i].func != NULL) {
						interrupt_handler_list[i]
						.func(interrupt_handler_list[i].cb);
					}
        		}
        	}
    	}
}

void MAX77655::interrupt_handler()
{
    	post_intr_work_thread->flags_set(POST_INTR_WORK_SIGNAL_ID);
}

