/*******************************************************************************
 * Copyright(C) Analog Devices Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files(the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Analog Devices Inc.
 * shall not be used except as stated in the Analog Devices Inc.
 * Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Analog Devices Inc.retains all ownership rights.
 *******************************************************************************
 */

#ifndef _MAX77655_H_
#define _MAX77655_H_

#include "mbed.h"
#include "MAX77655_regs.h"

#define MAX77655_NO_ERROR                    0
#define MAX77655_VALUE_NULL                 -1
#define MAX77655_WRITE_DATA_FAILED          -2
#define MAX77655_READ_DATA_FAILED           -3
#define MAX77655_INVALID_DATA               -4

#define MAX77655_I2C_ADDRESS         		0x88

/**
 * @brief MAX77655 Low IQ SIMO PMIC with 4-Outputs Delivering up to
 * 700mA Total Output Current
 *
 * @details The MAX77655 is a highly efficient, complete power supply for low-power, ultra-compact applications.
 *
 * @code
 * @endcode
 */

class MAX77655
{
private:
    I2C *i2c_handler;
    InterruptIn *irq_pin;	// interrupt pin

    /**
     * @brief   	Register Addresses
     * @details 	Enumerated MAX77655 register addresses
     */
    typedef enum {
        CNFG_GLBL_A    	= 0x00,    // Configuration Global A
        CNFG_GLBL_B    	= 0x01,    // Configuration Global B
        INT_GLBL	   	= 0x02,    // Interrupt Global Status
        INTM_GLBL   	= 0x03,    // Interrupt Mask
        STAT_GLBL    	= 0x04,    // Global Status
        ERCFLAG      	= 0x05,    // Flags
        CID          	= 0x06,    // Chip Identification Code
        CONFIG_SBB_TOP 	= 0x07,    // SIMO Buck-Boost Configuration
        CNFG_SBB0_A  	= 0x08,    // SIMO Buck-Boost 0 Configuration A
        CNFG_SBB0_B  	= 0x09,    // SIMO Buck-Boost 0 Configuration B
        CNFG_SBB1_A  	= 0x0A,    // SIMO Buck-Boost 1 Configuration A
        CNFG_SBB1_B  	= 0x0B,    // SIMO Buck-Boost 1 Configuration B
        CNFG_SBB2_A  	= 0x0C,    // SIMO Buck-Boost 2 Configuration A
        CNFG_SBB2_B  	= 0x0D,    // SIMO Buck-Boost 2 Configuration B
        CNFG_SBB3_A  	= 0x0E,    // SIMO Buck-Boost 3 Configuration A
        CNFG_SBB3_B  	= 0x0F,    // SIMO Buck-Boost 3 Configuration B
    } reg_t;

    void interrupt_handler();

    void (MAX77655::*funcptr)(void);

    void post_interrupt_work();

    Thread *post_intr_work_thread;

    struct handler {
        void (*func)(void *);
        void *cb;
    };

    handler interrupt_handler_list[8];

public:
    /**
    * @brief		MAX77655 constructor.
    */
    MAX77655(I2C *i2c, PinName IRQPin = NC);

    /**
    * @brief		MAX77655 destructor.
    */
    ~MAX77655();

    /**
    * @brief		Function pointer type to interrupt handler function
    */
    typedef void (*interrupt_handler_function)(void *);

    /**
    * @brief 		Read from a register.
    *
    * @param[in] 	reg Address of a register to be written.
    * @param[out] 	value Pointer to save result value.
    *
    * @returns 	0 on success, negative error code on failure.
    */
    int read_register(uint8_t reg, uint8_t *value);

    /**
    * @brief 		Write to a register.
    *
    * @param[in] 	reg Address of a register to be written.
    * @param[out] 	value Pointer of value to be written to register.
    *
    * @returns 	0 on success, negative error code on failure.
    */
    int write_register(uint8_t reg, const uint8_t *value);

	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : CNFG_GLBL_A (0x00)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Event Recorder Flags.
    */
    typedef enum {
        CNFG_GLBL_A_DBEN_nEN,
        CNFG_GLBL_A_nEN_MODE,
        CNFG_GLBL_A_MRT,
        CNFG_GLBL_A_BIAS_LPM,
        CNFG_GLBL_A_PU_DIS,
        CNFG_GLBL_A_RSVD
    }reg_bit_cnfg_glbl_a_t;	

    /**
  	* @brief		Set CNFG_GLBL_A (0x00) register.
  	*
	* @param[in]	bit_field 	Register bit field to be written.
  	* @param[in]	config 		Configuration bit field to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int set_cnfg_glbl_a(reg_bit_cnfg_glbl_a_t bit_field, uint8_t config);

    /**
  	* @brief		Get CNFG_GLBL_A (0x00) register.
  	*
	* @param[in]	bit_field 	Register bit field to be written.
  	* @param[out]	config 		Pointer of value to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int get_cnfg_glbl_a(reg_bit_cnfg_glbl_a_t bit_field, uint8_t *config);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_GLBL_B (0x01)
	*  - Bit Fields    : [2:0]
	*  - Default       : 0x0
	*  - Description   : Software Control Functions
	*/
    typedef enum {
        SFT_CTRL_NO_ACTION,
        SFT_CTRL_SOFTWARE_COLD_RESET,
        SFT_CTRL_SOFTWARE_OFF,
        SFT_CTRL_SOFTWARE_STANDBY,
        SFT_CTRL_RESERVED
    }decode_sft_ctrl_t;
	
	/**
	* @brief		Set CNFG_GLBL_B (0x01) register.
	*
	* @param[in]	config 	Register bit field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_sft_ctrl(decode_sft_ctrl_t config);

    /**
	* @brief		Get CNFG_GLBL_B (0x01) register.
	*
	* @param[in]	config 	Register bit field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_sft_ctrl(decode_sft_ctrl_t *config);
	
	/**
    * @brief 		Register Configuration.
	*		 		All Interrupt Flags combined from INT_GLBL0 (0x02)
    *
    * @details
    *  - Register      : INT_GLBL (0x02)
    *  - Bit Fields    :
    *  - Default       : 0x0
    *  - Description   : Enumerated interrupts.
    */
    typedef enum {
        INT_GLBL_nEN_F,
        INT_GLBL_nEN_R,
        INT_GLBL_TJAL1_R,
        INT_GLBL_TJAL2_R,
        INT_GLBL_SBB0_FM,
        INT_GLBL_SBB1_FM,
        INT_GLBL_SBB2_FM,
        INT_GLBL_SBB3_FM
    } reg_bit_int_glbl_t;

	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : INTM_GLBL (0x03)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : All interrupt mask bits.
    */
    typedef enum {
        INTM_GLBL_nEN_FM,
        INTM_GLBL_nEN_RM,
        INTM_GLBL_TJAL1_RM,
        INTM_GLBL_TJAL2_RM,
        INTM_GLBL_SBB0_FM,
        INTM_GLBL_SBB1_FM,
        INTM_GLBL_SBB2_FM,
        INTM_GLBL_SBB3_FM
    }reg_bit_int_mask_t;
	
	/**
    * @brief		Set bit field of INTM_GLBL (0x03) register.
    *
	* @param[in]	bit_field 	Register bit field to be set.
    * @param[out] 	maskBit 	0x0: Interrupt is unmasked,
    *                    		0x1: Interrupt is masked.
    *
    * @return		0 on success, error code on failure.
    */
    int set_interrupt_mask(reg_bit_int_mask_t bit_field, uint8_t maskBit);
	
	/**
    * @brief		Get bit field of INTM_GLBL (0x03) register.
    *
	* @param[in]	bit_field 	Register bit field to be written.
    * @param[out] 	maskBit 	0x0: Interrupt is unmasked,
    *                    		0x1: Interrupt is masked.
    *
    * @return		0 on success, error code on failure.
    */
    int get_interrupt_mask(reg_bit_int_mask_t bit_field, uint8_t *maskBit);

		/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : STAT_GLBL (0x04)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Global Status.
    */
    typedef enum {
        STAT_GLBL_STAT_EN,
        STAT_GLBL_TJAL1_S,
        STAT_GLBL_TJAL2_S,
        STAT_GLBL_RSVD,
        STAT_GLBL_SBB0_S,
        STAT_GLBL_SBB1_S,
        STAT_GLBL_SBB2_S,
        STAT_GLBL_SBB3_S
    }reg_bit_stat_glbl_t;
	
    /**
    * @brief		Get bit field of STAT_GLBL (0x04) register.
    *
	* @param[in]	bit_field 	STAT_GLBL register bit field to be written.
    * @param[out] 	status 		Pointer to save result of Status Global bit state.
    *
    * @return		0 on success, error code on failure.
    */
    int get_stat_glbl(reg_bit_stat_glbl_t bit_field, uint8_t *status);
	
	/**
    * @brief Register Configuration
    *
    * @details
    *  - Register      : ERCFLAG (0x05)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Event Recorder Flags.
    */
    typedef enum {
        ERCFLAG_TOVLD,
        ERCFLAG_OVLO,
        ERCFLAG_UVLO,
        ERCFLAG_MRST,
        ERCFLAG_SFT_OFF_F,
        ERCFLAG_SFT_CRST_F,
        ERCFLAG_RSVD
    }reg_bit_ercflag_t;

    /**
    * @brief		Get bit field of ERCFLAG (0x05) register.
    *
	* @param[in]	bit_field 	ERCFLAG register bit field to be written.
    * @param[out] 	flag 		Pointer to save result of ercglag bit states.	
	*							For individual bit
	*							0x0: ERCFLAG has not occurred,
    *               			0x1: ERCFLAG has occurred.
    *
    * @return		0 on success, error code on failure.
    */
    int get_ercflag(reg_bit_ercflag_t bit_field, uint8_t *flag);
	
    /**
    * @brief		Get bit field of CID (0x06) register.
    *
    * @return		CID on success, error code on failure.
    */
    int get_cid(void);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CONFIG_SBB_TOP (0x07)
	*  - Bit Fields    : [1:0]
	*  - Default       : 0x0
	*  - Description   : Configuration for SIMO Buck Boost
	*/
    typedef enum {
        DRV_SBB_FASTEST_TRANSITION_TIME,
        DRV_SBB_A_LITTLE_SLOWER_THAN_0X00,
        DRV_SBB_A_LITTLE_SLOWER_THAN_0X01,
        DRV_SBB_A_LITTLE_SLOWER_THAN_0X02
    }decode_drv_sbb_t;
	
	/**
	* @brief		Set CONFIG_SBB_TOP (0x07) register.
	*
	* @param[in]	config 		Configuration value to be written.
	*							0x0: Fastest transition time (~0.6ns)
	*							0x1: A little slower than 0b00 (~1.2ns)
	*							0x2: A little slower than 0b01 (~1.8ns)
	*							0x3: A little slower than 0b10 (~8ns) 
	*
	* @return		0 on success, error code on failure.
	*/
    int set_drv_sbb(decode_drv_sbb_t config);

    /**
	* @brief		Get CONFIG_SBB_TOP (0x07) register.
	*
	* @param[out]	config 		Configuration value to be read.
	*							0x0: Fastest transition time (~0.6ns)
	*							0x1: A little slower than 0b00 (~1.2ns)
	*							0x2: A little slower than 0b01 (~1.8ns)
	*							0x3: A little slower than 0b10 (~8ns) 
	*
	* @return		0 on success, error code on failure.
	*/
    int get_drv_sbb(decode_drv_sbb_t *config);
	
	/**
	* @brief		Set SIMO Buck-Boost Channel x Target Output Voltage.
	*				CNFG_SBB0_A (0x08), CNFG_SBB1_A (0x0A), CNFG_SBB2_A (0x0C) and CNFG_SBB3_A (0x0E)
	*
	* @param[in]	channel 	Channel number: 0, 1, 2 or 3.
	* @param[in]	voltV 		SIMO buck-boost channel x target output voltage field to be written.
	*							SBBx = 500mV + 25mV x TV_SBBx[7:0]
	*							0x00: 0.500V 0x01: 0.525V 0x02: 0.550V
								0x03: 0.575V ...
								0x8B: 3.975V 0x8C: 4.000V 0x8D–0xFF: Reserved
	*
	* @return		0 on success, error code on failure.
	*/
    int set_tv_sbb(uint8_t channel, float voltV);

    /**
	* @brief		Get SIMO Buck-Boost Channel x Target Output Voltage.
	*				CNFG_SBB0_A (0x08), CNFG_SBB1_A (0x0A), CNFG_SBB2_A (0x0C) and CNFG_SBB3_A (0x0E)
	*
	* @param[in]	channel 	Channel number: 0, 1, 2 or 3.
	* @param[out]	voltV 		SIMO buck-boost channel x target output voltage field to be read.
	*							SBBx = 500mV + 25mV x TV_SBBx[7:0]
	*							0x00: 0.500V 0x01: 0.525V 0x02: 0.550V
								0x03: 0.575V ...
								0x8B: 3.975V 0x8C: 4.000V 0x8D–0xFF: Reserved
	* @return		0 on success, error code on failure.
	*/
    int get_tv_sbb(uint8_t channel, float *voltV);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x09), CNFG_SBB1_B (0x0B), CNFG_SBB2_B (0x0D) and CNFG_SBB3_B (0x0F)
	*  - Bit Fields    : [3]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost Channel 0, 1, 2 or 3 Active-Discharge Enable.
	*/
    typedef enum {
        ADE_SBB_DISABLED,
        ADE_SBB_ENABLED
    }decode_ade_sbb_t;

    /**
	* @brief		Set SIMO Buck-Boost Channel x Active-Discharge Enable.
	*
	* @param[in]	channel Channel number: 0, 1, 2 or 3.
	* @param[in]	ade_sbb SIMO buck-boost channel x active-discharge enable bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_ade_sbb(uint8_t channel, decode_ade_sbb_t ade_sbb);

    /**
	* @brief		Get SIMO Buck-Boost Channel x Active-Discharge Enable.
	*
	* @param[in]	channel Channel number: 0, 1, 2 or 3.
	* @param[out]	ade_sbb SIMO buck-boost channel x active-discharge enable bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_ade_sbb(uint8_t channel, decode_ade_sbb_t *ade_sbb);
	
	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x09), CNFG_SBB1_B (0x0B), CNFG_SBB2_B (0x0D) and CNFG_SBB3_B (0x0F)
	*  - Bit Fields    : [2:0]
	*  - Default       : 0x0
	*  - Description   : Enable Control for SIMO Buck-Boost Channel 0, 1, 2 or 3.
	*/
    typedef enum {
        EN_SBB_FPS_SLOT_0,
        EN_SBB_FPS_SLOT_1,
        EN_SBB_FPS_SLOT_2,
        EN_SBB_FPS_SLOT_3,
        EN_SBB_OFF,
        EN_SBB_SAME_AS_0X04,
        EN_SBB_ON,
        EN_SBB_SAME_AS_0X06
    }decode_en_sbb_t;

    /**
	* @brief		Set Enable Control for SIMO Buck-Boost Channel x.
	*
	* @param[in]	channel Channel number: 0, 1, 2 or 3.
	* @param[in]	en_sbb 	Enable control for SIMO buck-boost channel x field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int set_en_sbb(uint8_t channel, decode_en_sbb_t en_sbb);

    /**
	* @brief		Get Enable Control for SIMO Buck-Boost Channel x.
	*
	* @param[in]	channel Channel number: 0, 1, 2 or 3.
	* @param[out]	en_sbb 	Enable control for SIMO buck-boost channel x field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int get_en_sbb(uint8_t channel, decode_en_sbb_t *en_sbb);

    /**
    * @brief	Disable all interrupts
    *
    * @return	0 on success, error code on failure
    */
    int irq_disable_all();

    /**
    * @brief		Set Interrupt Handler for a Specific Interrupt ID.
    *
    * @param[in]	id reg_bit_reg_bit_int_glbl_t 	id, one of INTR_ID_*.
    * @param[in]	func 					Interrupt handler function.
    * @param[in]	cb 						Interrupt handler data.
    */
    void set_interrupt_handler(reg_bit_int_glbl_t id, interrupt_handler_function func, void *cb);
};
#endif
