/*******************************************************************************
* Copyright (C) 2022 Maxim Integrated Products, Inc., All rights Reserved.
*
* This software is protected by copyright laws of the United States and
* of foreign countries. This material may also be protected by patent laws
* and technology transfer regulations of the United States and of foreign
* countries. This software is furnished under a license agreement and/or a
* nondisclosure agreement and may only be used or reproduced in accordance
* with the terms of those agreements. Dissemination of this information to
* any party or parties not specified in the license agreement and/or
* nondisclosure agreement is expressly prohibited.
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
*/

#include <Thread.h>
#include "MAX77654.h"
using namespace std;

#define POST_INTR_WORK_SIGNAL_ID			0x1

MAX77654::MAX77654(I2C *i2c, PinName IRQPin)
{
    if (i2c == NULL)
        return;

    i2c_handler = i2c;

    interrupt_handler_list = new handler[INT_CHG_END]{};
    
    if (IRQPin != NC) {
        IRQDisableAll();
        post_intr_work_thread = new Thread();
        post_intr_work_thread->start(Callback<void()>(this, &MAX77654::PostInterruptWork));

        this->irq_pin = new InterruptIn(IRQPin);
        this->irq_pin->fall(Callback<void()>(this, &MAX77654::InterruptHandler));
        this->irq_pin->enable_irq();
    } else {
        this->irq_pin = NULL;
    }
}

MAX77654::~MAX77654()
{
    if (post_intr_work_thread) {
        delete post_intr_work_thread;
    }

    if (irq_pin) {
        delete irq_pin;
    }

    if (interrupt_handler_list) {
        delete [] interrupt_handler_list;
    }
}

int MAX77654::read_register(uint8_t reg, uint8_t *value)
{
    int rtn_val;

    if (value == NULL) {
        return MAX77654_VALUE_NULL;
    }

    rtn_val = i2c_handler->write(MAX77654_I2C_ADDRESS, (const char *)&reg, 1, true);
    if (rtn_val != 0) {
        return MAX77654_WRITE_DATA_FAILED;
    }

    rtn_val = i2c_handler->read(MAX77654_I2C_ADDRESS, (char *) value, 1, false);
    if (rtn_val < 0) {
        return MAX77654_READ_DATA_FAILED;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::write_register(uint8_t reg, const uint8_t *value)
{
    int rtn_val;
    uint8_t local_data[2];

    if (value == NULL) {
        return MAX77654_VALUE_NULL;
    }

    local_data[0] = reg;

    memcpy(&local_data[1], value, 1);

    rtn_val = i2c_handler->write(MAX77654_I2C_ADDRESS, (const char *)local_data, sizeof(local_data));
    if (rtn_val != MAX77654_NO_ERROR) {
        return MAX77654_WRITE_DATA_FAILED;
    }

    return MAX77654_NO_ERROR;
}

#define SET_BIT_FIELD(address, reg_name, bit_field_name, value)                         \
            int ret;                                                                    \
            ret = read_register(address, (uint8_t *)&(reg_name));                       \
            if (ret) {                                                                  \
                return ret;                                                             \
            }                                                                           \
            bit_field_name = value;                                                     \
            ret = write_register(address, (uint8_t *)&(reg_name));                      \
            if (ret) {                                                                  \
                return ret;                                                             \
            }

int MAX77654::GetThermalOverload(ercflag_t *ercflag)
{
	int ret;
	reg_ercflag_t reg_ercflag = {0};

	ret = read_register(ERCFLAG, (uint8_t *)&(reg_ercflag));
	if (ret != MAX77654_NO_ERROR) {
		return ret;
	}

	*ercflag = (ercflag_t)reg_ercflag.bits.tovld;

	return MAX77654_NO_ERROR;
}

int MAX77654::GetSysOvervoltageLockout(ercflag_t *ercflag)
{
	int ret;
	reg_ercflag_t reg_ercflag = {0};

	ret = read_register(ERCFLAG, (uint8_t *)&(reg_ercflag));
	if (ret != MAX77654_NO_ERROR) {
		return ret;
	}

	*ercflag = (ercflag_t)reg_ercflag.bits.sysovlo;

	return MAX77654_NO_ERROR;
}

int MAX77654::GetSysUndervoltageLockout(ercflag_t *ercflag)
{
	int ret;
	reg_ercflag_t reg_ercflag = {0};

	ret = read_register(ERCFLAG, (uint8_t *)&(reg_ercflag));
	if (ret != MAX77654_NO_ERROR) {
		return ret;
	}

	*ercflag = (ercflag_t)reg_ercflag.bits.sysuvlo;

	return MAX77654_NO_ERROR;
}

int MAX77654::GetManualResetTimer(ercflag_t *ercflag)
{
	int ret;
	reg_ercflag_t reg_ercflag = {0};

	ret = read_register(ERCFLAG, (uint8_t *)&(reg_ercflag));
	if (ret != MAX77654_NO_ERROR) {
		return ret;
	}

	*ercflag = (ercflag_t)reg_ercflag.bits.mrst;

	return MAX77654_NO_ERROR;
}

int MAX77654::GetSoftwareOffFlag(ercflag_t *ercflag)
{
	int ret;
	reg_ercflag_t reg_ercflag = {0};

	ret = read_register(ERCFLAG, (uint8_t *)&(reg_ercflag));
	if (ret != MAX77654_NO_ERROR) {
		return ret;
	}

	*ercflag = (ercflag_t)reg_ercflag.bits.sft_off_f;

	return MAX77654_NO_ERROR;
}

int MAX77654::GetSoftwareColdResetFlag(ercflag_t *ercflag)
{
	int ret;
	reg_ercflag_t reg_ercflag = {0};

	ret = read_register(ERCFLAG, (uint8_t *)&(reg_ercflag));
	if (ret != MAX77654_NO_ERROR) {
		return ret;
	}

	*ercflag = (ercflag_t)reg_ercflag.bits.sft_crst_f;

	return MAX77654_NO_ERROR;
}

int MAX77654::GetWatchdogTimerOffFlag(ercflag_t *ercflag)
{
	int ret;
	reg_ercflag_t reg_ercflag = {0};

	ret = read_register(ERCFLAG, (uint8_t *)&(reg_ercflag));
	if (ret != MAX77654_NO_ERROR) {
		return ret;
	}

	*ercflag = (ercflag_t)reg_ercflag.bits.wdt_off;

	return MAX77654_NO_ERROR;
}

int MAX77654::GetWatchdogTimerResetFlag(ercflag_t *ercflag)
{
	int ret;
	reg_ercflag_t reg_ercflag = {0};

	ret = read_register(ERCFLAG, (uint8_t *)&(reg_ercflag));
	if (ret != MAX77654_NO_ERROR) {
		return ret;
	}

	*ercflag = (ercflag_t)reg_ercflag.bits.wdt_rst;

	return MAX77654_NO_ERROR;
}

int MAX77654::GetDeviceIdentification(didm_t *didm)
{
    int ret;
    reg_stat_glbl_t reg_stat_glbl = {0};

    ret = read_register(STAT_GLBL, (uint8_t *)&(reg_stat_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *didm = (didm_t)reg_stat_glbl.bits.didm;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetBOKInterruptStatus(bok_t* bok)
{
    int ret;
    reg_stat_glbl_t reg_stat_glbl = {0};

    ret = read_register(STAT_GLBL, (uint8_t *)&(reg_stat_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *bok = (bok_t)reg_stat_glbl.bits.bok;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetLDO0DropoutRisingStatus(dod0_s_t* dod0_s)
{
    int ret;
    reg_stat_glbl_t reg_stat_glbl = {0};

    ret = read_register(STAT_GLBL, (uint8_t *)&(reg_stat_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *dod0_s = (dod0_s_t)reg_stat_glbl.bits.dod0_s;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetLDO1DropoutRisingStatus(dod1_s_t* dod1_s)
{
    int ret;
    reg_stat_glbl_t reg_stat_glbl = {0};

    ret = read_register(STAT_GLBL, (uint8_t *)&(reg_stat_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *dod1_s = (dod1_s_t)reg_stat_glbl.bits.dod1_s;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetThermalAlarm2Status(tjal2_s_t* tjal2_s)
{
    int ret;
    reg_stat_glbl_t reg_stat_glbl = {0};

    ret = read_register(STAT_GLBL, (uint8_t *)&(reg_stat_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *tjal2_s = (tjal2_s_t)reg_stat_glbl.bits.tjal2_s;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetThermalAlarm1Status(tjal1_s_t* tjal1_s)
{
    int ret;
    reg_stat_glbl_t reg_stat_glbl = {0};

    ret = read_register(STAT_GLBL, (uint8_t *)&(reg_stat_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *tjal1_s = (tjal1_s_t)reg_stat_glbl.bits.tjal1_s;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetnENDebouncedStatus(stat_en_t* stat_en)
{
    int ret;
    reg_stat_glbl_t reg_stat_glbl = {0};

    ret = read_register(STAT_GLBL, (uint8_t *)&(reg_stat_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *stat_en = (stat_en_t)reg_stat_glbl.bits.stat_en;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetSoftwareVersionGateDrive(stat_irq_t* stat_irq)
{
    int ret;
    reg_stat_glbl_t reg_stat_glbl = {0};

    ret = read_register(STAT_GLBL, (uint8_t *)&(reg_stat_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *stat_irq = (stat_irq_t)reg_stat_glbl.bits.stat_irq;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetLDO1FaultInterruptMask(intm_t intm)
{
	reg_intm_glbl1_t reg_intm_glbl1 = {0};

    SET_BIT_FIELD(INTM_GLBL1, reg_intm_glbl1, reg_intm_glbl1.bits.ldo1_m, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetLDO1FaultInterruptMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl1_t reg_intm_glbl1 = {0};

    ret = read_register(INTM_GLBL1, (uint8_t *)&(reg_intm_glbl1));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl1.bits.ldo1_m;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetLDO0FaultInterruptMask(intm_t intm)
{
	reg_intm_glbl1_t reg_intm_glbl1 = {0};

    SET_BIT_FIELD(INTM_GLBL1, reg_intm_glbl1, reg_intm_glbl1.bits.ldo0_m, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetLDO0FaultInterruptMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl1_t reg_intm_glbl1 = {0};

    ret = read_register(INTM_GLBL1, (uint8_t *)&(reg_intm_glbl1));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl1.bits.ldo0_m;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetSBBTimeoutMask(intm_t intm)
{
	reg_intm_glbl1_t reg_intm_glbl1 = {0};

    SET_BIT_FIELD(INTM_GLBL1, reg_intm_glbl1, reg_intm_glbl1.bits.sbb_to_m, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetSBBTimeoutMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl1_t reg_intm_glbl1 = {0};

    ret = read_register(INTM_GLBL1, (uint8_t *)&(reg_intm_glbl1));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl1.bits.sbb_to_m;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetGPI2RisingInterruptMask(intm_t intm)
{
	reg_intm_glbl1_t reg_intm_glbl1 = {0};

    SET_BIT_FIELD(INTM_GLBL1, reg_intm_glbl1, reg_intm_glbl1.bits.gpi2_rm, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetGPI2RisingInterruptMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl1_t reg_intm_glbl1 = {0};

    ret = read_register(INTM_GLBL1, (uint8_t *)&(reg_intm_glbl1));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl1.bits.gpi2_rm;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetGPI2FallingInterruptMask(intm_t intm)
{
	reg_intm_glbl1_t reg_intm_glbl1 = {0};

    SET_BIT_FIELD(INTM_GLBL1, reg_intm_glbl1, reg_intm_glbl1.bits.gpi2_fm, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetGPI2FallingInterruptMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl1_t reg_intm_glbl1 = {0};

    ret = read_register(INTM_GLBL1, (uint8_t *)&(reg_intm_glbl1));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl1.bits.gpi2_fm;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetGPI1RisingInterruptMask(intm_t intm)
{
	reg_intm_glbl1_t reg_intm_glbl1 = {0};

    SET_BIT_FIELD(INTM_GLBL1, reg_intm_glbl1, reg_intm_glbl1.bits.gpi1_rm, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetGPI1RisingInterruptMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl1_t reg_intm_glbl1 = {0};

    ret = read_register(INTM_GLBL1, (uint8_t *)&(reg_intm_glbl1));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl1.bits.gpi1_rm;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetGPI1FallingInterruptMask(intm_t intm)
{
	reg_intm_glbl1_t reg_intm_glbl1 = {0};

    SET_BIT_FIELD(INTM_GLBL1, reg_intm_glbl1, reg_intm_glbl1.bits.gpi1_fm, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetGPI1FallingInterruptMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl1_t reg_intm_glbl1 = {0};

    ret = read_register(INTM_GLBL1, (uint8_t *)&(reg_intm_glbl1));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl1.bits.gpi1_fm;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetDropout0RisingInterruptMask(intm_t intm)
{
	reg_intm_glbl0_t reg_intm_glbl0 = {0};

    SET_BIT_FIELD(INTM_GLBL0, reg_intm_glbl0, reg_intm_glbl0.bits.dod0_rm, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetDropout0RisingInterruptMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl0_t reg_intm_glbl0 = {0};

    ret = read_register(INTM_GLBL0, (uint8_t *)&(reg_intm_glbl0));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl0.bits.dod0_rm;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetDropout1RisingInterruptMask(intm_t intm)
{
	reg_intm_glbl0_t reg_intm_glbl0 = {0};

    SET_BIT_FIELD(INTM_GLBL0, reg_intm_glbl0, reg_intm_glbl0.bits.dod1_rm, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetDropout1RisingInterruptMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl0_t reg_intm_glbl0 = {0};

    ret = read_register(INTM_GLBL0, (uint8_t *)&(reg_intm_glbl0));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl0.bits.dod1_rm;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetThermalAlarm2RisingInterruptMask(intm_t intm)
{
	reg_intm_glbl0_t reg_intm_glbl0 = {0};

    SET_BIT_FIELD(INTM_GLBL0, reg_intm_glbl0, reg_intm_glbl0.bits.tjal2_rm, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetThermalAlarm2RisingInterruptMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl0_t reg_intm_glbl0 = {0};

    ret = read_register(INTM_GLBL0, (uint8_t *)&(reg_intm_glbl0));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl0.bits.tjal2_rm;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetThermalAlarm1RisingInterruptMask(intm_t intm)
{
	reg_intm_glbl0_t reg_intm_glbl0 = {0};

    SET_BIT_FIELD(INTM_GLBL0, reg_intm_glbl0, reg_intm_glbl0.bits.tjal1_rm, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetThermalAlarm1RisingInterruptMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl0_t reg_intm_glbl0 = {0};

    ret = read_register(INTM_GLBL0, (uint8_t *)&(reg_intm_glbl0));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl0.bits.tjal1_rm;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetnENRisingInterruptMask(intm_t intm)
{
	reg_intm_glbl0_t reg_intm_glbl0 = {0};

    SET_BIT_FIELD(INTM_GLBL0, reg_intm_glbl0, reg_intm_glbl0.bits.nen_rm, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetnENRisingInterruptMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl0_t reg_intm_glbl0 = {0};

    ret = read_register(INTM_GLBL0, (uint8_t *)&(reg_intm_glbl0));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl0.bits.nen_rm;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetnENFallingInterruptMask(intm_t intm)
{
	reg_intm_glbl0_t reg_intm_glbl0 = {0};

    SET_BIT_FIELD(INTM_GLBL0, reg_intm_glbl0, reg_intm_glbl0.bits.nen_fm, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetnENFallingInterruptMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl0_t reg_intm_glbl0 = {0};

    ret = read_register(INTM_GLBL0, (uint8_t *)&(reg_intm_glbl0));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl0.bits.nen_fm;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetGPI0RisingInterruptMask(intm_t intm)
{
	reg_intm_glbl0_t reg_intm_glbl0 = {0};

    SET_BIT_FIELD(INTM_GLBL0, reg_intm_glbl0, reg_intm_glbl0.bits.gpi0_rm, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetGPI0RisingInterruptMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl0_t reg_intm_glbl0 = {0};

    ret = read_register(INTM_GLBL0, (uint8_t *)&(reg_intm_glbl0));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl0.bits.gpi0_rm;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetGPI0FallingInterruptMask(intm_t intm)
{
	reg_intm_glbl0_t reg_intm_glbl0 = {0};

    SET_BIT_FIELD(INTM_GLBL0, reg_intm_glbl0, reg_intm_glbl0.bits.gpi0_fm, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetGPI0FallingInterruptMask(intm_t* intm)
{
    int ret;
    reg_intm_glbl0_t reg_intm_glbl0 = {0};

    ret = read_register(INTM_GLBL0, (uint8_t *)&(reg_intm_glbl0));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_intm_glbl0.bits.gpi0_fm;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetnENInternalPullupResistor(pu_dis_t pu_dis)
{
	reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    SET_BIT_FIELD(CNFG_GLBL, reg_cnfg_glbl, reg_cnfg_glbl.bits.pu_dis, pu_dis);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetnENInternalPullupResistor(pu_dis_t* pu_dis)
{
    int ret;
    reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    ret = read_register(CNFG_GLBL, (uint8_t *)&(reg_cnfg_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *pu_dis = (pu_dis_t)reg_cnfg_glbl.bits.pu_dis;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetManualResetTime(t_mrst_t t_mrst)
{
	reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    SET_BIT_FIELD(CNFG_GLBL, reg_cnfg_glbl, reg_cnfg_glbl.bits.t_mrst, t_mrst);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetManualResetTime(t_mrst_t* t_mrst)
{
    int ret;
    reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    ret = read_register(CNFG_GLBL, (uint8_t *)&(reg_cnfg_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *t_mrst = (t_mrst_t)reg_cnfg_glbl.bits.t_mrst;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetMainBiasLowerPowerModeReq(sbia_lpm_t sbia_lpm)
{
	reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    SET_BIT_FIELD(CNFG_GLBL, reg_cnfg_glbl, reg_cnfg_glbl.bits.sbia_lpm, sbia_lpm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetMainBiasLowerPowerModeReq(sbia_lpm_t* sbia_lpm)
{
    int ret;
    reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    ret = read_register(CNFG_GLBL, (uint8_t *)&(reg_cnfg_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *sbia_lpm = (sbia_lpm_t)reg_cnfg_glbl.bits.sbia_lpm;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetMainBiasEnableReq(sbia_en_t sbia_en)
{
	reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    SET_BIT_FIELD(CNFG_GLBL, reg_cnfg_glbl, reg_cnfg_glbl.bits.sbia_en, sbia_en);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetMainBiasEnableReq(sbia_en_t* sbia_en)
{
    int ret;
    reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    ret = read_register(CNFG_GLBL, (uint8_t *)&(reg_cnfg_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *sbia_en = (sbia_en_t)reg_cnfg_glbl.bits.sbia_en;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetnEnInputMode(nen_mode_t nen_mode)
{
	reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    SET_BIT_FIELD(CNFG_GLBL, reg_cnfg_glbl, reg_cnfg_glbl.bits.nen_mode, nen_mode);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetnEnInputMode(nen_mode_t* nen_mode)
{
    int ret;
    reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    ret = read_register(CNFG_GLBL, (uint8_t *)&(reg_cnfg_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *nen_mode = (nen_mode_t)reg_cnfg_glbl.bits.nen_mode;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetDebounceTimerEnable(dben_nen_t dben_nen)
{
	reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    SET_BIT_FIELD(CNFG_GLBL, reg_cnfg_glbl, reg_cnfg_glbl.bits.dben_nen, dben_nen);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetDebounceTimerEnable(dben_nen_t* dben_nen)
{
    int ret;
    reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    ret = read_register(CNFG_GLBL, (uint8_t *)&(reg_cnfg_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *dben_nen = (dben_nen_t)reg_cnfg_glbl.bits.dben_nen;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetSoftwareResetFunctions(sft_ctrl_t sft_ctrl)
{
	reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    SET_BIT_FIELD(CNFG_GLBL, reg_cnfg_glbl, reg_cnfg_glbl.bits.sft_ctrl, sft_ctrl);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetSoftwareResetFunctions(sft_ctrl_t* sft_ctrl)
{
    int ret;
    reg_cnfg_glbl_t reg_cnfg_glbl = {0};

    ret = read_register(CNFG_GLBL, (uint8_t *)&(reg_cnfg_glbl));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *sft_ctrl = (sft_ctrl_t)reg_cnfg_glbl.bits.sft_ctrl;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetAlternateModeEnable(uint8_t channel, alt_gpio_t alt_gpio)
{
    if (channel == 0) {
    	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
        SET_BIT_FIELD(CNFG_GPIO0, reg_cnfg_gpio0, reg_cnfg_gpio0.bits.alt_gpio0, alt_gpio);
    }
    else if (channel == 1) {
    	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
        SET_BIT_FIELD(CNFG_GPIO1, reg_cnfg_gpio1, reg_cnfg_gpio1.bits.alt_gpio1, alt_gpio);
    }
    else if (channel == 2) {
    	reg_cnfg_gpio2_t reg_cnfg_gpio2 = {0};
        SET_BIT_FIELD(CNFG_GPIO2, reg_cnfg_gpio2, reg_cnfg_gpio2.bits.alt_gpio2, alt_gpio);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetAlternateModeEnable(uint8_t channel, alt_gpio_t* alt_gpio)
{
    int ret;

    if (channel == 0) {
    	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
        ret = read_register(CNFG_GPIO0, (uint8_t *)&(reg_cnfg_gpio0));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *alt_gpio = (alt_gpio_t)reg_cnfg_gpio0.bits.alt_gpio0;
    }
    else if (channel == 1) {
    	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
        ret = read_register(CNFG_GPIO1, (uint8_t *)&(reg_cnfg_gpio1));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *alt_gpio = (alt_gpio_t)reg_cnfg_gpio1.bits.alt_gpio1;
    }
    else if (channel == 2) {
    	reg_cnfg_gpio2_t reg_cnfg_gpio2 = {0};
        ret = read_register(CNFG_GPIO2, (uint8_t *)&(reg_cnfg_gpio2));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *alt_gpio = (alt_gpio_t)reg_cnfg_gpio2.bits.alt_gpio2;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::SetGPIDebounceTimerEnable(uint8_t channel, dben_gpi_t dben_gpi)
{
    if (channel == 0) {
    	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
        SET_BIT_FIELD(CNFG_GPIO0, reg_cnfg_gpio0, reg_cnfg_gpio0.bits.dben_gpi, dben_gpi);
    }
    else if (channel == 1) {
    	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
        SET_BIT_FIELD(CNFG_GPIO1, reg_cnfg_gpio1, reg_cnfg_gpio1.bits.dben_gpi, dben_gpi);
    }
    else if (channel == 2) {
    	reg_cnfg_gpio2_t reg_cnfg_gpio2 = {0};
        SET_BIT_FIELD(CNFG_GPIO2, reg_cnfg_gpio2, reg_cnfg_gpio2.bits.dben_gpi, dben_gpi);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetGPIDebounceTimerEnable(uint8_t channel, dben_gpi_t* dben_gpi)
{
    int ret;

    if (channel == 0) {
        reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
        ret = read_register(CNFG_GPIO0, (uint8_t *)&(reg_cnfg_gpio0));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *dben_gpi = (dben_gpi_t)reg_cnfg_gpio0.bits.dben_gpi;
    }
    else if (channel == 1) {
        reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
        ret = read_register(CNFG_GPIO1, (uint8_t *)&(reg_cnfg_gpio1));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *dben_gpi = (dben_gpi_t)reg_cnfg_gpio1.bits.dben_gpi;
    }
    else if (channel == 2) {
        reg_cnfg_gpio2_t reg_cnfg_gpio2 = {0};
        ret = read_register(CNFG_GPIO2, (uint8_t *)&(reg_cnfg_gpio2));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *dben_gpi = (dben_gpi_t)reg_cnfg_gpio2.bits.dben_gpi;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::SetGPODataOutput(uint8_t channel, gpo_do_t gpo_do)
{

    if (channel == 0) {
    	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
        SET_BIT_FIELD(CNFG_GPIO0, reg_cnfg_gpio0, reg_cnfg_gpio0.bits.gpo_do, gpo_do);
    }
    else if (channel == 1) {
    	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
        SET_BIT_FIELD(CNFG_GPIO1, reg_cnfg_gpio1, reg_cnfg_gpio1.bits.gpo_do, gpo_do);
    }
    else if (channel == 2) {
    	reg_cnfg_gpio2_t reg_cnfg_gpio2 = {0};
        SET_BIT_FIELD(CNFG_GPIO2, reg_cnfg_gpio2, reg_cnfg_gpio2.bits.gpo_do, gpo_do);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetGPODataOutput(uint8_t channel, gpo_do_t *gpo_do)
{
    int ret;

    if (channel == 0) {
    	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
        ret = read_register(CNFG_GPIO0, (uint8_t *)&(reg_cnfg_gpio0));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *gpo_do = (gpo_do_t)reg_cnfg_gpio0.bits.gpo_do;
    }
    else if (channel == 1) {
    	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
        ret = read_register(CNFG_GPIO1, (uint8_t *)&(reg_cnfg_gpio1));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *gpo_do = (gpo_do_t)reg_cnfg_gpio1.bits.gpo_do;
    }
    else if (channel == 2) {
    	reg_cnfg_gpio2_t reg_cnfg_gpio2 = {0};
        ret = read_register(CNFG_GPIO2, (uint8_t *)&(reg_cnfg_gpio2));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *gpo_do = (gpo_do_t)reg_cnfg_gpio2.bits.gpo_do;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::SetGPODriveType(uint8_t channel, gpo_drv_t gpo_drv)
{
    if (channel == 0) {
    	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
        SET_BIT_FIELD(CNFG_GPIO0, reg_cnfg_gpio0, reg_cnfg_gpio0.bits.gpo_drv, gpo_drv);
    }
    else if (channel == 1) {
    	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
        SET_BIT_FIELD(CNFG_GPIO1, reg_cnfg_gpio1, reg_cnfg_gpio1.bits.gpo_drv, gpo_drv);
    }
    else if (channel == 2) {
    	reg_cnfg_gpio2_t reg_cnfg_gpio2 = {0};
        SET_BIT_FIELD(CNFG_GPIO2, reg_cnfg_gpio2, reg_cnfg_gpio2.bits.gpo_drv, gpo_drv);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetGPODriveType(uint8_t channel, gpo_drv_t *gpo_drv)
{
    int ret;

    if (channel == 0) {
    	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
        ret = read_register(CNFG_GPIO0, (uint8_t *)&(reg_cnfg_gpio0));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *gpo_drv = (gpo_drv_t)reg_cnfg_gpio0.bits.gpo_drv;
    }
    else if (channel == 1) {
    	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
        ret = read_register(CNFG_GPIO1, (uint8_t *)&(reg_cnfg_gpio1));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *gpo_drv = (gpo_drv_t)reg_cnfg_gpio1.bits.gpo_drv;
    }
    else if (channel == 2) {
    	reg_cnfg_gpio2_t reg_cnfg_gpio2 = {0};
        ret = read_register(CNFG_GPIO2, (uint8_t *)&(reg_cnfg_gpio2));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *gpo_drv = (gpo_drv_t)reg_cnfg_gpio2.bits.gpo_drv;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::SetGPIOInputValue(uint8_t channel, gpo_di_t gpo_di)
{
    if (channel == 0) {
    	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
        SET_BIT_FIELD(CNFG_GPIO0, reg_cnfg_gpio0, reg_cnfg_gpio0.bits.gpo_di, gpo_di);
    }
    else if (channel == 1) {
    	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
        SET_BIT_FIELD(CNFG_GPIO1, reg_cnfg_gpio1, reg_cnfg_gpio1.bits.gpo_di, gpo_di);
    }
    else if (channel == 2) {
    	reg_cnfg_gpio2_t reg_cnfg_gpio2 = {0};
        SET_BIT_FIELD(CNFG_GPIO2, reg_cnfg_gpio2, reg_cnfg_gpio2.bits.gpo_di, gpo_di);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetGPIOInputValue(uint8_t channel, gpo_di_t *gpo_di)
{
    int ret;

    if (channel == 0) {
    	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
        ret = read_register(CNFG_GPIO0, (uint8_t *)&(reg_cnfg_gpio0));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *gpo_di = (gpo_di_t)reg_cnfg_gpio0.bits.gpo_di;
    }
    else if (channel == 1) {
    	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
        ret = read_register(CNFG_GPIO1, (uint8_t *)&(reg_cnfg_gpio1));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *gpo_di = (gpo_di_t)reg_cnfg_gpio1.bits.gpo_di;
    }
    else if (channel == 2) {
    	reg_cnfg_gpio2_t reg_cnfg_gpio2 = {0};
        ret = read_register(CNFG_GPIO2, (uint8_t *)&(reg_cnfg_gpio2));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *gpo_di = (gpo_di_t)reg_cnfg_gpio2.bits.gpo_di;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::SetGPIODirection(uint8_t channel, gpo_dir_t gpo_dir)
{
    if (channel == 0) {
    	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
        SET_BIT_FIELD(CNFG_GPIO0, reg_cnfg_gpio0, reg_cnfg_gpio0.bits.gpo_dir, gpo_dir);
    }
    else if (channel == 1) {
    	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
        SET_BIT_FIELD(CNFG_GPIO1, reg_cnfg_gpio1, reg_cnfg_gpio1.bits.gpo_dir, gpo_dir);
    }
    else if (channel == 2) {
    	reg_cnfg_gpio2_t reg_cnfg_gpio2 = {0};
        SET_BIT_FIELD(CNFG_GPIO2, reg_cnfg_gpio2, reg_cnfg_gpio2.bits.gpo_dir, gpo_dir);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetGPIODirection(uint8_t channel, gpo_dir_t *gpo_dir)
{
    int ret;
    
    if (channel == 0) {
    	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
        ret = read_register(CNFG_GPIO0, (uint8_t *)&(reg_cnfg_gpio0));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *gpo_dir = (gpo_dir_t)reg_cnfg_gpio0.bits.gpo_dir;
    }
    else if (channel == 1) {
    	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
        ret = read_register(CNFG_GPIO1, (uint8_t *)&(reg_cnfg_gpio1));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *gpo_dir = (gpo_dir_t)reg_cnfg_gpio1.bits.gpo_dir;
    }
    else if (channel == 2) {
    	reg_cnfg_gpio2_t reg_cnfg_gpio2 = {0};
        ret = read_register(CNFG_GPIO2, (uint8_t *)&(reg_cnfg_gpio2));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }

        *gpo_dir = (gpo_dir_t)reg_cnfg_gpio2.bits.gpo_dir;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetCID(void) {
    char rbuf[1];
    int ret;

    ret = read_register(CID, (uint8_t *)&(rbuf));
        if (ret != MAX77654_NO_ERROR) {
            return ret;
        }
 
    return *rbuf;
}

int MAX77654::SetWatchdogTimerPeriod(wdt_per_t wdt_per)
{
	reg_cnfg_wdt_t reg_cnfg_wdt = {0};

    SET_BIT_FIELD(CNFG_WDT, reg_cnfg_wdt, reg_cnfg_wdt.bits.wdt_per, wdt_per);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetWatchdogTimerPeriod(wdt_per_t *wdt_per)
{
    int ret;
    reg_cnfg_wdt_t reg_cnfg_wdt = {0};

    ret = read_register(CNFG_WDT, (uint8_t *)&(reg_cnfg_wdt));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *wdt_per = (wdt_per_t)reg_cnfg_wdt.bits.wdt_per;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetWatchdogTimerExpiredAction(wdt_mode_t wdt_mode)
{
	reg_cnfg_wdt_t reg_cnfg_wdt = {0};

    SET_BIT_FIELD(CNFG_WDT, reg_cnfg_wdt, reg_cnfg_wdt.bits.wdt_mode, wdt_mode);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetWatchdogTimerExpiredAction(wdt_mode_t *wdt_mode)
{
    int ret;
    reg_cnfg_wdt_t reg_cnfg_wdt = {0};

    ret = read_register(CNFG_WDT, (uint8_t *)&(reg_cnfg_wdt));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *wdt_mode = (wdt_mode_t)reg_cnfg_wdt.bits.wdt_mode;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetWatchdogTimerClearControl(wdt_clr_t wdt_clr)
{
	reg_cnfg_wdt_t reg_cnfg_wdt = {0};

    SET_BIT_FIELD(CNFG_WDT, reg_cnfg_wdt, reg_cnfg_wdt.bits.wdt_clr, wdt_clr);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetWatchdogTimerClearControl(wdt_clr_t *wdt_clr)
{
    int ret;
    reg_cnfg_wdt_t reg_cnfg_wdt = {0};

    ret = read_register(CNFG_WDT, (uint8_t *)&(reg_cnfg_wdt));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *wdt_clr = (wdt_clr_t)reg_cnfg_wdt.bits.wdt_clr;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetWatchdogTimerEnable(wdt_en_t wdt_en)
{
	reg_cnfg_wdt_t reg_cnfg_wdt = {0};

    SET_BIT_FIELD(CNFG_WDT, reg_cnfg_wdt, reg_cnfg_wdt.bits.wdt_en, wdt_en);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetWatchdogTimerEnable(wdt_en_t *wdt_en)
{
    int ret;
    reg_cnfg_wdt_t reg_cnfg_wdt = {0};

    ret = read_register(CNFG_WDT, (uint8_t *)&(reg_cnfg_wdt));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *wdt_en = (wdt_en_t)reg_cnfg_wdt.bits.wdt_en;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetFactorySetSafetyBit(wdt_lock_t wdt_lock)
{
	reg_cnfg_wdt_t reg_cnfg_wdt = {0};

    SET_BIT_FIELD(CNFG_WDT, reg_cnfg_wdt, reg_cnfg_wdt.bits.wdt_lock, wdt_lock);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetFactorySetSafetyBit(wdt_lock_t *wdt_lock)
{
    int ret;
    reg_cnfg_wdt_t reg_cnfg_wdt = {0};

    ret = read_register(CNFG_WDT, (uint8_t *)&(reg_cnfg_wdt));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *wdt_lock = (wdt_lock_t)reg_cnfg_wdt.bits.wdt_lock;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetMinimumVCHGINVoltageLoopStatus(stat_t *stat)
{
    int ret;
    reg_stat_chg_a_t reg_stat_chg_a = {0};

    ret = read_register(STAT_CHG_A, (uint8_t *)&(reg_stat_chg_a));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *stat = (stat_t)reg_stat_chg_a.bits.vchgin_min_stat;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetCHGINCurrentLimitLoopStatus(stat_t *stat)
{
    int ret;
    reg_stat_chg_a_t reg_stat_chg_a = {0};

    ret = read_register(STAT_CHG_A, (uint8_t *)&(reg_stat_chg_a));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *stat = (stat_t)reg_stat_chg_a.bits.ichgin_lim_stat;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetMinimumSYSVoltageLoopStatus(stat_t *stat)
{
    int ret;
    reg_stat_chg_a_t reg_stat_chg_a = {0};

    ret = read_register(STAT_CHG_A, (uint8_t *)&(reg_stat_chg_a));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *stat = (stat_t)reg_stat_chg_a.bits.vsys_min_stat;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetMaximumJunctionTempLoopStatus(stat_t *stat)
{
    int ret;
    reg_stat_chg_a_t reg_stat_chg_a = {0};

    ret = read_register(STAT_CHG_A, (uint8_t *)&(reg_stat_chg_a));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *stat = (stat_t)reg_stat_chg_a.bits.tj_reg_stat;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetBatteryTemperatureDetails(thm_dtls_t *thm_dtls)
{
    int ret;
    reg_stat_chg_a_t reg_stat_chg_a = {0};

    ret = read_register(STAT_CHG_A, (uint8_t *)&(reg_stat_chg_a));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *thm_dtls = (thm_dtls_t)reg_stat_chg_a.bits.thm_dtls;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetChargerDetails(chg_dtls_t *chg_dtls)
{
    int ret;
    reg_stat_chg_b_t reg_stat_chg_b = {0};

    ret = read_register(STAT_CHG_B, (uint8_t *)&(reg_stat_chg_b));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *chg_dtls = (chg_dtls_t)reg_stat_chg_b.bits.chg_dtls;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetCHGINStatusDetails(chgin_dtls_t *chgin_dtls)
{
    int ret;
    reg_stat_chg_b_t reg_stat_chg_b = {0};

    ret = read_register(STAT_CHG_B, (uint8_t *)&(reg_stat_chg_b));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *chgin_dtls = (chgin_dtls_t)reg_stat_chg_b.bits.chgin_dtls;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetQuickChargerStatus(chg_t *chg)
{
    int ret;
    reg_stat_chg_b_t reg_stat_chg_b = {0};

    ret = read_register(STAT_CHG_B, (uint8_t *)&(reg_stat_chg_b));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *chg = (chg_t)reg_stat_chg_b.bits.chg;

    return MAX77654_NO_ERROR;
}

int MAX77654::GetTimeSuspendedIndicator(time_sus_t *time_sus)
{
    int ret;
    reg_stat_chg_b_t reg_stat_chg_b = {0};

    ret = read_register(STAT_CHG_B, (uint8_t *)&(reg_stat_chg_b));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *time_sus = (time_sus_t)reg_stat_chg_b.bits.time_sus;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetSYSCONFIGMBit(intm_t intm)
{
	reg_int_m_chg_t reg_int_m_chg = {0};

    SET_BIT_FIELD(INT_M_CHG, reg_int_m_chg, reg_int_m_chg.bits.sys_cnfg_m, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetSYSCONFIGMBit(intm_t *intm)
{
    int ret;
    reg_int_m_chg_t reg_int_m_chg = {0};

    ret = read_register(INT_M_CHG, (uint8_t *)&(reg_int_m_chg));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_int_m_chg.bits.sys_cnfg_m;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetSYSCTRLMBit(intm_t intm)
{
	reg_int_m_chg_t reg_int_m_chg = {0};

    SET_BIT_FIELD(INT_M_CHG, reg_int_m_chg, reg_int_m_chg.bits.sys_ctrl_m, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetSYSCTRLMBit(intm_t *intm)
{
    int ret;
    reg_int_m_chg_t reg_int_m_chg = {0};

    ret = read_register(INT_M_CHG, (uint8_t *)&(reg_int_m_chg));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_int_m_chg.bits.sys_ctrl_m;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetCHGINCTRLMBit(intm_t intm)
{
	reg_int_m_chg_t reg_int_m_chg = {0};

    SET_BIT_FIELD(INT_M_CHG, reg_int_m_chg, reg_int_m_chg.bits.chgin_ctrl_m, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetCHGINCTRLMBit(intm_t *intm)
{ 
    int ret;
    reg_int_m_chg_t reg_int_m_chg = {0};

    ret = read_register(INT_M_CHG, (uint8_t *)&(reg_int_m_chg));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_int_m_chg.bits.chgin_ctrl_m;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetTJREGMBit(intm_t intm)
{
	reg_int_m_chg_t reg_int_m_chg = {0};

    SET_BIT_FIELD(INT_M_CHG, reg_int_m_chg, reg_int_m_chg.bits.tj_reg_m, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetTJREGMBit(intm_t *intm)
{
    int ret;
    reg_int_m_chg_t reg_int_m_chg = {0};

    ret = read_register(INT_M_CHG, (uint8_t *)&(reg_int_m_chg));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_int_m_chg.bits.tj_reg_m;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetCHGINMBit(intm_t intm)
{
	reg_int_m_chg_t reg_int_m_chg = {0};

    SET_BIT_FIELD(INT_M_CHG, reg_int_m_chg, reg_int_m_chg.bits.chgin_m, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetCHGINMBit(intm_t *intm)
{
    int ret;
    reg_int_m_chg_t reg_int_m_chg = {0};

    ret = read_register(INT_M_CHG, (uint8_t *)&(reg_int_m_chg));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_int_m_chg.bits.chgin_m;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetCHGMBit(intm_t intm)
{
	reg_int_m_chg_t reg_int_m_chg = {0};

    SET_BIT_FIELD(INT_M_CHG, reg_int_m_chg, reg_int_m_chg.bits.chg_m, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetCHGMBit(intm_t *intm)
{
    int ret;
    reg_int_m_chg_t reg_int_m_chg = {0};

    ret = read_register(INT_M_CHG, (uint8_t *)&(reg_int_m_chg));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_int_m_chg.bits.chg_m;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetTHMMBit(intm_t intm)
{
	reg_int_m_chg_t reg_int_m_chg = {0};

    SET_BIT_FIELD(INT_M_CHG, reg_int_m_chg, reg_int_m_chg.bits.thm_m, intm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetTHMMBit(intm_t *intm)
{
    int ret;
    reg_int_m_chg_t reg_int_m_chg = {0};

    ret = read_register(INT_M_CHG, (uint8_t *)&(reg_int_m_chg));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *intm = (intm_t)reg_int_m_chg.bits.thm_m;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetHOTJEITATemperature(thm_hot_t thm_hot)
{
	reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    SET_BIT_FIELD(CNFG_CHG_A, reg_cnfg_chg_a, reg_cnfg_chg_a.bits.thm_hot, thm_hot);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetHOTJEITATemperature(thm_hot_t *thm_hot)
{
    int ret;
    reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    ret = read_register(CNFG_CHG_A, (uint8_t *)&(reg_cnfg_chg_a));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *thm_hot = (thm_hot_t)reg_cnfg_chg_a.bits.thm_hot;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetWARMJEITATemperature(thm_warm_t thm_warm)
{
	reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    SET_BIT_FIELD(CNFG_CHG_A, reg_cnfg_chg_a, reg_cnfg_chg_a.bits.thm_warm, thm_warm);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetWARMJEITATemperature(thm_warm_t *thm_warm)
{
    int ret;
    reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    ret = read_register(CNFG_CHG_A, (uint8_t *)&(reg_cnfg_chg_a));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *thm_warm = (thm_warm_t)reg_cnfg_chg_a.bits.thm_warm;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetCOOLJEITATemperature(thm_cool_t thm_cool)
{
	reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    SET_BIT_FIELD(CNFG_CHG_A, reg_cnfg_chg_a, reg_cnfg_chg_a.bits.thm_cool, thm_cool);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetCOOLJEITATemperature(thm_cool_t *thm_cool)
{
    int ret;
    reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    ret = read_register(CNFG_CHG_A, (uint8_t *)&(reg_cnfg_chg_a));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *thm_cool = (thm_cool_t)reg_cnfg_chg_a.bits.thm_cool;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetCOLDJEITATemperature(thm_cold_t thm_cold)
{
	reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    SET_BIT_FIELD(CNFG_CHG_A, reg_cnfg_chg_a, reg_cnfg_chg_a.bits.thm_cold, thm_cold);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetCOLDJEITATemperature(thm_cold_t *thm_cold)
{
    int ret;
    reg_cnfg_chg_a_t reg_cnfg_chg_a = {0};

    ret = read_register(CNFG_CHG_A, (uint8_t *)&(reg_cnfg_chg_a));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *thm_cold = (thm_cold_t)reg_cnfg_chg_a.bits.thm_cold;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetMinimumCHGINVoltage(vchgin_min_t vchgin_min)
{
	reg_cnfg_chg_b_t reg_cnfg_chg_b = {0};

    SET_BIT_FIELD(CNFG_CHG_B, reg_cnfg_chg_b, reg_cnfg_chg_b.bits.vchgin_min, vchgin_min);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetMinimumCHGINVoltage(vchgin_min_t *vchgin_min)
{
    int ret;
    reg_cnfg_chg_b_t reg_cnfg_chg_b = {0};

    ret = read_register(CNFG_CHG_B, (uint8_t *)&(reg_cnfg_chg_b));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *vchgin_min = (vchgin_min_t)reg_cnfg_chg_b.bits.vchgin_min;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetCHGINInputCurrentLimit(ichgin_lim_t ichgin_lim)
{
	reg_cnfg_chg_b_t reg_cnfg_chg_b = {0};

    SET_BIT_FIELD(CNFG_CHG_B, reg_cnfg_chg_b, reg_cnfg_chg_b.bits.ichgin_lim, ichgin_lim);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetCHGINInputCurrentLimit(ichgin_lim_t *ichgin_lim)
{
    int ret;
    reg_cnfg_chg_b_t reg_cnfg_chg_b = {0};

    ret = read_register(CNFG_CHG_B, (uint8_t *)&(reg_cnfg_chg_b));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *ichgin_lim = (ichgin_lim_t)reg_cnfg_chg_b.bits.ichgin_lim;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetPrequalificationChargeCurrent(i_pq_t i_pq)
{
	reg_cnfg_chg_b_t reg_cnfg_chg_b = {0};

    SET_BIT_FIELD(CNFG_CHG_B, reg_cnfg_chg_b, reg_cnfg_chg_b.bits.i_pq, i_pq);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetPrequalificationChargeCurrent(i_pq_t *i_pq)
{
    int ret;
    reg_cnfg_chg_b_t reg_cnfg_chg_b = {0};

    ret = read_register(CNFG_CHG_B, (uint8_t *)&(reg_cnfg_chg_b));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *i_pq = (i_pq_t)reg_cnfg_chg_b.bits.i_pq;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetBatteryChargerEnable(chg_en_t chg_en)
{
	reg_cnfg_chg_b_t reg_cnfg_chg_b = {0};

    SET_BIT_FIELD(CNFG_CHG_B, reg_cnfg_chg_b, reg_cnfg_chg_b.bits.chg_en, chg_en);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetBatteryChargerEnable(chg_en_t *chg_en)
{
    int ret;
    reg_cnfg_chg_b_t reg_cnfg_chg_b = {0};

    ret = read_register(CNFG_CHG_B, (uint8_t *)&(reg_cnfg_chg_b));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *chg_en = (chg_en_t)reg_cnfg_chg_b.bits.chg_en;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetBatteryPQVoltageThreshold(chg_pq_t chg_pq)
{
	reg_cnfg_chg_c_t reg_cnfg_chg_c = {0};

    SET_BIT_FIELD(CNFG_CHG_C, reg_cnfg_chg_c, reg_cnfg_chg_c.bits.chg_pq, chg_pq);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetBatteryPQVoltageThreshold(chg_pq_t *chg_pq)
{
    int ret;
    reg_cnfg_chg_c_t reg_cnfg_chg_c = {0};

    ret = read_register(CNFG_CHG_C, (uint8_t *)&(reg_cnfg_chg_c));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *chg_pq = (chg_pq_t)reg_cnfg_chg_c.bits.chg_pq;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetChargerTerminationCurrent(i_term_t i_term)
{
	reg_cnfg_chg_c_t reg_cnfg_chg_c = {0};

    SET_BIT_FIELD(CNFG_CHG_C, reg_cnfg_chg_c, reg_cnfg_chg_c.bits.i_term, i_term);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetChargerTerminationCurrent(i_term_t *i_term)
{
    int ret;
    reg_cnfg_chg_c_t reg_cnfg_chg_c = {0};

    ret = read_register(CNFG_CHG_C, (uint8_t *)&(reg_cnfg_chg_c));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *i_term = (i_term_t)reg_cnfg_chg_c.bits.i_term;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetTopOffTimerValue(t_topoff_t t_topoff)
{
	reg_cnfg_chg_c_t reg_cnfg_chg_c = {0};

    SET_BIT_FIELD(CNFG_CHG_C, reg_cnfg_chg_c, reg_cnfg_chg_c.bits.t_topoff, t_topoff);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetTopOffTimerValue(t_topoff_t *t_topoff)
{
    int ret;
    reg_cnfg_chg_c_t reg_cnfg_chg_c = {0};

    ret = read_register(CNFG_CHG_C, (uint8_t *)&(reg_cnfg_chg_c));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *t_topoff = (t_topoff_t)reg_cnfg_chg_c.bits.t_topoff;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetDieJunctionTemperature(tj_reg_t tj_reg)
{
	reg_cnfg_chg_d_t reg_cnfg_chg_d = {0};

    SET_BIT_FIELD(CNFG_CHG_D, reg_cnfg_chg_d, reg_cnfg_chg_d.bits.tj_reg, tj_reg);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetDieJunctionTemperature(tj_reg_t *tj_reg)
{
    int ret;
    reg_cnfg_chg_d_t reg_cnfg_chg_d = {0};

    ret = read_register(CNFG_CHG_D, (uint8_t *)&(reg_cnfg_chg_d));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *tj_reg = (tj_reg_t)reg_cnfg_chg_d.bits.tj_reg;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetSystemVoltageRegulation(vsys_reg_t vsys_reg)
{
	reg_cnfg_chg_d_t reg_cnfg_chg_d = {0};

    SET_BIT_FIELD(CNFG_CHG_D, reg_cnfg_chg_d, reg_cnfg_chg_d.bits.vsys_reg, vsys_reg);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetSystemVoltageRegulation(vsys_reg_t *vsys_reg)
{
    int ret;
    reg_cnfg_chg_d_t reg_cnfg_chg_d = {0};

    ret = read_register(CNFG_CHG_D, (uint8_t *)&(reg_cnfg_chg_d));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *vsys_reg = (vsys_reg_t)reg_cnfg_chg_d.bits.vsys_reg;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetFastChargeCCValue(chg_cc_t chg_cc)
{
	reg_cnfg_chg_e_t reg_cnfg_chg_e = {0};

    SET_BIT_FIELD(CNFG_CHG_E, reg_cnfg_chg_e, reg_cnfg_chg_e.bits.chg_cc, chg_cc);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetFastChargeCCValue(chg_cc_t *chg_cc)
{
    int ret;
    reg_cnfg_chg_e_t reg_cnfg_chg_e = {0};

    ret = read_register(CNFG_CHG_E, (uint8_t *)&(reg_cnfg_chg_e));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *chg_cc = (chg_cc_t)reg_cnfg_chg_e.bits.chg_cc;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetFastChargSafetyTimer(t_fast_chg_t t_fast_chg)
{
	reg_cnfg_chg_e_t reg_cnfg_chg_e = {0};

    SET_BIT_FIELD(CNFG_CHG_E, reg_cnfg_chg_e, reg_cnfg_chg_e.bits.t_fast_chg, t_fast_chg);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetFastChargSafetyTimer(t_fast_chg_t *t_fast_chg)
{
    int ret;
    reg_cnfg_chg_e_t reg_cnfg_chg_e = {0};

    ret = read_register(CNFG_CHG_E, (uint8_t *)&(reg_cnfg_chg_e));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *t_fast_chg = (t_fast_chg_t)reg_cnfg_chg_e.bits.t_fast_chg;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetFastChargeCCJEITA(chg_cc_t chg_cc_jeita)
{
	reg_cnfg_chg_f_t reg_cnfg_chg_f = {0};

    SET_BIT_FIELD(CNFG_CHG_F, reg_cnfg_chg_f, reg_cnfg_chg_f.bits.chg_cc_jeita, chg_cc_jeita);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetFastChargeCCJEITA(chg_cc_t *chg_cc_jeita)
{
    int ret;
    reg_cnfg_chg_f_t reg_cnfg_chg_f = {0};

    ret = read_register(CNFG_CHG_F, (uint8_t *)&(reg_cnfg_chg_f));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *chg_cc_jeita = (chg_cc_t)reg_cnfg_chg_f.bits.chg_cc_jeita;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetThermistorEnable(thm_en_t thm_en)
{
	reg_cnfg_chg_f_t reg_cnfg_chg_f = {0};

    SET_BIT_FIELD(CNFG_CHG_F, reg_cnfg_chg_f, reg_cnfg_chg_f.bits.thm_en, thm_en);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetThermistorEnable(thm_en_t *thm_en)
{
    int ret;
    reg_cnfg_chg_f_t reg_cnfg_chg_f = {0};

    ret = read_register(CNFG_CHG_F, (uint8_t *)&(reg_cnfg_chg_f));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *thm_en = (thm_en_t)reg_cnfg_chg_f.bits.thm_en;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetFastChargeBatteryRegVolt(chg_cv_t chg_cv)
{
	reg_cnfg_chg_g_t reg_cnfg_chg_g = {0};

    SET_BIT_FIELD(CNFG_CHG_G, reg_cnfg_chg_g, reg_cnfg_chg_g.bits.chg_cv, chg_cv);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetFastChargeBatteryRegVolt(chg_cv_t *chg_cv)
{
    int ret;
    reg_cnfg_chg_g_t reg_cnfg_chg_g = {0};

    ret = read_register(CNFG_CHG_G, (uint8_t *)&(reg_cnfg_chg_g));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *chg_cv = (chg_cv_t)reg_cnfg_chg_g.bits.chg_cv;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetCHGINUSBSuspendMode(usbs_t usbs)
{
	reg_cnfg_chg_g_t reg_cnfg_chg_g = {0};

    SET_BIT_FIELD(CNFG_CHG_G, reg_cnfg_chg_g, reg_cnfg_chg_g.bits.usbs, usbs);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetCHGINUSBSuspendMode(usbs_t *usbs)
{
    int ret;
    reg_cnfg_chg_g_t reg_cnfg_chg_g = {0};

    ret = read_register(CNFG_CHG_G, (uint8_t *)&(reg_cnfg_chg_g));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *usbs = (usbs_t)reg_cnfg_chg_g.bits.usbs;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetFastChargeVoltageJEITA(chg_cv_t chg_cv_jeita)
{
	reg_cnfg_chg_h_t reg_cnfg_chg_h = {0};

    SET_BIT_FIELD(CNFG_CHG_H, reg_cnfg_chg_h, reg_cnfg_chg_h.bits.chg_cv_jeita, chg_cv_jeita);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetFastChargeVoltageJEITA(chg_cv_t *chg_cv_jeita)
{
    int ret;
    reg_cnfg_chg_h_t reg_cnfg_chg_h = {0};

    ret = read_register(CNFG_CHG_H, (uint8_t *)&(reg_cnfg_chg_h));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *chg_cv_jeita = (chg_cv_t)reg_cnfg_chg_h.bits.chg_cv_jeita;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetDischargeCurrentFullScale(imon_dischg_scale_t imon_dischg_scale)
{
	reg_cnfg_chg_i_t reg_cnfg_chg_i = {0};

    SET_BIT_FIELD(CNFG_CHG_I, reg_cnfg_chg_i, reg_cnfg_chg_i.bits.imon_dischg_scale, imon_dischg_scale);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetDischargeCurrentFullScale(imon_dischg_scale_t *imon_dischg_scale)
{
    int ret;
    reg_cnfg_chg_i_t reg_cnfg_chg_i = {0};

    ret = read_register(CNFG_CHG_I, (uint8_t *)&(reg_cnfg_chg_i));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *imon_dischg_scale = (imon_dischg_scale_t)reg_cnfg_chg_i.bits.imon_dischg_scale;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetAMUX(mux_sel_t mux_sel)
{
	reg_cnfg_chg_i_t reg_cnfg_chg_i = {0};

    SET_BIT_FIELD(CNFG_CHG_I, reg_cnfg_chg_i, reg_cnfg_chg_i.bits.mux_sel, mux_sel);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetAMUX(mux_sel_t *mux_sel)
{
    int ret;
    reg_cnfg_chg_i_t reg_cnfg_chg_i = {0};

    ret = read_register(CNFG_CHG_I, (uint8_t *)&(reg_cnfg_chg_i));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *mux_sel = (mux_sel_t)reg_cnfg_chg_i.bits.mux_sel;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetSBBTargetOutVoltage(uint8_t channel, tv_sbb_t tv_sbb)
{
    if (channel == 0) {
    	reg_cnfg_sbb0_a_t reg_cnfg_sbb0_a = {0};
        SET_BIT_FIELD(CNFG_SBB0_A, reg_cnfg_sbb0_a, reg_cnfg_sbb0_a.bits.tv_sbb0, tv_sbb);
    }
    else if (channel == 1) {
    	reg_cnfg_sbb1_a_t reg_cnfg_sbb1_a = {0};
        SET_BIT_FIELD(CNFG_SBB1_A, reg_cnfg_sbb1_a, reg_cnfg_sbb1_a.bits.tv_sbb1, tv_sbb);
    }
    else if (channel == 2) {
    	reg_cnfg_sbb2_a_t reg_cnfg_sbb2_a = {0};
        SET_BIT_FIELD(CNFG_SBB2_A, reg_cnfg_sbb2_a, reg_cnfg_sbb2_a.bits.tv_sbb2, tv_sbb);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetSBBTargetOutVoltage(uint8_t channel, tv_sbb_t *tv_sbb)
{
    int ret;

    if (channel == 0) {
    	reg_cnfg_sbb0_a_t reg_cnfg_sbb0_a = {0};
        ret = read_register(CNFG_SBB0_A, (uint8_t *)&(reg_cnfg_sbb0_a));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *tv_sbb = (tv_sbb_t)reg_cnfg_sbb0_a.bits.tv_sbb0;
    }
    else if (channel == 1) {
    	reg_cnfg_sbb1_a_t reg_cnfg_sbb1_a = {0};
        ret = read_register(CNFG_SBB1_A, (uint8_t *)&(reg_cnfg_sbb1_a));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *tv_sbb = (tv_sbb_t)reg_cnfg_sbb1_a.bits.tv_sbb1;
    }
    else if (channel == 2) {
    	reg_cnfg_sbb2_a_t reg_cnfg_sbb2_a = {0};
        ret = read_register(CNFG_SBB2_A, (uint8_t *)&(reg_cnfg_sbb2_a));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *tv_sbb = (tv_sbb_t)reg_cnfg_sbb2_a.bits.tv_sbb2;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::SetSIMOOperationMode(uint8_t channel, op_mode_t op_mode)
{
    if (channel == 0) {
    	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
        SET_BIT_FIELD(CNFG_SBB0_B, reg_cnfg_sbb0_b, reg_cnfg_sbb0_b.bits.op_mode, op_mode);
    }
    else if (channel == 1) {
    	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
        SET_BIT_FIELD(CNFG_SBB1_B, reg_cnfg_sbb1_b, reg_cnfg_sbb1_b.bits.op_mode, op_mode);
    }
    else if (channel == 2) {
    	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
        SET_BIT_FIELD(CNFG_SBB2_B, reg_cnfg_sbb2_b, reg_cnfg_sbb2_b.bits.op_mode, op_mode);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetSIMOOperationMode(uint8_t channel, op_mode_t *op_mode)
{
    int ret;

    if (channel == 0) {
    	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
        ret = read_register(CNFG_SBB0_B, (uint8_t *)&(reg_cnfg_sbb0_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
    }
        *op_mode = (op_mode_t)reg_cnfg_sbb0_b.bits.op_mode;
    }
    else if (channel == 1) {
    	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
        ret = read_register(CNFG_SBB1_B, (uint8_t *)&(reg_cnfg_sbb1_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *op_mode = (op_mode_t)reg_cnfg_sbb1_b.bits.op_mode;
    }
    else if (channel == 2) {
    	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
        ret = read_register(CNFG_SBB2_B, (uint8_t *)&(reg_cnfg_sbb2_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *op_mode = (op_mode_t)reg_cnfg_sbb2_b.bits.op_mode;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::SetSBBPeakCurrentLimit(uint8_t channel, ip_sbb_t ip_sbb)
{
    if (channel == 0) {
    	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
        SET_BIT_FIELD(CNFG_SBB0_B, reg_cnfg_sbb0_b, reg_cnfg_sbb0_b.bits.ip_sbb0, ip_sbb);
    }
    else if (channel == 1) {
    	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
        SET_BIT_FIELD(CNFG_SBB1_B, reg_cnfg_sbb1_b, reg_cnfg_sbb1_b.bits.ip_sbb1, ip_sbb);
    }
    else if (channel == 2) {
    	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
        SET_BIT_FIELD(CNFG_SBB2_B, reg_cnfg_sbb2_b, reg_cnfg_sbb2_b.bits.ip_sbb2, ip_sbb);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetSBBPeakCurrentLimit(uint8_t channel, ip_sbb_t *ip_sbb)
{
    int ret;

    if (channel == 0) {
    	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
        ret = read_register(CNFG_SBB0_B, (uint8_t *)&(reg_cnfg_sbb0_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
    }
        *ip_sbb = (ip_sbb_t)reg_cnfg_sbb0_b.bits.ip_sbb0;
    }
    else if (channel == 1) {
    	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
        ret = read_register(CNFG_SBB1_B, (uint8_t *)&(reg_cnfg_sbb1_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *ip_sbb = (ip_sbb_t)reg_cnfg_sbb1_b.bits.ip_sbb1;
    }
    else if (channel == 2) {
    	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
        ret = read_register(CNFG_SBB2_B, (uint8_t *)&(reg_cnfg_sbb2_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *ip_sbb = (ip_sbb_t)reg_cnfg_sbb2_b.bits.ip_sbb2;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::SetSBBActiveDischargeEnable(uint8_t channel, ade_sbb_t ade_sbb)
{
    if (channel == 0) {
    	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
        SET_BIT_FIELD(CNFG_SBB0_B, reg_cnfg_sbb0_b, reg_cnfg_sbb0_b.bits.ade_sbb0, ade_sbb);
    }
    else if (channel == 1) {
    	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
        SET_BIT_FIELD(CNFG_SBB1_B, reg_cnfg_sbb1_b, reg_cnfg_sbb1_b.bits.ade_sbb1, ade_sbb);
    }
    else if (channel == 2) {
    	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
        SET_BIT_FIELD(CNFG_SBB2_B, reg_cnfg_sbb2_b, reg_cnfg_sbb2_b.bits.ade_sbb2, ade_sbb);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetSBBActiveDischargeEnable(uint8_t channel, ade_sbb_t *ade_sbb)
{
    int ret;
    
    if (channel == 0) {
    	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
        ret = read_register(CNFG_SBB0_B, (uint8_t *)&(reg_cnfg_sbb0_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *ade_sbb = (ade_sbb_t)reg_cnfg_sbb0_b.bits.ade_sbb0;
    }
    else if (channel == 1) {
    	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
        ret = read_register(CNFG_SBB1_B, (uint8_t *)&(reg_cnfg_sbb1_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *ade_sbb = (ade_sbb_t)reg_cnfg_sbb1_b.bits.ade_sbb1;
    }
    else if (channel == 2) {
    	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
        ret = read_register(CNFG_SBB2_B, (uint8_t *)&(reg_cnfg_sbb2_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *ade_sbb = (ade_sbb_t)reg_cnfg_sbb2_b.bits.ade_sbb2;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::SetSBBEnableControl(uint8_t channel, en_sbb_t en_sbb)
{
    if (channel == 0) {
    	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
        SET_BIT_FIELD(CNFG_SBB0_B, reg_cnfg_sbb0_b, reg_cnfg_sbb0_b.bits.en_sbb0, en_sbb);
    }
    else if (channel == 1) {
    	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
        SET_BIT_FIELD(CNFG_SBB1_B, reg_cnfg_sbb1_b, reg_cnfg_sbb1_b.bits.en_sbb1, en_sbb);
    }
    else if (channel == 2) {
    	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
        SET_BIT_FIELD(CNFG_SBB2_B, reg_cnfg_sbb2_b, reg_cnfg_sbb2_b.bits.en_sbb2, en_sbb);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetSBBEnableControl(uint8_t channel, en_sbb_t *en_sbb)
{
    int ret;

    if (channel == 0) {
    	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
        ret = read_register(CNFG_SBB0_B, (uint8_t *)&(reg_cnfg_sbb0_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *en_sbb = (en_sbb_t)reg_cnfg_sbb0_b.bits.en_sbb0;
    }
    else if (channel == 1) {
    	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
        ret = read_register(CNFG_SBB1_B, (uint8_t *)&(reg_cnfg_sbb1_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *en_sbb = (en_sbb_t)reg_cnfg_sbb1_b.bits.en_sbb1;
    }
    else if (channel == 2) {
    	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
        ret = read_register(CNFG_SBB2_B, (uint8_t *)&(reg_cnfg_sbb2_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *en_sbb = (en_sbb_t)reg_cnfg_sbb2_b.bits.en_sbb2;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::SetCHGINInputCurrentLimit(ichgin_lim_def_t ichgin_lim_def)
{
	reg_cnfg_sbb_top_t reg_cnfg_sbb_top = {0};

    SET_BIT_FIELD(CNFG_SBB_TOP, reg_cnfg_sbb_top, reg_cnfg_sbb_top.bits.ichgin_lim_def, ichgin_lim_def);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetCHGINInputCurrentLimit(ichgin_lim_def_t *ichgin_lim_def)
{
    int ret;
    reg_cnfg_sbb_top_t reg_cnfg_sbb_top = {0};

    ret = read_register(CNFG_SBB_TOP, (uint8_t *)&(reg_cnfg_sbb_top));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *ichgin_lim_def = (ichgin_lim_def_t)reg_cnfg_sbb_top.bits.ichgin_lim_def;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetSBBDriveStrength(drv_sbb_t drv_sbb)
{
	reg_cnfg_sbb_top_t reg_cnfg_sbb_top = {0};

    SET_BIT_FIELD(CNFG_SBB_TOP, reg_cnfg_sbb_top, reg_cnfg_sbb_top.bits.drv_sbb, drv_sbb);

    return MAX77654_NO_ERROR;
}

int MAX77654::GetSBBDriveStrength(drv_sbb_t *drv_sbb)
{
    int ret;
    reg_cnfg_sbb_top_t reg_cnfg_sbb_top = {0};

    ret = read_register(CNFG_SBB_TOP, (uint8_t *)&(reg_cnfg_sbb_top));
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    *drv_sbb = (drv_sbb_t)reg_cnfg_sbb_top.bits.drv_sbb;

    return MAX77654_NO_ERROR;
}

int MAX77654::SetLDOTargetOutVoltage(uint8_t channel, tv_ldo_t tv_ldo)
{
    if (channel == 0) {
    	reg_cnfg_ldo0_a_t reg_cnfg_ldo0_a = {0};
        SET_BIT_FIELD(CNFG_LDO0_A, reg_cnfg_ldo0_a, reg_cnfg_ldo0_a.bits.tv_ldo0, tv_ldo);
    }
    else if (channel == 1) {
    	reg_cnfg_ldo1_a_t reg_cnfg_ldo1_a = {0};
        SET_BIT_FIELD(CNFG_LDO1_A, reg_cnfg_ldo1_a, reg_cnfg_ldo1_a.bits.tv_ldo1, tv_ldo);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetLDOTargetOutVoltage(uint8_t channel, tv_ldo_t *tv_ldo)
{
    int ret;

    if (channel == 0) {
    	reg_cnfg_ldo0_a_t reg_cnfg_ldo0_a = {0};
        ret = read_register(CNFG_LDO0_A, (uint8_t *)&(reg_cnfg_ldo0_a));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *tv_ldo = (tv_ldo_t)reg_cnfg_ldo0_a.bits.tv_ldo0;
    }
    else if (channel == 1) {
    	reg_cnfg_ldo1_a_t reg_cnfg_ldo1_a = {0};
        ret = read_register(CNFG_LDO1_A, (uint8_t *)&(reg_cnfg_ldo1_a));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *tv_ldo = (tv_ldo_t)reg_cnfg_ldo1_a.bits.tv_ldo1;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::SetLDOMode(uint8_t channel, ldo_md_t ldo_md)
{
    if (channel == 0) {
    	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};
        SET_BIT_FIELD(CNFG_LDO0_B, reg_cnfg_ldo0_b, reg_cnfg_ldo0_b.bits.ldo0_md, ldo_md);
    }
    else if (channel == 1) {
    	reg_cnfg_ldo1_b_t reg_cnfg_ldo1_b = {0};
        SET_BIT_FIELD(CNFG_LDO1_B, reg_cnfg_ldo1_b, reg_cnfg_ldo1_b.bits.ldo1_md, ldo_md);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetLDOMode(uint8_t channel, ldo_md_t *ldo_md)
{
    int ret;

    if (channel == 0) {
    	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};
        ret = read_register(CNFG_LDO0_B, (uint8_t *)&(reg_cnfg_ldo0_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *ldo_md = (ldo_md_t)reg_cnfg_ldo0_b.bits.ldo0_md;
    }
    else if (channel == 1) {
    	reg_cnfg_ldo1_b_t reg_cnfg_ldo1_b = {0};
        ret = read_register(CNFG_LDO1_B, (uint8_t *)&(reg_cnfg_ldo1_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *ldo_md = (ldo_md_t)reg_cnfg_ldo1_b.bits.ldo1_md;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::SetLDOActiveDischargeEnable(uint8_t channel, ade_ldo_t ade_ldo)
{
    if (channel == 0) {
    	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};
        SET_BIT_FIELD(CNFG_LDO0_B, reg_cnfg_ldo0_b, reg_cnfg_ldo0_b.bits.ade_ldo0, ade_ldo);
    }
    else if (channel == 1) {
    	reg_cnfg_ldo1_b_t reg_cnfg_ldo1_b = {0};
        SET_BIT_FIELD(CNFG_LDO1_B, reg_cnfg_ldo1_b, reg_cnfg_ldo1_b.bits.ade_ldo1, ade_ldo);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetLDOActiveDischargeEnable(uint8_t channel, ade_ldo_t *ade_ldo)
{
    int ret;

    if (channel == 0) {
    	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};
        ret = read_register(CNFG_LDO0_B, (uint8_t *)&(reg_cnfg_ldo0_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *ade_ldo = (ade_ldo_t)reg_cnfg_ldo0_b.bits.ade_ldo0;
    }
    else if (channel == 1) {
    	reg_cnfg_ldo1_b_t reg_cnfg_ldo1_b = {0};
        ret = read_register(CNFG_LDO1_B, (uint8_t *)&(reg_cnfg_ldo1_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *ade_ldo = (ade_ldo_t)reg_cnfg_ldo1_b.bits.ade_ldo1;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::SetLDOEnableControl(uint8_t channel, en_ldo_t en_ldo)
{
    if (channel == 0) {
    	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};
        SET_BIT_FIELD(CNFG_LDO0_B, reg_cnfg_ldo0_b, reg_cnfg_ldo0_b.bits.en_ldo0, en_ldo);
    }
    else if (channel == 1) {
    	reg_cnfg_ldo1_b_t reg_cnfg_ldo1_b = {0};
        SET_BIT_FIELD(CNFG_LDO1_B, reg_cnfg_ldo1_b, reg_cnfg_ldo1_b.bits.en_ldo1, en_ldo);
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::GetLDOEnableControl(uint8_t channel, en_ldo_t *en_ldo)
{
    int ret;

    if (channel == 0) {
    	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};
        ret = read_register(CNFG_LDO0_B, (uint8_t *)&(reg_cnfg_ldo0_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *en_ldo = (en_ldo_t)reg_cnfg_ldo0_b.bits.en_ldo0;
    }
    else if (channel == 1) {
    	reg_cnfg_ldo1_b_t reg_cnfg_ldo1_b = {0};
        ret = read_register(CNFG_LDO1_B, (uint8_t *)&(reg_cnfg_ldo1_b));
        if (ret != MAX77654_NO_ERROR) {
        return ret;
        }
        *en_ldo = (en_ldo_t)reg_cnfg_ldo1_b.bits.en_ldo1;
    }
    else {
        return MAX77654_INVALID_CHANNEL_NUMBER;
    }

    return MAX77654_NO_ERROR;
}

int MAX77654::IRQDisableAll()
{
    int ret;
    uint8_t reg = 0;
    uint8_t status = 0;

    //Disable Masks in INTM_GLBL1
    ret = write_register(INTM_GLBL1, &reg);
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }
    //Disable Masks in INTM_GLBL0
    ret = write_register(INTM_GLBL0, &reg);
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }
    //Disable Masks in INT_M_CHG
    ret = write_register(INT_M_CHG, &reg);
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    // Clear Interrupt Flags in INT_GLBL1
    ret = read_register(INT_GLBL1, &status);
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }
    // Clear Interrupt Flags in INT_GLBL0
    ret = read_register(INT_GLBL0, &status);
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }
    // Clear Interrupt Flags in INT_CHG
    ret = read_register(INT_CHG, &status);
    if (ret != MAX77654_NO_ERROR) {
        return ret;
    }

    return 0;
}

void MAX77654::SetInterruptHandler(int_glbl_t id, interrupt_handler_function func, void *cb)
{
    interrupt_handler_list[id].func = func;
    interrupt_handler_list[id].cb = cb;
}

void MAX77654::PostInterruptWork()
{
    int ret;
    uint8_t reg = 0, inten = 0, not_inten = 0, mask = 0;


    while (true) {

        ThisThread::flags_wait_any(POST_INTR_WORK_SIGNAL_ID);

        // Check Interrupt Flags in INT_GLBL0
        ret = read_register(INT_GLBL0, &reg);
        if (ret != MAX77654_NO_ERROR) {
            return;
        }

        ret = read_register(INTM_GLBL0, &inten);
        if (ret != MAX77654_NO_ERROR) {
            return;
        }

        not_inten = ~inten; // 0 means unmasked.

        for (int i = 0; i < INT_GLBL1_GPI1_F; i++) {
            mask = (1 << i);
            if ((reg & mask) && (not_inten & mask)) {
                if (interrupt_handler_list[i].func != NULL) {
                    interrupt_handler_list[i]
                    .func(interrupt_handler_list[i].cb);
                }
            }
        }

        // Check Interrupt Flags in INT_GLBL1
        ret = read_register(INT_GLBL1, &reg);
		if (ret != MAX77654_NO_ERROR) {
			return;
		}

		ret = read_register(INTM_GLBL1, &inten);
		if (ret != MAX77654_NO_ERROR) {
			return;
		}

		not_inten = ~inten; // 0 means unmasked.

		for (int i = INT_GLBL1_GPI1_F; i < INT_CHG_THM_I; i++) {
			mask = (1 << (i - INT_GLBL1_GPI1_F));
			if ((reg & mask) && (not_inten & mask)) {
				if (interrupt_handler_list[i].func != NULL) {
					interrupt_handler_list[i]
					.func(interrupt_handler_list[i].cb);
				}
			}
		}

		// Check Interrupt Flags in INT_CHG
        ret = read_register(INT_CHG, &reg);
		if (ret != MAX77654_NO_ERROR) {
			return;
		}

		ret = read_register(INT_M_CHG, &inten);
		if (ret != MAX77654_NO_ERROR) {
			return;
		}

		not_inten = ~inten; // 0 means unmasked.

		for (int i = INT_CHG_THM_I; i < INT_CHG_END; i++) {
			mask = (1 << (i - INT_CHG_THM_I));
			if ((reg & mask) && (not_inten & mask)) {
				if (interrupt_handler_list[i].func != NULL) {
					interrupt_handler_list[i]
					.func(interrupt_handler_list[i].cb);
				}
			}
		}
    }
}
void MAX77654::InterruptHandler()
{
    post_intr_work_thread->flags_set(POST_INTR_WORK_SIGNAL_ID);
}
