/*******************************************************************************
* Copyright (C) 2022 Maxim Integrated Products, Inc., All rights Reserved.
*
* This software is protected by copyright laws of the United States and
* of foreign countries. This material may also be protected by patent laws
* and technology transfer regulations of the United States and of foreign
* countries. This software is furnished under a license agreement and/or a
* nondisclosure agreement and may only be used or reproduced in accordance
* with the terms of those agreements. Dissemination of this information to
* any party or parties not specified in the license agreement and/or
* nondisclosure agreement is expressly prohibited.
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
*/

#ifndef _MAX77654_H_
#define _MAX77654_H_

#include "mbed.h"
#include "MAX77654_regs.h"

#define MAX77654_NO_ERROR                    0
#define MAX77654_VALUE_NULL                 -1
#define MAX77654_WRITE_DATA_FAILED          -2
#define MAX77654_READ_DATA_FAILED           -3
#define MAX77654_INVALID_DATA               -4
#define MAX77654_INVALID_CHANNEL_NUMBER     -5

#define MAX77654_I2C_ADDRESS              	0x90

/**
 * @brief MAX77654 Ultra-Low Power PMIC Featuring Single-Inductor, 3-Output Buck-Boost,
 * 2-LDOs, Power Path Charger for Small Li+, and Ship Mode
 *
 * @details The MAX77654 provides highly-integrated battery charging and power supply
 * solutions for low-power applications where size and efficiency are critical.
 * <br>https://www.maximintegrated.com/en/products/power/power-management-ics/MAX77654.html
 *
 * @code
 * @endcode
 */

class MAX77654
{
private:
    I2C *i2c_handler;
    InterruptIn *irq_pin;	// interrupt pin

    /**
     * @brief   Register Addresses
     * @details Enumerated MAX77654 register addresses
     */
    typedef enum {
        /*Global*/
        INT_GLBL0    = 0x00,    // Interrupt Status 0
        INT_GLBL1    = 0x04,    // Interrupt Status 1
        ERCFLAG      = 0x05,    // Flags
        STAT_GLBL    = 0x06,    // Global Status
        INTM_GLBL1   = 0x08,    // Interrupt Mask 1
        INTM_GLBL0   = 0x09,    // Interrupt Mask 0
        CNFG_GLBL    = 0x10,    // Configuration Global
        CNFG_GPIO0   = 0x11,    // GPIO0 Configuration
        CNFG_GPIO1   = 0x12,    // GPIO1 Configuration
        CNFG_GPIO2   = 0x13,    // GPIO2 Configuration
        CID          = 0x14,    // Chip Identification Code
        CNFG_WDT     = 0x17,    // Configuration WatchDog Timer
        /*Charger*/
        INT_CHG      = 0x01,    // Charger Interrupt Status
        STAT_CHG_A   = 0x02,    // Charger Status A
        STAT_CHG_B   = 0x03,    // Charger Status B
        INT_M_CHG    = 0x07,    // Charger Interrupt Mask
        CNFG_CHG_A   = 0x20,    // Charger Configuration A
        CNFG_CHG_B   = 0x21,    // Charger Configuration B
        CNFG_CHG_C   = 0x22,    // Charger Configuration C
        CNFG_CHG_D   = 0x23,    // Charger Configuration D
        CNFG_CHG_E   = 0x24,    // Charger Configuration E
        CNFG_CHG_F   = 0x25,    // Charger Configuration F
        CNFG_CHG_G   = 0x26,    // Charger Configuration G
        CNFG_CHG_H   = 0x27,    // Charger Configuration H
        CNFG_CHG_I   = 0x28,    // Charger Configuration I
        /*SBB*/
        CNFG_SBB0_A  = 0x29,    // SIMO Buck-Boost 0 Configuration A
        CNFG_SBB0_B  = 0x2A,    // SIMO Buck-Boost 0 Configuration B
        CNFG_SBB1_A  = 0x2B,    // SIMO Buck-Boost 1 Configuration A
        CNFG_SBB1_B  = 0x2C,    // SIMO Buck-Boost 1 Configuration B
        CNFG_SBB2_A  = 0x2D,    // SIMO Buck-Boost 2 Configuration A
        CNFG_SBB2_B  = 0x2E,    // SIMO Buck-Boost 2 Configuration B
        CNFG_SBB_TOP = 0x2F,    // SIMO Buck-Boost Configuration
        /*LDO*/
        CNFG_LDO0_A   = 0x38,   // LDO Configuration A
        CNFG_LDO0_B   = 0x39,   // LDO Configuration B
        CNFG_LDO1_A   = 0x3A,   // LDO Configuration B
        CNFG_LDO1_B   = 0x3B    // LDO Configuration B
    } reg_t;

    /**
    * @brief	Interrupt handler function
    */
    void InterruptHandler();

    void (MAX77654::*funcptr)(void);

    /**
    * @brief	Post interrupt jobs after interrupt is detected.
    */
    void PostInterruptWork();

    Thread *post_intr_work_thread;

    struct handler {
        void (*func)(void *);
        void *cb;
    };

    handler *interrupt_handler_list;

public:
	/**
    * @brief Register Configuration
	*		 All Interrupt Flags combined from INT_GLBL0, INT_GLBL1 and INT_CHG
    *
    * @details
    *  - Register      : ERCFLAG (0x05)
    *  - Bit Fields    : [7:0]
    *  - Default       : 0x0
    *  - Description   : Enumerated interrupt flags
    */
    typedef enum {
        INT_GLBL0_GPI0_F,
        INT_GLBL0_GPI0_R,
        INT_GLBL0_NEN_F,
        INT_GLBL0_NEN_R,
        INT_GLBL0_TJAL1_R,
        INT_GLBL0_TJAL2_R,
        INT_GLBL0_DOD1_R,
        INT_GLBL0_DOD0_R,
        INT_GLBL1_GPI1_F,
        INT_GLBL1_GPI1_R,
        INT_GLBL1_GPI2_F,
        INT_GLBL1_GPI2_R,
        INT_GLBL1_SBB_TO,
        INT_GLBL1_LDO0_F,
        INT_GLBL1_LDO1_F,
        INT_CHG_THM_I,
        INT_CHG_CGH_I,
        INT_CHG_CHGIN_I,
        INT_CHG_TJ_REG_I,
        INT_CHG_CHGIN_CTRL_I,
        INT_CHG_SYS_CTRL_I,
        INT_CHG_SYS_CNFG_I,
		INT_CHG_END
    } int_glbl_t;

    /**
     * MAX77654 constructor.
     */
    MAX77654(I2C *i2c, PinName IRQPin = NC);

    /**
     * MAX77654 destructor.
     */
    ~MAX77654();

    /**
    * @brief	Function pointer type to interrupt handler function
    */
    typedef void (*interrupt_handler_function)(void *);

    /**
    * @brief Read from a register.
    *
    * @param[in] reg Address of a register to be read.
    * @param[out] value Pointer to save result value.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int read_register(uint8_t reg, uint8_t *value);

    /**
    * @brief Write to a register.
    *
    * @param[in] reg Address of a register to be written.
    * @param[out] value Pointer of value to be written to register.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int write_register(uint8_t reg, const uint8_t *value);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : ERCFLAG (0x05)
     *  - Bit Fields    : [7:0]
     *  - Default       : 0x0
     *  - Description   : Event Recorder Flags.
     */
    typedef enum {
    	ERC_FLAG_NOT_OCCURRED,
		ERC_FLAG_OCCURRED
    }ercflag_t;

    /**
    * @brief		Get Thermal Overload Flag.
    *
    * @param[out]	ercflag Thermal overload bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetThermalOverload(ercflag_t *ercflag);

    /**
    * @brief		Get Sys Domain Overvoltage Lockout Flag.
    *
    * @param[out]	ercflag Sys domain overvoltage lockout bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetSysOvervoltageLockout(ercflag_t *ercflag);

    /**
    * @brief		Get Sys Domain Undervoltage Lockout Flag.
    *
    * @param[out]	ercflag Sys domain undervoltage lockout bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetSysUndervoltageLockout(ercflag_t *ercflag);

    /**
    * @brief		Get Manual Reset Timer Flag.
    *
    * @param[out]	ercflag Manual reset timer bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetManualResetTimer(ercflag_t *ercflag);

    /**
    * @brief		Get Software Off Flag.
    *
    * @param[out]	ercflag Software off bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetSoftwareOffFlag(ercflag_t *ercflag);

    /**
    * @brief		Get Software Cold Reset.
    *
    * @param[out]	ercflag Software cold reset bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetSoftwareColdResetFlag(ercflag_t *ercflag);

    /**
    * @brief		Get Watchdog Timer Off Flag.
    *
    * @param[out]	ercflag Watchdog timer off bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetWatchdogTimerOffFlag(ercflag_t *ercflag);

    /**
    * @brief		Get Watchdog Timer Reset Flag.
    *
    * @param[out]	ercflag Watchdog timer reset bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetWatchdogTimerResetFlag(ercflag_t *ercflag);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_GLBL (0x06)
	*  - Bit Fields    : [7]
	*  - Default       : 0x0
	*  - Description   : Device Identification Bits for Metal Options.
	*/
    typedef enum {
        DIDM_MAX77654,
        DIDM_RESERVED
    }didm_t;

    /**
    * @brief		Get Device Identification Bits for Metal Options.
    *
    * @param[out]	didm Device identification bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetDeviceIdentification(didm_t *didm);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_GLBL (0x06)
	*  - Bit Fields    : [6]
	*  - Default       : 0x0
	*  - Description   : BOK Interrupt Status.
	*/
    typedef enum {
        BOK_MAIN_BIAS_NOT_READY,
        BOK_MAIN_BIAS_READY
    }bok_t;

    /**
    * @brief		Get BOK Interrupt Status.
    *
    * @param[out]	bok Main bias status bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetBOKInterruptStatus(bok_t *bok);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_GLBL (0x06)
	*  - Bit Fields    : [5]
	*  - Default       : 0x0
	*  - Description   : LDO0 Dropout Detector Rising Status.
	*/
    typedef enum {
        DOD0_S_LDO0_NOT_IN_DROPOUT,
        DOD0_S_LDO0_IN_DROPOUT
    }dod0_s_t;

    /**
    * @brief		Get LDO0 Dropout Detector Rising Status.
    *
    * @param[out]	dod0_s LDO0 dropout status bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetLDO0DropoutRisingStatus(dod0_s_t *dod0_s);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_GLBL (0x06)
	*  - Bit Fields    : [4]
	*  - Default       : 0x0
	*  - Description   : LDO1 Dropout Detector Rising Status.
	*/
    typedef enum {
        DOD1_S_LDO1_NOT_IN_DROPOUT,
        DOD1_S_LDO1_IN_DROPOUT
    }dod1_s_t;

    /**
    * @brief		Get LDO1 Dropout Detector Rising Status.
    *
    * @param[out]	dod1_s LDO1 dropout status bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetLDO1DropoutRisingStatus(dod1_s_t *dod1_s);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_GLBL (0x06)
	*  - Bit Fields    : [3]
	*  - Default       : 0x0
	*  - Description   : Thermal Alarm 2 Status.
	*/
    typedef enum {
        TJAL2_S_JUNC_TEMP_LESS_THAN_TJA2,
        TJAL2_S_JUNC_TEMP_GREATER_THAN_TJA2
    }tjal2_s_t;

    /**
    * @brief		Get Thermal Alarm 2 Status.
    *
    * @param[out]	tjal2_s Thermal alarm 2 status bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetThermalAlarm2Status(tjal2_s_t *tjal2_s);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_GLBL (0x06)
	*  - Bit Fields    : [2]
	*  - Default       : 0x0
	*  - Description   : Thermal Alarm 1 Status.
	*/
    typedef enum {
        TJAL1_S_JUNC_TEMP_LESS_THAN_TJA1,
        TJAL1_S_JUNC_TEMP_GREATER_THAN_TJA1
    }tjal1_s_t;

    /**
    * @brief		Get Thermal Alarm 1 Status.
    *
    * @param[out]	tjal1_s Thermal alarm 1 status bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetThermalAlarm1Status(tjal1_s_t *tjal1_s);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_GLBL (0x06)
	*  - Bit Fields    : [1]
	*  - Default       : 0x0
	*  - Description   : Debounced Status for the nEN input.
	*/
    typedef enum {
        STAT_EN_NEN_NOT_ACTIVE,
        STAT_EN_NEN_ACTIVE
    }stat_en_t;

    /**
    * @brief		Get Debounced Status for the nEN input.
    *
    * @param[out]	stat_en Debounced status bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetnENDebouncedStatus(stat_en_t *stat_en);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_GLBL (0x06)
	*  - Bit Fields    : [0]
	*  - Default       : 0x0
	*  - Description   : Software Version of the nIRQ MOSFET gate drive.
	*/
    typedef enum {
        STAT_IRQ_UNMASKED_GATE_DRIVE_LOW,
        STAT_IRQ_UNMASKED_GATE_DRIVE_HIGH
    }stat_irq_t;

    /**
    * @brief		Get Software Version of the nIRQ MOSFET gate drive.
    *
    * @param[out]	stat_en Software version bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetSoftwareVersionGateDrive(stat_irq_t *stat_irq);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : INT_M_CHG (0x07), INTM_GLBL1 (0x08), and INTM_GLBL0 (0x09)
	*  - Bit Fields    : [7:0]
	*  - Default       : 0x0
	*  - Description   : Interrupt Masks.
	*/
    typedef enum {
        INTM_INTERRUPT_UNMASKED,
        INTM_INTERRUPT_MASKED
    }intm_t;
    
    /**
    * @brief		Set LDO1 Fault Interrupt Mask.
    *
    * @param[in]	intm LDO1 fault interrupt mask bit to be written.
    *
    * @return		0 on success, error code on failure.
    */
    int SetLDO1FaultInterruptMask(intm_t intm);

    /**
    * @brief		Get LDO1 Fault Interrupt Mask.
    *
    * @param[out]	intm LDO1 fault interrupt mask bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetLDO1FaultInterruptMask(intm_t *intm);

    /**
    * @brief		Set LDO0 Fault Interrupt Mask.
    *
    * @param[in]	intm LDO0 fault interrupt mask bit to be written.
    *
    * @return		0 on success, error code on failure.
    */
    int SetLDO0FaultInterruptMask(intm_t intm);

    /**
    * @brief		Get LDO1 Fault Interrupt Mask.
    *
    * @param[out]	intm LDO1 fault interrupt mask bit to be read.
    *
    * @return		0 on success, error code on failure.
    */
    int GetLDO0FaultInterruptMask(intm_t *intm);

    /**
    * @brief		Set SBB Timeout Mask.
    *
    * @param[in]	intm SBB timeout interrupt mask bit to be written.
    *
    * @return		0 on success, error code on failure.
    */
    int SetSBBTimeoutMask(intm_t intm);

    /**
	* @brief		Get SBB Timeout Mask.
	*
	* @param[out]	intm SBB timeout interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetSBBTimeoutMask(intm_t *intm);

    /**
    * @brief		Set GPI2 Rising Interrupt Mask.
    *
    * @param[in]	intm GPI2 rising interrupt mask bit to be written.
    *
    * @return		0 on success, error code on failure.
    */
    int SetGPI2RisingInterruptMask(intm_t intm);

    /**
	* @brief		Get GPI2 Rising Mask.
	*
	* @param[out]	intm GPI2 rising interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetGPI2RisingInterruptMask(intm_t *intm);

    /**
	* @brief		Set GPI2 Falling Interrupt Mask.
	*
	* @param[in]	intm GPI2 falling interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetGPI2FallingInterruptMask(intm_t intm);

    /**
	* @brief		Get GPI2 Falling Mask.
	*
	* @param[out]	intm GPI2 falling interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetGPI2FallingInterruptMask(intm_t *intm);

    /**
    * @brief		Set GPI1 Rising Interrupt Mask.
    *
    * @param[in]	intm GPI1 rising interrupt mask bit to be written.
    *
    * @return		0 on success, error code on failure.
    */
    int SetGPI1RisingInterruptMask(intm_t intm);

    /**
	* @brief		Get GPI1 Rising Mask.
	*
	* @param[out]	intm GPI1 rising interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetGPI1RisingInterruptMask(intm_t *intm);

    /**
	* @brief		Set GPI1 Falling Interrupt Mask.
	*
	* @param[in]	intm GPI1 falling interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetGPI1FallingInterruptMask(intm_t intm);

    /**
	* @brief		Get GPI1 Falling Mask.
	*
	* @param[out]	intm GPI1 falling interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetGPI1FallingInterruptMask(intm_t *intm);

    /**
	* @brief		Set LDO Dropout0 Detector Rising Interrupt Mask.
	*
	* @param[in]	intm LDO dropout0 detector rising interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetDropout0RisingInterruptMask(intm_t intm);

    /**
	* @brief		Get LDO Dropout0 Detector Rising Interrupt Mask.
	*
	* @param[out]	intm LDO dropout0 detector rising interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetDropout0RisingInterruptMask(intm_t *intm);

    /**
	* @brief		Set LDO Dropout1 Detector Rising Interrupt Mask.
	*
	* @param[in]	intm LDO dropout1 detector rising interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetDropout1RisingInterruptMask(intm_t intm);

    /**
	* @brief		Get LDO Dropout1 Detector Rising Interrupt Mask.
	*
	* @param[out]	intm LDO dropout1 detector rising interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetDropout1RisingInterruptMask(intm_t *intm);

    /**
	* @brief		Set Thermal Alarm 2 Rising Interrupt Mask.
	*
	* @param[in]	intm Thermal alarm 2 rising interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetThermalAlarm2RisingInterruptMask(intm_t intm);

    /**
	* @brief		Get Thermal Alarm 2 Rising Interrupt Mask.
	*
	* @param[out]	intm Thermal alarm 2 rising interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetThermalAlarm2RisingInterruptMask(intm_t *intm);

    /**
	* @brief		Set Thermal Alarm 1 Rising Interrupt Mask.
	*
	* @param[in]	intm Thermal alarm 1 rising interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetThermalAlarm1RisingInterruptMask(intm_t intm);

    /**
	* @brief		Get Thermal Alarm 1 Rising Interrupt Mask.
	*
	* @param[out]	intm Thermal alarm 1 rising interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetThermalAlarm1RisingInterruptMask(intm_t *intm);

    /**
	* @brief		Set nEN Rising Interrupt Mask.
	*
	* @param[in]	intm nEN rising interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetnENRisingInterruptMask(intm_t intm);

    /**
	* @brief		Get nEN Rising Interrupt Mask.
	*
	* @param[out]	intm nEN rising interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetnENRisingInterruptMask(intm_t *intm);

    /**
	* @brief		Set nEN Falling Interrupt Mask.
	*
	* @param[in]	intm nEN falling interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetnENFallingInterruptMask(intm_t intm);

    /**
	* @brief		Get nEN Falling Interrupt Mask.
	*
	* @param[out]	intm nEN falling interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetnENFallingInterruptMask(intm_t *intm);

    /**
	* @brief		Set GPI0 Rising Interrupt Mask.
	*
	* @param[in]	intm GPI rising interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetGPI0RisingInterruptMask(intm_t intm);

    /**
	* @brief		Get GPI0 Rising Interrupt Mask.
	*
	* @param[out]	intm GPI rising interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetGPI0RisingInterruptMask(intm_t *intm);

    /**
	* @brief		Set GPI0 Falling Interrupt Mask.
	*
	* @param[in]	intm GPI falling interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetGPI0FallingInterruptMask(intm_t intm);

    /**
	* @brief		Get GPI0 Falling Interrupt Mask.
	*
	* @param[out]	intm GPI falling interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetGPI0FallingInterruptMask(intm_t *intm);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_GLBL (0x10)
	*  - Bit Fields    : [7]
	*  - Default       : 0x0
	*  - Description   : nEN Internal Pullup Resistor.
	*/
    typedef enum {
        PU_DIS_STRONG_NEN_PULLUP_200K,
        PU_DIS_WEAK_NEN_PULLUP_10M
    }pu_dis_t;

    /**
  	* @brief		Set nEN Internal Pullup Resistor.
  	*
  	* @param[in]	pu_dis nEN internal pullup resistor bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetnENInternalPullupResistor(pu_dis_t pu_dis);

    /**
  	* @brief		Get nEN Internal Pullup Resistor.
  	*
  	* @param[out]	pu_dis nEN internal pullup resistor bit to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int GetnENInternalPullupResistor(pu_dis_t *pu_dis);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_GLBL (0x10)
	*  - Bit Fields    : [6]
	*  - Default       : 0x0
	*  - Description   : Manual Reset Time (tMRST).
	*/
    typedef enum {
        T_MRST_TIME_8S,
        T_MRST_TIME_16S
    }t_mrst_t;

    /**
  	* @brief		Set the Manual Reset Time (tMRST).
  	*
  	* @param[in]	t_mrst Manual reset time bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetManualResetTime(t_mrst_t t_mrst);

    /**
  	* @brief		Get the Manual Reset Time (tMRST).
  	*
  	* @param[out]	t_mrst Manual reset time bit to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int GetManualResetTime(t_mrst_t *t_mrst);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_GLBL (0x10)
	*  - Bit Fields    : [5]
	*  - Default       : 0x0
	*  - Description   : Main Bias Low-Power Mode Software Request.
	*/
    typedef enum {
        SBIA_LPM_NORMAL_POWER_MODE,
        SBIA_LPM_LOWER_POWER_MODE
    }sbia_lpm_t;

    /**
  	* @brief		Set Main Bias Low-Power Mode Software Request.
  	*
  	* @param[in]	sbia_lpm main bias low-power mode software request bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetMainBiasLowerPowerModeReq(sbia_lpm_t sbia_lpm);

    /**
  	* @brief		Get Main Bias Low-Power Mode Software Request.
  	*
  	* @param[out]	sbia_lpm Main bias low-power mode software request bit to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int GetMainBiasLowerPowerModeReq(sbia_lpm_t *sbia_lpm);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_GLBL (0x10)
	*  - Bit Fields    : [4]
	*  - Default       : 0x0
	*  - Description   : Main Bias Enable Software Request.
	*/
    typedef enum {
        SBIA_EN_ON_OFF_CONTROLLER,
        SBIA_EN_ALWAYS_ON
    }sbia_en_t;

    /**
  	* @brief		Set Main Bias Enable Software Request.
  	*
  	* @param[in]	sbia_en Main bias enable software request bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetMainBiasEnableReq(sbia_en_t sbia_en);

    /**
  	* @brief		Get Main Bias Enable Software Request.
  	*
  	* @param[out]	sbia_en Main bias enable software request bit to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int GetMainBiasEnableReq(sbia_en_t *sbia_en);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_GLBL (0x10)
	*  - Bit Fields    : [3]
	*  - Default       : 0x0
	*  - Description   : nEN Input (ON-KEY) Default Configuration Mode.
	*/
    typedef enum {
        NEN_MODE_PUSH_BUTTON_MODE,
        NEN_MODE_SLIDE_SWITCH_MODE
    }nen_mode_t;

    /**
  	* @brief		Set nEN Input (ON-KEY) Default Configuration Mode.
  	*
  	* @param[in]	nen_mode nEN input default configuration mode bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetnEnInputMode(nen_mode_t nen_mode);

    /**
  	* @brief		Get nEN Input (ON-KEY) Default Configuration Mode.
  	*
  	* @param[out]	nen_mode nEN input default configuration mode bit to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int GetnEnInputMode(nen_mode_t *nen_mode);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_GLBL (0x10)
	*  - Bit Fields    : [2]
	*  - Default       : 0x0
	*  - Description   : Debounce Timer Enable for the nEN Pin.
	*/
    typedef enum {
        DBEN_NEN_DEBOUNCE_500US,
        DBEN_NEN_DEBOUNCE_30MS
    }dben_nen_t;

    /**
  	* @brief		Set Debounce Timer Enable for the nEN Pin.
  	*
  	* @param[in]	dben_nen Debounce timer enable for the nEN pin bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetDebounceTimerEnable(dben_nen_t dben_nen);

    /**
  	* @brief		Get Debounce Timer Enable for the nEN Pin.
  	*
  	* @param[out]	dben_nen Debounce timer enable for the nEN pin bit to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int GetDebounceTimerEnable(dben_nen_t *dben_nen);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_GLBL (0x10)
	*  - Bit Fields    : [1:0]
	*  - Default       : 0x0
	*  - Description   : Software Reset Functions.
	*/
    typedef enum {
        SFT_CTRL_NO_ACTION,
        SFT_CTRL_SOFTWARE_COLD_RESET,
        SFT_CTRL_SOFTWARE_OFF,
        SFT_CTRL_FACTORY_SHIP_MODE_ENTER
    }sft_ctrl_t;

    /**
  	* @brief		Set Software Reset Functions.
  	*
  	* @param[in]	sft_ctrl Software reset functions field to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetSoftwareResetFunctions(sft_ctrl_t sft_ctrl);

    /**
  	* @brief		Get Software Reset Functions.
  	*
  	* @param[out]	sft_ctrl Software reset functions field to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int GetSoftwareResetFunctions(sft_ctrl_t *sft_ctrl);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_GPIO0 (0x11), CNFG_GPIO1 (0x12) and CNFG_GPIO2 (0x13)
	*  - Bit Fields    : [5]
	*  - Default       : 0x0
	*  - Description   : Alternate Mode Enable for GPIO 0, 1 or 2.
	*/
    typedef enum {
        ALT_GPIO_STANDARD_GPI_OR_GPO,
        ALT_GPIO_FLEX_POWER_OR_SSB2_ENABLE_OR_BIAS_LPM
    }alt_gpio_t;

    /**
  	* @brief		Set Alternate Mode Enable for GPIO.
  	*
  	* @param[in]	channel Channel number: 0, 1 or 2
  	* @param[in]	alt_gpio Alternate mode enable for GPIO bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetAlternateModeEnable(uint8_t channel, alt_gpio_t alt_gpio);

    /**
  	* @brief		Get Alternate Mode Enable for GPIO.
  	*
  	* @param[in]	channel Channel number: 0, 1 or 2
  	* @param[out]	alt_gpio Alternate mode enable for GPIO bit to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int GetAlternateModeEnable(uint8_t channel, alt_gpio_t *alt_gpio);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_GPIO0 (0x11), CNFG_GPIO1 (0x12) and CNFG_GPIO2 (0x13)
	*  - Bit Fields    : [4]
	*  - Default       : 0x0
	*  - Description   : General Purpose Input Debounce Timer Enable for GPIO 0, 1 or 2.
	*/
    typedef enum {
        DBEN_GPI_NO_DEBOUNCE,
        DBEN_GPI_DEBOUNCE_30MS
    }dben_gpi_t;

    /**
  	* @brief		Set General Purpose Input Debounce Timer Enable.
  	*
  	* @param[in]	channel Channel number: 0, 1 or 2
  	* @param[in]	dben_gpi General purpose input debounce timer enable bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetGPIDebounceTimerEnable(uint8_t channel, dben_gpi_t dben_gpi);

    /**
  	* @brief		Get General Purpose Input Debounce Timer Enable.
  	*
  	* @param[in]	channel Channel number: 0, 1 or 2
  	* @param[out]	dben_gpi General purpose input debounce timer enable bit to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int GetGPIDebounceTimerEnable(uint8_t channel, dben_gpi_t *dben_gpi);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_GPIO0 (0x11), CNFG_GPIO1 (0x12) and CNFG_GPIO2 (0x13)
	*  - Bit Fields    : [3]
	*  - Default       : 0x0
	*  - Description   : General Purpose Output Data Output for GPIO 0, 1 or 2.
	*/
    typedef enum {
        GPO_DO_LOGIC_LOW,
        GPO_DO_LOGIC_HIGH
    }gpo_do_t;

    /**
  	* @brief		Set General Purpose Output Data Output.
  	*
  	* @param[in]	channel Channel number: 0, 1 or 2
  	* @param[in]	gpo_do General purpose output data output bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetGPODataOutput(uint8_t channel, gpo_do_t gpo_do);

    /**
  	* @brief		Get General Purpose Output Data Output.
  	*
  	* @param[in]	channel Channel number: 0, 1 or 2
  	* @param[out]	gpo_do General purpose output data output bit to be read.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int GetGPODataOutput(uint8_t channel, gpo_do_t *gpo_do);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_GPIO0 (0x11), CNFG_GPIO1 (0x12) and CNFG_GPIO2 (0x13)
	*  - Bit Fields    : [2]
	*  - Default       : 0x0
	*  - Description   : General Purpose Output Driver Type for GPIO 0, 1 or 2.
	*/
    typedef enum {
        GPO_DRV_OPEN_DRAIN,
        GPO_DRV_PUSH_PULL
    }gpo_drv_t;

    /**
  	* @brief		Set General Purpose Output Driver Type.
  	*
  	* @param[in]	channel Channel number: 0, 1 or 2
  	* @param[in]	gpo_drv General purpose output driver type bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetGPODriveType(uint8_t channel, gpo_drv_t gpo_drv);

    /**
	* @brief		Get General Purpose Output Driver Type.
	*
	* @param[in]	channel Channel number: 0, 1 or 2
	* @param[out]	gpo_drv General purpose output driver type bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetGPODriveType(uint8_t channel, gpo_drv_t *gpo_drv);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_GPIO0 (0x11), CNFG_GPIO1 (0x12) and CNFG_GPIO2 (0x13)
	*  - Bit Fields    : [1]
	*  - Default       : 0x0
	*  - Description   : GPIO Digital Input Value for GPIO 0, 1 or 2.
	*/
    typedef enum {
        GPO_DI_INPUT_LOGIC_LOW,
        GPO_DI_INPUT_LOGIC_HIGH
    }gpo_di_t;

    /**
  	* @brief		Set GPIO Digital Input Value.
  	*
  	* @param[in]	channel Channel number: 0, 1 or 2
  	* @param[in]	gpo_di GPIO digital input value bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetGPIOInputValue(uint8_t channel, gpo_di_t gpo_di);

    /**
	* @brief		Get GPIO Digital Input Value.
	*
	* @param[in]	channel Channel number: 0, 1 or 2
	* @param[out]	gpo_di GPIO digital input value bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetGPIOInputValue(uint8_t channel, gpo_di_t *gpo_di);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_GPIO0 (0x11), CNFG_GPIO1 (0x12) and CNFG_GPIO2 (0x13)
	*  - Bit Fields    : [0]
	*  - Default       : 0x0
	*  - Description   : GPIO Direction for GPIO 0, 1 or 2.
	*/
    typedef enum {
        GPO_DIR_GPO,
        GPO_DIR_GPI
    }gpo_dir_t;

    /**
  	* @brief		Set GPIO Direction.
  	*
  	* @param[in]	channel Channel number: 0, 1 or 2
  	* @param[in]	gpo_dir GPIO direction bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetGPIODirection(uint8_t channel, gpo_dir_t gpo_dir);

    /**
	* @brief		Get GPIO Direction.
	*
	* @param[in]	channel Channel number
	* @param[out]	gpo_dir GPIO direction bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetGPIODirection(uint8_t channel, gpo_dir_t *gpo_dir);

    /**
     * @brief   Read the CID
     * @details Read and return Chip Identification Code register value
     * @returns CID if no errors, error code on failure.
     */
    int GetCID(void);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_WDT (0x17)
	*  - Bit Fields    : [5:4]
	*  - Default       : 0x0
	*  - Description   : Watchdog Timer Period.
	*/
    typedef enum {
        WDT_PER_SECOND_16S,
        WDT_PER_SECOND_32S,
        WDT_PER_SECOND_64S,
        WDT_PER_SECOND_128S
    }wdt_per_t;

    /**
  	* @brief		Set Watchdog Timer Period.
  	*
  	* @param[in]	wdt_per Watchdog timer period bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetWatchdogTimerPeriod(wdt_per_t wdt_per);

    /**
	* @brief		Get Watchdog Timer Period.
	*
	* @param[out]	wdt_per Watchdog timer period bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetWatchdogTimerPeriod(wdt_per_t *wdt_per);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_WDT (0x17)
	*  - Bit Fields    : [3]
	*  - Default       : 0x0
	*  - Description   : Watchdog Timer Expired Action.
	*/
    typedef enum {
        WDT_MODE_POWER_OFF,
        WDT_MODE_POWER_RESET
    }wdt_mode_t;

    /**
  	* @brief		Set Watchdog Timer Expired Action.
  	*
  	* @param[in]	wdt_mode Watchdog timer expired action bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetWatchdogTimerExpiredAction(wdt_mode_t wdt_mode);

    /**
	* @brief		Get Watchdog Timer Expired Action.
	*
	* @param[out]	wdt_mode Watchdog timer expired action bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetWatchdogTimerExpiredAction(wdt_mode_t *wdt_mode);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_WDT (0x17)
	*  - Bit Fields    : [2]
	*  - Default       : 0x0
	*  - Description   : Watchdog Timer Clear Control.
	*/
    typedef enum {
        WDT_CLR_PERIOD_NOT_RESET,
        WDT_CLR_PERIOD_RESET
    }wdt_clr_t;

    /**
  	* @brief		Set Watchdog Timer Clear Control.
  	*
  	* @param[in]	wdt_clr Watchdog timer clear control bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetWatchdogTimerClearControl(wdt_clr_t wdt_clr);

    /**
	* @brief		Get Watchdog Timer Clear Control.
	*
	* @param[out]	wdt_clr Watchdog timer clear control bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetWatchdogTimerClearControl(wdt_clr_t *wdt_clr);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_WDT (0x17)
	*  - Bit Fields    : [1]
	*  - Default       : 0x0
	*  - Description   : Watchdog Timer Enable.
	*/
    typedef enum {
        WDT_EN_DISABLED,
        WDT_EN_ENABLED
    }wdt_en_t;

    /**
  	* @brief		Set Watchdog Timer Enable.
  	*
  	* @param[in]	wdt_en Watchdog timer enable bit to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetWatchdogTimerEnable(wdt_en_t wdt_en);

    /**
	* @brief		Get Watchdog Timer Enable.
	*
	* @param[out]	wdt_en Watchdog timer enable bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetWatchdogTimerEnable(wdt_en_t *wdt_en);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_WDT (0x17)
	*  - Bit Fields    : [0]
	*  - Default       : 0x0
	*  - Description   : Factory-Set Safety Bit for the Watchdog Timer.
	*/
    typedef enum {
        WDT_LOCK_UNLOCKED,
        WDT_LOCK_LOCKED
    }wdt_lock_t;

    /**
  	* @brief		Set Factory-Set Safety Bit for the Watchdog Timer.
  	*
  	* @param[in]	wdt_lock Factory-set safety bit for the watchdog timer to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
    int SetFactorySetSafetyBit(wdt_lock_t wdt_lock);

    /**
	* @brief		Get Factory-Set Safety Bit for the Watchdog Timer.
	*
	* @param[out]	wdt_lock Factory-set safety bit for the watchdog timer to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetFactorySetSafetyBit(wdt_lock_t *wdt_lock);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_CHG_A (0x02)
	*  - Bit Fields    : [6:3]
	*  - Default       : 0x0
	*  - Description   : Status bits if it is engaged or not.
	*/
    typedef enum {
        STAT_EN_NOT_ENGAGED,
        STAT_EN_ENGAGED
    }stat_t;

    /**
	* @brief		Get Minimum Input Voltage Regulation Loop Status.
	*
	* @param[out]	stat Status bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetMinimumVCHGINVoltageLoopStatus(stat_t *stat);

    /**
	* @brief		Get Input Current Limit Loop Status.
	*
	* @param[out]	stat Status bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetCHGINCurrentLimitLoopStatus(stat_t *stat);

    /**
	* @brief		Get Minimum System Voltage Regulation Loop Status.
	*
	* @param[out]	stat Status bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetMinimumSYSVoltageLoopStatus(stat_t *stat);

    /**
	* @brief		Get Maximum Junction Temperature Regulation Loop Status.
	*
	* @param[out]	stat Status bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetMaximumJunctionTempLoopStatus(stat_t *stat);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_CHG_A (0x02)
	*  - Bit Fields    : [2:0]
	*  - Default       : 0x0
	*  - Description   : Battery Temperature Details.
	*/
    typedef enum {
        THM_DTLS_THERMISTOR_DISABLED,
        THM_DTLS_BATTERY_COLD,
        THM_DTLS_BATTERY_COOL,
        THM_DTLS_BATTERY_WARM,
        THM_DTLS_BATTERY_HOT,
        THM_DTLS_BATTERY_NORMAL,
        THM_DTLS_RESERVED_0x06,
        THM_DTLS_RESERVED_0x07
    }thm_dtls_t;

    /**
	* @brief		Get Battery Temperature Details.
	* 				Valid only when CHGIN_DTLS[1:0] = 0b11.
	*
	* @param[out]	thm_dtls Battery temperature details field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetBatteryTemperatureDetails(thm_dtls_t *thm_dtls);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_CHG_B (0x03)
	*  - Bit Fields    : [7:4]
	*  - Default       : 0x0
	*  - Description   : Charger Details.
	*/
    typedef enum {
        CHG_DTLS_OFF,
        CHG_DTLS_PREQUALIFICATION_MODE,
        CHG_DTLS_FAST_CHARGE_CC,
        CHG_DTLS_JEITA_FAST_CHARGE_CC,
        CHG_DTLS_FAST_CHARGE_CV,
        CHG_DTLS_JEITA_FAST_CHARGE_CV,
        CHG_DTLS_TOP_OFF_MODE,
        CHG_DTLS_JEITA_MODIFIED_TOP_OFF_MODE,
        CHG_DTLS_DONE,
        CHG_DTLS_JEITA_MODIFIED_DONE,
        CHG_DTLS_PREQUALIFICATION_TIMER_FAULT,
        CHG_DTLS_FAST_CHARGE_TIMER_FAULT,
        CHG_DTLS_BATTERY_TEMPERATURE_FAULT,
        CHG_DTLS_RESERVED_0x0D,
        CHG_DTLS_RESERVED_0x0E,
        CHG_DTLS_RESERVED_0x0F
    }chg_dtls_t;

    /**
	* @brief		Get Charger Details.
	*
	* @param[out]	chg_dtls Charger details field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetChargerDetails(chg_dtls_t *chg_dtls);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_CHG_B (0x03)
	*  - Bit Fields    : [3:2]
	*  - Default       : 0x0
	*  - Description   : CHGIN Status Detail.
	*/
    typedef enum {
        CHGIN_DTLS_VOLTAGE_BELOW_UVLO,
        CHGIN_DTLS_VOLTAGE_ABOVE_OVP,
        CHGIN_DTLS_DEBOUNCED,
        CHGIN_DTLS_OKAY
    }chgin_dtls_t;

    /**
	* @brief		Get CHGIN Status Detail.
	*
	* @param[out]	chgin_dtls CHGIN status detail field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetCHGINStatusDetails(chgin_dtls_t *chgin_dtls);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_CHG_B (0x03)
	*  - Bit Fields    : [1]
	*  - Default       : 0x0
	*  - Description   : Quick Charger Status.
	*/
    typedef enum {
        CHG_CHARGING_NOT_HAPPENING,
        CHG_CHARGING_HAPPENING
    }chg_t;

    /**
	* @brief		Get Quick Charger Status.
	*
	* @param[out]	chg Quick charger status bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetQuickChargerStatus(chg_t *chg);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : STAT_CHG_B (0x03)
	*  - Bit Fields    : [0]
	*  - Default       : 0x0
	*  - Description   : Time Suspend Indicator.
	*/
    typedef enum {
        TIME_SUS_CHARGER_TIMER_NOT_SUSPENDED,
        TIME_SUS_CHARGER_TIMER_SUSPENDED
    }time_sus_t;
    
    /**
	* @brief		Get Time Suspend Indicator.
	*
	* @param[out]	time_sus Time suspend indicator bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetTimeSuspendedIndicator(time_sus_t *time_sus);

    /**
	* @brief		Set SYS_CNFG_I.
	* 				Setting this bit prevents the SYS_CNFG_I bit from causing hardware IRQs
	*
	* @param[in]	intm Interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetSYSCONFIGMBit(intm_t intm);

    /**
	* @brief		Get SYS_CNFG_I.
	* 				Setting this bit prevents the SYS_CNFG_I bit from causing hardware IRQs
	*
	* @param[out]	intm Interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetSYSCONFIGMBit(intm_t *intm);

    /**
	* @brief		Set SYS_CTRL_I.
	* 				Setting this bit prevents the SYS_CTRL_I bit from causing hardware IRQs
	*
	* @param[in]	intm Interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetSYSCTRLMBit(intm_t intm);

    /**
	* @brief		Get SYS_CTRL_I.
	* 				Setting this bit prevents the SYS_CTRL_I bit from causing hardware IRQs
	*
	* @param[out]	intm Interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetSYSCTRLMBit(intm_t *intm);

    /**
	* @brief		Set CHGIN_CTRL_I.
	* 				Setting this bit prevents the CHGIN_CTRL_I bit from causing hardware IRQs
	*
	* @param[in]	intm Interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetCHGINCTRLMBit(intm_t intm);

    /**
	* @brief		Get CHGIN_CTRL_I.
	* 				Setting this bit prevents the CHGIN_CTRL_I bit from causing hardware IRQs
	*
	* @param[out]	intm Interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetCHGINCTRLMBit(intm_t *intm);

    /**
	* @brief		Set TJREG_I.
	* 				Setting this bit prevents the TJREG_I bit from causing hardware IRQs
	*
	* @param[in]	intm Interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetTJREGMBit(intm_t intm);

    /**
	* @brief		Get TJREG_I.
	* 				Setting this bit prevents the TJREG_I bit from causing hardware IRQs
	*
	* @param[out]	intm Interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetTJREGMBit(intm_t *intm);

    /**
	* @brief		Set CHGIN_I.
	* 				Setting this bit prevents the CHGIN_I bit from causing hardware IRQs
	*
	* @param[in]	intm Interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetCHGINMBit(intm_t intm);

    /**
	* @brief		Get CHGIN_I.
	* 				Setting this bit prevents the CHGIN_I bit from causing hardware IRQs
	*
	* @param[out]	intm Interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetCHGINMBit(intm_t *intm);

    /**
	* @brief		Set CHG_I.
	* 				Setting this bit prevents the CHG_I bit from causing hardware IRQs
	*
	* @param[in]	intm Interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetCHGMBit(intm_t intm);

    /**
	* @brief		Get CHG_I.
	* 				Setting this bit prevents the CHG_I bit from causing hardware IRQs
	*
	* @param[out]	intm Interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetCHGMBit(intm_t *intm);

    /**
	* @brief		Set THM_I.
	* Setting this bit prevents the THM_I bit from causing hardware IRQs
	*
	* @param[in]	intm Interrupt mask bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetTHMMBit(intm_t intm);

    /**
	* @brief		Get THM_I.
	* Setting this bit prevents the THM_I bit from causing hardware IRQs
	*
	* @param[out]	intm Interrupt mask bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetTHMMBit(intm_t *intm);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_A (0x20)
	*  - Bit Fields    : [7:6]
	*  - Default       : 0x0
	*  - Description   : VHOT JEITA Temperature Threshold.
	*/
    typedef enum {
        THM_HOT_VOLT_0_411V,
        THM_HOT_VOLT_0_367V,
        THM_HOT_VOLT_0_327V,
        THM_HOT_VOLT_0_291V
    }thm_hot_t;

    /**
	* @brief		Set the VHOT JEITA Temperature Threshold.
	*
	* @param[in]	thm_hot The VHOT JEITA temperature threshold field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetHOTJEITATemperature(thm_hot_t thm_hot);

    /**
	* @brief		Get the VHOT JEITA Temperature Threshold.
	*
	* @param[out]	thm_hot The VHOT JEITA temperature threshold field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetHOTJEITATemperature(thm_hot_t *thm_hot);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_A (0x20)
	*  - Bit Fields    : [5:4]
	*  - Default       : 0x0
	*  - Description   : VWARM JEITA Temperature Threshold.
	*/
    typedef enum {
        THM_WARM_VOLT_0_511V,
        THM_WARM_VOLT_0_459V,
        THM_WARM_VOLT_0_411V,
        THM_WARM_VOLT_0_367V
    }thm_warm_t;

    /**
	* @brief		Set the VWARM JEITA Temperature Threshold.
	*
	* @param[in]	thm_warm The VWARM JEITA temperature threshold field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetWARMJEITATemperature(thm_warm_t thm_warm);

    /**
	* @brief		Get the VWARM JEITA Temperature Threshold.
	*
	* @param[out]	thm_warm The VWARM JEITA temperature threshold field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetWARMJEITATemperature(thm_warm_t *thm_warm);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_A (0x20)
	*  - Bit Fields    : [3:2]
	*  - Default       : 0x0
	*  - Description   : VCOOL JEITA Temperature Threshold.
	*/
    typedef enum {
        THM_COOL_VOLT_0_923V,
        THM_COOL_VOLT_0_867V,
        THM_COOL_VOLT_0_807V,
        THM_COOL_VOLT_0_747V
    }thm_cool_t;

    /**
	* @brief		Set the VCOOL JEITA Temperature Threshold.
	*
	* @param[in]	thm_cool The VCOOL JEITA temperature threshold field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetCOOLJEITATemperature(thm_cool_t thm_cool);

    /**
	* @brief		Get the VCOOL JEITA Temperature Threshold.
	*
	* @param[out]	thm_cool The VCOOL JEITA temperature threshold field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetCOOLJEITATemperature(thm_cool_t *thm_cool);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_A (0x20)
	*  - Bit Fields    : [1:0]
	*  - Default       : 0x0
	*  - Description   : VCOLD JEITA Temperature Threshold.
	*/
    typedef enum {
        THM_COLD_VOLT_1_024V,
        THM_COLD_VOLT_0_976V,
        THM_COLD_VOLT_0_923V,
        THM_COLD_VOLT_0_867V
    }thm_cold_t;

    /**
	* @brief		Set the VCOLD JEITA Temperature Threshold.
	*
	* @param[in]	thm_cold The VCOLD JEITA temperature threshold field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetCOLDJEITATemperature(thm_cold_t thm_cold);

    /**
	* @brief		Get the VCOLD JEITA Temperature Threshold.
	*
	* @param[out]	thm_cold The VCOLD JEITA temperature threshold field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetCOLDJEITATemperature(thm_cold_t *thm_cold);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_B (0x21)
	*  - Bit Fields    : [7:5]
	*  - Default       : 0x0
	*  - Description   : Minimum CHGIN Regulation Voltage.
	*/
    typedef enum {
        VCHGIN_MIN_VOLT_4_0V,
        VCHGIN_MIN_VOLT_4_1V,
        VCHGIN_MIN_VOLT_4_2V,
        VCHGIN_MIN_VOLT_4_3V,
        VCHGIN_MIN_VOLT_4_4V,
        VCHGIN_MIN_VOLT_4_5V,
        VCHGIN_MIN_VOLT_4_6V,
        VCHGIN_MIN_VOLT_4_7V
    }vchgin_min_t;

    /**
	* @brief		Set Minimum CHGIN Regulation Voltage.
	*
	* @param[in]	vchgin_min Minimum CHGIN regulation voltage field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetMinimumCHGINVoltage(vchgin_min_t vchgin_min);

    /**
	* @brief		Get Minimum CHGIN Regulation Voltage.
	*
	* @param[out]	vchgin_min Minimum CHGIN regulation voltage field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetMinimumCHGINVoltage(vchgin_min_t *vchgin_min);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_B (0x21)
	*  - Bit Fields    : [4:2]
	*  - Default       : 0x0
	*  - Description   : CHGIN Input Current Limit.
	*/
    typedef enum {
        ICHGIN_LIM_AMP_95MA,
        ICHGIN_LIM_AMP_190MA,
        ICHGIN_LIM_AMP_285MA,
        ICHGIN_LIM_AMP_385MA,
        ICHGIN_LIM_AMP_475MA,
        ICHGIN_LIM_AMP_RESERVED_0x05,
        ICHGIN_LIM_AMP_RESERVED_0x06,
        ICHGIN_LIM_AMP_RESERVED_0x07
    }ichgin_lim_t;

    /**
	* @brief		Set CHGIN Input Current Limit.
	*
	* @param[in]	ichgin_lim CHGIN input current limit field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetCHGINInputCurrentLimit(ichgin_lim_t ichgin_lim);

    /**
	* @brief		Get CHGIN Input Current Limit.
	*
	* @param[out]	ichgin_lim CHGIN input current limit field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetCHGINInputCurrentLimit(ichgin_lim_t *ichgin_lim);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_B (0x21)
	*  - Bit Fields    : [1]
	*  - Default       : 0x0
	*  - Description   : Prequalification Charge Current as a percentage of IFAST-CHG.
	*/
    typedef enum {
        I_PQ_PERCENT_10,
        I_PQ_PERCENT_20
    }i_pq_t;

    /**
	* @brief		Set the Prequalification Charge Current as a percentage of IFAST-CHG.
	*
	* @param[in]	i_pq The prequalification charge current bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetPrequalificationChargeCurrent(i_pq_t i_pq);

    /**
	* @brief		Get the Prequalification Charge Current as a percentage of IFAST-CHG.
	*
	* @param[out]	i_pq the prequalification charge current bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetPrequalificationChargeCurrent(i_pq_t *i_pq);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_B (0x21)
	*  - Bit Fields    : [0]
	*  - Default       : 0x0
	*  - Description   : Charger Enable.
	*/
    typedef enum {
        CHG_EN_BATTERY_CHARGER_DISABLED,
        CHG_EN_BATTERY_CHARGER_ENABLED
    }chg_en_t;

    /**
	* @brief		Set Charger Enable.
	*
	* @param[in]	chg_en Charger enable bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetBatteryChargerEnable(chg_en_t chg_en);

    /**
	* @brief		Get Charger Enable.
	*
	* @param[out]	chg_en Charger enable bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetBatteryChargerEnable(chg_en_t *chg_en);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_C (0x22)
	*  - Bit Fields    : [7:5]
	*  - Default       : 0x0
	*  - Description   : Battery Prequalification Voltage Threshold (VPQ).
	*/
    typedef enum {
        CHG_PQ_VOLT_2_3V,
        CHG_PQ_VOLT_2_4V,
        CHG_PQ_VOLT_2_5V,
        CHG_PQ_VOLT_2_6V,
        CHG_PQ_VOLT_2_7V,
        CHG_PQ_VOLT_2_8V,
        CHG_PQ_VOLT_2_9V,
        CHG_PQ_VOLT_3_0V
    }chg_pq_t;

    /**
	* @brief		Set Battery Prequalification Voltage Threshold (VPQ).
	*
	* @param[in]	chg_pq Battery prequalification voltage threshold field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetBatteryPQVoltageThreshold(chg_pq_t chg_pq);

    /**
	* @brief		Get Battery Prequalification Voltage Threshold (VPQ).
	*
	* @param[out]	chg_pq Battery prequalification voltage threshold field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetBatteryPQVoltageThreshold(chg_pq_t *chg_pq);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_C (0x22)
	*  - Bit Fields    : [4:3]
	*  - Default       : 0x0
	*  - Description   : Charger Termination Current (ITERM).
	*/
    typedef enum {
        I_TERM_PERCENT_5,
        I_TERM_PERCENT_7_5,
        I_TERM_PERCENT_10,
        I_TERM_PERCENT_15
    }i_term_t;

    /**
	* @brief		Set Charger Termination Current (ITERM).
	* 				I_TERM[1:0] sets the charger termination current
	* 				as a percentage of the fast charge current IFAST-CHG.
	*
	* @param[in]	i_term Charger termination current field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetChargerTerminationCurrent(i_term_t i_term);

    /**
	* @brief		Get Charger Termination Current (ITERM).
	* 				I_TERM[1:0] sets the charger termination current
	* 				as a percentage of the fast charge current IFAST-CHG.
	*
	* @param[out]	i_term Charger termination current field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetChargerTerminationCurrent(i_term_t *i_term);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_C (0x22)
	*  - Bit Fields    : [2:0]
	*  - Default       : 0x0
	*  - Description   : Top-off Timer Value.
	*/
    typedef enum {
        T_TOPOFF_MINUTE_0M,
        T_TOPOFF_MINUTE_5M,
        T_TOPOFF_MINUTE_10M,
        T_TOPOFF_MINUTE_15M,
        T_TOPOFF_MINUTE_20M,
        T_TOPOFF_MINUTE_25M,
        T_TOPOFF_MINUTE_30M,
        T_TOPOFF_MINUTE_35M
    }t_topoff_t;

    /**
	* @brief		Set Top-off Timer Value.
	*
	* @param[in]	t_topoff Top-off timer value field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetTopOffTimerValue(t_topoff_t t_topoff);

    /**
	* @brief		Get Top-off Timer Value.
	*
	* @param[out]	t_topoff Top-off timer value field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetTopOffTimerValue(t_topoff_t *t_topoff);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_D (0x23)
	*  - Bit Fields    : [7:5]
	*  - Default       : 0x0
	*  - Description   : Die Junction Temperature Regulation Point, TJ-REG.
	*/
    typedef enum {
        TJ_REG_DEGREE_60C,
        TJ_REG_DEGREE_70C,
        TJ_REG_DEGREE_80C,
        TJ_REG_DEGREE_90C,
        TJ_REG_DEGREE_100C,
        TJ_REG_DEGREE_100C_0x05,
        TJ_REG_DEGREE_100C_0x06,
        TJ_REG_DEGREE_100C_0x07
    }tj_reg_t;

    /**
	* @brief		Set the Die Junction Temperature Regulation Point, TJ-REG.
	*
	* @param[in]	tj_reg The die junction temperature regulation point field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetDieJunctionTemperature(tj_reg_t tj_reg);

    /**
	* @brief		Get the Die Junction Temperature Regulation Point, TJ-REG.
	*
	* @param[out]	tj_reg The die junction temperature regulation point field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetDieJunctionTemperature(tj_reg_t *tj_reg);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_D (0x23)
	*  - Bit Fields    : [4:0]
	*  - Default       : 0x0
	*  - Description   : System Voltage Regulation (VSYS-REG).
	*/
    typedef enum {
        VSYS_REG_VOLT_4_100V,
        VSYS_REG_VOLT_4_125V,
        VSYS_REG_VOLT_4_150V,
        VSYS_REG_VOLT_4_175V,
        VSYS_REG_VOLT_4_200V,
        VSYS_REG_VOLT_4_225V,
        VSYS_REG_VOLT_4_250V,
        VSYS_REG_VOLT_4_275V,
        VSYS_REG_VOLT_4_300V,
        VSYS_REG_VOLT_4_325V,
        VSYS_REG_VOLT_4_350V,
        VSYS_REG_VOLT_4_375V,
        VSYS_REG_VOLT_4_400V,
        VSYS_REG_VOLT_4_425V,
        VSYS_REG_VOLT_4_450V,
        VSYS_REG_VOLT_4_475V,
        VSYS_REG_VOLT_4_500V,
        VSYS_REG_VOLT_4_525V,
        VSYS_REG_VOLT_4_550V,
        VSYS_REG_VOLT_4_575V,
        VSYS_REG_VOLT_4_600V,
        VSYS_REG_VOLT_4_625V,
        VSYS_REG_VOLT_4_650V,
        VSYS_REG_VOLT_4_675V,
        VSYS_REG_VOLT_4_700V,
        VSYS_REG_VOLT_4_725V,
        VSYS_REG_VOLT_4_750V,
        VSYS_REG_VOLT_4_775V,
        VSYS_REG_VOLT_4_800V,
        VSYS_REG_VOLT_4_800V_0x1D,
        VSYS_REG_VOLT_4_800V_0x1E,
        VSYS_REG_VOLT_4_800V_0x1F
    }vsys_reg_t;

    /**
	* @brief		Set System Voltage Regulation (VSYS-REG).
	*
	* @param[in]	vsys_reg System voltage regulation field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetSystemVoltageRegulation(vsys_reg_t vsys_reg);

    /**
	* @brief		Get System Voltage Regulation (VSYS-REG).
	*
	* @param[out]	vsys_reg System voltage regulation field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetSystemVoltageRegulation(vsys_reg_t *vsys_reg);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_E (0x24), and CNFG_CHG_F (0x25)
	*  - Bit Fields    : [7:2]
	*  - Default       : 0x0
	*  - Description   : Fast-Charge Constant Current Value, IFAST-CHG or IFAST-CHG-JEITA.
	*/
    typedef enum {
        CHG_CC_AMP_7_5MA,
        CHG_CC_AMP_15_0MA,
        CHG_CC_AMP_22_5MA,
        CHG_CC_AMP_30_0MA,
        CHG_CC_AMP_37_5MA,
        CHG_CC_AMP_45_0MA,
        CHG_CC_AMP_52_5MA,
        CHG_CC_AMP_60_0MA,
        CHG_CC_AMP_67_5MA,
        CHG_CC_AMP_75_0MA,
        CHG_CC_AMP_82_5MA,
        CHG_CC_AMP_90_0MA,
        CHG_CC_AMP_97_5MA,
        CHG_CC_AMP_105_0MA,
        CHG_CC_AMP_112_5MA,
        CHG_CC_AMP_120_0MA,
        CHG_CC_AMP_127_5MA,
        CHG_CC_AMP_135_0MA,
        CHG_CC_AMP_142_5MA,
        CHG_CC_AMP_150_0MA,
        CHG_CC_AMP_157_5MA,
        CHG_CC_AMP_165_0MA,
        CHG_CC_AMP_172_5MA,
        CHG_CC_AMP_180_0MA,
        CHG_CC_AMP_187_5MA,
        CHG_CC_AMP_195_0MA,
        CHG_CC_AMP_202_5MA,
        CHG_CC_AMP_210_0MA,
        CHG_CC_AMP_217_5MA,
        CHG_CC_AMP_225_0MA,
        CHG_CC_AMP_232_5MA,
        CHG_CC_AMP_240_0MA,
        CHG_CC_AMP_247_5MA,
        CHG_CC_AMP_255_0MA,
        CHG_CC_AMP_262_5MA,
        CHG_CC_AMP_270_0MA,
        CHG_CC_AMP_277_5MA,
        CHG_CC_AMP_285_0MA,
        CHG_CC_AMP_292_5MA,
        CHG_CC_AMP_300_0MA,
        CHG_CC_AMP_300_0MA_0x28,
        CHG_CC_AMP_300_0MA_0x29,
        CHG_CC_AMP_300_0MA_0x2A,
        CHG_CC_AMP_300_0MA_0x2B,
        CHG_CC_AMP_300_0MA_0x2C,
        CHG_CC_AMP_300_0MA_0x2D,
        CHG_CC_AMP_300_0MA_0x2E,
        CHG_CC_AMP_300_0MA_0x2F,
        CHG_CC_AMP_300_0MA_0x30,
        CHG_CC_AMP_300_0MA_0x31,
        CHG_CC_AMP_300_0MA_0x32,
        CHG_CC_AMP_300_0MA_0x33,
        CHG_CC_AMP_300_0MA_0x34,
        CHG_CC_AMP_300_0MA_0x35,
        CHG_CC_AMP_300_0MA_0x36,
        CHG_CC_AMP_300_0MA_0x37,
        CHG_CC_AMP_300_0MA_0x38,
        CHG_CC_AMP_300_0MA_0x39,
        CHG_CC_AMP_300_0MA_0x3A,
        CHG_CC_AMP_300_0MA_0x3B,
        CHG_CC_AMP_300_0MA_0x3C,
        CHG_CC_AMP_300_0MA_0x3D,
        CHG_CC_AMP_300_0MA_0x3E,
        CHG_CC_AMP_300_0MA_0x3F
    }chg_cc_t;

    /**
	* @brief		Set the Fast-Charge Constant Current Value, IFAST-CHG.
	*
	* @param[in]	chg_cc the fast-charge constant current value field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetFastChargeCCValue(chg_cc_t chg_cc);

    /**
	* @brief		Get the Fast-Charge Constant Current Value, IFAST-CHG.
	*
	* @param[out]	chg_cc the fast-charge constant current value field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetFastChargeCCValue(chg_cc_t *chg_cc);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_E (0x24)
	*  - Bit Fields    : [1:0]
	*  - Default       : 0x0
	*  - Description   : Fast-charge Safety timer, tFC.
	*/
    typedef enum {
        T_FAST_CHG_TIMER_DISABLED,
        T_FAST_CHG_HOUR_3H,
        T_FAST_CHG_HOUR_5H,
        T_FAST_CHG_HOUR_7H
    }t_fast_chg_t;

    /**
	* @brief		Set the Fast-charge Safety timer, tFC.
	*
	* @param[in]	t_fast_chg Fast-charge safety timer field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetFastChargSafetyTimer(t_fast_chg_t t_fast_chg);

    /**
	* @brief		Get the Fast-charge Safety timer, tFC.
	*
	* @param[out]	t_fast_chg Fast-charge safety timer field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetFastChargSafetyTimer(t_fast_chg_t *t_fast_chg);

    /**
	* @brief		Set IFAST-CHG-JEITA
	* 				when the battery is either cool or warm as defined by the
	* 				VCOOL and VWARM temperature thresholds..
	*
	* @param[in]	chg_cc_jeita IFAST-CHG-JEITA field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetFastChargeCCJEITA(chg_cc_t chg_cc_jeita);

    /**
	* @brief		Get IFAST-CHG-JEITA
	* 				when the battery is either cool or warm as defined by the
	* 				VCOOL and VWARM temperature thresholds.
	*
	* @param[out]	chg_cc_jeita IFAST-CHG-JEITA field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetFastChargeCCJEITA(chg_cc_t *chg_cc_jeita);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_F (0x25)
	*  - Bit Fields    : [1]
	*  - Default       : 0x0
	*  - Description   : Thermistor Enable Bit.
	*/
    typedef enum {
        THM_EN_THERMISTOR_DISABLED,
        THM_EN_THERMISTOR_ENABLED
    }thm_en_t;

    /**
	* @brief		Set Thermistor Enable Bit.
	*
	* @param[in]	thm_en Thermistor enable bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetThermistorEnable(thm_en_t thm_en);

    /**
	* @brief		Get Thermistor enable bit.
	*
	* @param[out]	thm_en Thermistor enable bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetThermistorEnable(thm_en_t *thm_en);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_G (0x26) and CNFG_CHG_H (0x27)
	*  - Bit Fields    : [7:2]
	*  - Default       : 0x0
	*  - Description   : Fast-Charge Battery Regulation Voltage, VFAST-CHG or VFAST-CHG-JEITA.
	*/
    typedef enum {
        CHG_CV_VOLT_3_600V,
        CHG_CV_VOLT_3_625V,
        CHG_CV_VOLT_3_650V,
        CHG_CV_VOLT_3_675V,
        CHG_CV_VOLT_3_700V,
        CHG_CV_VOLT_3_725V,
        CHG_CV_VOLT_3_750V,
        CHG_CV_VOLT_3_775V,
        CHG_CV_VOLT_3_800V,
        CHG_CV_VOLT_3_825V,
        CHG_CV_VOLT_3_850V,
        CHG_CV_VOLT_3_875V,
        CHG_CV_VOLT_3_900V,
        CHG_CV_VOLT_3_925V,
        CHG_CV_VOLT_3_950V,
        CHG_CV_VOLT_3_975V,
        CHG_CV_VOLT_4_000V,
        CHG_CV_VOLT_4_025V,
        CHG_CV_VOLT_4_050V,
        CHG_CV_VOLT_4_075V,
        CHG_CV_VOLT_4_100V,
        CHG_CV_VOLT_4_125V,
        CHG_CV_VOLT_4_150V,
        CHG_CV_VOLT_4_175V,
        CHG_CV_VOLT_4_200V,
        CHG_CV_VOLT_4_225V,
        CHG_CV_VOLT_4_250V,
        CHG_CV_VOLT_4_275V,
        CHG_CV_VOLT_4_300V,
        CHG_CV_VOLT_4_325V,
        CHG_CV_VOLT_4_350V,
        CHG_CV_VOLT_4_375V,
        CHG_CV_VOLT_4_400V,
        CHG_CV_VOLT_4_425V,
        CHG_CV_VOLT_4_450V,
        CHG_CV_VOLT_4_475V,
        CHG_CV_VOLT_4_500V,
        CHG_CV_VOLT_4_525V,
        CHG_CV_VOLT_4_550V,
        CHG_CV_VOLT_4_575V,
        CHG_CV_VOLT_4_600V,
        CHG_CV_VOLT_4_600V_0x29,
        CHG_CV_VOLT_4_600V_0x2A,
        CHG_CV_VOLT_4_600V_0x2B,
        CHG_CV_VOLT_4_600V_0x2C,
        CHG_CV_VOLT_4_600V_0x2D,
        CHG_CV_VOLT_4_600V_0x2E,
        CHG_CV_VOLT_4_600V_0x2F,
        CHG_CV_VOLT_4_600V_0x30,
        CHG_CV_VOLT_4_600V_0x31,
        CHG_CV_VOLT_4_600V_0x32,
        CHG_CV_VOLT_4_600V_0x33,
        CHG_CV_VOLT_4_600V_0x34,
        CHG_CV_VOLT_4_600V_0x35,
        CHG_CV_VOLT_4_600V_0x36,
        CHG_CV_VOLT_4_600V_0x37,
        CHG_CV_VOLT_4_600V_0x38,
        CHG_CV_VOLT_4_600V_0x39,
        CHG_CV_VOLT_4_600V_0x3A,
        CHG_CV_VOLT_4_600V_0x3B,
        CHG_CV_VOLT_4_600V_0x3C,
        CHG_CV_VOLT_4_600V_0x3D,
        CHG_CV_VOLT_4_600V_0x3E,
        CHG_CV_VOLT_4_600V_0x3F
    }chg_cv_t;

    /**
	* @brief		Set Fast-Charge Battery Regulation Voltage, VFAST-CHG.
	*
	* @param[in]	chg_cv Fast-charge battery regulation voltage field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetFastChargeBatteryRegVolt(chg_cv_t chg_cv);

    /**
	* @brief		Get Fast-Charge Battery Regulation Voltage, VFAST-CHG.
	*
	* @param[out]	chg_cv Fast-charge battery regulation voltage field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetFastChargeBatteryRegVolt(chg_cv_t *chg_cv);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_G (0x26)
	*  - Bit Fields    : [1]
	*  - Default       : 0x0
	*  - Description   : USB suspend mode.
	*/
    typedef enum {
        USBS_CHGIN_NOT_SUSPENDED,
        USBS_CHGIN_SUSPENDED
    }usbs_t;

    /**
	* @brief		Set USBS. Setting this bit places CHGIN in USB suspend mode.
	*
	* @param[in]	usbs USB suspend mode bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetCHGINUSBSuspendMode(usbs_t usbs);

    /**
	* @brief		Get USBS. Setting this bit places CHGIN in USB suspend mode.
	*
	* @param[out]	usbs USB suspend mode bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetCHGINUSBSuspendMode(usbs_t *usbs);

    /**
	* @brief		Set the modified VFAST-CHG-JEITA for when the battery is either
	* 				cool or warm as defined by the VCOOL and VWARM temperature thresholds.
	*
	* @param[in]	chg_cv_jeita Modified VFAST-CHG-JEITA field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetFastChargeVoltageJEITA(chg_cv_t chg_cv_jeita);

    /**
	* @brief		Get the modified VFAST-CHG-JEITA for when the battery is either
	* 				cool or warm as defined by the VCOOL and VWARM temperature thresholds.
	*
	* @param[out]	chg_cv_jeita Modified VFAST-CHG-JEITA field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetFastChargeVoltageJEITA(chg_cv_t *chg_cv_jeita);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_I (0x28)
	*  - Bit Fields    : [7:4]
	*  - Default       : 0x0
	*  - Description   : Battery Discharge Current Full-Scale Current Value.
	*/
    typedef enum {
        IMON_DISCHG_SCALE_AMP_8_2MA,
        IMON_DISCHG_SCALE_AMP_40_5MA,
        IMON_DISCHG_SCALE_AMP_72_3MA,
        IMON_DISCHG_SCALE_AMP_103_4MA,
        IMON_DISCHG_SCALE_AMP_134_1MA,
        IMON_DISCHG_SCALE_AMP_164_1MA,
        IMON_DISCHG_SCALE_AMP_193_7MA,
        IMON_DISCHG_SCALE_AMP_222_7MA,
        IMON_DISCHG_SCALE_AMP_251_2MA,
        IMON_DISCHG_SCALE_AMP_279_3MA,
        IMON_DISCHG_SCALE_AMP_300_0MA,
        IMON_DISCHG_SCALE_AMP_300_0MA_0x0B,
        IMON_DISCHG_SCALE_AMP_300_0MA_0x0C,
        IMON_DISCHG_SCALE_AMP_300_0MA_0x0D,
        IMON_DISCHG_SCALE_AMP_300_0MA_0x0E,
        IMON_DISCHG_SCALE_AMP_300_0MA_0x0F
    }imon_dischg_scale_t;

    /**
	* @brief		Set the Battery Discharge Current Full-Scale Current Value.
	*
	* @param[in]	imon_dischg_scale the battery discharge current full-scale current value field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetDischargeCurrentFullScale(imon_dischg_scale_t imon_dischg_scale);

    /**
	* @brief		Get the Battery Discharge Current Full-Scale Current Value.
	*
	* @param[out]	imon_dischg_scale the battery discharge current full-scale current value field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetDischargeCurrentFullScale(imon_dischg_scale_t *imon_dischg_scale);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_CHG_I (0x28)
	*  - Bit Fields    : [3:0]
	*  - Default       : 0x0
	*  - Description   : Analog channel to connect to AMUX.
	*/
    typedef enum {
        MUX_SEL_MULTIPLEXER_DISABLED, 
		MUX_SEL_CHGIN_VOLTAGE_MONITOR, 
		MUX_SEL_CHGIN_CURRENT_MONITOR, 
		MUX_SEL_BATTERY_VOLTAGE_MONITOR,
		MUX_SEL_BATTERY_CHARGE_CURRENT_MONITOR, 
		MUX_SEL_BATTERY_DISCHARGE_CURRENT_MONITOR_NORMAL, 
		MUX_SEL_BATTERY_DISCHARGE_CURRENT_MONITOR_NULL, 
		MUX_SEL_THM_VOLTAGE_MONITOR,
		MUX_SEL_TBIAS_VOLTAGE_MONITOR,
		MUX_SEL_AGND_VOLTAGE_MONITOR,
		MUX_SEL_SYS_VOLTAGE_MONITOR,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0B,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0C,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0D,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0E,
		MUX_SEL_SYS_VOLTAGE_MONITOR_0x0F
    }mux_sel_t;

    /**
	* @brief		Set the Analog Channel to connect to AMUX.
	*
	* @param[in]	imon_dischg_scale the battery discharge current full-scale current value field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetAMUX(mux_sel_t mux_sel);

    /**
	* @brief		Get the Battery Discharge Current Full-Scale Current Value.
	*
	* @param[out]	imon_dischg_scale the battery discharge current full-scale current value field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetAMUX(mux_sel_t *mux_sel);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_A (0x29), CNFG_SBB1_A (0x2B) and CNFG_SBB2_A (0x2D)
	*  - Bit Fields    : [6:0]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost Channel 0, 1 or 2 Target Output Voltage.
	*/
    typedef enum {
        TV_SBB_VOLT_0_800V,
        TV_SBB_VOLT_0_850V,
        TV_SBB_VOLT_0_900V,
        TV_SBB_VOLT_0_950V,
        TV_SBB_VOLT_1_000V,
        TV_SBB_VOLT_1_050V,
        TV_SBB_VOLT_1_100V,
        TV_SBB_VOLT_1_150V,
        TV_SBB_VOLT_1_200V,
        TV_SBB_VOLT_1_250V,
        TV_SBB_VOLT_1_300V,
        TV_SBB_VOLT_1_350V,
        TV_SBB_VOLT_1_400V,
        TV_SBB_VOLT_1_450V,
        TV_SBB_VOLT_1_500V,
        TV_SBB_VOLT_1_550V,
        TV_SBB_VOLT_1_600V,
        TV_SBB_VOLT_1_650V,
        TV_SBB_VOLT_1_700V,
        TV_SBB_VOLT_1_750V,
        TV_SBB_VOLT_1_800V,
        TV_SBB_VOLT_1_850V,
        TV_SBB_VOLT_1_900V,
        TV_SBB_VOLT_1_950V,
        TV_SBB_VOLT_2_000V,
        TV_SBB_VOLT_2_050V,
        TV_SBB_VOLT_2_100V,
        TV_SBB_VOLT_2_150V,
        TV_SBB_VOLT_2_200V,
        TV_SBB_VOLT_2_250V,
        TV_SBB_VOLT_2_300V,
        TV_SBB_VOLT_2_350V,
        TV_SBB_VOLT_2_400V,
        TV_SBB_VOLT_2_450V,
        TV_SBB_VOLT_2_500V,
        TV_SBB_VOLT_2_550V,
        TV_SBB_VOLT_2_600V,
        TV_SBB_VOLT_2_650V,
        TV_SBB_VOLT_2_700V,
        TV_SBB_VOLT_2_750V,
        TV_SBB_VOLT_2_800V,
        TV_SBB_VOLT_2_850V,
        TV_SBB_VOLT_2_900V,
        TV_SBB_VOLT_2_950V,
        TV_SBB_VOLT_3_000V,
        TV_SBB_VOLT_3_050V,
        TV_SBB_VOLT_3_100V,
        TV_SBB_VOLT_3_150V,
        TV_SBB_VOLT_3_200V,
        TV_SBB_VOLT_3_250V,
        TV_SBB_VOLT_3_300V,
        TV_SBB_VOLT_3_350V,
        TV_SBB_VOLT_3_400V,
        TV_SBB_VOLT_3_450V,
        TV_SBB_VOLT_3_500V,
        TV_SBB_VOLT_3_550V,
        TV_SBB_VOLT_3_600V,
        TV_SBB_VOLT_3_650V,
        TV_SBB_VOLT_3_700V,
        TV_SBB_VOLT_3_750V,
        TV_SBB_VOLT_3_800V,
        TV_SBB_VOLT_3_850V,
        TV_SBB_VOLT_3_900V,
        TV_SBB_VOLT_3_950V,
        TV_SBB_VOLT_4_000V,
        TV_SBB_VOLT_4_050V,
        TV_SBB_VOLT_4_100V,
        TV_SBB_VOLT_4_150V,
        TV_SBB_VOLT_4_200V,
        TV_SBB_VOLT_4_250V,
        TV_SBB_VOLT_4_300V,
        TV_SBB_VOLT_4_350V,
        TV_SBB_VOLT_4_400V,
        TV_SBB_VOLT_4_450V,
        TV_SBB_VOLT_4_500V,
        TV_SBB_VOLT_4_550V,
        TV_SBB_VOLT_4_600V,
        TV_SBB_VOLT_4_650V,
        TV_SBB_VOLT_4_700V,
        TV_SBB_VOLT_4_750V,
        TV_SBB_VOLT_4_800V,
        TV_SBB_VOLT_4_850V,
        TV_SBB_VOLT_4_900V,
        TV_SBB_VOLT_4_950V,
        TV_SBB_VOLT_5_000V,
        TV_SBB_VOLT_5_050V,
        TV_SBB_VOLT_5_100V,
        TV_SBB_VOLT_5_150V,
        TV_SBB_VOLT_5_200V,
        TV_SBB_VOLT_5_250V,
        TV_SBB_VOLT_5_300V,
        TV_SBB_VOLT_5_350V,
        TV_SBB_VOLT_5_400V,
        TV_SBB_VOLT_5_450V,
        TV_SBB_VOLT_5_500V,
        TV_SBB_VOLT_5_500V_0x5D,
        TV_SBB_VOLT_5_500V_0x5E,
        TV_SBB_VOLT_5_500V_0x5F,
        TV_SBB_VOLT_5_500V_0x60,
        TV_SBB_VOLT_5_500V_0x61,
        TV_SBB_VOLT_5_500V_0x62,
        TV_SBB_VOLT_5_500V_0x63,
        TV_SBB_VOLT_5_500V_0x64,
        TV_SBB_VOLT_5_500V_0x65,
        TV_SBB_VOLT_5_500V_0x66,
        TV_SBB_VOLT_5_500V_0x67,
        TV_SBB_VOLT_5_500V_0x68,
        TV_SBB_VOLT_5_500V_0x69,
        TV_SBB_VOLT_5_500V_0x6A,
        TV_SBB_VOLT_5_500V_0x6B,
        TV_SBB_VOLT_5_500V_0x6C,
        TV_SBB_VOLT_5_500V_0x6D,
        TV_SBB_VOLT_5_500V_0x6E,
        TV_SBB_VOLT_5_500V_0x6F,
        TV_SBB_VOLT_5_500V_0x70,
        TV_SBB_VOLT_5_500V_0x71,
        TV_SBB_VOLT_5_500V_0x72,
        TV_SBB_VOLT_5_500V_0x73,
        TV_SBB_VOLT_5_500V_0x74,
        TV_SBB_VOLT_5_500V_0x75,
        TV_SBB_VOLT_5_500V_0x76,
        TV_SBB_VOLT_5_500V_0x77,
        TV_SBB_VOLT_5_500V_0x78,
        TV_SBB_VOLT_5_500V_0x79,
        TV_SBB_VOLT_5_500V_0x7A,
        TV_SBB_VOLT_5_500V_0x7B,
        TV_SBB_VOLT_5_500V_0x7C,
        TV_SBB_VOLT_5_500V_0x7D,
        TV_SBB_VOLT_5_500V_0x7E,
        TV_SBB_VOLT_5_500V_0x7F
    }tv_sbb_t;

    /**
	* @brief		Set SIMO Buck-Boost Channel x Target Output Voltage.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	tv_sbb SIMO buck-boost channel x target output voltage field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetSBBTargetOutVoltage(uint8_t channel, tv_sbb_t tv_sbb);

    /**
	* @brief		Get SIMO Buck-Boost Channel x Target Output Voltage.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	tv_sbb SIMO buck-boost channel x target output voltage field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetSBBTargetOutVoltage(uint8_t channel, tv_sbb_t *tv_sbb);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x2A), CNFG_SBB1_B (0x2C) (0x2B) and CNFG_SBB2_B (0x2E)
	*  - Bit Fields    : [6]
	*  - Default       : 0x0
	*  - Description   : Operation mode of SBB0, 1 or 2.
	*/
    typedef enum {
        OP_MODE_BUCK_BOOST_MODE,
        OP_MODE_BUCK_MODE
    }op_mode_t;

    /**
	* @brief		Set Operation mode of SBBx.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	op_mode Operation mode of SBBx bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetSIMOOperationMode(uint8_t channel, op_mode_t op_mode);

    /**
	* @brief		Get Operation mode of SBBx.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	op_mode Operation mode of SBBx bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetSIMOOperationMode(uint8_t channel, op_mode_t *op_mode);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x2A), CNFG_SBB1_B (0x2C) (0x2B) and CNFG_SBB2_B (0x2E)
	*  - Bit Fields    : [5:4]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost Channel 0, 1 or 2 Peak Current Limit.
	*/
    typedef enum {
        IP_SBB_AMP_1_000A,
        IP_SBB_AMP_0_750A,
        IP_SBB_AMP_0_500A,
        IP_SBB_AMP_0_333A
    }ip_sbb_t;

    /**
	* @brief		Set SIMO Buck-Boost Channel x Peak Current Limit.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	ip_sbb SIMO buck-boost channel 2 peak current limit field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetSBBPeakCurrentLimit(uint8_t channel, ip_sbb_t ip_sbb);

    /**
	* @brief		Get SIMO Buck-Boost Channel x Peak Current Limit.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	ip_sbb SIMO buck-boost channel 2 peak current limit field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetSBBPeakCurrentLimit(uint8_t channel, ip_sbb_t *ip_sbb);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x2A), CNFG_SBB1_B (0x2C) (0x2B) and CNFG_SBB2_B (0x2E)
	*  - Bit Fields    : [3]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost Channel 0, 1 or 2 Active-Discharge Enable.
	*/
    typedef enum {
        ADE_SBB_DISABLED,
        ADE_SBB_ENABLED
    }ade_sbb_t;

    /**
	* @brief		Set SIMO Buck-Boost Channel x Active-Discharge Enable.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	ade_sbb SIMO buck-boost channel 2 active-discharge enable bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetSBBActiveDischargeEnable(uint8_t channel, ade_sbb_t ade_sbb);

    /**
	* @brief		Get SIMO Buck-Boost Channel x Active-Discharge Enable.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	ade_sbb SIMO buck-boost channel 2 active-discharge enable bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetSBBActiveDischargeEnable(uint8_t channel, ade_sbb_t *ade_sbb);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x2A), CNFG_SBB1_B (0x2C) (0x2B) and CNFG_SBB2_B (0x2E)
	*  - Bit Fields    : [5:4]
	*  - Default       : 0x0
	*  - Description   : Enable Control for SIMO Buck-Boost Channel 0, 1 or 2.
	*/
    typedef enum {
        EN_SBB_FPS_SLOT_0,
        EN_SBB_FPS_SLOT_1,
        EN_SBB_FPS_SLOT_2,
        EN_SBB_FPS_SLOT_3,
        EN_SBB_OFF,
        EN_SBB_SAME_AS_0X04,
        EN_SBB_ON,
        EN_SBB_SAME_AS_0X06,
    }en_sbb_t;

    /**
	* @brief		Set Enable Control for SIMO Buck-Boost Channel x.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	en_sbb Enable control for SIMO buck-boost channel x field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetSBBEnableControl(uint8_t channel, en_sbb_t en_sbb);

    /**
	* @brief		Get Enable Control for SIMO Buck-Boost Channel x.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	en_sbb Enable control for SIMO buck-boost channel x field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetSBBEnableControl(uint8_t channel, en_sbb_t *en_sbb);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB_TOP (0x2F)
	*  - Bit Fields    : [7]
	*  - Default       : 0x0
	*  - Description   : Changes how CNFG_CHG_B.ICHGIN_LIM is interpreted.
	*/
    typedef enum {
        ICHIN_LIM_AMP_95MA,
        ICHIN_LIM_AMP_475MA
    }ichgin_lim_def_t;

    /**
	* @brief		Set Changes how CNFG_CHG_B.ICHGIN_LIM is interpreted.
	*
	* @param[in]	ichgin_lim_def CNFG_CHG_B.ICHGIN_LIM bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetCHGINInputCurrentLimit(ichgin_lim_def_t ichgin_lim_def);

    /**
	* @brief		Get Changes how CNFG_CHG_B.ICHGIN_LIM is interpreted.
	*
	* @param[out]	ichgin_lim_def CNFG_CHG_B.ICHGIN_LIM bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetCHGINInputCurrentLimit(ichgin_lim_def_t *ichgin_lim_def);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB_TOP (0x2F)
	*  - Bit Fields    : [1:0]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost (all channels) Drive Strength Trim.
	*/
    typedef enum {
        DRV_SBB_FASTEST_TRANSITION_TIME,
        DRV_SBB_A_LITTLE_SLOWER_THAN_0X00,
        DRV_SBB_A_LITTLE_SLOWER_THAN_0X01,
        DRV_SBB_A_LITTLE_SLOWER_THAN_0X02
    }drv_sbb_t;

    /**
	* @brief		Set SIMO Buck-Boost (all channels) Drive Strength Trim.
	*
	* @param[in]	drv_sbb SIMO buck-boost drive strength trim field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetSBBDriveStrength(drv_sbb_t drv_sbb);

    /**
	* @brief		Get SIMO Buck-Boost (all channels) Drive Strength Trim.
	*
	* @param[out]	drv_sbb SIMO buck-boost drive strength trim field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetSBBDriveStrength(drv_sbb_t *drv_sbb);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_A (0x38) and CNFG_LDO1_A (0x3A)
	*  - Bit Fields    : [6:0]
	*  - Default       : 0x0
	*  - Description   : LDO Target Output Voltage.
	*/
    typedef enum {
        TV_LDO_VOLT_0_800V,
        TV_LDO_VOLT_0_825V,
        TV_LDO_VOLT_0_850V,
        TV_LDO_VOLT_0_875V,
        TV_LDO_VOLT_0_900V,
        TV_LDO_VOLT_0_925V,
        TV_LDO_VOLT_0_950V,
        TV_LDO_VOLT_0_975V,
        TV_LDO_VOLT_1_000V,
        TV_LDO_VOLT_1_025V,
        TV_LDO_VOLT_1_050V,
        TV_LDO_VOLT_1_075V,
        TV_LDO_VOLT_1_100V,
        TV_LDO_VOLT_1_125V,
        TV_LDO_VOLT_1_150V,
        TV_LDO_VOLT_1_175V,
        TV_LDO_VOLT_1_200V,
        TV_LDO_VOLT_1_225V,
        TV_LDO_VOLT_1_250V,
        TV_LDO_VOLT_1_275V,
        TV_LDO_VOLT_1_300V,
        TV_LDO_VOLT_1_325V,
        TV_LDO_VOLT_1_350V,
        TV_LDO_VOLT_1_375V,
        TV_LDO_VOLT_1_400V,
        TV_LDO_VOLT_1_425V,
        TV_LDO_VOLT_1_450V,
        TV_LDO_VOLT_1_475V,
        TV_LDO_VOLT_1_500V,
        TV_LDO_VOLT_1_525V,
        TV_LDO_VOLT_1_550V,
        TV_LDO_VOLT_1_575V,
        TV_LDO_VOLT_1_600V,
        TV_LDO_VOLT_1_625V,
        TV_LDO_VOLT_1_650V,
        TV_LDO_VOLT_1_675V,
        TV_LDO_VOLT_1_700V,
        TV_LDO_VOLT_1_725V,
        TV_LDO_VOLT_1_750V,
        TV_LDO_VOLT_1_775V,
        TV_LDO_VOLT_1_800V,
        TV_LDO_VOLT_1_825V,
        TV_LDO_VOLT_1_850V,
        TV_LDO_VOLT_1_875V,
        TV_LDO_VOLT_1_900V,
        TV_LDO_VOLT_1_925V,
        TV_LDO_VOLT_1_950V,
        TV_LDO_VOLT_1_975V,
        TV_LDO_VOLT_2_000V,
        TV_LDO_VOLT_2_025V,
        TV_LDO_VOLT_2_050V,
        TV_LDO_VOLT_2_075V,
        TV_LDO_VOLT_2_100V,
        TV_LDO_VOLT_2_125V,
        TV_LDO_VOLT_2_150V,
        TV_LDO_VOLT_2_175V,
        TV_LDO_VOLT_2_200V,
        TV_LDO_VOLT_2_225V,
        TV_LDO_VOLT_2_250V,
        TV_LDO_VOLT_2_275V,
        TV_LDO_VOLT_2_300V,
        TV_LDO_VOLT_2_325V,
        TV_LDO_VOLT_2_350V,
        TV_LDO_VOLT_2_375V,
        TV_LDO_VOLT_2_400V,
        TV_LDO_VOLT_2_425V,
        TV_LDO_VOLT_2_450V,
        TV_LDO_VOLT_2_475V,
        TV_LDO_VOLT_2_500V,
        TV_LDO_VOLT_2_525V,
        TV_LDO_VOLT_2_550V,
        TV_LDO_VOLT_2_575V,
        TV_LDO_VOLT_2_600V,
        TV_LDO_VOLT_2_625V,
        TV_LDO_VOLT_2_650V,
        TV_LDO_VOLT_2_675V,
        TV_LDO_VOLT_2_700V,
        TV_LDO_VOLT_2_725V,
        TV_LDO_VOLT_2_750V,
        TV_LDO_VOLT_2_775V,
        TV_LDO_VOLT_2_800V,
        TV_LDO_VOLT_2_825V,
        TV_LDO_VOLT_2_850V,
        TV_LDO_VOLT_2_875V,
        TV_LDO_VOLT_2_900V,
        TV_LDO_VOLT_2_925V,
        TV_LDO_VOLT_2_950V,
        TV_LDO_VOLT_2_975V,
        TV_LDO_VOLT_3_000V,
        TV_LDO_VOLT_3_025V,
        TV_LDO_VOLT_3_050V,
        TV_LDO_VOLT_3_075V,
        TV_LDO_VOLT_3_100V,
        TV_LDO_VOLT_3_125V,
        TV_LDO_VOLT_3_150V,
        TV_LDO_VOLT_3_175V,
        TV_LDO_VOLT_3_200V,
        TV_LDO_VOLT_3_225V,
        TV_LDO_VOLT_3_250V,
        TV_LDO_VOLT_3_275V,
        TV_LDO_VOLT_3_300V,
        TV_LDO_VOLT_3_325V,
        TV_LDO_VOLT_3_350V,
        TV_LDO_VOLT_3_375V,
        TV_LDO_VOLT_3_400V,
        TV_LDO_VOLT_3_425V,
        TV_LDO_VOLT_3_450V,
        TV_LDO_VOLT_3_475V,
        TV_LDO_VOLT_3_500V,
        TV_LDO_VOLT_3_525V,
        TV_LDO_VOLT_3_550V,
        TV_LDO_VOLT_3_575V,
        TV_LDO_VOLT_3_600V,
        TV_LDO_VOLT_3_625V,
        TV_LDO_VOLT_3_650V,
        TV_LDO_VOLT_3_675V,
        TV_LDO_VOLT_3_700V,
        TV_LDO_VOLT_3_725V,
        TV_LDO_VOLT_3_750V,
        TV_LDO_VOLT_3_775V,
        TV_LDO_VOLT_3_800V,
        TV_LDO_VOLT_3_825V,
        TV_LDO_VOLT_3_850V,
        TV_LDO_VOLT_3_875V,
        TV_LDO_VOLT_3_900V,
        TV_LDO_VOLT_3_925V,
        TV_LDO_VOLT_3_950V,
        TV_LDO_VOLT_3_975V
    }tv_ldo_t;

    /**
	* @brief		Set LDO Target Output Voltage.
	*
	* @param[in]	channel Channel number: 0 or 1
	* @param[in]	tv_ldo LDO target output voltage field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetLDOTargetOutVoltage(uint8_t channel, tv_ldo_t tv_ldo);

    /**
	* @brief		Get LDO Target Output Voltage.
	*
	* @param[in]	channel Channel number: 0 or 1
	* @param[out]	tv_ldo LDO target output voltage field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetLDOTargetOutVoltage(uint8_t channel, tv_ldo_t *tv_ldo);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_B (0x39) and CNFG_LDO1_B (0x3B)
	*  - Bit Fields    : [4]
	*  - Default       : 0x0
	*  - Description   : Operation Mode of LDO0 or 1.
	*/
    typedef enum {
        LDO_MD_LDO_MODE,
        LDO_MD_LSW_MODE
    }ldo_md_t;

    /**
	* @brief		Set Operation Mode of LDOx.
	*
	* @param[in]	channel Channel number: 0 or 1
	* @param[in]	ldo_md Operation mode of LDOx bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetLDOMode(uint8_t channel, ldo_md_t ldo_md);

    /**
	* @brief		Get Operation Mode of LDOx.
	*
	* @param[in]	channel Channel number: 0 or 1
	* @param[out]	ldo_md Operation mode of LDOx bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetLDOMode(uint8_t channel, ldo_md_t *ldo_md);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_B (0x39) and CNFG_LDO1_B (0x3B)
	*  - Bit Fields    : [3]
	*  - Default       : 0x0
	*  - Description   : LDO0 or 1  Active-Discharge Enable.
	*/
    typedef enum {
        ADE_LDO_DISABLED,
        ADE_LDO_ENABLED
    }ade_ldo_t;

    /**
	* @brief		Set LDOx Active-Discharge Enable.
	*
	* @param[in]	channel Channel number: 0 or 1
	* @param[in]	ade_ldo LDOx active-discharge enable bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
    int SetLDOActiveDischargeEnable(uint8_t channel, ade_ldo_t ade_ldo);

    /**
	* @brief		Get LDOx Active-Discharge Enable.
	*
	* @param[in]	channel Channel number: 0 or 1
	* @param[out]	ade_ldo LDOx active-discharge enable bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
    int GetLDOActiveDischargeEnable(uint8_t channel, ade_ldo_t *ade_ldo);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_B (0x39) and CNFG_LDO1_B (0x3B)
	*  - Bit Fields    : [2:0]
	*  - Default       : 0x0
	*  - Description   : Enable Control for LDO0 or 1.
	*/
    typedef enum {
        EN_LDO_FPS_SLOT_0,
        EN_LDO_FPS_SLOT_1,
        EN_LDO_FPS_SLOT_2,
        EN_LDO_FPS_SLOT_3,
        EN_LDO_OFF,
        EN_LDO_SAME_AS_0X04,
        EN_LDO_ON,
        EN_LDO_SAME_AS_0X06,
    }en_ldo_t;

    /**
    * @brief		Set Enable Control for LDOx.
    *
    * @param[in]	channel Channel number: 0 or 1.
    * @param[in]	en_ldo Enable control for LDOx field to be written.
    *
    * @return		0 on success, error code on failure
    */
    int SetLDOEnableControl(uint8_t channel, en_ldo_t en_ldo);

    /**
    * @brief		Get Enable Control for LDOx.
    *
    * @param[in]	channel Channel number: 0 or 1.
    * @param[in]	en_ldo Enable control for LDOx field to be read.
    *
    * @return		0 on success, error code on failure
    */
    int GetLDOEnableControl(uint8_t channel, en_ldo_t *en_ldo);

    /**
     * @brief	Disable all interrupts
     *
     * @return	0 on success, error code on failure
     */
     int IRQDisableAll();

     /**
     * @brief		Set Interrupt Handler for a Specific Interrupt ID.
     *
     * @param[in]	id Interrupt id, one of INTR_ID_*.
     * @param[in]	func Interrupt handler function.
     * @param[in]	cb Interrupt handler data.
     */
     void SetInterruptHandler(int_glbl_t id, interrupt_handler_function func, void *cb);
};
#endif /*_MAX77654_H_*/
