/*******************************************************************************
 * Copyright(C) Analog Devices Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files(the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Analog Devices Inc.
 * shall not be used except as stated in the Analog Devices Inc.
 * Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Analog Devices Inc.retains all ownership rights.
 *******************************************************************************
 */

#ifndef _MAX77643_2_H_
#define _MAX77643_2_H_

#include "mbed.h"
#include "MAX77643_2_regs.h"

#define MAX77643_2_NO_ERROR                    0
#define MAX77643_2_VALUE_NULL                 -1
#define MAX77643_2_WRITE_DATA_FAILED          -2
#define MAX77643_2_READ_DATA_FAILED           -3
#define MAX77643_2_INVALID_DATA               -4

#define MAX77643_2_I2C_ADDRESS         		0x90

/**
 * @brief MAX77643_2 Ultra Configurable PMIC Featuring 93% Peak Efficiency Single-Inductor,
 * 3-Output Buck-Boost, 1-LDO for Long Battery Life Applications
 *
 * @details The MAX77643_2/MAX77643_2 provide power supply solutions for low-power applications where size and efficiency
 * are critical. The MAX77643_2's SIMO and LDO output voltages are individually programmable through resistors.
 *
 * @code
 * @endcode
 */

class MAX77643_2
{
private:
	I2C *i2c_handler;
	InterruptIn *irq_pin;	// interrupt pin

	/**
	 * @brief   	Register Addresses
	 * @details 	Enumerated MAX77643_2 register addresses
	 */
	typedef enum {
		/*Global*/
		INT_GLBL0    	= 0x00,    // Interrupt Status 0
		INT_GLBL1   	= 0x01,    // Interrupt Status 1
		ERCFLAG      	= 0x02,    // Flags
		STAT_GLBL    	= 0x03,    // Global Status
		INTM_GLBL0   	= 0x04,    // Interrupt Mask 0
		INTM_GLBL1   	= 0x05,    // Interrupt Mask 1
		CNFG_GLBL0    	= 0x06,    // Configuration Global 0
		CNFG_GLBL1    	= 0x07,    // Configuration Global 1
		CNFG_GPIO0   	= 0x08,    // GPIO0 Configuration
		CNFG_GPIO1   	= 0x09,    // GPIO1 Configuration
		CID          	= 0x10,    // Chip Identification Code
		CNFG_WDT     	= 0x17,    // Configuration WatchDog Timer
		/*SBB*/
		CNFG_SBB_TOP 	= 0x28,    // SIMO Buck-Boost Configuration
		CNFG_SBB0_A  	= 0x29,    // SIMO Buck-Boost 0 Configuration A
		CNFG_SBB0_B  	= 0x2A,    // SIMO Buck-Boost 0 Configuration B
		CNFG_SBB1_A  	= 0x2B,    // SIMO Buck-Boost 1 Configuration A
		CNFG_SBB1_B  	= 0x2C,    // SIMO Buck-Boost 1 Configuration B
		CNFG_SBB2_A  	= 0x2D,    // SIMO Buck-Boost 2 Configuration A
		CNFG_SBB2_B  	= 0x2E,    // SIMO Buck-Boost 2 Configuration B
		CNFG_DVS_SBB0_A = 0x2F,    // SIMO Buck-Boost 0 DVS Configuration A
		/*LDO*/
		CNFG_LDO0_A		= 0x38,	   // LDO0 Output Voltage
		CNFG_LDO0_B		= 0x39	   // LDO0 Output Voltage Configuration
	} reg_t;

	void interrupt_handler();

	void (MAX77643_2::*funcptr)(void);

	void post_interrupt_work();

	Thread *post_intr_work_thread;

	struct handler {
		void (*func)(void *);
		void *cb;
	};

	handler *interrupt_handler_list;

public:
	/**
	 * @brief		MAX77643_2 constructor.
	 */
	MAX77643_2(I2C *i2c, PinName IRQPin = NC);

	/**
	 * @brief		MAX77643_2 destructor.
	 */
	~MAX77643_2();

	/**
	 * @brief		Function pointer type to interrupt handler function
	 */
	typedef void (*interrupt_handler_function)(void *);

	/**
	 * @brief 		Read from a register.
	 *
	 * @param[in] 	reg Address of a register to be written.
	 * @param[out] 	value Pointer to save result value.
	 *
	 * @returns 	0 on success, negative error code on failure.
	 */
	int read_register(uint8_t reg, uint8_t *value);

	/**
	 * @brief 		Write to a register.
	 *
	 * @param[in] 	reg Address of a register to be written.
	 * @param[out] 	value Pointer of value to be written to register.
	 *
	 * @returns 	0 on success, negative error code on failure.
	 */
	int write_register(uint8_t reg, const uint8_t *value);	

	/**
	 * @brief 		Register Configuration.
	*		 		All Interrupt Flags combined from INT_GLBL0 (0x00) and INT_GLBL1(0x01)
	*
	* @details
	*  - Register      : INT_GLBL0 (0x00) and INT_GLBL1(0x01)
	*  - Bit Fields    :
	*  - Default       : 0x0
	*  - Description   : Enumerated interrupts.
	*/
	typedef enum {
		INT_GLBL0_GPI0_F,
		INT_GLBL0_GPI0_R,
		INT_GLBL0_NEN_F,
		INT_GLBL0_NEN_R,
		INT_GLBL0_TJAL1_R,
		INT_GLBL0_TJAL2_R,
		INT_GLBL0_DOD_R,
		INT_GLBL0_RSVD,
		INT_GLBL1_GPI1_F,
		INT_GLBL1_GPI1_R,
		INT_GLBL1_SBB0_F,
		INT_GLBL1_SBB1_F,
		INT_GLBL1_SBB2_F,
		INT_GLBL1_LDO_F,
		INT_GLBL1_RSVD,
		INT_CHG_END
	} reg_bit_int_glbl_t;
	
	/**
	 * @brief Register Configuration
	 *
	 * @details
	 *  - Register      : ERCFLAG (0x02)
	 *  - Bit Fields    : [7:0]
	 *  - Default       : 0x0
	 *  - Description   : Event Recorder Flags.
	 */
	typedef enum {
		ERCFLAG_TOVLD,
		ERCFLAG_INOVLO,
		ERCFLAG_INUVLO,
		ERCFLAG_MRST_F,
		ERCFLAG_SFT_OFF_F,
		ERCFLAG_SFT_CRST_F,
		ERCFLAG_WDT_EXP_F,
		ERCFLAG_SBB_FAULT_F
	}reg_bit_ercflag_t;

	/**
	 * @brief		Get bit field of ERCFLAG (0x02) register.
	 *
	* @param[in]	bit_field 	ERCFLAG register bit field to be written.
	* @param[out] 	flag 		Pointer to save result of ercglag bit states.	
	*							For individual bit
	*							0x0: ERCFLAG has not occurred,
	*               			0x1: ERCFLAG has occurred.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_ercflag(reg_bit_ercflag_t bit_field, uint8_t *flag);

	/**
	 * @brief Register Configuration
	 *
	 * @details
	 *  - Register      : STAT_GLBL (0x03)
	 *  - Bit Fields    : [7:0]
	 *  - Default       : 0x0
	 *  - Description   : Global Status.
	 */
	typedef enum {
		STAT_GLBL_STAT_IRQ,
		STAT_GLBL_STAT_EN,
		STAT_GLBL_TJAL1_S,
		STAT_GLBL_TJAL2_S,
		STAT_GLBL_DOD_S,
		STAT_GLBL_RSVD,
		STAT_GLBL_BOK,
		STAT_GLBL_DIDM
	}reg_bit_stat_glbl_t;
	
	/**
	 * @brief		Get bit field of STAT_GLBL (0x03) register.
	 *
	* @param[in]	bit_field 	STAT_GLBL register bit field to be written.
	* @param[out] 	status 		Pointer to save result of Status Global bit state.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_stat_glbl(reg_bit_stat_glbl_t bit_field, uint8_t *status);

	/**
	 * @brief Register Configuration
	 *
	 * @details
	 *  - Register      : INTM_GLBL0 (0x04) and INTM_GLBL1 (0x05)
	 *  - Bit Fields    : [7:0]
	 *  - Default       : 0x0
	 *  - Description   : All interrupt mask bits.
	 */
	typedef enum {
		INTM_GLBL0_GPI0_FM,
		INTM_GLBL0_GPI0_RM,
		INTM_GLBL0_nEN_FM,
		INTM_GLBL0_nEN_RM,
		INTM_GLBL0_TJAL1_RM,
		INTM_GLBL0_TJAL2_RM,
		INTM_GLBL0_DOD_RM,
		INTM_GLBL0_RSVD,
		INTM_GLBL1_GPI1_FM,
		INTM_GLBL1_GPI1_RM,
		INTM_GLBL1_SBB0_FM,
		INTM_GLBL1_SBB1_FM,
		INTM_GLBL1_SBB2_FM,
		INTM_GLBL1_LDO_M,
		INTM_GLBL1_RSVD,
		INTM_NUM_OF_BIT
	}reg_bit_int_mask_t;
	
	/**
	 * @brief		Set bit field of INTM_GLBL0 (0x04) or INTM_GLBL1 (0x05) register.
	 *
	* @param[in]	bit_field 	Register bit field to be set.
	* @param[out] 	maskBit 	0x0: Interrupt is unmasked,
	*                    		0x1: Interrupt is masked.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_interrupt_mask(reg_bit_int_mask_t bit_field, uint8_t maskBit);

	/**
	 * @brief		Get bit field of INTM_GLBL0 (0x04) or INTM_GLBL1 (0x05) register.
	 *
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out] 	maskBit 	0x0: Interrupt is unmasked,
	*                    		0x1: Interrupt is masked.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_interrupt_mask(reg_bit_int_mask_t bit_field, uint8_t *maskBit);

	/**
	 * @brief Register Configuration
	 *
	 * @details
	 *  - Register      : CNFG_GLBL0 (0x06)
	 *  - Bit Fields    : [7:0]
	 *  - Default       : 0x0
	 *  - Description   : Event Recorder Flags.
	 */
	typedef enum {
		CNFG_GLBL0_SFT_CTRL,
		CNFG_GLBL0_DBEN_nEN,
		CNFG_GLBL0_nEN_MODE,
		CNFG_GLBL0_SBIA_LPM,
		CNFG_GLBL0_T_MRST,
		CNFG_GLBL0_PU_DIS
	}reg_bit_cnfg_glbl0_t;	

	/**
  	* @brief		Set CNFG_GLBL0 (0x06) register.
  	*
	* @param[in]	bit_field 	Register bit field to be written.
  	* @param[in]	config 		Register bit field to be written.
  	*
  	* @return		0 on success, error code on failure.
  	*/
	int set_cnfg_glbl0(reg_bit_cnfg_glbl0_t bit_field, uint8_t config);

	/**
	* @brief		Get CNFG_GLBL0 (0x06) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	config 		Pointer of value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_cnfg_glbl0(reg_bit_cnfg_glbl0_t bit_field, uint8_t *config);

	/**
	 * @brief Register Configuration
	 *
	 * @details
	 *  - Register      : CNFG_GLBL1 (0x07)
	 *  - Bit Fields    : [7:0]
	 *  - Default       : 0x0
	 *  - Description   : Event Recorder Flags.
	 */
	typedef enum {
		CNFG_GLBL1_AUTO_WKT,
		CNFG_GLBL1_SBB_F_SHUTDN,
		CNFG_GLBL1_RSVD
	}reg_bit_cnfg_glbl1_t;	

	/**
	* @brief		Set CNFG_GLBL1 (0x07) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[in]	config 		Register bit field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_cnfg_glbl1(reg_bit_cnfg_glbl1_t bit_field, uint8_t config);

	/**
	* @brief		Get CNFG_GLBL1 (0x07) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	config 		Pointer of value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_cnfg_glbl1(reg_bit_cnfg_glbl1_t bit_field, uint8_t *config);

	/**
	 * @brief Register Configuration
	 *
	 * @details
	 *  - Register      : CNFG_GPIO0 (0x08) or CNFG_GPIO1 (0x09)
	 *  - Bit Fields    : [7:0]
	 *  - Default       : 0x0
	 *  - Description   : Event Recorder Flags.
	 */
	typedef enum {
		CNFG_GPIO_DIR,
		CNFG_GPIO_DI,
		CNFG_GPIO_DRV,
		CNFG_GPIO_DO,
		CNFG_GPIO_DBEN_GPI,
		CNFG_GPIO_ALT_GPIO,
		CNFG_GPIO_RSVD
	}reg_bit_cnfg_gpio_t;	

	/**
	* @brief		Set either CNFG_GPIO0 (0x08) or CNFG_GPIO1 (0x09).
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[in]	channel 	Channel number: 0 or 1
	* @param[in]	config 		Register bit field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_cnfg_gpio(reg_bit_cnfg_gpio_t bit_field, uint8_t channel, uint8_t config);

	/**
	* @brief		Get either CNFG_GPIO0 (0x08) or CNFG_GPIO1 (0x09).
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[in]	channel 	Channel number: 0 or 1
	* @param[out]	config 		Pointer of value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_cnfg_gpio(reg_bit_cnfg_gpio_t bit_field, uint8_t channel, uint8_t *config);

	/**
	 * @brief		Get bit field of CID (0x10) register.
	* 
	* @param[in]	config 		Register bit field to be written
	* 
	* @return		CID on success, error code on failure.
	*/
	int get_cid(void);

	/**
	 * @brief Register Configuration
	 *
	 * @details
	 *  - Register      : CNFG_WDT (0x17)
	 *  - Bit Fields    : [7:0]
	 *  - Default       : 0x0
	 *  - Description   : Watchdog Timer Configuration.
	 */
	typedef enum {
		CNFG_WDT_WDT_LOCK,
		CNFG_WDT_WDT_EN,
		CNFG_WDT_WDT_CLR,
		CNFG_WDT_WDT_MODE,
		CNFG_WDT_WDT_PER,
		CNFG_WDT_RSVD
	}reg_bit_cnfg_wdt_t;	
	
	/**
	* @brief		Set CNFG_WDT (0x17) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[in]	config 		Field value to be written.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_cnfg_wdt(reg_bit_cnfg_wdt_t bit_field, uint8_t config);

	/**
	* @brief		Get CNFG_WDT (0x17) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	config 		Pointer of value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_cnfg_wdt(reg_bit_cnfg_wdt_t bit_field, uint8_t *config);

	/*SBB*/

	/**
	 * @brief Register Configuration
	 *
	 * @details
	 *  - Register      : CNFG_SBB_TOP (0x28)
	 *  - Bit Fields    : [7:0]
	 *  - Default       : 0x0
	 *  - Description   : Watchdog Timer Configuration.
	 */
	typedef enum {
		CNFG_SBB_TOP_DRV_SBB,
		CNFG_SBB_TOP_DIS_LPM
	}reg_bit_cnfg_sbb_top_t;	
	
	/**
	* @brief		Set CNFG_SBB_TOP (0x28) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[in]	config 		Configuration value to be written.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_cnfg_sbb_top(reg_bit_cnfg_sbb_top_t bit_field, uint8_t config);

	/**
	* @brief		Get CNFG_SBB_TOP (0x28) register.
	*
	* @param[in]	bit_field 	Register bit field to be written.
	* @param[out]	config 		Configuration value to be read.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_cnfg_sbb_top(reg_bit_cnfg_sbb_top_t bit_field, uint8_t *config);

	/**
	* @brief		Set SIMO Buck-Boost Channel x Target Output Voltage.
	*				CNFG_SBB0_A (0x29), CNFG_SBB1_A (0x2B) and CNFG_SBB2_A (0x2D)
	*
	* @param[in]	channel 	Channel number: 0, 1 or 2.
	* @param[in]	voltV 		SIMO buck-boost channel x target output voltage field to be written.
	*							SBBx = 500mV + 25mV x TV_SBBx[7:0]
	*							0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*							0.650V, 0.675V, 0.700V, ... 
	*							5.425V, 5.450V, 5.475V, 5.500V.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_tv_sbb(uint8_t channel, float voltV);

	/**
	* @brief		Get SIMO Buck-Boost Channel x Target Output Voltage.
	*				CNFG_SBB0_A (0x29), CNFG_SBB1_A (0x2B) and CNFG_SBB2_A (0x2D)
	*
	* @param[in]	channel 	Channel number: 0, 1 or 2.
	* @param[out]	voltV 		SIMO buck-boost channel x target output voltage field to be read.
	*							SBBx = 500mV + 25mV x TV_SBBx[7:0]
	*							0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*							0.650V, 0.675V, 0.700V, ... 
	*							5.425V, 5.450V, 5.475V, 5.500V.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_tv_sbb(uint8_t channel, float *voltV);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x2A), CNFG_SBB1_B (0x2C) and CNFG_SBB2_B (0x2E)
	*  - Bit Fields    : [7:6]
	*  - Default       : 0x0
	*  - Description   : Operation mode of SBB0, 1 or 2.
	*/
	typedef enum {
		OP_MODE_AUTOMATIC,
		OP_MODE_BUCK_MODE,
		OP_MODE_BOOST_MODE,
		OP_MODE_BUCK_BOOST_MODE
	}decode_op_mode_t;

	/**
	* @brief		Set Operation mode of SBBx.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	mode 	Operation mode of SBBx bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_op_mode(uint8_t channel, decode_op_mode_t mode);

	/**
	* @brief		Get Operation mode of SBBx.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	mode 	Operation mode of SBBx bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_op_mode(uint8_t channel, decode_op_mode_t *mode);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x2A), CNFG_SBB1_B (0x2C) and CNFG_SBB2_B (0x2E)
	*  - Bit Fields    : [5:4]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost Channel 0, 1 or 2 Peak Current Limit.
	*/
	typedef enum {
		IP_SBB_AMP_1_000A,
		IP_SBB_AMP_0_750A,
		IP_SBB_AMP_0_500A,
		IP_SBB_AMP_0_333A
	}decode_ip_sbb_t;

	/**
	* @brief		Set SIMO Buck-Boost Channel x Peak Current Limit.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	ip_sbb 	SIMO buck-boost channel 2 peak current limit field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_ip_sbb(uint8_t channel, decode_ip_sbb_t ip_sbb);

	/**
	* @brief		Get SIMO Buck-Boost Channel x Peak Current Limit.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	ip_sbb 	SIMO buck-boost channel 2 peak current limit field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_ip_sbb(uint8_t channel, decode_ip_sbb_t *ip_sbb);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x2A), CNFG_SBB1_B (0x2C) and CNFG_SBB2_B (0x2E)
	*  - Bit Fields    : [3]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost Channel 0, 1 or 2 Active-Discharge Enable.
	*/
	typedef enum {
		ADE_SBB_DISABLED,
		ADE_SBB_ENABLED
	}decode_ade_sbb_t;

	/**
	* @brief		Set SIMO Buck-Boost Channel x Active-Discharge Enable.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	ade_sbb SIMO buck-boost channel 2 active-discharge enable bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_ade_sbb(uint8_t channel, decode_ade_sbb_t ade_sbb);

	/**
	* @brief		Get SIMO Buck-Boost Channel x Active-Discharge Enable.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	ade_sbb SIMO buck-boost channel 2 active-discharge enable bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_ade_sbb(uint8_t channel, decode_ade_sbb_t *ade_sbb);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_SBB0_B (0x2A), CNFG_SBB1_B (0x2C) and CNFG_SBB2_B (0x2E)
	*  - Bit Fields    : [2:0]
	*  - Default       : 0x0
	*  - Description   : Enable Control for SIMO Buck-Boost Channel 0, 1 or 2.
	*/
	typedef enum {
		EN_SBB_FPS_SLOT_0,
		EN_SBB_FPS_SLOT_1,
		EN_SBB_FPS_SLOT_2,
		EN_SBB_FPS_SLOT_3,
		EN_SBB_OFF,
		EN_SBB_SAME_AS_0X04,
		EN_SBB_ON,
		EN_SBB_SAME_AS_0X06
	}decode_en_sbb_t;

	/**
	* @brief		Set Enable Control for SIMO Buck-Boost Channel x.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[in]	en_sbb 	Enable control for SIMO buck-boost channel x field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_en_sbb(uint8_t channel, decode_en_sbb_t en_sbb);

	/**
	* @brief		Get Enable Control for SIMO Buck-Boost Channel x.
	*
	* @param[in]	channel Channel number: 0, 1 or 2.
	* @param[out]	en_sbb 	Enable control for SIMO buck-boost channel x field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_en_sbb(uint8_t channel, decode_en_sbb_t *en_sbb);

	/**
	* @brief		Set SIMO Buck-Boost Channel 0 Target Output Voltage.
	*				Bit 7:0 of CNFG_DVS_SBB0_A (0x2F).
	*
	* @param[in]	voltV 	SIMO buck-boost channel 0 target output voltage field to be written.
	*						SBBx = 500mV + 25mV x TV_SBBx[7:0]
	*						0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*						0.650V, 0.675V, 0.700V, ... 
	*						5.425V, 5.450V, 5.475V, 5.500V.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_tv_sbb_dvs(float voltV);

	/**
	* @brief		Get SIMO Buck-Boost Channel 0 Target Output Voltage.
	*				Bit 7:0 of CNFG_DVS_SBB0_A (0x2F).
	*
	* @param[out]	voltV 	SIMO buck-boost channel 0 target output voltage field to be read.
	*						SBBx = 500mV + 25mV x TV_SBBx[7:0]
	*						0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*						0.650V, 0.675V, 0.700V, ... 
	*						5.425V, 5.450V, 5.475V, 5.500V.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_tv_sbb_dvs(float *voltV);

	/*LDO*/

	/**
	* @brief		Set LDO Output Channel x Target Output Voltage. Bit 6:0.
	*				CNFG_LDO0_A (0x38)
	*
	* @param[in]	voltV 		LDO Output Channel x target output voltage field to be read.
	*							LDOx = 500mV + 25mV x TV_LDOx[6:0]
	*							0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*							0.650V, 0.675V, 0.700V, ... 
	*							3.650, 3.675.
	*
	*							When TV_LDO[7] = 0, TV_LDO[6:0] sets the
	*							LDO's output voltage range from 0.5V to 3.675V.
	*							When TV_LDO[7] = 1, TV_LDO[6:0] sets the
	*							LDO's output voltage from 1.825V to 5V.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_tv_ldo(float voltV);

	/**
	* @brief		Get LDO Output Channel x Target Output Voltage. Bit 6:0.
	*				CNFG_LDO0_A (0x38)
	*
	* @param[out]	voltV 		LDO Output Channel x target output voltage field to be read.
	*							LDOx = 500mV + 25mV x TV_LDOx[6:0]
	*							0.500V, 0.525V, 0.550V, 0.575V, 0.600V, 0.625V, 
	*							0.650V, 0.675V, 0.700V, ... 
	*							3.650, 3.675.
	*
	*							When TV_LDO[7] = 0, TV_LDO[6:0] sets the
	*							LDO's output voltage range from 0.5V to 3.675V.
	*							When TV_LDO[7] = 1, TV_LDO[6:0] sets the
	*							LDO's output voltage from 1.825V to 5V.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_tv_ldo(float *voltV);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_A (0x38)
	*  - Bit Fields    : [7]
	*  - Default       : 0x0
	*  - Description   : SIMO Buck-Boost Channel 0 Peak Current Limit.
	*/
	typedef enum {
		TV_OFS_LDO_NO_OFFSET,
		TV_OFS_LDO_NO_1_325V
	}decode_tv_ofs_ldo_t;
	
	/**
	* @brief		Set LDO Output Channel 0 Target Output Voltage. Bit 7.
	*				CNFG_LDO0_A (0x38)
	*
	* @param[in]	offset 		LDO Output Channel 0 target output voltage offset field to be read.
	*							0b0 = No Offset
	*							0b1 = 1.325V Offset
	*
	* @return		0 on success, error code on failure.
	*/
	int set_tv_ofs_ldo(decode_tv_ofs_ldo_t offset);

	/**
	* @brief		Get LDO Output Channel 0 Target Output Voltage. Bit 7.
	*				CNFG_LDO0_A (0x38)
	*
	* @param[out]	offset 		LDO Output Channel 0 target output voltage offset field to be read.
	*							0b0 = No Offset
	*							0b1 = 1.325V Offset
	*
	* @return		0 on success, error code on failure.
	*/
	int get_tv_ofs_ldo(decode_tv_ofs_ldo_t *offset);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_B (0x39) 
	*  - Bit Fields    : [2:0]
	*  - Default       : 0x0
	*  - Description   : Enable Control for LDO 0.
	*/
	typedef enum {
		EN_LDO_FPS_SLOT_0,
		EN_LDO_FPS_SLOT_1,
		EN_LDO_FPS_SLOT_2,
		EN_LDO_FPS_SLOT_3,
		EN_LDO_OFF,
		EN_LDO_SAME_AS_0X04,
		EN_LDO_ON,
		EN_LDO_SAME_AS_0X06
	}decode_en_ldo_t;

	/**
	* @brief		Set Enable Control for LDO Channel 0.
	*
	* @param[in]	en_ldo 	Enable control for LDO channel 0 field to be written.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_en_ldo(decode_en_ldo_t en_ldo);

	/**
	* @brief		Get Enable Control for LDO Channel 0.
	*
	* @param[out]	en_ldo 	Enable control for LDO channel x field to be read.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_en_ldo(decode_en_ldo_t *en_ldo);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_B (0x39) 
	*  - Bit Fields    : [3]
	*  - Default       : 0x0
	*  - Description   : LDO Channel 0 Active-Discharge Enable.
	*/
	typedef enum {
		ADE_LDO_DISABLED,
		ADE_LDO_ENABLED
	}decode_ade_ldo_t;

	/**
	* @brief		Set LDO Channel 0 Active-Discharge Enable.
	*
	* @param[in]	ade_ldo LDO channel 0 active-discharge enable bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_ade_ldo(decode_ade_ldo_t ade_ldo);

	/**
	* @brief		Get LDO Channel 0 Active-Discharge Enable.
	*
	* @param[out]	ade_ldo LDO channel 0 active-discharge enable bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_ade_ldo(decode_ade_ldo_t *ade_ldo);

	/**
	* @brief Register Configuration
	*
	* @details
	*  - Register      : CNFG_LDO0_B (0x39)
	*  - Bit Fields    : [4]
	*  - Default       : 0x0
	*  - Description   : Operation mode of LDO 0.
	*/
	typedef enum {
		LDO_MD_LDO_MODE,
		LDO_MD_LSW_MODE
	}decode_ldo_md_t;

	/**
	* @brief		Set Operation mode of LDO0.
	*
	* @param[in]	mode 	Operation mode of LDO0 bit to be written.
	*
	* @return		0 on success, error code on failure.
	*/
	int set_ldo_md(decode_ldo_md_t mode);

	/**
	* @brief		Get Operation mode of LDO0.
	*
	* @param[out]	mode 	Operation mode of LDO0 bit to be read.
	*
	* @return		0 on success, error code on failure.
	*/
	int get_ldo_md(decode_ldo_md_t *mode);

	/**
	 * @brief	Disable all interrupts
	 *
	 * @return	0 on success, error code on failure
	 */
	int irq_disable_all();

	/**
	 * @brief		Set Interrupt Handler for a Specific Interrupt ID.
	 *
	 * @param[in]	id reg_bit_reg_bit_int_glbl_t 	id, one of INTR_ID_*.
	 * @param[in]	func 					Interrupt handler function.
	 * @param[in]	cb 						Interrupt handler data.
	 */
	void set_interrupt_handler(reg_bit_int_glbl_t id, interrupt_handler_function func, void *cb);
};
#endif
