/*******************************************************************************
 * Copyright(C) Analog Devices Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files(the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Analog Devices Inc.
 * shall not be used except as stated in the Analog Devices Inc.
 * Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Analog Devices Inc.retains all ownership rights.
 *******************************************************************************
 */

#include <Thread.h>
#include "MAX77643_2.h"
#include <math.h>

#define POST_INTR_WORK_SIGNAL_ID			0x1
#define TO_UINT8							0xFF
#define TO_UINT16							0xFFFF

MAX77643_2::MAX77643_2(I2C *i2c, PinName IRQPin)
{
	if (i2c == NULL)
		return;

	i2c_handler = i2c;
	interrupt_handler_list = new handler[INTM_NUM_OF_BIT] {};

	if (IRQPin != NC) {
		irq_disable_all();
		post_intr_work_thread = new Thread();
		post_intr_work_thread->start(Callback<void()>(this, &MAX77643_2::post_interrupt_work));

		this->irq_pin = new InterruptIn(IRQPin);
		this->irq_pin->fall(Callback<void()>(this, &MAX77643_2::interrupt_handler));
		this->irq_pin->enable_irq();
	} else {
		this->irq_pin = NULL;
	}
}

MAX77643_2::~MAX77643_2()
{
	if (post_intr_work_thread)
		delete post_intr_work_thread;

	if (irq_pin)
		delete irq_pin;

	if (interrupt_handler_list)
		delete [] interrupt_handler_list;
}

int MAX77643_2::read_register(uint8_t reg, uint8_t *value)
{
	int rtn_val;

	if (value == NULL)
		return MAX77643_2_VALUE_NULL;

	rtn_val = i2c_handler->write(MAX77643_2_I2C_ADDRESS, (const char *)&reg, 1, true);
	if (rtn_val != 0)
		return MAX77643_2_WRITE_DATA_FAILED;
	rtn_val = i2c_handler->read(MAX77643_2_I2C_ADDRESS, (char *) value, 1, false);
	if (rtn_val < 0)
		return MAX77643_2_READ_DATA_FAILED;

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::write_register(uint8_t reg, const uint8_t *value)
{
	int rtn_val;
	unsigned char local_data[2];

	if (value == NULL)
		return MAX77643_2_VALUE_NULL;

	local_data[0] = reg;

	memcpy(&local_data[1], value, 1);

	rtn_val = i2c_handler->write(MAX77643_2_I2C_ADDRESS, (const char *)local_data, sizeof(local_data));
	if (rtn_val != MAX77643_2_NO_ERROR)
		return MAX77643_2_WRITE_DATA_FAILED;

	return MAX77643_2_NO_ERROR;
}

#define SET_BIT_FIELD(address, reg_name, bit_field_name, value)                         	\
	int ret_val;                                                                    \
        ret_val = read_register(address, (uint8_t *)&(reg_name));                       \
        if (ret_val) {                                                                  \
        	return ret_val;                                                             \
	}                                                                           	\
	bit_field_name = value;                                                     	\
	ret_val = write_register(address, (uint8_t *)&(reg_name));                      \
	if (ret_val) {                                                                  \
	return ret_val;                                                             \
	}

int MAX77643_2::get_ercflag(reg_bit_ercflag_t bit_field, uint8_t *flag)
{
	int ret;
	reg_ercflag_t reg_ercflag = {0};

	ret = read_register(ERCFLAG, (uint8_t *)&(reg_ercflag));
	if (ret != MAX77643_2_NO_ERROR) return ret;
	
	switch (bit_field)
	{
	case ERCFLAG_TOVLD:
		*flag = (uint8_t)reg_ercflag.bits.tovld;
		break;
	case ERCFLAG_INOVLO:
		*flag = (uint8_t)reg_ercflag.bits.inovlo;
		break;
	case ERCFLAG_INUVLO:
		*flag = (uint8_t)reg_ercflag.bits.inuvlo;
		break;
	case ERCFLAG_MRST_F:
		*flag = (uint8_t)reg_ercflag.bits.mrst_f;
		break;
	case ERCFLAG_SFT_OFF_F:
		*flag = (uint8_t)reg_ercflag.bits.sft_off_f;
		break;
	case ERCFLAG_SFT_CRST_F:
		*flag = (uint8_t)reg_ercflag.bits.sft_crst_f;
		break;
	case ERCFLAG_WDT_EXP_F:
		*flag = (uint8_t)reg_ercflag.bits.wdt_exp_f;
		break;
	case ERCFLAG_SBB_FAULT_F:
		*flag = (uint8_t)reg_ercflag.bits.sbb_fault_f;
		break;
	default:
		ret = MAX77643_2_INVALID_DATA;
		break;		
	}

	return ret;
}

int MAX77643_2::get_stat_glbl(reg_bit_stat_glbl_t bit_field, uint8_t *status)
{
	int ret;
	reg_stat_glbl_t reg_stat_glbl = {0};

	ret = read_register(STAT_GLBL, (uint8_t *)&(reg_stat_glbl));
	if (ret != MAX77643_2_NO_ERROR) return ret;

	switch (bit_field)
	{
	case STAT_GLBL_STAT_IRQ:
		*status = (uint8_t)reg_stat_glbl.bits.stat_irq;
		break;
	case STAT_GLBL_STAT_EN:
		*status = (uint8_t)reg_stat_glbl.bits.stat_en;
		break;
	case STAT_GLBL_TJAL1_S:
		*status = (uint8_t)reg_stat_glbl.bits.tjal1_s;
		break;
	case STAT_GLBL_TJAL2_S:
		*status = (uint8_t)reg_stat_glbl.bits.tjal2_s;
		break;
	case STAT_GLBL_DOD_S:
		*status = (uint8_t)reg_stat_glbl.bits.dod_s;
		break;
	case STAT_GLBL_BOK:
		*status = (uint8_t)reg_stat_glbl.bits.bok;
		break;
	case STAT_GLBL_DIDM:
		*status = (uint8_t)reg_stat_glbl.bits.didm;
		break;
	default:
		ret = MAX77643_2_INVALID_DATA;
		break;		
	}

	return ret;
}

int MAX77643_2::set_interrupt_mask(reg_bit_int_mask_t bit_field, uint8_t maskBit)
{
	int ret;
	uint8_t reg_addr = 0;
	reg_intm_glbl0_t reg_intm_glbl0 = {0};
	reg_intm_glbl1_t reg_intm_glbl1 = {0};
 
	//INTM_GLBL0 (0x04) and INTM_GLBL1 (0x05)
	reg_addr = (uint8_t)floor((static_cast<uint8_t>(bit_field)) / 8) + 0x04; 
	
	if (reg_addr == INTM_GLBL0)
		ret = read_register(INTM_GLBL0, (uint8_t *)&(reg_intm_glbl0));
	else if (reg_addr == INTM_GLBL1)
		ret = read_register(INTM_GLBL1, (uint8_t *)&(reg_intm_glbl1));
	else 
		return MAX77643_2_INVALID_DATA;
	
	if (ret != MAX77643_2_NO_ERROR) return ret;
	
	switch (bit_field)
	{
	case INTM_GLBL0_GPI0_FM:
		reg_intm_glbl0.bits.gpi0_fm = maskBit;
		break;
	case INTM_GLBL0_GPI0_RM:
		reg_intm_glbl0.bits.gpi0_rm = maskBit;
		break;
	case INTM_GLBL0_nEN_FM:
		reg_intm_glbl0.bits.nen_fm = maskBit;
		break;
	case INTM_GLBL0_nEN_RM:
		reg_intm_glbl0.bits.nen_rm = maskBit;
		break;
	case INTM_GLBL0_TJAL1_RM:
		reg_intm_glbl0.bits.tjal1_rm = maskBit;
		break;
	case INTM_GLBL0_TJAL2_RM:
		reg_intm_glbl0.bits.tjal2_rm = maskBit;
		break;
	case INTM_GLBL0_DOD_RM:
		reg_intm_glbl0.bits.dod_rm = maskBit;
		break;
	case INTM_GLBL1_GPI1_FM:
		reg_intm_glbl1.bits.gpi1_fm = maskBit;
		break;
	case INTM_GLBL1_GPI1_RM:
		reg_intm_glbl1.bits.gpi1_rm = maskBit;
		break;
	case INTM_GLBL1_SBB0_FM:
		reg_intm_glbl1.bits.sbb0_fm = maskBit;
		break;
	case INTM_GLBL1_SBB1_FM:
		reg_intm_glbl1.bits.sbb1_fm = maskBit;
		break;
	case INTM_GLBL1_SBB2_FM:
		reg_intm_glbl1.bits.sbb2_fm = maskBit;
		break;
	case INTM_GLBL1_LDO_M:
		reg_intm_glbl1.bits.ldo_m = maskBit;
		break;
	default:
		return MAX77643_2_INVALID_DATA;
		break;		
	}
    
	if (reg_addr == INTM_GLBL0)
		return write_register(INTM_GLBL0, (uint8_t *)&(reg_intm_glbl0));
	else if (reg_addr == INTM_GLBL1)
		return write_register(INTM_GLBL1, (uint8_t *)&(reg_intm_glbl1));
	else 
		return MAX77643_2_INVALID_DATA;
}

int MAX77643_2::get_interrupt_mask(reg_bit_int_mask_t bit_field, uint8_t *maskBit)
{
	int ret;
	uint8_t reg_addr = 0;
	reg_intm_glbl0_t reg_intm_glbl0 = {0};
	reg_intm_glbl1_t reg_intm_glbl1 = {0};
 
	//INTM_GLBL0 (0x04) and INTM_GLBL1 (0x05)
	reg_addr = (uint8_t)floor((static_cast<uint8_t>(bit_field)) / 8) + 0x04; 
	
	if (reg_addr == INTM_GLBL0)
		ret = read_register(INTM_GLBL0, (uint8_t *)&(reg_intm_glbl0));
	else if (reg_addr == INTM_GLBL1)
		ret = read_register(INTM_GLBL1, (uint8_t *)&(reg_intm_glbl1));
	else 
		return MAX77643_2_INVALID_DATA;
	
	if (ret != MAX77643_2_NO_ERROR) return ret;

	switch (bit_field)
	{	
	case INTM_GLBL0_GPI0_FM:
		*maskBit = (uint8_t)reg_intm_glbl0.bits.gpi0_fm;
		break;
	case INTM_GLBL0_GPI0_RM:
		*maskBit = (uint8_t)reg_intm_glbl0.bits.gpi0_rm;
		break;
	case INTM_GLBL0_nEN_FM:
		*maskBit = (uint8_t)reg_intm_glbl0.bits.nen_fm;
		break;
	case INTM_GLBL0_nEN_RM:
		*maskBit = (uint8_t)reg_intm_glbl0.bits.nen_rm;
		break;
	case INTM_GLBL0_TJAL1_RM:
		*maskBit = (uint8_t)reg_intm_glbl0.bits.tjal1_rm;
		break;
	case INTM_GLBL0_TJAL2_RM:
		*maskBit = (uint8_t)reg_intm_glbl0.bits.tjal2_rm;
		break;
	case INTM_GLBL0_DOD_RM:
		*maskBit = (uint8_t)reg_intm_glbl0.bits.dod_rm;
		break;	
	case INTM_GLBL1_GPI1_FM:
		*maskBit = (uint8_t)reg_intm_glbl1.bits.gpi1_fm;
		break;
	case INTM_GLBL1_GPI1_RM:
		*maskBit = (uint8_t)reg_intm_glbl1.bits.gpi1_rm;
		break;
	case INTM_GLBL1_SBB0_FM:
		*maskBit = (uint8_t)reg_intm_glbl1.bits.sbb0_fm;
		break;
	case INTM_GLBL1_SBB1_FM:
		*maskBit = (uint8_t)reg_intm_glbl1.bits.sbb1_fm;
		break;
	case INTM_GLBL1_SBB2_FM:
		*maskBit = (uint8_t)reg_intm_glbl1.bits.sbb2_fm;
		break;
	case INTM_GLBL1_LDO_M:
		*maskBit = (uint8_t)reg_intm_glbl1.bits.ldo_m;
		break;
	default:
		return MAX77643_2_INVALID_DATA;
		break;		
	}

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::set_cnfg_glbl0(reg_bit_cnfg_glbl0_t bit_field, uint8_t config)
{	
	int ret;
	reg_cnfg_glbl0_t reg_cnfg_glbl0 = {0};
	
	ret = read_register(CNFG_GLBL0, (uint8_t *)&(reg_cnfg_glbl0));
	if (ret != MAX77643_2_NO_ERROR) return ret;
	
	switch (bit_field)
	{
	case CNFG_GLBL0_SFT_CTRL:
		reg_cnfg_glbl0.bits.sft_ctrl = config;
		break;
	case CNFG_GLBL0_DBEN_nEN:
		reg_cnfg_glbl0.bits.dben_nen = config;
		break;
	case CNFG_GLBL0_nEN_MODE:
		reg_cnfg_glbl0.bits.nen_mode = config;
		break;
	case CNFG_GLBL0_SBIA_LPM:
		reg_cnfg_glbl0.bits.sbia_lpm = config;
		break;
	case CNFG_GLBL0_T_MRST:
		reg_cnfg_glbl0.bits.t_mrst = config;
		break;
	case CNFG_GLBL0_PU_DIS:
		reg_cnfg_glbl0.bits.pu_dis = config;
		break;
	default:
		return MAX77643_2_INVALID_DATA;
		break;		
	}
                                  	
	return write_register(CNFG_GLBL0, (uint8_t *)&(reg_cnfg_glbl0));
}

int MAX77643_2::get_cnfg_glbl0(reg_bit_cnfg_glbl0_t bit_field, uint8_t *config)
{
	int ret;
	reg_cnfg_glbl0_t reg_cnfg_glbl0 = {0};

	ret = read_register(CNFG_GLBL0, (uint8_t *)&(reg_cnfg_glbl0));
	if (ret != MAX77643_2_NO_ERROR) return ret;

	switch (bit_field)
	{
	case CNFG_GLBL0_SFT_CTRL:
		*config = (uint8_t)reg_cnfg_glbl0.bits.sft_ctrl;
		break;
	case CNFG_GLBL0_DBEN_nEN:
		*config = (uint8_t)reg_cnfg_glbl0.bits.dben_nen;
		break;
	case CNFG_GLBL0_nEN_MODE:
		*config = (uint8_t)reg_cnfg_glbl0.bits.nen_mode;
		break;
	case CNFG_GLBL0_SBIA_LPM:
		*config = (uint8_t)reg_cnfg_glbl0.bits.sbia_lpm;
		break;
	case CNFG_GLBL0_T_MRST:
		*config = (uint8_t)reg_cnfg_glbl0.bits.t_mrst;
		break;
	case CNFG_GLBL0_PU_DIS:
		*config = (uint8_t)reg_cnfg_glbl0.bits.pu_dis;
		break;
	default:
		ret = MAX77643_2_INVALID_DATA;
		break;		
	}

    return ret;
}

int MAX77643_2::set_cnfg_glbl1(reg_bit_cnfg_glbl1_t bit_field, uint8_t config)
{	
	int ret;
	reg_cnfg_glbl1_t reg_cnfg_glbl1;
	
	ret = read_register(CNFG_GLBL1, (uint8_t *)&(reg_cnfg_glbl1));
	if (ret != MAX77643_2_NO_ERROR) return ret;
	
	switch (bit_field)
	{
	case CNFG_GLBL1_AUTO_WKT:
		reg_cnfg_glbl1.bits.auto_wkt = config;
		break;
	case CNFG_GLBL1_SBB_F_SHUTDN:
		reg_cnfg_glbl1.bits.sbb_f_shutdn = config;
		break;
	default:
		return MAX77643_2_INVALID_DATA;
		break;		
	}
                                  	
	return write_register(CNFG_GLBL1, (uint8_t *)&(reg_cnfg_glbl1));
}

int MAX77643_2::get_cnfg_glbl1(reg_bit_cnfg_glbl1_t bit_field, uint8_t *config)
{
	int ret;
	reg_cnfg_glbl1_t reg_cnfg_glbl1 = {0};

	ret = read_register(CNFG_GLBL1, (uint8_t *)&(reg_cnfg_glbl1));
	if (ret != MAX77643_2_NO_ERROR) return ret;

	switch (bit_field)
	{
	case CNFG_GLBL1_AUTO_WKT:
		*config = (uint8_t)reg_cnfg_glbl1.bits.auto_wkt;
		break;
	case CNFG_GLBL1_SBB_F_SHUTDN:
		*config = (uint8_t)reg_cnfg_glbl1.bits.sbb_f_shutdn;
		break;
	default:
		ret = MAX77643_2_INVALID_DATA;
		break;		
	}

	return ret;
}

int MAX77643_2::set_cnfg_gpio(reg_bit_cnfg_gpio_t bit_field, uint8_t channel, uint8_t config)
{
	int ret;
	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
	
	if (channel == 0)
	{
		ret = read_register(CNFG_GPIO0, (uint8_t *)&(reg_cnfg_gpio0));
		if (ret != MAX77643_2_NO_ERROR) return ret;
		
		switch (bit_field)
		{
			case CNFG_GPIO_DIR:
				reg_cnfg_gpio0.bits.gpo_dir = config;
				break;
			case CNFG_GPIO_DI:
				reg_cnfg_gpio0.bits.gpo_di = config;
				break;
			case CNFG_GPIO_DRV:
				reg_cnfg_gpio0.bits.gpo_drv = config;
				break;
			case CNFG_GPIO_DO:
				reg_cnfg_gpio0.bits.gpo_do = config;
				break;
			case CNFG_GPIO_DBEN_GPI:
				reg_cnfg_gpio0.bits.dben_gpi = config;
				break;
			case CNFG_GPIO_ALT_GPIO:
				reg_cnfg_gpio0.bits.alt_gpio = config;
				break;
			default:
				return MAX77643_2_INVALID_DATA;
				break;		
		}
		
		return write_register(CNFG_GPIO0, (uint8_t *)&(reg_cnfg_gpio0));
	}
	else if (channel == 1)
	{
		ret = read_register(CNFG_GPIO1, (uint8_t *)&(reg_cnfg_gpio1));
		if (ret != MAX77643_2_NO_ERROR) return ret;
		
		switch (bit_field)
		{
			case CNFG_GPIO_DIR:
				reg_cnfg_gpio1.bits.gpo_dir = config;
				break;
			case CNFG_GPIO_DI:
				reg_cnfg_gpio1.bits.gpo_di = config;
				break;
			case CNFG_GPIO_DRV:
				reg_cnfg_gpio1.bits.gpo_drv = config;
				break;
			case CNFG_GPIO_DO:
				reg_cnfg_gpio1.bits.gpo_do = config;
				break;
			case CNFG_GPIO_DBEN_GPI:
				reg_cnfg_gpio1.bits.dben_gpi = config;
				break;
			case CNFG_GPIO_ALT_GPIO:
				reg_cnfg_gpio1.bits.alt_gpio = config;
				break;
			default:
				return MAX77643_2_INVALID_DATA;
				break;		
		}
		
		return write_register(CNFG_GPIO1, (uint8_t *)&(reg_cnfg_gpio1));
	}
	else {
		return MAX77643_2_INVALID_DATA;
	}
}

int MAX77643_2::get_cnfg_gpio(reg_bit_cnfg_gpio_t bit_field, uint8_t channel, uint8_t *config)
{
	int ret;
	reg_cnfg_gpio0_t reg_cnfg_gpio0 = {0};
	reg_cnfg_gpio1_t reg_cnfg_gpio1 = {0};
	
	if (channel == 0)
	{
		ret = read_register(CNFG_GPIO0, (uint8_t *)&(reg_cnfg_gpio0));
		if (ret != MAX77643_2_NO_ERROR) return ret;
	
		switch (bit_field)
		{
			case CNFG_GPIO_DIR:
				*config = (uint8_t)reg_cnfg_gpio0.bits.gpo_dir;
				break;
			case CNFG_GPIO_DI:
				*config = (uint8_t)reg_cnfg_gpio0.bits.gpo_di;
				break;
			case CNFG_GPIO_DRV:
				*config = (uint8_t)reg_cnfg_gpio0.bits.gpo_drv;
				break;
			case CNFG_GPIO_DO:
				*config = (uint8_t)reg_cnfg_gpio0.bits.gpo_do;
				break;
			case CNFG_GPIO_DBEN_GPI:
				*config = (uint8_t)reg_cnfg_gpio0.bits.dben_gpi;
				break;
			case CNFG_GPIO_ALT_GPIO:
				*config = (uint8_t)reg_cnfg_gpio0.bits.alt_gpio;
				break;
			default:
				return MAX77643_2_INVALID_DATA;
				break;		
		}
	}
	else if (channel == 1)
	{
		ret = read_register(CNFG_GPIO1, (uint8_t *)&(reg_cnfg_gpio1));
		if (ret != MAX77643_2_NO_ERROR) return ret;
		
		switch (bit_field)
		{
			case CNFG_GPIO_DIR:
				*config = (uint8_t)reg_cnfg_gpio1.bits.gpo_dir;
				break;
			case CNFG_GPIO_DI:
				*config = (uint8_t)reg_cnfg_gpio1.bits.gpo_di;
				break;
			case CNFG_GPIO_DRV:
				*config = (uint8_t)reg_cnfg_gpio1.bits.gpo_drv;
				break;
			case CNFG_GPIO_DO:
				*config = (uint8_t)reg_cnfg_gpio1.bits.gpo_do;
				break;
			case CNFG_GPIO_DBEN_GPI:
				*config = (uint8_t)reg_cnfg_gpio1.bits.dben_gpi;
				break;
			case CNFG_GPIO_ALT_GPIO:
				*config = (uint8_t)reg_cnfg_gpio1.bits.alt_gpio;
				break;
			default:
				return MAX77643_2_INVALID_DATA;
				break;		
		}
	}
	else {
		return MAX77643_2_INVALID_DATA;
	}

	return ret;
}

int MAX77643_2::get_cid(void) {
	char rbuf[1] = {0};
	int ret;

	ret = read_register(CID, (uint8_t *)&(rbuf));
	if (ret != MAX77643_2_NO_ERROR) return ret;
 
	return *rbuf;
}

int MAX77643_2::set_cnfg_wdt(reg_bit_cnfg_wdt_t bit_field, uint8_t config)
{
	int ret;
	reg_cnfg_wdt_t reg_cnfg_wdt = {0};
	
	ret = read_register(CNFG_WDT, (uint8_t *)&(reg_cnfg_wdt));
	if (ret != MAX77643_2_NO_ERROR) return ret;
	
	switch (bit_field)
	{
		case CNFG_WDT_WDT_LOCK:
			reg_cnfg_wdt.bits.wdt_lock = config;
			break;
		case CNFG_WDT_WDT_EN:
			reg_cnfg_wdt.bits.wdt_en = config;
			break;
		case CNFG_WDT_WDT_CLR:
			reg_cnfg_wdt.bits.wdt_clr = config;
			break;
		case CNFG_WDT_WDT_MODE:
			reg_cnfg_wdt.bits.wdt_mode = config;
			break;
		case CNFG_WDT_WDT_PER:
			reg_cnfg_wdt.bits.wdt_per = config;
			break;
		default:
			return MAX77643_2_INVALID_DATA;
			break;		
	}

	return write_register(CNFG_WDT, (uint8_t *)&(reg_cnfg_wdt));
}

int MAX77643_2::get_cnfg_wdt(reg_bit_cnfg_wdt_t bit_field, uint8_t *config)
{
	int ret;
	reg_cnfg_wdt_t reg_cnfg_wdt = {0};

	ret = read_register(CNFG_WDT, (uint8_t *)&(reg_cnfg_wdt));
	if (ret != MAX77643_2_NO_ERROR) return ret;

	switch (bit_field)
	{
		case CNFG_WDT_WDT_LOCK:
			*config = (uint8_t)reg_cnfg_wdt.bits.wdt_lock;
			break;
		case CNFG_WDT_WDT_EN:
			*config = (uint8_t)reg_cnfg_wdt.bits.wdt_en;
			break;
		case CNFG_WDT_WDT_CLR:
			*config = (uint8_t)reg_cnfg_wdt.bits.wdt_clr;
			break;
		case CNFG_WDT_WDT_MODE:
			*config = (uint8_t)reg_cnfg_wdt.bits.wdt_mode;
			break;
		case CNFG_WDT_WDT_PER:
			*config = (uint8_t)reg_cnfg_wdt.bits.wdt_per;
			break;
		default:
			return MAX77643_2_INVALID_DATA;
			break;		
	}

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::set_cnfg_sbb_top(reg_bit_cnfg_sbb_top_t bit_field, uint8_t config)
{
	int ret;
	reg_cnfg_sbb_top_t reg_cnfg_sbb_top = {0};
	
	ret = read_register(CNFG_SBB_TOP, (uint8_t *)&(reg_cnfg_sbb_top));
	if (ret != MAX77643_2_NO_ERROR) return ret;
	
	switch (bit_field)
	{
		case CNFG_SBB_TOP_DRV_SBB:
			reg_cnfg_sbb_top.bits.drv_sbb = config;
			break;
		case CNFG_SBB_TOP_DIS_LPM:
			reg_cnfg_sbb_top.bits.dis_lpm = config;
			break;
		default:
			return MAX77643_2_INVALID_DATA;
		break;
	}

	return write_register(CNFG_SBB_TOP, (uint8_t *)&(reg_cnfg_sbb_top));
}

int MAX77643_2::get_cnfg_sbb_top(reg_bit_cnfg_sbb_top_t bit_field, uint8_t *config)
{
	int ret;
	reg_cnfg_sbb_top_t reg_cnfg_sbb_top = {0};

	ret = read_register(CNFG_SBB_TOP, (uint8_t *)&(reg_cnfg_sbb_top));
	if (ret != MAX77643_2_NO_ERROR) return ret;

	switch (bit_field)
	{
		case CNFG_SBB_TOP_DRV_SBB:
			*config = (uint8_t)reg_cnfg_sbb_top.bits.drv_sbb;
			break;
		case CNFG_SBB_TOP_DIS_LPM:
			*config = (uint8_t)reg_cnfg_sbb_top.bits.dis_lpm;
			break;
		default:
			return MAX77643_2_INVALID_DATA;
		break;
	}

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::set_tv_sbb(uint8_t channel, float voltV)
{
	uint8_t value;
	reg_cnfg_sbb0_a_t reg_cnfg_sbb0_a = {0};
	reg_cnfg_sbb1_a_t reg_cnfg_sbb1_a = {0};
	reg_cnfg_sbb2_a_t reg_cnfg_sbb2_a = {0};
	float voltmV = voltV * 1000;
	
	if (voltmV < 500) voltmV = 500;
	else if (voltmV > 5500) voltmV = 5500;

	value = (voltmV - 500) / 25;

	if (channel == 0) {
		SET_BIT_FIELD(CNFG_SBB0_A, reg_cnfg_sbb0_a, reg_cnfg_sbb0_a.bits.tv_sbb0, value);
	}		
	else if (channel == 1) {
		SET_BIT_FIELD(CNFG_SBB1_A, reg_cnfg_sbb1_a, reg_cnfg_sbb1_a.bits.tv_sbb1, value);
	}
	else if (channel == 2) {
		SET_BIT_FIELD(CNFG_SBB2_A, reg_cnfg_sbb2_a, reg_cnfg_sbb2_a.bits.tv_sbb2, value);
	}
	else {
		return MAX77643_2_INVALID_DATA;
	}
	
	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::get_tv_sbb(uint8_t channel, float *voltV)
{
	int ret;
	uint8_t bit_value;
	reg_cnfg_sbb0_a_t reg_cnfg_sbb0_a = {0};
	reg_cnfg_sbb1_a_t reg_cnfg_sbb1_a = {0};
	reg_cnfg_sbb2_a_t reg_cnfg_sbb2_a = {0};

	if (channel == 0) { 
		ret = read_register(CNFG_SBB0_A, (uint8_t *)&(reg_cnfg_sbb0_a));
		if (ret != MAX77643_2_NO_ERROR) return ret;
		
		bit_value = (uint8_t)reg_cnfg_sbb0_a.bits.tv_sbb0;
	}
	else if (channel == 1) {
		ret = read_register(CNFG_SBB1_A, (uint8_t *)&(reg_cnfg_sbb1_a));
		if (ret != MAX77643_2_NO_ERROR) return ret;
	
		bit_value = (uint8_t)reg_cnfg_sbb1_a.bits.tv_sbb1;
	}
	else if (channel == 2) {
		ret = read_register(CNFG_SBB2_A, (uint8_t *)&(reg_cnfg_sbb2_a));
		if (ret != MAX77643_2_NO_ERROR) return ret;
	
		bit_value = (uint8_t)reg_cnfg_sbb2_a.bits.tv_sbb2;
	}
	else return MAX77643_2_INVALID_DATA;
	
	if (bit_value > 200) bit_value = 200;
	*voltV = (bit_value * 0.025f) + 0.5f;

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::set_op_mode(uint8_t channel, decode_op_mode_t mode)
{	
	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
	
	if (channel == 0) {
		SET_BIT_FIELD(CNFG_SBB0_B, reg_cnfg_sbb0_b, reg_cnfg_sbb0_b.bits.op_mode0, mode);
	}
	else if (channel == 1) {
		SET_BIT_FIELD(CNFG_SBB1_B, reg_cnfg_sbb1_b, reg_cnfg_sbb1_b.bits.op_mode1, mode);
	}
	else if (channel == 2) {
		SET_BIT_FIELD(CNFG_SBB2_B, reg_cnfg_sbb2_b, reg_cnfg_sbb2_b.bits.op_mode2, mode);
	}
	else {
		return MAX77643_2_INVALID_DATA;
	}
	
	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::get_op_mode(uint8_t channel, decode_op_mode_t *mode)
{
	int ret;
	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};

	if (channel == 0) {
		ret = read_register(CNFG_SBB0_B, (uint8_t *)&(reg_cnfg_sbb0_b));
		if (ret != MAX77643_2_NO_ERROR) return ret;

		*mode = (decode_op_mode_t)reg_cnfg_sbb0_b.bits.op_mode0;
	}
	else if (channel == 1) {
		ret = read_register(CNFG_SBB1_B, (uint8_t *)&(reg_cnfg_sbb1_b));
		if (ret != MAX77643_2_NO_ERROR) return ret;

		*mode = (decode_op_mode_t)reg_cnfg_sbb1_b.bits.op_mode1;
	}
	else if (channel == 2) {
		ret = read_register(CNFG_SBB2_B, (uint8_t *)&(reg_cnfg_sbb2_b));
		if (ret != MAX77643_2_NO_ERROR) return ret;

		*mode = (decode_op_mode_t)reg_cnfg_sbb2_b.bits.op_mode2;
	}
	else { 
		return MAX77643_2_INVALID_DATA;
	}

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::set_ip_sbb(uint8_t channel, decode_ip_sbb_t ip_sbb)
{	
	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
	
	if (channel == 0) {
		SET_BIT_FIELD(CNFG_SBB0_B, reg_cnfg_sbb0_b, reg_cnfg_sbb0_b.bits.ip_sbb0, ip_sbb);
	}
	else if (channel == 1) {
		SET_BIT_FIELD(CNFG_SBB1_B, reg_cnfg_sbb1_b, reg_cnfg_sbb1_b.bits.ip_sbb1, ip_sbb);
	}
	else if (channel == 2) {
		SET_BIT_FIELD(CNFG_SBB2_B, reg_cnfg_sbb2_b, reg_cnfg_sbb2_b.bits.ip_sbb2, ip_sbb);
	}
	else {
		return MAX77643_2_INVALID_DATA;
	}
	
	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::get_ip_sbb(uint8_t channel, decode_ip_sbb_t *ip_sbb)
{
	int ret;
	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};

	if (channel == 0) {
		ret = read_register(CNFG_SBB0_B, (uint8_t *)&(reg_cnfg_sbb0_b));
		if (ret != MAX77643_2_NO_ERROR) return ret;

		*ip_sbb = (decode_ip_sbb_t)reg_cnfg_sbb0_b.bits.ip_sbb0;
	}
	else if (channel == 1) {
		ret = read_register(CNFG_SBB1_B, (uint8_t *)&(reg_cnfg_sbb1_b));
		if (ret != MAX77643_2_NO_ERROR) return ret;

		*ip_sbb = (decode_ip_sbb_t)reg_cnfg_sbb1_b.bits.ip_sbb1;
	}
	else if (channel == 2) {
		ret = read_register(CNFG_SBB2_B, (uint8_t *)&(reg_cnfg_sbb2_b));
		if (ret != MAX77643_2_NO_ERROR) return ret;

		*ip_sbb = (decode_ip_sbb_t)reg_cnfg_sbb2_b.bits.ip_sbb2;
	}
	else {
		return MAX77643_2_INVALID_DATA;
	}

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::set_ade_sbb(uint8_t channel, decode_ade_sbb_t ade_sbb)
{	
	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
	
	if (channel == 0) {
		SET_BIT_FIELD(CNFG_SBB0_B, reg_cnfg_sbb0_b, reg_cnfg_sbb0_b.bits.ade_sbb0, ade_sbb);
	}
	else if (channel == 1) {
		SET_BIT_FIELD(CNFG_SBB1_B, reg_cnfg_sbb1_b, reg_cnfg_sbb1_b.bits.ade_sbb1, ade_sbb);
	}
	else if (channel == 2) {
		SET_BIT_FIELD(CNFG_SBB2_B, reg_cnfg_sbb2_b, reg_cnfg_sbb2_b.bits.ade_sbb2, ade_sbb);
	}
	else {
		return MAX77643_2_INVALID_DATA;
	}
	
	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::get_ade_sbb(uint8_t channel, decode_ade_sbb_t *ade_sbb)
{
	int ret;
	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};

	if (channel == 0) {
		ret = read_register(CNFG_SBB0_B, (uint8_t *)&(reg_cnfg_sbb0_b));
		if (ret != MAX77643_2_NO_ERROR) return ret;

		*ade_sbb = (decode_ade_sbb_t)reg_cnfg_sbb0_b.bits.ade_sbb0;
	}
	else if (channel == 1) {
		ret = read_register(CNFG_SBB1_B, (uint8_t *)&(reg_cnfg_sbb1_b));
		if (ret != MAX77643_2_NO_ERROR) return ret;

		*ade_sbb = (decode_ade_sbb_t)reg_cnfg_sbb1_b.bits.ade_sbb1;
	}
	else if (channel == 2) {
		ret = read_register(CNFG_SBB2_B, (uint8_t *)&(reg_cnfg_sbb2_b));
		if (ret != MAX77643_2_NO_ERROR) return ret;

		*ade_sbb = (decode_ade_sbb_t)reg_cnfg_sbb2_b.bits.ade_sbb2;
	}
	else {
		return MAX77643_2_INVALID_DATA;
	}

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::set_en_sbb(uint8_t channel, decode_en_sbb_t en_sbb)
{	
	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};
	
	if (channel == 0) {
		SET_BIT_FIELD(CNFG_SBB0_B, reg_cnfg_sbb0_b, reg_cnfg_sbb0_b.bits.en_sbb0, en_sbb);
	}
	else if (channel == 1) {
		SET_BIT_FIELD(CNFG_SBB1_B, reg_cnfg_sbb1_b, reg_cnfg_sbb1_b.bits.en_sbb1, en_sbb);
	}
	else if (channel == 2) {
		SET_BIT_FIELD(CNFG_SBB2_B, reg_cnfg_sbb2_b, reg_cnfg_sbb2_b.bits.en_sbb2, en_sbb);
	}
	else {
		return MAX77643_2_INVALID_DATA;
	}
	
	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::get_en_sbb(uint8_t channel, decode_en_sbb_t *en_sbb)
{
	int ret;
	reg_cnfg_sbb0_b_t reg_cnfg_sbb0_b = {0};
	reg_cnfg_sbb1_b_t reg_cnfg_sbb1_b = {0};
	reg_cnfg_sbb2_b_t reg_cnfg_sbb2_b = {0};

	if (channel == 0) {
		ret = read_register(CNFG_SBB0_B, (uint8_t *)&(reg_cnfg_sbb0_b));
		if (ret != MAX77643_2_NO_ERROR) return ret;

		*en_sbb = (decode_en_sbb_t)reg_cnfg_sbb0_b.bits.en_sbb0;
	}
	else if (channel == 1) {
		ret = read_register(CNFG_SBB1_B, (uint8_t *)&(reg_cnfg_sbb1_b));
		if (ret != MAX77643_2_NO_ERROR) return ret;

		*en_sbb = (decode_en_sbb_t)reg_cnfg_sbb1_b.bits.en_sbb1;
	}
	else if (channel == 2) {
		ret = read_register(CNFG_SBB2_B, (uint8_t *)&(reg_cnfg_sbb2_b));
		if (ret != MAX77643_2_NO_ERROR) return ret;

		*en_sbb = (decode_en_sbb_t)reg_cnfg_sbb2_b.bits.en_sbb2;
	}
	else return MAX77643_2_INVALID_DATA;

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::set_tv_sbb_dvs(float voltV)
{
	uint8_t value;
	reg_cnfg_dvs_sbb0_a_t reg_cnfg_dvs_sbb0_a = {0};
	float voltmV = voltV * 1000;

	if (voltmV < 500) voltmV = 500;
	else if (voltmV > 5500) voltmV = 5500;	

	value = (voltmV - 500) / 25;
	
	SET_BIT_FIELD(CNFG_DVS_SBB0_A, reg_cnfg_dvs_sbb0_a, reg_cnfg_dvs_sbb0_a.bits.tv_sbb0_dvs, value);
	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::get_tv_sbb_dvs(float *voltV)
{
	int ret;
	uint8_t bit_value;
	reg_cnfg_dvs_sbb0_a_t reg_cnfg_dvs_sbb0_a = {0};

	ret = read_register(CNFG_DVS_SBB0_A, (uint8_t *)&(reg_cnfg_dvs_sbb0_a));
	if (ret != MAX77643_2_NO_ERROR) return ret;

	bit_value = (uint8_t)reg_cnfg_dvs_sbb0_a.bits.tv_sbb0_dvs;
	
	if (bit_value > 200) bit_value = 200;
	
	*voltV = (bit_value * 0.025f) + 0.5f;
	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::set_tv_ldo(float voltV)
{
	int ret;
	uint8_t value;
	reg_cnfg_ldo0_a_t reg_cnfg_ldo0_a = {0};
	float voltmV = voltV * 1000;
	const float offsetmV = 1325;
	const float incrementmV = 25;
	float lower_limit_voltmV = 500, upper_limit_voltmV = 3675;

	ret = read_register(CNFG_LDO0_A, (uint8_t *)&(reg_cnfg_ldo0_a));
	if (ret != MAX77643_2_NO_ERROR) return ret;

	if ((uint8_t)reg_cnfg_ldo0_a.bits.tv_ofs_ldo == 1) { // 1.325V Offset
		lower_limit_voltmV += offsetmV;
		upper_limit_voltmV += offsetmV;
	}

	voltmV	= (voltmV < lower_limit_voltmV) ? lower_limit_voltmV : upper_limit_voltmV;
	value	= (voltmV - lower_limit_voltmV) / incrementmV;

	SET_BIT_FIELD(CNFG_LDO0_A, reg_cnfg_ldo0_a, reg_cnfg_ldo0_a.bits.tv_ldo0, value);

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::get_tv_ldo(float *voltV)
{
	int ret;
	uint8_t bit_value;
	reg_cnfg_ldo0_a_t reg_cnfg_ldo0_a = {0};
	float lower_limitV		= 0.5f;
	const float incrementV	= 0.025f;
	const float offsetV		= 1.325f;

	ret = read_register(CNFG_LDO0_A, (uint8_t *)&(reg_cnfg_ldo0_a));
	if (ret != MAX77643_2_NO_ERROR) return ret;

	bit_value = (uint8_t)reg_cnfg_ldo0_a.bits.tv_ldo0;
	if ((uint8_t)reg_cnfg_ldo0_a.bits.tv_ofs_ldo == 0) //No Offset
		*voltV = (bit_value * incrementV) + lower_limitV;
	else //1.325V Offset 
		*voltV = (bit_value * incrementV) + (lower_limitV + offsetV);

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::set_tv_ofs_ldo(decode_tv_ofs_ldo_t offset)
{	
	reg_cnfg_ldo0_a_t reg_cnfg_ldo0_a = {0};
	
	SET_BIT_FIELD(CNFG_LDO0_A, reg_cnfg_ldo0_a, reg_cnfg_ldo0_a.bits.tv_ofs_ldo, offset);
	
	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::get_tv_ofs_ldo(decode_tv_ofs_ldo_t *offset)
{
	int ret;
	reg_cnfg_ldo0_a_t reg_cnfg_ldo0_a = {0};

	ret = read_register(CNFG_LDO0_A, (uint8_t *)&(reg_cnfg_ldo0_a));
	if (ret != MAX77643_2_NO_ERROR) return ret;

	*offset = (decode_tv_ofs_ldo_t)reg_cnfg_ldo0_a.bits.tv_ofs_ldo;

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::set_en_ldo(decode_en_ldo_t en_ldo)
{	
	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};
	
	SET_BIT_FIELD(CNFG_LDO0_B, reg_cnfg_ldo0_b, reg_cnfg_ldo0_b.bits.en_ldo, en_ldo);
	
	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::get_en_ldo(decode_en_ldo_t *en_ldo)
{
	int ret;
	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};

	ret = read_register(CNFG_LDO0_B, (uint8_t *)&(reg_cnfg_ldo0_b));
	if (ret != MAX77643_2_NO_ERROR) return ret;

	*en_ldo = (decode_en_ldo_t)reg_cnfg_ldo0_b.bits.en_ldo;

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::set_ade_ldo(decode_ade_ldo_t ade_ldo)
{	
	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};
	
	SET_BIT_FIELD(CNFG_LDO0_B, reg_cnfg_ldo0_b, reg_cnfg_ldo0_b.bits.ade_ldo, ade_ldo);
	
	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::get_ade_ldo(decode_ade_ldo_t *ade_ldo)
{
	int ret;
	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};

	ret = read_register(CNFG_LDO0_B, (uint8_t *)&(reg_cnfg_ldo0_b));
	if (ret != MAX77643_2_NO_ERROR) return ret;

	*ade_ldo = (decode_ade_ldo_t)reg_cnfg_ldo0_b.bits.ade_ldo;

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::set_ldo_md(decode_ldo_md_t mode)
{	
	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};
	
	SET_BIT_FIELD(CNFG_LDO0_B, reg_cnfg_ldo0_b, reg_cnfg_ldo0_b.bits.ldo_md, mode);
	
	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::get_ldo_md(decode_ldo_md_t *mode)
{
	int ret;
	reg_cnfg_ldo0_b_t reg_cnfg_ldo0_b = {0};

	ret = read_register(CNFG_LDO0_B, (uint8_t *)&(reg_cnfg_ldo0_b));
	if (ret != MAX77643_2_NO_ERROR) return ret;

	*mode = (decode_ldo_md_t)reg_cnfg_ldo0_b.bits.ldo_md;

	return MAX77643_2_NO_ERROR;
}

int MAX77643_2::irq_disable_all()
{
	int ret;
	uint8_t reg = 0;
	uint8_t status = 0;

	//Disable Masks in INTM_GLBL1
	ret = write_register(INTM_GLBL1, &reg);
	if (ret != MAX77643_2_NO_ERROR) return ret;

	//Disable Masks in INTM_GLBL0
	ret = write_register(INTM_GLBL0, &reg);
	if (ret != MAX77643_2_NO_ERROR) return ret;

	// Clear Interrupt Flags in INT_GLBL1
	ret = read_register(INT_GLBL1, &status);
	if (ret != MAX77643_2_NO_ERROR) return ret;

	// Clear Interrupt Flags in INT_GLBL0
	ret = read_register(INT_GLBL0, &status);
	if (ret != MAX77643_2_NO_ERROR) return ret;

	return MAX77643_2_NO_ERROR;
}

void MAX77643_2::set_interrupt_handler(reg_bit_int_glbl_t id, interrupt_handler_function func, void *cb)
{
	interrupt_handler_list[id].func = func;
	interrupt_handler_list[id].cb = cb;
}

void MAX77643_2::post_interrupt_work()
{
	int ret;
	uint8_t reg = 0,  inten = 0, not_inten = 0, mask = 0;

	while (true) {

		ThisThread::flags_wait_any(POST_INTR_WORK_SIGNAL_ID);

		// Check Interrupt Flags in INT_GLBL0
		ret = read_register(INT_GLBL0, &reg);
		if (ret != MAX77643_2_NO_ERROR) return;

		ret = read_register(INTM_GLBL0, &inten);
		if (ret != MAX77643_2_NO_ERROR) return;

		not_inten = ~inten; // 0 means unmasked.

		for (int i = 0; i < INT_GLBL1_GPI1_F; i++) {
			mask = (1 << i);
			if ((reg & mask) && (not_inten & mask)) {
			if (interrupt_handler_list[i].func != NULL) {
				interrupt_handler_list[i]
				.func(interrupt_handler_list[i].cb);
			}
			}
		}

		// Check Interrupt Flags in INT_GLBL1
		ret = read_register(INT_GLBL1, &reg);
		if (ret != MAX77643_2_NO_ERROR) return;

		ret = read_register(INTM_GLBL1, &inten);
		if (ret != MAX77643_2_NO_ERROR) return;

		not_inten = ~inten; // 0 means unmasked.

		for (int i = INT_GLBL1_GPI1_F; i < INT_CHG_END; i++) {
			mask = (1 << (i - INT_GLBL1_GPI1_F));
			if ((reg & mask) && (not_inten & mask)) {
				if (interrupt_handler_list[i].func != NULL) {
					interrupt_handler_list[i]
					.func(interrupt_handler_list[i].cb);
				}
			}
		}
	}
}

void MAX77643_2::interrupt_handler()
{
	post_intr_work_thread->flags_set(POST_INTR_WORK_SIGNAL_ID);
}

