/*******************************************************************************
 * Copyright(C) Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files(the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc.shall not be used except as stated in the Maxim Integrated
 * Products, Inc.Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc.retains all
 * ownership rights.
 *******************************************************************************
 */

#ifndef MAX7032_MAX7032_H_
#define MAX7032_MAX7032_H_

#include "mbed.h"
#include "Max7032_regs.h"
#include "rtos.h"

#define MAX7032_SPI                MXC_SPIM2

#define MAX7032_SPI         MXC_SPIM2
#define MAX7032_SPI_MOSI    P5_1
#define MAX7032_SPI_MISO    P5_2
#define MAX7032_SPI_SCK     P5_0
#define MAX7032_SPI_CS      P5_5

#define MAX7032_PIN_POWER   P4_0
#define MAX7032_PIN_RSSI    AIN_1
#define MAX7032_TRX_PIN     P3_3

#define MAX7032_MBED_DATA_PIN       P5_6
#define MAX7032_GPIO_DATA_PORT      PORT_5
#define MAX7032_GPIO_DATA_PIN       PIN_6

#define Q_START_CONF_LEN    17

class MAX7032
{
private:
    max7032_reg_map_t *reg_map;
    SPI *spi_handler;
    DigitalOut *ssel;
    DigitalOut *power_pin;
    DigitalOut *trx_pin;
    DigitalOut *data_send;
    PinName data_pin;
    DigitalOut *dio;

    DigitalIn *data_read;

    uint8_t preset_mode;
    float f_xtal ;
    float f_rf;
    float data_rate;
    float fsk_dev;
    const float fsk_dev_max = 0.1f;  //100khz

protected:


public:

    //Constructors
    MAX7032(SPI *spi, DigitalOut *cs, DigitalOut *powerPin, PinName dataPin, DigitalOut *dioPin, DigitalOut *trxPin);

    MAX7032(SPI *spi, DigitalOut *powerPin, PinName dataPin, DigitalOut *dioPin, DigitalOut *trxPin);

    //Destructor
    ~MAX7032();

    typedef enum {
        POWER_ADDR      = 0x00,
        CONTRL_ADDR     = 0x01,
        CONF0_ADDR      = 0x02,
        CONF1_ADDR      = 0x03,
        OSC_ADDR        = 0x05,
        TOFFMSB_ADDR    = 0x06,
        TOFFLSB_ADDR    = 0x07,
        TCPU_ADDR       = 0x08,
        TRFMSB_ADDR     = 0x09,
        TRFLSB_ADDR     = 0x0A,
        TONMSB_ADDR     = 0x0B,
        TONLSB_ADDR     = 0x0C,
        TXLOWMSB_ADDR   = 0x0D,
        TXLOWLSB_ADDR   = 0x0E,
        TXHIGHMSB_ADDR  = 0x0F,
        TXHIGHLSB_ADDR  = 0x10,
        STATUS_ADDR     = 0x1A,
     }register_address_t;

    typedef enum {
        X0,
        RSSIO,
        PA,
        PkDet,
        BaseB,
        MIXER,
        AGC,
        LNA,
        SLEEP,
        CKOUT,
        FCAL,
        PCAL,
        X1,
        TRK_EN,
        GAIN,
        AGCLK,
        ONPS0,
        ONPS1,
        OFPS0,
        OFPS1,
        DRX,
        MGAIN,
        T_R,
        MODE,
        DT0,
        DT1,
        DT2,
        CDIV0,
        CDIV1,
        CLKOF,
        ACAL,
        X2,
        NUM_OF_BIT
    }reg_bits_t;

    typedef enum
    {
        Manchester = 0,
        NRZ = 1
    }encoding_t;

    encoding_t encoding;

    /**
     * @brief Operation state of the RF receiver
     *
     * @details
     *  - Default       : 0x00
     *  - Description   : Indicates whether initialization is successful
     */
    typedef enum {
        INITIALIZED = 0,
        UNINITIALIZED = 1,
        UNKNOWN = 2,
    } operation_mode_t;

    operation_mode_t operation_mode;

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : CONF0 (0x02)
     *  - Bit Fields    : [7]
     *  - Default       : 0x0
     *  - Description   : ASK/FSK Selection for both receive and transmit
     */
    typedef enum {
        ASK_FSK_SEL_ASK,    /**< 0x0: ASK modulation */
        ASK_FSK_SEL_FSK     /**< 0x1: FSK modulation */
    } ask_fsk_sel_t;


    /* REGISTER SET & GET FUNCTION DECLARATIONS */


    /**
     * @brief Sets the specified bit.
     *
     * @param[in] bit_field   Bit to be set.
     * @param[in] val         Value to be set.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_bit_field(reg_bits_t bit_field, uint8_t val);

    /**
     * @brief Gets the specified bit.
     *
     * @param[in] bit_field   Bit to get value.
     * @param[in] val         The address to store the value of the Bit.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_bit_field(reg_bits_t bit_field, uint8_t *val);

    /**
     * @brief Sets Low-Noise Amplifier Power-Configuration Bit.
     *
     * @param[in] lna   0x0: Disable, 0x1: Enable.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_lna(uint8_t lna);

    /**
     * @brief Gets Low-Noise Amplifier Power-Configuration Bit.
     *
     * @returns LNA Power-Configuration Bit's State, negative error code on failure.
     */
    int get_lna();

    /**
     * @brief Sets Automatic Gain Control Power-Configuration Bit.
     *
     * @param[in] agc   0x0: Disable, 0x1: Enable.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_agc(uint8_t agc);

    /**
     * @brief Gets Automatic Gain Control Power-Configuration Bit.
     *
     * @returns AGC Power-Configuration Bit's State, negative error code on failure.
     */
    int get_agc();

    /**
     * @brief Sets Mixer Power-Configuration Bit.
     *
     * @param[in] mixer   0x0: Disable, 0x1: Enable.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_mixer(uint8_t mixer);

    /**
     * @brief Gets Mixer Power-Configuration Bit.
     *
     * @returns Mixer Power-Configuration Bit's State, negative error code on failure.
     */
    int get_mixer();

    /**
     * @brief Sets Baseband Power-Configuration Bit.
     *
     * @param[in] baseb   0x0: Disable, 0x1: Enable.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_baseb(uint8_t baseb);

    /**
     * @brief Gets Baseband Power-Configuration Bit.
     *
     * @returns Baseband Power-Configuration Bit's State, negative error code on failure.
     */
    int get_baseb();

    /**
     * @brief Sets Peak-detector Power-Configuration Bit.
     *
     * @param[in] pkdet   0x0: Disable, 0x1: Enable.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_pkdet(uint8_t pkdet);

    /**
     * @brief Gets Peak-detector Power-Configuration Bit.
     *
     * @returns Peak-detector Power-Configuration Bit's State, negative error code on failure.
     */
    int get_pkdet();

    /**
     * @brief Sets Transmitter Power Amplifier Power-Configuration Bit.
     *
     * @param[in] pa   0x0: Disable, 0x1: Enable.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_pa(uint8_t pa);

    /**
     * @brief Gets Transmitter Power Amplifier Power-Configuration Bit.
     *
     * @returns Transmitter Power Amplifier Power-Configuration Bit's State, negative error code on failure.
     */
    int get_pa();

    /**
     * @brief Sets RSSI Amplifier Power-Configuration Bit.
     *
     * @param[in] rssio   0x0: Disable, 0x1: Enable.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_rssio(uint8_t rssio);

    /**
     * @brief Gets RSSI Amplifier Power-Configuration Bit.
     *
     * @returns RSSI Amplifier Power-Configuration Bit's State, negative error code on failure.
     */
    int get_rssio();

    /**
     * @brief Sets AGC Locking Feature Control Bit.
     *
     * @param[in] agclk   0x0: Disable, 0x1: Enable.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_agclk(uint8_t agclk);

    /**
     * @brief Gets AGC Locking Feature Control Bit.
     *
     * @returns AGC Locking Feature Control Bit's State, negative error code on failure.
     */
    int get_agclk();

    /**
     * @brief Sets Gain State Control Bit.
     *
     * @param[in] gain   0x0: Force manual low-gain state if MGAIN = 1,
     *                   0x1: Force manual high-gain state if MGAIN = 1.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_gain(uint8_t gain);

    /**
     * @brief Gets Gain State Control Bit.
     *
     * @returns Gain State Control Bit's State, negative error code on failure.
     */
    int get_gain();

    /**
     * @brief Sets Manual Peak-Detector Tracking Control Bit.
     *
     * @param[in] trk_en   0x0: Release peak-detector tracking,
     *                     0x1: Force manual peak-detector tracking.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_trk_en(uint8_t trk_en);

    /**
     * @brief Gets Manual Peak-Detector Tracking Control Bit.
     *
     * @returns Manual Peak-Detector Tracking Control Bit's State, negative error code on failure.
     */
    int get_trk_en();

    /**
     * @brief Sets Polling Timer Calibration Control Bit.
     *
     * @description   Sets PCAL Bit to 1 to Perform polling timer calibration.
     *                Automatically reset to zero once calibration is completed
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_pcal();

    /**
     * @brief Gets Polling Timer Calibration Control Bit.
     *
     * @returns Polling Timer Calibration Control Bit's State, negative error code on failure.
     */
    int get_pcal();

    /**
     * @brief Sets FSK calibration Control Bit.
     *
     * @description   Sets FCAL Bit to 1 to Perform FSK calibration.
     *                Automatically reset to zero once calibration is completed
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_fcal();

    /**
     * @brief Gets FSK calibration Control Bit.
     *
     * @returns FSK calibration Control Bit's State, negative error code on failure.
     */
    int get_fcal();

    /**
     * @brief Sets Crystal Clock Output Enable Control Bit.
     *
     * @param[in] sleep   0x0: Disable crystal clock output,
     *                    0x1: Enable crystal clock output.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_ckout(uint8_t ckout);

    /**
     * @brief Gets Crystal Clock Output Enable Control Bit.
     *
     * @returns Sleep Mode Control Bit's State, negative error code on failure.
     */
    int get_ckout();

    /**
     * @brief Sets Sleep Mode Control Bit.
     *
     * @param[in] sleep   0x0: Normal operation,
     *                    0x1: Deep-sleep mode, regardless the state of ENABLE pin.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_sleep(uint8_t sleep);

    /**
     * @brief Gets Sleep Mode Control Bit.
     *
     * @returns Sleep Mode Control Bit's State, negative error code on failure.
     */
    int get_sleep();

    /**
     * @brief Sets FSK or ASK modulation Configuration Bit.
     *
     * @param[in] ask_fsk_sel   0x0: Enable ASK for both receive and transmit,
     *                          0x1: Enable FSK for both receive and transmit.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_mode(ask_fsk_sel_t ask_fsk_sel);

    /**
     * @brief Gets FSK or ASK modulation Configuration Bit.
     *
     * @param[in] ask_fsk_sel
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_mode(ask_fsk_sel_t* ask_fsk_sel);

    /**
     * @brief Gets FSK or ASK modulation Configuration Bit.
     *
     * @returns FSK or ASK modulation Configuration Bit's State, negative error code on failure.
     */
    int get_mode();

    /**
     * @brief Sets Transmit or Receive Configuration Bit.
     *
     * @param[in] t_r   0x0: Enable receive mode of the transceiver when pin T/R = 0,
     *                  0x1: Enable transmit mode of the transceiver, regardless the state of pin T/R.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_t_r(uint8_t t_r);

    /**
     * @brief Gets Transmit or Receive Configuration Bit.
     *
     * @returns Transmit or Receive Configuration Bit's State, negative error code on failure.
     */
    int get_t_r();

    /**
     * @brief Sets Manual Gain Mode Configuration Bit.
     *
     * @param[in] mgain   0x0: Disable manual-gain mode,
     *                    0x1: Enable manual-gain mode.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_mgain(uint8_t mgain);

    /**
     * @brief Gets Manual Gain Mode Configuration Bit.
     *
     * @returns Manual Gain Mode Configuration Bit's State, negative error code on failure.
     */
    int get_mgain();

    /**
     * @brief Sets Discontinuous Receive Mode Configuration Bit.
     *
     * @param[in] drx   0x0: Disable DRX,
     *                  0x1: Enable DRX.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_drx(uint8_t drx);

    /**
     * @brief Gets Discontinuous Receive Mode Configuration Bit.
     *
     * @returns Discontinuous Receive Mode Configuration Bit's State, negative error code on failure.
     */
    int get_drx();

    /**
     * @brief Sets Off-timer Prescaler Configuration Bits.
     *
     * @param[in] ofps   Off-timer Prescaler Configuration Value.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_ofps(uint8_t ofps);

    /**
     * @brief Gets Off-timer Prescaler Configuration Bits.
     *
     * @returns Off-timer Prescaler Configuration Bit's value, negative error code on failure.
     */
    int get_ofps();

    /**
     * @brief Sets On-timer Prescaler Configuration Bits.
     *
     * @param[in] onps   On-timer Prescaler Configuration Value.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_onps(uint8_t onps);

    /**
     * @brief Gets On-timer Prescaler Configuration Bits.
     *
     * @returns On-timer Prescaler Configuration Bit's value, negative error code on failure.
     */
    int get_onps();

    /**
     * @brief Sets Automatic FSK Calibration Configuration Bit.
     *
     * @param[in] acal   0x0: Disable automatic FSK calibration,
     *                   0x1: Enable automatic FSK calibration when coming out of the sleep state in DRX mode.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_acal(uint8_t acal);

    /**
     * @brief Gets Automatic FSK Calibration Configuration Bit.
     *
     * @returns Automatic FSK Calibration Configuration Bit's State, negative error code on failure.
     */
    int get_acal();

    /**
     * @brief Sets Continuous Clock Output (even during tOFF or when ENABLE pin is low) Configuration Bit.
     *
     * @param[in] clkof   0x0: Continuous clock output; if CKOUT = 1, clock output is active during tON (DRX mode) or
     *                         when ENABLE pin is high (continuous receive mode),
     *                    0x1: Enable continuous clock output when CKOUT = 1.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_clkof(uint8_t clkof);

    /**
     * @brief Gets Continuous Clock Output Configuration Bit.
     *
     * @returns Continuous Clock Output Configuration Bit's State, negative error code on failure.
     */
    int get_clkof();

    /**
     * @brief Sets Crystal Divider Configuration Bits.
     *
     * @param[in] cdiv   Crystal Divider Configuration Value.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_cdiv(uint8_t cdiv);

    /**
     * @brief Gets Crystal Divider Configuration Bits Value.
     *
     * @returns Crystal Divider Configuration Bit's value, negative error code on failure.
     */
    int get_cdiv();

    /**
     * @brief Sets AGC Dwell Timer Configuration Bits.
     *
     * @param[in] dt   AGC Dwell Timer Configuration Value.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_dt(uint8_t dt);

    /**
     * @brief Gets AGC Dwell Timer Configuration Bits Value.
     *
     * @returns AGC Dwell Timer Configuration Bit's value, negative error code on failure.
     */
    int get_dt();

    /**
     * @brief Sets the OSC Register.
     *
     * @param[in] osc   OSC Register Value.
     *
     * @description This register must be set to the integer result of fXTAL/100kHz.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_osc(uint8_t osc);

    /**
     * @brief Gets the OSC Register Value.
     *
     * @returns OSC Register Value, negative error code on failure.
     */
    int get_osc();

    /**
     * @brief Sets the tOFF Registers.
     *
     * @param[in] toff   OSC Registers Value.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_toff(uint16_t toff);

    /**
     * @brief Gets the tOFF Registers Value.
     *
     * @returns tOFF Registers Value, negative error code on failure.
     */
    uint16_t get_toff();

    /**
     * @brief Sets the tCPU Register.
     *
     * @param[in] tcpu   tCPU Register Value.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_tcpu(uint8_t tcpu);

    /**
     * @brief Gets the tCPU Register Value.
     *
     * @returns tCPU Register Value, negative error code on failure.
     */
    uint8_t get_tcpu();

    /**
     * @brief Sets the tRF Registers.
     *
     * @param[in] trf   tRF Registers Value.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_trf(uint16_t trf);

    /**
     * @brief Gets the tRF Registers Value.
     *
     * @returns tRF Registers Value, negative error code on failure.
     */
    uint16_t get_trf();

    /**
     * @brief Sets the tON Registers.
     *
     * @param[in] ton   tON Registers Value.
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_ton(uint16_t ton);

    /**
     * @brief Gets the tON Registers Value.
     *
     * @returns tON Registers Value, negative error code on failure.
     */
    uint16_t get_ton();

    /**
    * @brief Sets the TxLOW Registers.
    *
    * @param[in] txlow_val TxLOW Registers Value.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int set_txlow(int txlow_val);

    /**
     * @brief Gets the TxLOW Registers Value.
     *
     * @returns the TxLOW Registers Value, negative error code on failure.
     */
    int get_txlow();

    /**
    * @brief Sets the TxHIGH Registers.
    *
    * @param[in] txhigh_val TxHIGH Registers Value.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int set_txhigh(int txhigh_val);

    /**
     * @brief Gets the TxHIGH Registers Value.
     *
     * @returns the TxHIGH Registers Value, negative error code on failure.
     */
    int get_txhigh();

    /**
     * @brief Gets Lock Detect Status Bit.
     *
     * @returns Lock Detect Status Bit's State, negative error code on failure.
     */
    int get_lckd();

    /**
     * @brief Gets AGC Gain State Status Bit.
     *
     * @returns AGC Gain State Status Bit's State, negative error code on failure.
     */
    int get_gains();

    /**
     * @brief Gets Clock/Crystal Alive Status Bit.
     *
     * @returns Clock/Crystal Alive Status Bit's State, negative error code on failure.
     */
    int get_clkon();

    /**
     * @brief Gets Polling Timer Calibration Done Status Bit.
     *
     * @returns Polling Timer Calibration Done Status Bit's State, negative error code on failure.
     */
    int get_pcald();

    /**
     * @brief Gets FSK Calibration Done Status Bit.
     *
     * @returns FSK Calibration Done Status Bit's State, negative error code on failure.
     */
    int get_fcald();

    /* END OF REGISTER SET & GET FUNCTION DECLARATIONS */

    /* DEVICE CONFIGURATION FUNCTION DECLARATIONS */
    typedef enum {
        DISABLE,        /**< 0x0: Disable */
        F_XTAL,         /**< 0x1: fXTAL */
        F_XTAL_X0_5,    /**< 0x2: fXTAL/2 */
        F_XTAL_X0_25,   /**< 0x3: fXTAL/4 */
        F_XTAL_X0_125   /**< 0x4: fXTAL/8 */
    } cdiv_t;

    /**
    * @brief Configures Crystal Clock Output.
    *
    * @param[in] cdiv Clock Output Configuration.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int adjust_clockout(cdiv_t cdiv);

    /**
    * @brief Gets Clock Output Configuration.
    *
    * @param[in] cdiv   Clock Output Configuration.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int get_clockout_conf(cdiv_t *cdiv);

    /**
    * @brief Configures Internal Clock Frequency Divisor.
    *
    * @param[in] osc_freq Oscillator Frequency.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int adjust_osc_freq(float osc_freq);

    /**
    * @brief Gets Oscillator Frequency.
    *
    * @returns Oscillator Frequency.
    */
    float get_osc_freq();

    typedef enum {
        US_120,   /**< 0x0: 120 us */
        US_480,   /**< 0x1: 480 us */
        US_1920,  /**< 0x2: 1920 us */
        US_7680   /**< 0x3: 7680 us */
    } timer_base_t;

    /**
    * @brief Configures the off timer.
    *
    * @param[in] time_base Timer Base Configuration.
    * @param[in] timer_val tOFF Registers Value.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int adjust_off_timer(timer_base_t time_base, int timer_val);

    /**
    * @brief Configures the off timer.
    *
    * @param[in] toff_time Off Time in microseconds.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int adjust_off_timer(int toff_time);

    /**
     * @brief Gets the off time in us.
     *
     * @returns the off time in us, negative error code on failure.
     */
    int get_off_timer();

    /**
    * @brief Configures the CPU recovery timer.
    *
    * @param[in] tcpu_time desired CPU recovery time in us.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int adjust_cpu_recovery_timer(int tcpu_time);

    /**
     * @brief Gets the CPU recovery time in us.
     *
     * @returns the CPU recovery time in us, negative error code on failure.
     */
    int get_cpu_recovery_timer();

    /**
    * @brief Configures the RF settling timer.
    *
    * @param[in] trf_time RF settling time in us.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int adjust_rf_settling_timer(int trf_time);

    /**
     * @brief Gets the RF settling time in us.
     *
     * @returns the RF settling time in us, negative error code on failure.
     */
    int get_rf_settling_timer();

    /**
    * @brief Configures the on timer.
    *
    * @param[in] time_base Timer Base Configuration.
    * @param[in] timer_val tON Registers Value.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int adjust_on_timer(timer_base_t time_base, int timer_val);

    /**
    * @brief Configures the on timer.
    *
    * @param[in] ton_time On Time in microseconds.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int adjust_on_timer(int ton_time);

    /**
     * @brief Gets the on time in us.
     *
     * @returns the on time in us, negative error code on failure.
     */
    int get_on_timer();

    /**
    * @brief Sets the devices center frequency.
    *
    * @param[in] center_freq desired center frequency
    *
    * @returns 0 on success, negative error code on failure.
    */
    int set_center_freq(float center_freq);

    /**
    * @brief Gets the devices center frequency.
    *
    * @returns 0 on success, negative error code on failure.
    */
    float get_center_freq();

    /**
    * @brief Sets the devices data rate.
    *
    * @param[in] data_rate_set desired data rate
    *
    * @returns 0 on success, negative error code on failure.
    */
    int set_data_rate(float data_rate_set);

    /**
    * @brief Gets the devices data rate.
    *
    * @returns 0 on success, negative error code on failure.
    */
    float get_data_rate();

    /**
    * @brief Sets the FSK deviation.
    *
    * @param[in] fsk_dev_set desired FSK deviation
    *
    * @returns 0 on success, negative error code on failure.
    */
    int set_fsk_dev(float fsk_dev_set);

    /**
    * @brief Gets the FSK deviation.
    *
    * @returns 0 on success, negative error code on failure.
    */
    float get_fsk_dev();

    /**
    * @brief Sets the encoding type.
    *
    * @param[in] encoding_set desired encoding type
    *
    * @returns 0 on success, negative error code on failure.
    */
    int set_encoding(encoding_t encoding_set);

    /**
    * @brief Gets the encoding type.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int get_encoding();

    /**
    * @brief Configures the AGC Dwell Timer.
    *
    * @param[in] k_val K value for dwell timer configuration.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int adjust_agc_dwell_timer(uint8_t k_val);

    /**
    * @brief Configures the AGC Dwell Timer.
    *
    * @param[in] dwell_time desired dwell time.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int adjust_agc_dwell_timer(int dwell_time);

    /**
     * @brief Gets the AGC Dwell Time Value.
     *
     * @returns the AGC Dwell Time Value, negative error code on failure.
     */
    int get_agc_dwell_timer();

    /* END OF DEVICE CONFIGURATION FUNCTION DECLARATIONS */

    /* PUBLIC FUNCTION DECLARATIONS */

    /**
    * @brief Set power on/off
    *
    * @param[in] power 0 : power ON, 1: power OFF
    *
    * @returns 0 on success, negative error code on failure.
    */
    int set_power_on_off(uint8_t power);

    /**
    * @brief Read from a register. Since 3-wire spi is not supported by mbed, the read_register function must be implemented by the user.
    *
    * @param[in] reg Address of a register to be read.
    * @param[out] value Pointer to save result value.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int read_register(uint8_t reg, uint8_t *value);

    /**
    * @brief Write to a register.
    *
    * @param[in] reg Address of a register to be written.
    * @param[out] value Pointer of value to be written to register.
    * @param[in] len Size of result to be written.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int write_register(uint8_t reg, const uint8_t *value, uint8_t len);

    /**
    * @brief Initial programming steps after power on or soft reset.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int initial_programming(void);

    /**
    * @brief Loads the Quick Start configuration to the device for RX mode.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int rx_quich_start(void);

    /**
    * @brief Loads the Quick Start configuration to the device for TX mode.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int tx_quich_start(void);

    typedef enum{
        RECEIVE_MODE,
        TRANSMIT_MODE
    }trx_state_t;

    /**
    * @brief Sets the Device to Receiver or Transmitter Mode.
    *
    * @param[in] trx_state desired mode.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int set_trx_state(trx_state_t trx_state);

    /**
     * @brief Gets the Mode the Device is in.
     *
     * @returns the mode the Device is in, negative error code on failure.
     */
    int get_trx_state();

    /**
    * @brief Transmits the RF data.
    *
    * @param[in] data Address of the data to be transmit.
    * @param[in] data_len Size of the data to be transmit.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int rf_transmit_data(uint8_t *data, uint8_t data_len);

    /**
    * @brief Receives the RF data.
    *
    * @param[in] coded_data the address where the received data will be stored.
    * @param[in] coded_data_len the length of the data to be received.
    *
    * @description The coded data is sampled at twice the data rate. The coded data must be filtered.
    * @returns 0 on success, negative error code on failure.
    */
    int rf_receive_data(uint8_t *coded_data, uint8_t coded_data_len);

    /**
    * @brief Sets the T/R Pin if defined.
    *
    * @param[in] pin_state desired pin state.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int set_trx_pin(trx_state_t pin_state);
};

#endif /* MAX7032_MAX7032_H_ */
