/*******************************************************************************
* Copyright (C) 2019 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
*/
// *********************************************************************
// @file MAX541.h
// *********************************************************************

// Prevent multiple declaration
#ifndef MAX541_H
#define MAX541_H

#include "mbed.h"

/**
* @brief MAX541 Ultra-Small, 16-Bit, Unbuffered Output Voltage DAC with SPI Interface
 *
 * @version 1.0000.0
 *
 * @details The MAX541 is an SPI programmable voltage DAC.
 * This driver enables writing of voltage ouptut values to the output register.
 *
 * Datasheet: https://www.maximintegrated.com/MAX541
 *
 * @code
 * #include "mbed.h"
 * #include "MAX541.h"
 *
 * // Connect MAX541 to Arduino compatible connector as shown:
 * //                  MAX541BCPA+
 * //              --------__--------
 * //     output --| 1 OUT    VDD 8 |-- IOREF (3V3 or 5V logic supply)
 * //        GND --| 2 AGND  DGND 7 |-- GND
 * //       V3.3 --| 3 REF    DIN 6 |-- D11/SPI_MOSI
 * // D10/SPI_SS --| 4 CS    SCLK 5 |-- D13/SPI_SCLK
 * //              ------------------
 *
 * // example code declare SPI interface
 * #if defined(TARGET_MAX32625MBED)
 * SPI spi_max541(SPI1_MOSI, SPI1_MISO, SPI1_SCK); // mosi, miso, sclk spi1 TARGET_MAX32625MBED: P1_1 P1_2 P1_0 Arduino 10-pin header D11 D12 D13
 * DigitalOut spi_max541_cs(SPI1_SS); // TARGET_MAX32625MBED: P1_3 Arduino 10-pin header D10
 * // alternate spi connection
 * // SPI spi2_max541(SPI2_MOSI, SPI2_MISO, SPI2_SCK); // mosi, miso, sclk spi2 TARGET_MAX32625MBED: P2_5 P2_6 P2_4 Arduino 2x3-pin header; microSD
 * // DigitalOut spi2_max541_cs(SPI2_SS); // TARGET_MAX32625MBED: P2_7 Arduino 2x3-pin header
 * #elif defined(TARGET_MAX32600MBED)
 * SPI spi_max541(SPI2_MOSI, SPI2_MISO, SPI2_SCK);
 * DigitalOut spi_max541_cs(SPI2_SS);
 * #else
 * SPI spi_max541(D11, D12, D13); // Arduino 10-pin header
 * DigitalOut spi_max541_cs(D10); // Arduino 10-pin header
 * #endif
 *
 * int main()
 * {
 *     MAX541 max541(spi_max541, spi_max541_cs);
 *     max541.VRef = 3.30;
 *
 *     double voltageV = 1.0f;
 *     max541.Set_Voltage(voltageV);
 *
 *     //wait(1.0);
 * }
 * @endcode
 */
class MAX541
{
public:

    /** ********************************************************
     * @brief Constructor for MAX541 Class.
     *
     * @details Allows user to use existing SPI object
     *
     * On Entry:
     *     @param[in] spi - pointer to existing SPI object
     *     @param[in] cs_pin - pointer to a DigitalOut pin object
     *
     * On Exit:
     *
     * @return None
     */
    MAX541(SPI &spi, DigitalOut &cs_pin);

    /** ********************************************************
     * @brief Default destructor for MAX541 Class.
     *
     * @details Destroys SPI object if owner
     *
     * On Entry:
     *
     * On Exit:
     *
     * @return None
     */
    ~MAX541();

private:
    /** SPI object */
    SPI &m_spi;
    int m_SPI_SCLK_Hz;
    int m_SPI_dataMode;

    /** Selector pin object */
    DigitalOut &m_cs_pin;

public:
    /** Reference voltage, in Volts.
     * Default value is 3.3 unless otherwise specified.
     * For best performance, an external 2.5V reference should be used.
     */
    double VRef;

    /** Full-scale (maximum) raw DAC code, in LSBs.
     */
    // TODO: warning: non-static data member initializers only available with -std=c++11 or -std=gnu++11
    const float max541codeFS = (float)((uint32_t)0xffff);

    /** Raw DAC code, in LSBs.
     *
     * The DAC output voltage is a function of raw DAC code:
     * VOUT = VREF * (code / 65536)
     *
     * The full-scale voltage is VREF * 65535 / 65536.
     */
    uint16_t max541_code;

    /** Set the MAX541 output voltage.
     *
     * @pre VRef = Full-Scale reference voltage, in Volts
     * @param[in] voltageV = output voltage, in Volts
     */
    void Set_Voltage(double voltageV);

    /** Get the MAX541 output voltage.
     *
     * @returns output voltage, in Volts
     */
    double Get_Voltage() const;

    /** Set the MAX541 output code.
     *
     * @param[in] max541_mosiData16 = raw DAC code, in LSBs
     */
    void Set_Code(int16_t max541_mosiData16);

    /** Get the MAX541 output code.
     *
     * @returns raw DAC code, in LSBs
     */
    uint16_t Get_Code(void) const;

};

#endif/* MAX541_H */
