/*******************************************************************************
* Copyright (C) 2018 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
* @file MAX5216.h
*******************************************************************************
*/
#ifndef MAX5216_H
#define MAX5216_H

#include "mbed.h"

const unsigned int MAX521X_IC_BIT_MASK[] = {
        0X00003FFF,  // 14 Bits for MAX5214
        0X0000FFFF   // 16 Bits for MAX5216
    };

const int MAX521X_NUM_BYTES_SPI[] = {
        2,  // 2 bytes for MAX5214
        3   // 3 bytes for MAX5216
    };

/**
* @brief 16-bit, 14-bit digital-to-analog converters (DACs)
*        for the MAX5216, MAX5214.
* @version 1.0000.1
*
* @details The MAX5214/MAX5216 accept a wide 2.7V to 5.5V supply
* voltage range. Power consumption is extremely low
* to accommodate most low-power and low-voltage applications.
* These devices feature a 3-wire SPI-/QSPI™-/
* MICROWIRE-/DSP-compatible serial interface
* This driver is compatible with the
* MAX5216, MAX5214.
*
* @code 
* #include "mbed.h"
* #include "max32630fthr.h"
* #include "MAX5216.h"
* #include "USBSerial.h"
* MAX32630FTHR pegasus(MAX32630FTHR::VIO_3V3); 
* DigitalOut rLED(LED1);
* DigitalOut gLED(LED2);
* DigitalOut bLED(LED3);
* DigitalOut selectPin(P3_0); // Pin 3_0 is used to drive chip enable low
* SPI spi(P5_1, P5_2, P5_0); // mosi, miso, sclk
* 
* int main()
* {
*     selectPin = 0;
*     MAX5216 dac(spi, selectPin, MAX5216::MAX5216_IC);
*     spi.format(8,0);
*     spi.frequency(1000000);
*     dac.write_command(MAX5216::WrtThru_Reg, 0xFFFF);
*     wait(1.0);
*     dac.write_command(MAX5216::WrtThru_Reg, 0x7FFF);
*     wait(1.0);
*     dac.write_cmmand(MAX5216::PwrDwn_Reg, MAX5216::PwrDwnHiZ);
* }
* @endcode
*/

class MAX5216{
    public:
    /**
    * @brief IC's supported with this driver
    * @details MAX5214, MAX5216
    */ 
    typedef enum
    {
        MAX5214_IC = 0,
        MAX5216_IC = 1
    }MAX521X_ic_t;

    /**
    * @brief Commands supported by the DAC
    * @details The upper 2 bits define the commands
    */ 
    typedef enum {
        PwrDwn_Reg =  (0x2<<6),                   // Power Down
        WrtThru_Reg = (0x1<<6)                    // Write Through
    } setting_t;
    
    /**
    * @brief Power Down modes
    * @details 2 bits are used to define the power down modes
    */ 
    typedef enum {
        PwrDwnNormalOp = (0x00),                  // DAC powers up and returns to its previous code setting.
        PwrDwnHiZ =      (0x01),                  // DAC powers down; OUT is high impedance.
        PwrDwn100K =     (0x02),                  // DAC powers down; OUT connects to ground through an internal 100k resistor.
        PwrDwn1K =       (0x03)                   // DAC powers down; OUT connects to ground through an internal 1k resistor.
    } pwrDwnMode_t; 
    /**********************************************************//**
    * @brief Constructor for MAX5216 Class.  
    * 
    * @details Requires an existing SPI object as well as a DigitalOut object. 
    * The DigitalOut object is used for a chip enable signal
    *
    * On Entry:
    *     @param[in] spi - pointer to existing SPI object
    *     @param[in] pin - pointer to a DigitalOut pin object
    *     @param[in] ic_variant - which type of MAX521x is used
    *
    * On Exit:
    *
    * @return None
    **************************************************************/
    MAX5216(SPI &spi, DigitalOut &pin, MAX521X_ic_t ic_variant); 
    
    /** 
    * @brief Send write command
    * @param setting - WrtThru_Reg or Power Down Mode command
    * @param value - 14 or 16 bit Value to write if WrtThru_Reg commanded
    *                Power Down Mode if PwrDwn_Reg commanded
    * @return void
    */
    void write_command(MAX5216::setting_t setting, uint32_t value, pwrDwnMode_t mode = PwrDwnNormalOp);

    /************************************************************
    * @brief Default destructor for MAX5216 Class.  
    *
    * @details Destroys SPI object if owner 
    *
    * On Entry:
    *
    * On Exit:
    *
    * @return None
    **************************************************************/
    ~MAX5216();
    private:
    // SPI object
    SPI &m_spi;
    // Selector pin object
    DigitalOut &m_pin;
    // Identifies which IC variant is being used
    MAX521X_ic_t m_ic_variant;
};

#endif