/*******************************************************************************
 * Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */

#ifndef _MAX44000_H_
#define _MAX44000_H_

#include "mbed.h"

/**
 * MAX44000 Ambient Light and Infrared Proximity Sensor
 *
 * @code
 * #include <stdio.h>
 * #include "mbed.h"
 * #include "MAX44000.h"
 *
 * I2C i2c(I2C_SDA, I2C_SCL);
 * MAX44000 max44000(&i2c);
 *
 * int main(void) {
 *
 *     while(1) {
 *         if (max44000.init(MAX44000::MODE_ALS_PROX, MAX44000::ALSTIM_1X, MAX44000::ALSPGA_128X, MAX44000::DRV_110) != 0) {
 *             printf("Error communicating with MAX44000\n");
 *         } else {
 *             printf("Initialized MAX44000\n");
 *             break;
 *         }
 *         wait(1);
 *     }
 *
 *     while(1) {
 *         int alsValue = max44000.readALS();
 *         if(alsValue < 0) {
 *             printf("Error reading ALS value\n");
 *             continue;
 *         }
 *
 *         int proxValue = max44000.readReg(MAX44000::REG_PRX_DATA);
 *         if(proxValue < 0) {
 *             printf("Error reading proximity value\n");
 *             continue;
 *         }
 *
 *         printf("ALS = 0x%04X  Proximity = 0x%02X\n", alsValue, proxValue);
 *         wait(1);
 *     }
 * }
 * @endcode
 */
class MAX44000
{

public:

    /**
     * @brief   Register Addresses
     * @details Enumerated MAX44000 register addresses
     */
    typedef enum {
        REG_STATUS,         ///< Interrupt Status
        REG_MAIN_CONFIG,    ///< Main Configuration
        REG_RX_CONFIG,      ///< Receive Configuration
        REG_TX_CONFIG,      ///< Transmit Configuration
        REG_ALS_DATA_HIGH,  ///< ADC High Byte (ALS)
        REG_ALS_DATA_LOW,   ///< ADC Low Byte (ALS)
        REG_ALS_UPTHR_HIGH, ///< ALS Upper Threshold (High Byte)
        REG_ALS_UPTHR_LOW,  ///< ALS Upper Threshold (Low Byte)
        REG_ALS_LOTHR_HIGH, ///< ALS Lower Threshold (High Byte)
        REG_ALS_LOTHR_LOW,  ///< ALS Lower Threshold (Low Byte)
        REG_PST,            ///< Threshold Persist Timer
        REG_PRX_IND,        ///< PROX Threshold Indicator
        REG_PRX_THR,        ///< PROX Threshold
        REG_TRIM_GAIN_GREEN,     ///< Digital Gain Trim of Green Channel
        REG_TRIM_GAIN_IR = 0x0F, ///< Digital Gain Trim of Infrared Channel
        REG_PRX_DATA     = 0x16  ///< ADC Byte (PROX)
    } registers_t;


    /**
     * @brief   Operating Modes
     * @details Enumerated MAX44000 operating modes
     */
    typedef enum {
        MODE_SHUTDOWN, ///< Analog circuits are shut down, but the digital register retains values
        MODE_ALS_GMIR, ///< Standard ALS mode, stores difference between green and IR channels
        MODE_ALS_G,    ///< ALS green channel only
        MODE_ALS_IR,   ///< Infrared channel only
        MODE_ALS_PROX, ///< ALS and PROX are interleaved continuously
        MODE_PROX,     ///< PROX only
        MODE_RSVD_110, ///< Reserved, do not use
        MODE_RSVD_111  ///< Reserved, do not use
    } modes_t;


    /**
     * @brief   ALS ADC Conversion Time
     * @details MAX44000 Ambient Light Sensor ADC Conversion Time
     */
    typedef enum {
        ALSTIM_1X,  ///< 100ms, 16,384 counts, 14 bit resolution
        ALSTIM_4X,  ///< 25ms,    4096 counts, 12 bit resolution
        ALSTIM_16X, ///< 6.25ms,  1024 counts, 10 bit resolution
        ALSTIM_64X  ///< 1.5625ms, 256 counts,  8 bit resolution
    } alstim_t;


    /**
    * alspga_t - MAX44000 Ambient Light Measurement Gain
    */
    typedef enum {
        ALSPGA_1X,  ///< 0.03125 lux/lsb
        ALSPGA_4X,  ///< 0.125 lux/lsb
        ALSPGA_16X, ///< 0.5 lux/lsb
        ALSPGA_128X ///< 4 lux/lsb
    } alspga_t;


    /**
     * @brief   LED Driver Current
     * @details MAX44000 LED Driver Current Settings
     */
    typedef enum {
        DRV_0,   ///< LED driver disabled
        DRV_10,  ///< 10mA
        DRV_20,  ///< 20mA
        DRV_30,  ///< 30mA
        DRV_40,  ///< 40mA
        DRV_50,  ///< 50mA
        DRV_60,  ///< 60mA
        DRV_70,  ///< 70mA
        DUP_40,  ///< 40mA
        DUP_50,  ///< 50mA
        DUP_60,  ///< 60mA
        DUP_70,  ///< 70mA
        DRV_80,  ///< 80mA
        DRV_90,  ///< 90mA
        DRV_100, ///< 100mA
        DRV_110  ///< 110mA
    } drive_t;


    /**
     * @brief   Persist Times
     * @details MAX44000 ALS/PROX Threshold Persist Timer Settings
     */
    typedef enum {
        PST_1, ///< 1 trigger before interrupt
        PST_2, ///< 2 consecutive triggers before interrupt
        PST_4, ///< 4 consecutive triggers before interrupt
        PST_16 ///< 16 consecutive triggers before interrupt
    } persist_t;


    /**
     * MAX44000 constructor.
     *
     * @param sda mbed pin to use for SDA line of I2C interface.
     * @param scl mbed pin to use for SCL line of I2C interface.
     */
    MAX44000(PinName sda, PinName scl);

    /**
     * MAX44000 constructor.
     *
     * @param i2c I2C object to use.
     */
    MAX44000(I2C *i2c);

    /**
     * MAX44000 destructor.
     */
    ~MAX44000();

    /**
     * @brief   Initialize MAX44000.
     * @details Gets the device ID and saves the calibration values.
     * @param   mode Operating Mode
     * @param   alstim Ambient Light ADC Conversion Time
     * @param   alspga Ambient Light Measurement Gain
     * @param   drive LED Driver Current
     * @returns 0 if no errors, -1 if error.
     */
    int init(MAX44000::modes_t mode, MAX44000::alstim_t alstim, MAX44000::alspga_t alspga, MAX44000::drive_t drive);

    /**
     * @brief   Write Register
     * @details Writes data to MAX44000 register
     *
     * @param   reg_addr Register to write
     * @param   reg_data Data to write
     * @returns 0 if no errors, -1 if error.
     */
    int writeReg(MAX44000::registers_t reg_addr, char reg_data);

    /**
     * @brief   Read Register
     * @details Reads data from MAX44000 register
     *
     * @param   reg_addr Register to read
     * @returns data if no errors, -1 if error.
     */
    int readReg(MAX44000::registers_t reg_addr);

    /**
     * @brief   Read ALS Data
     * @details Reads both ALS data registers and returns combined value
     *
     * @returns data if no errors, -1 if error.
     */
    int readALS(void);


private:

    I2C *i2c_;
    bool i2c_owner;

};

#endif /* _MAX44000_H_ */
