/*******************************************************************************
 * Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */

#include "MAX44000.h"

/***** Definitions *****/
#define I2C_ADDR            (0x94) // 1001_010x

//******************************************************************************
MAX44000::MAX44000(PinName sda, PinName scl)
{
    i2c_ = new I2C(sda, scl);
    i2c_owner = true;

    i2c_->frequency(400000);
}

//******************************************************************************
MAX44000::MAX44000(I2C *i2c) :
    i2c_(i2c)
{
    i2c_owner = false;
}

//******************************************************************************
MAX44000::~MAX44000()
{
    if(i2c_owner) {
        delete i2c_;
    }
}

//******************************************************************************
int MAX44000::init(MAX44000::modes_t mode, MAX44000::alstim_t alstim, MAX44000::alspga_t alspga, MAX44000::drive_t drive)
{
    char data[2];

    data[0] = REG_RX_CONFIG;
    data[1] = 0xF0 | ((alstim & 0x03)<<2) | (alspga & 0x03);
    if (i2c_->write(I2C_ADDR, data, 2) != 0) {
        return -1;
    }

    data[0] = REG_TX_CONFIG;
    data[1] = (drive & 0x0F);
    if (i2c_->write(I2C_ADDR, data, 2) != 0) {
        return -1;
    }

    data[0] = REG_MAIN_CONFIG;
    data[1] = ((mode & 0x07)<<2);
    if (i2c_->write(I2C_ADDR, data, 2) != 0) {
        return -1;
    }

    return 0;
}

//******************************************************************************
int MAX44000::writeReg(MAX44000::registers_t reg_addr, char reg_data)
{
    char data[2];

    data[0] = reg_addr;
    data[1] = reg_data;
    if (i2c_->write(I2C_ADDR, data, 2) != 0) {
        return -1;
    }

    return 0;
}

//******************************************************************************
int MAX44000::readReg(MAX44000::registers_t reg_addr)
{
    char data;

    data = reg_addr;
    if (i2c_->write(I2C_ADDR, &data, 1, true) != 0) {
        return -1;
    }

    if (i2c_->read(I2C_ADDR, &data, 1) != 0) {
        return -1;
    }

    return (0x0 + data);
}

//******************************************************************************
int MAX44000::readALS(void)
{
    char data;
    int alsData;

    data = REG_ALS_DATA_HIGH;
    if (i2c_->write(I2C_ADDR, &data, 1, true) != 0) {
        return -1;
    }

    if (i2c_->read(I2C_ADDR, &data, 1, true) != 0) {  // must use repeated start to protect low byte data
        return -1;
    }

    if (data & 0x40) { // if the overflow bit is set
        return -1;
    }

    alsData = (data << 8);
    data = REG_ALS_DATA_LOW;
    if (i2c_->write(I2C_ADDR, &data, 1, true) != 0) {
        return -1;
    }

    if (i2c_->read(I2C_ADDR, &data, 1) != 0) {
        return -1;
    }

    alsData += data;
    return alsData;
}
