/*******************************************************************************
 * Copyright(C) Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files(the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc.shall not be used except as stated in the Maxim Integrated
 * Products, Inc.Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc.retains all
 * ownership rights.
 *******************************************************************************
 */

#ifndef MAX4147X_MAX4147X_H_
#define MAX4147X_MAX4147X_H_

#include "mbed.h"
#include "rtos.h"
#include "Max41470_regs.h"
#include "Max41473_4_regs.h"

#define MAX4147X_SPI                MXC_SPIM2

#define MAX4147X_PIN_POWER          P4_0
#define MAX4147X_PIN_RSSI           P3_1
#define MAX4147X_PIN_DATA           P5_6

#define MAX4147X_I2C_SCL            P3_5
#define MAX4147X_I2C_SDA            P3_4

#define MAX4147X_SPI_SCK            P5_0
#define MAX4147X_SPI_MOSI           P5_1
#define MAX4147X_SPI_MISO           P5_2
#define MAX4147X_SPI_SSEL           P3_0

#define MAX4147X_I2C_ADDRESS        0xD6

#define Q_CONF_LEN                  15

extern const uint8_t default_register_value_0[Q_CONF_LEN];
extern const uint8_t default_register_value_1[Q_CONF_LEN];

template <class REG>
class MAX4147X
{
private:
    REG *reg_map;
    I2C *i2c_handler;
    SPI *spi_handler;
    DigitalOut *ssel;
    DigitalOut *power_pin;

    DigitalIn *data_read; // data sent pin

    uint8_t preset_mode;
    float crystal_frequency ;
    float center_frequency;
    float baud_rate;
    float baud_rate_ratio;

    typedef enum {
        DEMOD_ADDR =            0x00,
        AGC_ADDR =              0x01,
        IF_CHF_SEL_ADDR =       0x02,
        PDF_CFG_ADDR =          0x03,
        ATH_CFG1_ADDR =         0x04,
        ATH_CFG2_ADDR =         0x05,
        ATH_CFG3_ADDR =         0x06,
        AFC_CFG1_ADDR =         0x07,
        AFC_CFG2_ADDR =         0x08,
        LO_CTR_FREQ3_ADDR =     0x09,
        LO_CTR_FREQ2_ADDR =     0x0A,
        LO_CTR_FREQ1_ADDR =     0x0B,
        PREAMBLE_CFG1_ADDR =    0x0C,
        PREAMBLE_WORD1_ADDR =   0x0D,
        PREAMBLE_WORD2_ADDR =   0x0E,
        RSSI_ADDR =             0x10,
        FEI_ADDR =              0x11,
        PDF_OUT_ADDR =          0x12,
        ISR_ADDR =              0x13,
        CDR_CFG1_ADDR =         0x35,
        STATE_CTRL1_ADDR =      0x14,
        STATE_CTRL2_ADDR =      0x15,
        STATE_CTRL3_ADDR =      0x16,
        WUT1_ADDR =             0x17,
        WUT2_ADDR =             0x18,
        AFE_CTL1_ADDR =         0x19,
        IR_ADJUST_ADDR =        0x1A,
        PART_NUM_ADDR =         0x1E,
        REV_NUM_ADDR =          0x1F,
        STATUS_ADDR =           0x27
     }register_address_t;

     /**
      * @brief Sets LO Center Frequency, lower byte of 24-bit word (Register: LO_CTR_FREQ1 ,0x0B)
      *
      * @param[in] lo_ctr_freq_lower
      *
      * @returns 0 on success, negative error code on failure.
      *
      * @description LO Center Frequency, lower byte of 24-bit word
      */
     int set_lo_ctr_freq_lower(uint8_t lo_ctr_freq_lower);

     /**
      * @brief Sets LO Center Frequency, middle byte of 24-bit word (Register: LO_CTR_FREQ2 ,0x0A)
      *
      * @param[in] lo_ctr_freq_middle
      *
      * @returns 0 on success, negative error code on failure.
      *
      * @description LO Center Frequency, middle byte of 24-bit word
      */
     int set_lo_ctr_freq_middle(uint8_t lo_ctr_freq_middle);

     /**
      * @brief Sets LO Center Frequency, upper byte of 24-bit word (Register: LO_CTR_FREQ3 ,0x09)
      *
      * @param[in] lo_ctr_freq_upper
      *
      * @returns 0 on success, negative error code on failure.
      *
      * @description LO Center Frequency, upper byte of 24-bit word
      */
     int set_lo_ctr_freq_upper(uint8_t lo_ctr_freq_upper);

     /**
      * @brief Sets Preamble Bit Pattern Length before Manchester Coding (Register: PREAMBLE_CFG1 ,0x0C)
      *
      * @param[in] preamb_len
      *
      * @returns 0 on success, negative error code on failure.
      *
      * @description Preamble Bit Pattern Length before Manchester Coding
      *              Bit Pattern Length = Register Field Value +1
      */
     int set_preamb_len(uint8_t preamb_len);

     /**
      * @brief Sets Lower Byte of the Preamble Bit Pattern before Manchester Coding (Register: PREAMBLE_WORD1 ,0x0D)
      *
      * @param[in] preamb_word_lower
      *
      * @returns 0 on success, negative error code on failure.
      *
      * @description Lower Byte of the Preamble Bit Pattern before Manchester Coding
      */
     int set_preamb_word_lower(uint8_t preamb_word_lower);

     /**
      * @brief Sets Upper Byte of the Preamble Bit Pattern before Manchester Coding (Register: PREAMBLE_WORD2 ,0x0E)
      *
      * @param[in] preamb_word_upper
      *
      * @returns 0 on success, negative error code on failure.
      *
      * @description Upper Byte of the Preamble Bit Pattern before Manchester Coding
      */
     int set_preamb_word_upper(uint8_t preamb_word_upper);


protected:

    int io_read(uint8_t *data, uint32_t length);

public:

    //Constructors
    MAX4147X(REG *reg, SPI *spi, DigitalOut *cs, DigitalOut *powerPin, DigitalIn *dataPin);

    MAX4147X(REG *reg, SPI *spi, DigitalOut *powerPin, DigitalIn *dataPin);

    MAX4147X(REG *reg, I2C *i2c, DigitalOut *powerPin, DigitalIn *dataPin);

    typedef enum
    {
        Manchester = 0,
        NRZ = 1
    }encoding_t;

    encoding_t encoding;

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : DEMOD (0x00)
     *  - Bit Fields    : [7:6]
     *  - Default       : 0x1
     *  - Description   : RSSI Peak Detector Discharge Time
     */
    typedef enum {
        RSSI_DT_DEFAULT_VALUE_X0_5,  /**< 0x0: 1/2 default value */
        RSSI_DT_DEFAULT_VALUE,       /**< 0x1: default value */
        RSSI_DT_DEFAULT_VALUE_X2,    /**< 0x2: 0x2: 2x default value */
        RSSI_DT_DEFAULT_VALUE_X4,    /**< 0x3: 0x3: 4x default value */
    } rssi_dt_t;

    /**
     * @brief Set RSSI Peak Detector Discharge Time
     *
     * @param[in] rssi_dt
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_rssi_dt(rssi_dt_t rssi_dt);

    /**
     * @brief Get RSSI Peak Detector Discharge Time
     *
     * @param[in] rssi_dt
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_rssi_dt(rssi_dt_t *rssi_dt);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : DEMOD (0x00)
     *  - Bit Fields    : [5:3]
     *  - Default       : 0x0
     *  - Description   : Demodulator Parameter #2 to be used only in FSK mode. Must be programmed according to
     *                    the table of FSK Demodulator Configuration.
     */
    typedef enum {
        DEMOD_FSK_CONF_0,  /**< 0x0: FSK Demod Config Index = 0, 14 */
        DEMOD_FSK_CONF_1,  /**< 0x1: FSK Demod Config Index = 1, 15 */
        DEMOD_FSK_CONF_2,  /**< 0x2: FSK Demod Config Index = 2, 16 */
        DEMOD_FSK_CONF_3,  /**< 0x3: FSK Demod Config Index = 3, 6, 17, 20 */
        DEMOD_FSK_CONF_4,  /**< 0x4: FSK Demod Config Index = 4, 7, 9, 11, 18, 21, 23, 25 */
        DEMOD_FSK_CONF_5,  /**< 0x5: FSK Demod Config Index = 5, 8, 10, 12, 19, 22, 24, 26 */
        DEMOD_FSK_CONF_6,  /**< 0x6: FSK Demod Config Index = 13, 27 */
        DEMOD_FSK_CONF_7   /**< 0x7: Invalid value */
    } demod_fsk_t;

    /**
     * @brief Set Demodulator Parameter #2
     *
     * @param[in] demod_fsk
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_demod_fsk(demod_fsk_t demod_fsk);

    /**
     * @brief Get Demodulator Parameter #2
     *
     * @param[in] demod_fsk
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_demod_fsk(demod_fsk_t *demod_fsk);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : DEMOD (0x00)
     *  - Bit Fields    : [2:0]
     *  - Default       : 0x4
     *  - Description   : Demodulator Parameter #1.
     *                    Conditions / Recommended Value
     *                    ASK_FSK_SEL=1 / 4 - CHF_SEL
     *                    ASK_FSK_SEL=0, ATH_TYPE=0 / min(2+SRC_LG, 7)
     *                    ASK_FSK_SEL=0, ATH_TYPE=1 / min(3+SRC_LG,7)
     */
    typedef enum {
        DEMOD_TCTRL_CONF_0,  /**< 0x0: 1/16 Default */
        DEMOD_TCTRL_CONF_1,  /**< 0x1: 1/8 Default */
        DEMOD_TCTRL_CONF_2,  /**< 0x2: 1/4 Default */
        DEMOD_TCTRL_CONF_3,  /**< 0x3: 1/2 Default */
        DEMOD_TCTRL_CONF_4,  /**< 0x4: Default value */
        DEMOD_TCTRL_CONF_5,  /**< 0x5: 2x Default */
        DEMOD_TCTRL_CONF_6,  /**< 0x6: 4x Default */
        DEMOD_TCTRL_CONF_7   /**< 0x7: 8x Default */
    } demod_tctrl_t;

    /**
     * @brief Set Demodulator Parameter #1
     *
     * @param[in] demod_tctrl
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_demod_tctrl(demod_tctrl_t demod_tctrl);

    /**
     * @brief Get Demodulator Parameter #1
     *
     * @param[in] demod_tctrl
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_demod_tctrl(demod_tctrl_t *demod_tctrl);

    /**
     * @brief Sets AGC-Release Threshold Fine Tune (Register: AGC ,0x01)
     *
     * @param[in] agc_threl
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description AGC-Release Threshold Fine Tune. Recommended value is 0x9 when data rate is lower than 52kbps, or 0xF when data rate is higher than 52kbps.
     */
    int set_agc_threl(uint8_t agc_threl);

    /**
     * @brief Gets AGC-Release Threshold Fine Tune (Register: AGC ,0x01)
     *
     * @param[in] agc_threl
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description AGC-Release Threshold Fine Tune. Recommended value is 0x9 when data rate is lower than 52kbps, or 0xF when data rate is higher than 52kbps.
     */
    int get_agc_threl(uint8_t *agc_threl);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : AGC (0x01)
     *  - Bit Fields    : [1:0]
     *  - Default       : 0x2
     *  - Description   : AGC Operation Mode
     */
    typedef enum {
        AGC_EN_BO_AGC_DISABLED_MAX_GAIN,     /**< 0x0: AGC disabled, max gain */
        AGC_EN_BO_AGC_DISABLED_BO_ADC_BUF,   /**< 0x1: AGC disabled, back off ADC buffer */
        AGC_EN_BO_AGC_ENABLED,               /**< 0x2: AGC enabled */
        AGC_EN_BO_AGC_ENABLED_BO_ADC_BUF     /**< 0x3: AGC enabled, back off ADC buffer */
    } agc_en_bo_t;

    /**
     * @brief Set AGC Operation Mode
     *
     * @param[in] agc_op_mode
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_agc_en_bo(agc_en_bo_t agc_op_mode);

    /**
     * @brief Get AGC Operation Mode
     *
     * @param[in] agc_op_mode
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_agc_en_bo(agc_en_bo_t *agc_op_mode);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : IF_CHF_SEL (0x02)
     *  - Bit Fields    : [4]
     *  - Default       : 0x0
     *  - Description   : ASK/FSK Selection
     */
    typedef enum {
        ASK_FSK_SEL_ASK,    /**< 0x0: ASK demodulation */
        ASK_FSK_SEL_FSK     /**< 0x1: FSK demodulation */
    } ask_fsk_sel_t;

    /**
     * @brief Sets Demodulation Mode to ASK or FSK
     *
     * @param[in] ask_fsk_sel
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_ask_fsk_sel(ask_fsk_sel_t ask_fsk_sel);

    /**
     * @brief Gets Demodulation Mode
     *
     * @param[in] ask_fsk_sel
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_ask_fsk_sel(ask_fsk_sel_t *ask_fsk_sel);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : IF_CHF_SEL (0x02)
     *  - Bit Fields    : [3]
     *  - Default       : 0x0
     *  - Description   : Intermediate Frequency Selection
     */
    typedef enum {
        IF_SEL_400_KHZ,     /**< 0x0: 400kHz */
        IF_SEL_200_KHZ      /**< 0x1: 200kHz */
    } if_sel_t;

    /**
     * @brief Sets Intermediate Frequency
     *
     * @param[in] if_sel
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_if_sel(if_sel_t if_sel);

    /**
     * @brief Gets Intermediate Frequency
     *
     * @param[in] if_sel
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_if_sel(if_sel_t *if_sel);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : IF_CHF_SEL (0x02)
     *  - Bit Fields    : [2:0]
     *  - Default       : 0x0
     *  - Description   : Channel Filter Selection
     */
    typedef enum {
        CHF_SEL_RXBW_340_170_KHZ,   /**< 0x0: RXBW = 340kHz or 170kHz */
        CHF_SEL_RXBW_120_60_KHZ,    /**< 0x1: RXBW = 120kHz or 60kHz */
        CHF_SEL_RXBW_52_26_KHZ,     /**< 0x2: RXBW = 52kHz or 26kHz */
        CHF_SEL_RXBW_24_12_KHZ,     /**< 0x3: RXBW = 24kHz or 12kHz */
        CHF_SEL_RXBW_12_6_KHZ,      /**< 0x4: RXBW = 12kHz or 6kHz */
        CHF_SEL_INV0,               /**< 0x5: Invalid value */
        CHF_SEL_INV1,               /**< 0x6: Invalid value */
        CHF_SEL_INV2                /**< 0x7: Invalid value */
    } chf_sel_t;

    /**
     * @brief Sets Channel Filter
     *
     * @param[in] chf_sel
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_chf_sel(chf_sel_t chf_sel);

    /**
     * @brief Gets Channel Filter Selection
     *
     * @param[in] chf_sel
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_chf_sel(chf_sel_t *chf_sel);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : PDF_CFG (0x03)
     *  - Bit Fields    : [7]
     *  - Default       : 0x0
     *  - Description   : Output Buffer Selection. Low delay buffer can only be selected
     *                    when (SRC_LG >= 3) or (SRC_LG = 2 and SRC_SM is even).
     */
    typedef enum {
        LF_BUF_DEFAULT_SELECTION,   /**< 0x0: Default selection */
        LF_BUF_LOW_DELAY_BUF        /**< 0x1: Low delay buffer */
    } ld_buf_t;

    /**
     * @brief Sets Output Buffer
     *
     * @param[in] ld_buf
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_ld_buf(ld_buf_t ld_buf);

    /**
     * @brief Gets Output Buffer Selection
     *
     * @param[in] ld_buf
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_ld_buf(ld_buf_t *ld_buf);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : PDF_CFG (0x03)
     *  - Bit Fields    : [6]
     *  - Default       : 0x0
     *  - Description   : Post Demodulation Filter Bandwidth Control
     */
    typedef enum {
        LF_BW_DEFAULT_BW,       /**< 0x0: Default selection */
        LF_BW_DEFAULT_BW_X1_67  /**< 0x1: Low delay buffer */
    } ld_bw_t;

    /**
     * @brief Sets Post Demodulation Filter Bandwidth
     *
     * @param[in] ld_bw
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_ld_bw(ld_bw_t ld_bw);

    /**
     * @brief Gets Post Demodulation Filter Bandwidth
     *
     * @param[in] ld_bw
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_ld_bw(ld_bw_t *ld_bw);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : PDF_CFG (0x03)
     *  - Bit Fields    : [5:3]
     *  - Default       : 0x2
     *  - Description   : "Large" adjustment to the Sample Rate Converter used to
     *                    calculate the recommended data rate. See Configuration
     *                    Guidance Tables and Recommended Data Rate Equation
     */
    typedef enum {
        SRC_LG_CONF_0,  /**< 0x0: 4x Default */
        SRC_LG_CONF_1,  /**< 0x1: 2x Default */
        SRC_LG_CONF_2,  /**< 0x2: Default rate */
        SRC_LG_CONF_3,  /**< 0x3: 1/2 Default */
        SRC_LG_CONF_4,  /**< 0x4: 1/4 Default */
        SRC_LG_CONF_5,  /**< 0x5: 1/8 Default */
        SRC_LG_CONF_6,  /**< 0x6: 1/16 Default */
        SRC_LG_CONF_7   /**< 0x7: 1/32 Default */
    } src_lg_t;

    /**
     * @brief Sets "Large" adjustment to the Sample Rate Converter
     *
     * @param[in] src_lg
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_src_lg(src_lg_t src_lg);

    /**
     * @brief Gets "Large" adjustment of the Sample Rate Converter
     *
     * @param[in] src_lg
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_src_lg(src_lg_t *src_lg);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : PDF_CFG (0x03)
     *  - Bit Fields    : [2:0]
     *  - Default       : 0x0
     *  - Description   : "Small" adjustment to the Sample Rate Converter used to
     *                    calculate the recommended data rate. See Configuration
     *                    Guidance Tables and Recommended Data Rate Equation
     */
    typedef enum {
        SRC_SM_CONF_0,  /**< 0x0: Default rate */
        SRC_SM_CONF_1,  /**< 0x1: 8/9 Default */
        SRC_SM_CONF_2,  /**< 0x2: 8/10 Default */
        SRC_SM_CONF_3,  /**< 0x3: 8/11 Default */
        SRC_SM_CONF_4,  /**< 0x4: 8/12 Default */
        SRC_SM_CONF_5,  /**< 0x5: 8/13 Default */
        SRC_SM_CONF_6,  /**< 0x6: 8/14 Default */
        SRC_SM_CONF_7   /**< 0x7: 8/15 Default */
    } src_sm_t;

    /**
     * @brief Sets "Small" adjustment to the Sample Rate Converter
     *
     * @param[in] src_sm
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_src_sm(src_sm_t src_sm);

    /**
     * @brief Gets "Small" adjustment of the Sample Rate Converter
     *
     * @param[in] src_sm
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_src_sm(src_sm_t *src_sm);

    /**
     * @brief Sets Parameter #1 for ASK Threshold Generation (Register: ATH_CFG1 ,0x04)
     *
     * @param[in] ath_lb
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Parameter #1 for ASK Threshold Generation: lower bound of threshold in 8-bit signed, two's complement format. Valid value from -128 to 0.
     */
    int set_ath_lb(uint8_t ath_lb);

    /**
     * @brief Gets Parameter #1 of ASK Threshold Generation (Register: ATH_CFG1 ,0x04)
     *
     * @param[in] ath_lb
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Parameter #1 for ASK Threshold Generation: lower bound of threshold in 8-bit signed, two's complement format. Valid value from -128 to 0.
     */
    int get_ath_lb(uint8_t *ath_lb);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : ATH_CFG2 (0x05)
     *  - Bit Fields    : [6:5]
     *  - Default       : 0x0
     *  - Description   : Parameter #4 for ASK Threshold Generation: peak-hold time control
     *                    in the "adaptive Peak Detector" (aPD) method
     */
    typedef enum {
        ATH_DT_DEFAULT_DISCHARGE_TIME,  /**< 0x0: Default discharge time, suggested for Manchester data, close to Rb */
        ATH_DT_DISCHARGE_TIME_X2,       /**< 0x1: 2x Discharge time, suggested for Manchester data, lower than Rb */
        ATH_DT_DISCHARGE_TIME_X4,       /**< 0x2: 4x Discharge time */
        ATH_DT_DISCHARGE_TIME_X8        /**< 0x3: 8x Discharge time, suggested for NRZ data */
    } ath_dt_t;

    /**
     * @brief Sets Parameter #4 for ASK Threshold Generation
     *
     * @param[in] ath_dt
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_ath_dt(ath_dt_t ath_dt);

    /**
     * @brief Gets Parameter #4 of ASK Threshold Generation
     *
     * @param[in] ath_dt
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_ath_dt(ath_dt_t *ath_dt);

    /**
     * @brief Sets Parameter #2 for ASK Threshold Generation (Register: ATH_CFG2 ,0x05)
     *
     * @param[in] ath_tc
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description: Parameter #2 for ASK Threshold Generation: to be programmed according to SRC_LG
     */
    int set_ath_tc(uint8_t ath_tc);

    /**
     * @brief Gets Parameter #2 of ASK Threshold Generation (Register: ATH_CFG2 ,0x05)
     *
     * @param[in] ath_tc
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description: Parameter #2 for ASK Threshold Generation: to be programmed according to SRC_LG
     */
    int get_ath_tc(uint8_t *ath_tc);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : ATH_CFG3 (0x06)
     *  - Bit Fields    : [6]
     *  - Default       : 0x0
     *  - Description   : ASK Threshold Adjustment Method
     */
    typedef enum {
        ATH_TYPE_PRELPF_MANCHESTER,     /**< 0x0: Precharged lowpass filter (preLPF) (Manchester) */
        ATH_TYPE_APD_NRZ                /**< 0x1: Adaptive peak detector (aPD)(NRZ) */
    } ath_type_t;

    /**
     * @brief Sets ASK Threshold Adjustment Method
     *
     * @param[in] ath_type
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_ath_type(ath_type_t ath_type);

    /**
     * @brief Gets ASK Threshold Adjustment Method
     *
     * @param[in] ath_type
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_ath_type(ath_type_t *ath_type);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : ATH_CFG3 (0x06)
     *  - Bit Fields    : [5]
     *  - Default       : 0x0
     *  - Description   : Parameter #5 for ASK Threshold Generation: bandwidth control for precharged LPF (preLPF)
     */
    typedef enum {
        ATH_BW_DEFAULT,     /**< 0x0: Default bandwidth */
        ATH_BW_DEFAULT_X2   /**< 0x1: 2x default */
    } ath_bw_t;

    /**
     * @brief Sets Parameter #5 for ASK Threshold Generation
     *
     * @param[in] ath_bw
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_ath_bw(ath_bw_t ath_bw);

    /**
     * @brief Gets Parameter #5 of ASK Threshold Generation
     *
     * @param[in] ath_bw
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_ath_bw(ath_bw_t *ath_bw);

    /**
     * @brief Sets Parameter #3 for ASK Threshold Generation (Register: ATH_CFG3 ,0x06)
     *
     * @param[in] ath_gc
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Parameter #3 for ASK Threshold Generation: to be programmed according to IF_SEL and CHF_SEL
     */
    int set_ath_gc(uint8_t ath_gc);

    /**
     * @brief Gets Parameter #3 of ASK Threshold Generation (Register: ATH_CFG3 ,0x06)
     *
     * @param[in] ath_gc
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Parameter #3 for ASK Threshold Generation: to be programmed according to IF_SEL and CHF_SEL
     */
    int get_ath_gc(uint8_t *ath_gc);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : AFC_CFG1 (0x07)
     *  - Bit Fields    : [4:2]
     *  - Default       : 0x0
     *  - Description   : AFC Frequency Offset Limit
     */
    typedef enum {
        AFC_MO_CONF_0,  /**< 0x0: AFC disabled */
        AFC_MO_CONF_1,  /**< 0x1: 1/7 Max offset */
        AFC_MO_CONF_2,  /**< 0x2: 2/7 Max offset */
        AFC_MO_CONF_3,  /**< 0x3: 3/7 Max offset */
        AFC_MO_CONF_4,  /**< 0x4: 4/7 Max offset */
        AFC_MO_CONF_5,  /**< 0x5: 5/7 Max offset */
        AFC_MO_CONF_6,  /**< 0x6: 6/7 Max offset */
        AFC_MO_CONF_7,  /**< 0x7: Max offset */
    } afc_mo_t;

    /**
     * @brief Sets AFC Frequency Offset Limit
     *
     * @param[in] afc_mo
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_afc_mo(afc_mo_t afc_mo);

    /**
     * @brief Gets AFC Frequency Offset Limit
     *
     * @param[in] ath_bw
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_afc_mo(afc_mo_t *afc_mo);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : AFC_CFG1 (0x07)
     *  - Bit Fields    : [1:0]
     *  - Default       : 0x2
     *  - Description   : AFC Loop Gain Control
     */
    typedef enum {
        AFC_LG_CONF_DEFAULT_X0_25,  /**< 0x0: 1/4 Default */
        AFC_LG_CONF_DEFAULT_X0_5,   /**< 0x1: 1/2 Default */
        AFC_LG_CONF_DEFAULT,        /**< 0x2: Default gain, FSK typical setting */
        AFC_LG_CONF_DEFAULT_X2      /**< 0x3: 2x Default, ASK typical setting */
    } afc_lg_t;

    /**
     * @brief Sets AFC Loop Gain Control
     *
     * @param[in] afc_lg
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_afc_lg(afc_lg_t afc_lg);

    /**
     * @brief Gets AFC Loop Gain Control
     *
     * @param[in] afc_lg
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_afc_lg(afc_lg_t *afc_lg);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : AFC_CFG2 (0x08)
     *  - Bit Fields    : [6]
     *  - Default       : 0x0
     *  - Description   : Control bit to Freeze AFC after Preamble Detected. Not used in ASK mode.
     */
    typedef enum {
        PAD_FREEZE_AFC_NOT_FREEZE,  /**< 0x0: Not to freeze AFC */
        PAD_FREEZE_AFC_FREEZE       /**< 0x1: Freeze AFC (stop PLL frequency update) once preamble is detected */
    } pad_freeze_afc_t;

    /**
     * @brief Sets Control bit to Freeze AFC after Preamble Detected
     *
     * @param[in] pad_freeze_afc
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_pad_freeze_afc(pad_freeze_afc_t pad_freeze_afc);

    /**
     * @brief Gets Control bit of Freeze AFC after Preamble Detected
     *
     * @param[in] pad_freeze_afc
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_pad_freeze_afc(pad_freeze_afc_t *pad_freeze_afc);

    /**
     * @brief Gets LO Center Frequency, upper byte of 24-bit word (Register: LO_CTR_FREQ3 ,0x09)
     *
     * @param[in] lo_ctr_freq_upper
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description LO Center Frequency, upper byte of 24-bit word
     */
    int get_lo_ctr_freq_upper(uint8_t *lo_ctr_freq_upper);

    /**
     * @brief Gets LO Center Frequency, middle byte of 24-bit word (Register: LO_CTR_FREQ2 ,0x0A)
     *
     * @param[in] lo_ctr_freq_middle
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description LO Center Frequency, middle byte of 24-bit word
     */
    int get_lo_ctr_freq_middle(uint8_t *lo_ctr_freq_middle);

    /**
     * @brief Gets LO Center Frequency, lower byte of 24-bit word (Register: LO_CTR_FREQ1 ,0x0B)
     *
     * @param[in] lo_ctr_freq_lower
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description LO Center Frequency, lower byte of 24-bit word
     */
    int get_lo_ctr_freq_lower(uint8_t *lo_ctr_freq_lower);

    /**
     * @brief Gets Preamble Bit Pattern Length before Manchester Coding (Register: PREAMBLE_CFG1 ,0x0C)
     *
     * @param[in] preamb_len
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Preamble Bit Pattern Length before Manchester Coding
     *              Bit Pattern Length = Register Field Value +1
     */
    int get_preamb_len(uint8_t *preamb_len);

    /**
     * @brief Gets Lower Byte of the Preamble Bit Pattern before Manchester Coding (Register: PREAMBLE_WORD1 ,0x0D)
     *
     * @param[in] preamb_word_lower
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Lower Byte of the Preamble Bit Pattern before Manchester Coding
     */
    int get_preamb_word_lower(uint8_t *preamb_word_lower);

    /**
     * @brief Gets Upper Byte of the Preamble Bit Pattern before Manchester Coding (Register: PREAMBLE_WORD2 ,0x0E)
     *
     * @param[in] preamb_word_upper
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Upper Byte of the Preamble Bit Pattern before Manchester Coding
     */
    int get_preamb_word_upper(uint8_t *preamb_word_upper);

    /**
     * @brief Gets Received Signal Strength Indicator (RSSI) (Register: RSSI ,0x10)
     *
     * @param[in] rssi
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Received Signal Strength Indicator (RSSI)
     *              8-bit unsigned integer
     */
    int get_rssi(uint8_t *rssi);

    /**
     * @brief Gets AFC Frequency Error Indicator (FEI) (Register: FEI ,0x11)
     *
     * @param[in] fei
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description AFC Frequency Error Indicator (FEI)
     *              8-bit signed integer in two's complement format
     */
    int get_fei(uint8_t *fei);

    /**
     * @brief Gets Post Demodulation Filter (PDF) Read Out (Register: PDF_OUT, 0x12)
     *
     * @param[in] pdf_out
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Post Demodulation Filter (PDF) Read Out
     *              8-bit signed integer in two's complement format
     */
    int get_pdf_out(uint8_t *pdf_out);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : ISR (0x13)
     *  - Bit Fields    : [0]
     *  - Default       : 0x0
     *  - Description   : Interrupt Status Register Bit 0: preamble detector in self-polling mode
     */
    typedef enum {
        PREAMB_DET_NO_INT_EVENT,    /**< 0x0: No interrupt event */
        PREAMB_DET_PREAMB_DETECTED  /**< 0x1: Preamble detected in self-polling */
    } preamb_det_t;

    /**
     * @brief Gets Post Demodulation Filter (PDF) Read Out
     *
     * @param[in] preamb_det
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_preamb_det(preamb_det_t *preamb_det);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : CDR_CFG1 (0x35)
     *  - Bit Fields    : [1:0]
     *  - Default       : 0x0
     *  - Description   :
     */
    typedef enum {
        CDR_MODE_CDR_DISABLED,                  /**< 0x0: CDR disabled */
        CDR_MODE_CLKOUT_EN_DATAOUT_UNTIMED,     /**< 0x1: Clock out enabled, DATAOUT untimed */
        CDR_MODE_CLKOUT_DIS_DATAOUT_RETIMED,    /**< 0x2: Clock out disabled, DATAOUT retimed */
        CDR_MODE_CLKOUT_EN_DATAOUT_RETIMED      /**< 0x3: Clock out enabled, DATAOUT retimed */
    } cdr_mode_t;

    /**
     * @brief Sets CDR_MODE
     *
     * @param[in] cdr_mode
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_cdr_mode(cdr_mode_t cdr_mode);

    /**
     * @brief Gets CDR_MODE
     *
     * @param[in] cdr_mode
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_cdr_mode(cdr_mode_t *cdr_mode);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : STATE_CTRL1 (0x14)
     *  - Bit Fields    : [2]
     *  - Default       : 0x0
     *  - Description   : XO Enable Bit
     */
    typedef enum {
        EN_XO_DIS_XO,   /**< 0x0: Disable XO */
        EN_XO_EN_XO     /**< 0x1: Enable XO */
    } en_xo_t;

    /**
     * @brief Sets XO Enable Bit
     *
     * @param[in] en_xo
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_en_xo(en_xo_t en_xo);

    /**
     * @brief Gets XO Enable Bit
     *
     * @param[in] en_xo
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_en_xo(en_xo_t *en_xo);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : STATE_CTRL1 (0x14)
     *  - Bit Fields    : [1]
     *  - Default       : 0x0
     *  - Description   : Wake-Up Timer (WUT) Enable Bit
     */
    typedef enum {
        WUT_EN_DIS_WUT,     /**< 0x0: Disable WUT */
        WUT_EN_EN_WUT       /**< 0x1: Enable WUT */
    } wut_en_t;

    /**
     * @brief Sets Wake-Up Timer (WUT) Enable Bit
     *
     * @param[in] wut_en
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_wut_en(wut_en_t wut_en);

    /**
     * @brief Gets Wake-Up Timer (WUT) Enable Bit
     *
     * @param[in] wut_en
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_wut_en(wut_en_t *wut_en);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : STATE_CTRL1 (0x14)
     *  - Bit Fields    : [0]
     *  - Default       : 0x0
     *  - Description   : Slave Receiver Enable Bit
     */
    typedef enum {
        SLAVE_RX_EN_DIS_RECEIVER,   /**< 0x0: Disable Receiver */
        SLAVE_RX_EN_EN_RECEIVER     /**< 0x1: Enable Receiver */
    } slave_rx_en_t;

    /**
     * @brief Sets Slave Receiver Enable Bit
     *
     * @param[in] slave_rx_en
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_slave_rx_en(slave_rx_en_t slave_rx_en);

    /**
     * @brief Gets Slave Receiver Enable Bit
     *
     * @param[in] slave_rx_en
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_slave_rx_en(slave_rx_en_t *slave_rx_en);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : STATE_CTRL2 (0x15)
     *  - Bit Fields    : [1:0]
     *  - Default       :
     *  - Description   : Receiver State Machine Register
     */
    typedef enum {
        RX_STATE_STANDBY,           /**< 0x0: Standby */
        RX_STATE_SLAVE_RECEIVER,    /**< 0x1: Slave receiver */
        RX_STATE_W_IN_SELFPOLLING,  /**< 0x2: Wait in self-polling */
        RX_STATE_POLLING_RECEIVER   /**< 0x3: Polling receiver */
    } rx_state_t;

    /**
     * @brief Gets Receiver State Machine Register
     *
     * @param[in] rx_state
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_rx_state(rx_state_t *rx_state);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : STATE_CTRL3 (0x16)
     *  - Bit Fields    : [1:0]
     *  - Default       : 0x03
     *  - Description   : Receiver Front-End Turn-On Time
     */
    typedef enum {
        RX_RESET_TIME_80US,     /**< 0x0: 0.08ms */
        RX_RESET_TIME_160US,    /**< 0x1: 0.16ms */
        RX_RESET_TIME_240US,    /**< 0x2: 0.24ms */
        RX_RESET_TIME_320US     /**< 0x3: 0.32ms */
    } rx_reset_time_t;

    /**
     * @brief Sets Receiver Front-End Turn-On Time
     *
     * @param[in] rx_reset_time
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_rx_reset_time(rx_reset_time_t rx_reset_time);

    /**
     * @brief Gets Receiver Front-End Turn-On Time
     *
     * @param[in] rx_reset_time
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_rx_reset_time(rx_reset_time_t *rx_reset_time);

    /**
     * @brief Sets Duration in POLLINGRX State (Register: WUT1, 0x17)
     *
     * @param[in] tdet
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Duration in POLLINGRX State: from 0.48ms to 20.88ms, in step size of 0.08ms
     *              Duration (ms) = 0.48 + 0.08 x (tdet)
     */
    int set_tdet(uint8_t tdet);

    /**
     * @brief Gets Duration in POLLINGRX State (Register: WUT1, 0x17)
     *
     * @param[in] tdet
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Duration in POLLINGRX State: from 0.48ms to 20.88ms, in step size of 0.08ms
     *              Duration (ms) = 0.48 + 0.08 x (tdet)
     */
    int get_tdet(uint8_t *tdet);

    /**
     * @brief Sets WUT Duty Cycle Control (Register: WUT2, 0x18)
     *
     * @param[in] tsby_tdet_ratio
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Duty Cycle = 1 / (2 + tsby_tdet_ratio)
     */
    int set_tsby_tdet_ratio(uint8_t tsby_tdet_ratio);

    /**
     * @brief Gets WUT Duty Cycle Control (WUT2, 0x18)
     *
     * @param[in] tsby_tdet_ratio
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Duty Cycle = 1 / (2 + tsby_tdet_ratio)
     */
    int get_tsby_tdet_ratio(uint8_t *tsby_tdet_ratio);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : AFE_CTL1 (0x19)
     *  - Bit Fields    : [7:6]
     *  - Default       : 0x02
     *  - Description   : Start Delay before Applying XO Clock to Digital
     */
    typedef enum {
        XOCLKDELAY_NO_DELAY,    /**< 0x0: No delay */
        XOCLKDELAY_16_CYCLE,    /**< 0x1: 16 cycle delay */
        XOCLKDELAY_32_CYCLE,    /**< 0x2: 32 cycle delay */
        XOCLKDELAY_64_CYCLE     /**< 0x3: 64 cycle delay */
    } xoclkdelay_t;

    /**
     * @brief Sets Start Delay before Applying XO Clock to Digital
     *
     * @param[in] xoclkdelay
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_xoclkdelay(xoclkdelay_t xoclkdelay);

    /**
     * @brief Gets Start Delay before Applying XO Clock to Digital
     *
     * @param[in] xoclkdelay
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_xoclkdelay(xoclkdelay_t *xoclkdelay);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : AFE_CTL1 (0x19)
     *  - Bit Fields    : [5:4]
     *  - Default       : 0x01
     *  - Description   : XO Clock Divider Ratio
     */
    typedef enum {
        XOCLKDIV_DIVIDE_BY_4,   /**< 0x0: Divide by 4 */
        XOCLKDIV_DIVIDE_BY_5,   /**< 0x1: Divide by 5 */
        XOCLKDIV_DIVIDE_BY_6,   /**< 0x2: Divide by 6 */
        XOCLKDIV_INVALID        /**< 0x3: Invalid Value */
    } xoclkdiv_t;

    /**
     * @brief Sets Start Delay before Applying XO Clock to Digital
     *
     * @param[in] xoclkdiv
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_xoclkdiv(xoclkdiv_t xoclkdiv);

    /**
     * @brief Gets Start Delay before Applying XO Clock to Digital
     *
     * @param[in] xoclkdelay
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_xoclkdiv(xoclkdiv_t *xoclkdiv);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : AFE_CTL1 (0x19)
     *  - Bit Fields    : [3]
     *  - Default       : 0x00
     *  - Description   : LO Injection Control
     */
    typedef enum {
        MIX_HS_LSBAR_TARGET_HT_LO_FREQ,     /**< 0x0: Targeted RF frequency higher than LO frequency */
        MIX_HS_LSBAR_TARGET_LT_LO_FREQ      /**< 0x1: Targeted RF frequency lower than LO frequency */
    } mix_hs_lsbar_t;

    /**
     * @brief Sets LO Injection Control
     *
     * @param[in] mix_hs_ls_bar
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_mix_hs_lsbar(mix_hs_lsbar_t mix_hs_lsbar);

    /**
     * @brief Gets LO Injection Control
     *
     * @param[in] mix_hs_ls_bar
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_mix_hs_lsbar(mix_hs_lsbar_t *mix_hs_lsbar);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : AFE_CTL1 (0x19)
     *  - Bit Fields    : [2:1]
     *  - Default       : 0x01
     *  - Description   : LO Divider Control
     */
    typedef enum {
        LODIV_PLL_DISABLED,         /**< 0x0: PLL disabled */
        LODIV_860MHZ_TO_960MHZ,     /**< 0x1: 860MHz to 960MHz */
        LODIV_425MHZ_TO_480MHZ,     /**< 0x2: 425MHz to 480MHz */
        LODIV_286MHZ_TO_320MHZ,     /**< 0x3: 286MHz to 320MHz */
    } lodiv_t;

    /**
     * @brief Sets LO Divider Control
     *
     * @param[in] lodiv
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_lodiv(lodiv_t lodiv);

    /**
     * @brief Gets LO Divider Control
     *
     * @param[in] lodiv
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_lodiv(lodiv_t *lodiv);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : AFE_CTL1 (0x19)
     *  - Bit Fields    : [0]
     *  - Default       : 0x01
     *  - Description   : PLL Mode Control: always program to 1
     */
    typedef enum {
        FRACMODE_INTEGER_N_PLL,     /**< 0x0: Integer-N PLL */
        FRACMODE_FRACTIONAL_N_PLL   /**< 0x1: Fractional-N PLL */
    } fracmode_t;

    /**
     * @brief Sets PLL Mode Control
     *
     * @param[in] fracmode
     *
     * @returns 0 on success, negative error code on failure.
     */
    int set_fracmode(fracmode_t fracmode);

    /**
     * @brief Gets PLL Mode Control
     *
     * @param[in] fracmode
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_fracmode(fracmode_t *fracmode);

    /**
     * @brief Sets Image Rejection Adjustment (Register: IR_ADJUST ,0x1A)
     *
     * @param[in] ir_adjust
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Image Rejection Adjustment. See the Image Rejection Calibration section for more information.
     */
    int set_ir_adjust(uint8_t ir_adjust);

    /**
     * @brief Gets Image Rejection Adjustment (Register: IR_ADJUST ,0x1A)
     *
     * @param[in] ir_adjust
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Image Rejection Adjustment. See the Image Rejection Calibration section for more information.
     */
    int get_ir_adjust(uint8_t *ir_adjust);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : PART_NUM (0x1E)
     *  - Bit Fields    : [7:0]
     *  - Default       :
     *  - Description   : Part Number Designator.
     *                    Read of part number requires EN_XO = 1
     */
    typedef enum {
        PART_NUM_MAX41470 = 0x70,   /**< 0x70 = MAX41470 */
        PART_NUM_MAX41473 = 0x73,   /**< 0x73 = MAX41473 */
        PART_NUM_MAX41474 = 0x74    /**< 0x74 = MAX41474 */
    } part_num_t;

    /**
     * @brief Gets Part Number Designator
     *
     * @param[in] part_num
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_part_num(part_num_t *part_num);

    /**
     * @brief Gets Revision Number of Chip (Register: REV_NUM ,0x1F)
     *
     * @param[in] rev_num
     *
     * @returns 0 on success, negative error code on failure.
     *
     * @description Revision Number of Chip
     */
    int get_rev_num(uint8_t *rev_num);

    /**
     * @brief Register Configuration
     *
     * @details
     *  - Register      : STATUS (0x27)
     *  - Bit Fields    : [0]
     *  - Default       :
     *  - Description   : PLL Lock Status
     */
    typedef enum {
        PLL_LOCK_NOT_LOCKED,    /**< 0x0: PLL is not locked */
        PLL_LOCK_LOCKED         /**< 0x1: PLL is locked */
    } pll_lock_t;

    /**
     * @brief Gets PLL Lock Status
     *
     * @param[in] pll_lock
     *
     * @returns 0 on success, negative error code on failure.
     */
    int get_pll_lock(pll_lock_t *pll_lock);

    /**
    * @brief Set power on/off
    *
    * @param[in] power 0 : power ON, 1: power OFF
    *
    * @returns 0 on success, negative error code on failure.
    */
    int set_power_on_off(uint8_t power);

    /**
     * @brief Operation state of the RF receiver
     *
     * @details
     *  - Default       : 0x00
     *  - Description   :
     */
    typedef enum {
        INITIALIZED = 0,
        UNINITIALIZED = 1,
        UNKNOWN = 2,
    } operation_mode_t;


    // Indicates whether initialization is successful
    operation_mode_t operation_mode;

    /* PUBLIC FUNCTION DECLARATIONS */

    /**
    * @brief Read from a register. Since 3-wire spi is not supported by mbed, the read_register function must be implemented by the user.
    *
    * @param[in] reg Address of a register to be read.
    * @param[out] value Pointer to save result value.
    * @param[in] len Size of result to be read.
    *
    * @returns 0 on success, negative error code on failure.
    */
    virtual int read_register(uint8_t reg, uint8_t *value, uint8_t len);

    /**
    * @brief Write to a register.
    *
    * @param[in] reg Address of a register to be written.
    * @param[out] value Pointer of value to be written to register.
    * @param[in] len Size of result to be written.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int write_register(uint8_t reg, const uint8_t *value, uint8_t len);

    /**
    * @brief Configures the crystal frequency of the chip (Fxtal)
    *
    * @param[in] freq crystal frequency values between 12.8 MHz and 19.2 MHz
    *
    * @returns 0 on success, negative error code on failure.
    */
    int set_crystal_frequency(float freq);

    /**
    * @brief Get the crystal frequency of the chip (Fxtal)
    *
    * @returns crystal frequency values between 12.8 MHz and 19.2 MHz
    */
    float get_crystal_frequency();

    /**
    * @brief Adjust baud rate
    *
    * @param[in] baud_rate preferred baud rate
    *
    * @returns 0 on success, negative error code on failure.
    *
    * @description It changes only the values of SRC_LG and SRC_SM
    *              Baud_rate for Manchester = (200/(2^(IF_SEL+CHF_SEL+SRC_LG)))/(4/(8+SRC_SM))
    *              Baud_rate for NRZ        = (200/(2^(IF_SEL+CHF_SEL+SRC_LG)))/(8/(8+SRC_SM))
    */
    int adjust_baud_rate(float baud_rate);

    /**
    * @brief Gets baud rate
    *
    * @returns baud rate.
    */
    float get_baud_rate();

    /**
    * @brief Adjust center/carrier frequency of the chip
    *
    * @param[in] freq center/carrier frequency value between 250 MHz and 950 MHz
    *
    * @returns 0 on success, negative error code on failure.
    */
    int set_center_frequency(float freq);

    /**
    * @brief Gets center/carrier frequency of the chip
    *
    * @returns center/carrier frequency value between 250 MHz and 950 MHz
    */
    float get_center_frequency();

    /**
    * @brief Initial programming steps after power on or soft reset.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int initial_programming(void);

    /**
    * @brief Configures the crystal divider to maintain the internal 3.2MHz time base.
    *
    * @returns 0 on success, negative error code on failure.
    *
    * @description The available crystal frequencies are 12.8MHz, 16.0MHz (default), and 19.2MHz.
    *              Crystal Freq(MHz) / Divider Ratio / XOCLKDIV
    *                   12.8         /      4        /   0
    *                   16.0         /      5        /   1
    *                   19.2         /      6        /   2
    */
    int adjust_crystal_divider(void);

    /**
    * @brief Read data for selected Preset/I2C/SPI mode
    *
    * @param[in] data data pointer for data to be read
    * @param[in] length legth of data to be read
    *
    * @returns 0 on success, negative error code on failure.
    */
    int read_data(uint8_t *data, uint32_t length);

    /**
    * @brief Updates Parameter #1 for ASK Threshold Generation(ath_lb) according to parameters
    *        Intermediate Frequency, Channel Filter, Sample Rate Converter and ASK/FSK Selection.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int update_ath_lb(void);

    /**
    * @brief Updates Parameter #2 for ASK Threshold Generation(ath_tc) according to parameters
    *        "Large" adjustment for the Sample Rate Converter(src_lg) and ASK/FSK Selection.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int update_ath_tc(void);

    /**
    * @brief Updates Parameter #3 for ASK Threshold Generation(ath_gc) according to parameters
    *        Intermediate Frequency, Channel Filter and ASK/FSK Selection.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int update_ath_gc(void);

    /**
    * @brief Updates Parameter #4 for ASK Threshold Generation(ath_dt) according to parameters
    *        Baud Rate Ratio, Encoding and ASK/FSK Selection.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int update_ath_dt(void);

    /**
    * @brief Updates Parameter #5 for ASK Threshold Generation(ath_bw) according to parameters
    *        Baud Rate Ratio, Encoding and ASK/FSK Selection.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int update_ath_bw(void);

    /**
    * @brief Updates AFC Frequency Offset Limit(afc_mo)
    *
    * @returns 0 on success, negative error code on failure.
    */
    int update_afc_mo(void);

    /**
    * @brief Updates AFC Loop Gain Control(afc_lg) according to ASK/FSK Selection.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int update_afc_lg(void);

    /**
    * @brief Updates Demodulator Parameter #1(demod_tctrl) according to parameters
    *        "Large" adjustment for the Sample Rate Converter(src_lg), Channel Filter(chf_sel),
    *        ASK Threshold Adjustment Method(ath_type) and ASK/FSK Selection.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int update_demod_tctrl(void);

    /**
    * @brief Updates AGC-Release Threshold Fine Tune(agc_threl) according to Data Rate,
    *        Encoding and ASK/FSK Selection.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int update_agc_threl(void);

    /**
    * @brief Updates ASK Threshold Adjustment Method(ath_type) according to Encoding and
    *        ASK/FSK Selection.
    *
    * @returns 0 on success, negative error code on failure.
    */
    int update_ath_type(void);

    /**
    * @brief Adjust demodulation type
    *
    * @param[in] demodulation_type ASK(0x00) or FSK(0x01)
    *
    * @returns 0 on success, negative error code on failure.
    *
    * @description Adjusts the Register settings of the Receiver according to the modulation type.
    */
    int adjust_demodulation(ask_fsk_sel_t demodulation_type);

    /**
    * @brief Adjust encoding type
    *
    * @param[in] encoding_type Manchester(0x00) or NRZ(0x01)
    *
    * @returns 0 on success, negative error code on failure.
    *
    * @description Adjusts the register settings of the Receiver according to the encoding type.
    */
    int adjust_encoding_type(encoding_t encoding_type);

    /**
    * @brief Gets encoding type
    *
    * @returns encoding type
    *
    * @description
    */
    int get_encoding_type();

    /**
    * @brief Adjust receiver bandwidth(chf_sel). Used only in ASK mode.
    *
    * @param[in] receiver_bw Receiver bandwidth(kHz)
    *
    * @returns 0 on success, negative error code on failure.
    *
    * @description Adjusts the register settings of the receiver according to the desired bandwidth.
    */
    int adjust_receiver_bandwidth(int receiver_bw);

    /**
    * @brief Adjust intermediate frequency selection(if_sel). Used only in ASK mode.
    *
    * @param[in] if_sel intermediate frequency. 400kHz(0x00) or 200kHz(0x01).
    *
    * @returns 0 on success, negative error code on failure.
    *
    * @description Adjusts the register settings of the receiver according to the desired intermediate frequency.
    */
    int adjust_if_sel(if_sel_t if_sel);

    /**
    * @brief Adjust receiver bandwidth(chf_sel). Used only in FSK mode.
    *
    * @param[in] receiver_bw Receiver bandwidth(kHz).
    * @param[in] deviation FSK deviation(kHz).
    *
    * @returns 0 on success, negative error code on failure.
    *
    * @description Adjusts the register settings of the receiver according to the desired bandwidth and FSK deviation.
    */
    int adjust_receiver_bandwidth(int receiver_bw, float deviation);

    /**
    * @brief Adjust intermediate frequency selection(if_sel). Used only in FSK mode.
    *
    * @param[in] if_sel intermediate frequency. 400kHz(0x00) or 200kHz(0x01).
    * @param[in] deviation FSK deviation(kHz).
    *
    * @returns 0 on success, negative error code on failure.
    *
    * @description Adjusts the register settings of the receiver according to the desired intermediate frequency and FSK deviation.
    */
    int adjust_if_sel(if_sel_t if_sel, float deviation);

    /**
    * @brief Adjust Demodulator Parameter #2 according to desired FSK deviation.
    *
    * @param[in] deviation FSK deviation(kHz).
    *
    * @returns 0 on success, negative error code on failure.
    *
    * @description Adjusts the register settings of the receiver according to the desired FSK deviation.
    *              There are a total of 28 options for configuring the FSK demodulator. Please refer to the Demodulator
    *              Configuration section in the datasheet
    */
    int adjust_fsk_deviation(float deviation);

    /**
    * @brief Adjust Receiver's preamble detector
    *
    * @param[in] preamble Preamble bit pattern.
    * @param[in] preamb_len pattern length.
    *
    * @returns 0 on success, negative error code on failure.
    *
    * @description Adjusts the receiver's preamble detector to be used with the self-polling operation. The preamble must be Manchester encoded.
    */
    int adjust_preamble(int preamble, uint8_t preamb_len);
};

/** MAX41470 Device Class
*
* Hold configurations for the MAX41470
*/
class MAX41470 : public MAX4147X <max41470_reg_map_t>
{
    max41470_reg_map_t regmap;
    public:
    MAX41470(SPI *spi, DigitalOut *cs, DigitalOut *powerPin, DigitalIn *dataPin) : MAX4147X<max41470_reg_map_t>(&regmap, spi, cs, powerPin, dataPin) {}

    MAX41470(SPI *spi, DigitalOut *powerPin, DigitalIn *dataPin) : MAX4147X<max41470_reg_map_t>(&regmap, spi, powerPin, dataPin) {}
};

/** MAX41473 Device Class
*
* Hold configurations for the MAX41473
*/
class MAX41473 : public MAX4147X <max41473_4_reg_map_t>
{
max41473_4_reg_map_t regmap;
    public:
    MAX41473(I2C *i2c, DigitalOut *powerPin, DigitalIn *dataPin) : MAX4147X<max41473_4_reg_map_t>(&regmap, i2c, powerPin, dataPin) {}
};

/** MAX41474 Device Class
*
* Hold configurations for the MAX41474
*/
class MAX41474 : public MAX41473
{
    public:
    MAX41474(I2C *i2c, DigitalOut *powerPin, DigitalIn *dataPin) : MAX41473(i2c, powerPin, dataPin) {}
};

#endif /* MAX4147X_MAX4147X_H_ */
