/**********************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/


#include "mbed.h"
#include "bmi160.h"
#include "max32630hsp.h"

MAX32630HSP icarus(MAX32630HSP::VIO_3V3);

DigitalOut rLED(LED1, LED_OFF);
DigitalOut gLED(LED2, LED_OFF);
DigitalOut bLED(LED3, LED_OFF);

Serial pc(USBTX,USBRX);

I2C i2cBus(I2C2_SDA, I2C2_SCL);

BMI160_I2C imu(i2cBus, BMI160_I2C::I2C_ADRS_SDO_LO);

void dumpImuRegisters(BMI160 &imu);
void printRegister(BMI160 &imu, BMI160::Registers reg);
void printBlock(BMI160 &imu, BMI160::Registers startReg, BMI160::Registers stopReg);

int main()
{
    i2cBus.frequency(400000);

    pc.printf("\033[H");  //home
    pc.printf("\033[0J");  //erase from cursor to end of screen

    uint32_t failures = 0;

    if(imu.setSensorPowerMode(BMI160::GYRO, BMI160::NORMAL) != BMI160::RTN_NO_ERROR)
    {
        pc.printf("Failed to set gyroscope power mode\r\n");
        failures++;
    }
    wait_ms(100);

    if(imu.setSensorPowerMode(BMI160::ACC, BMI160::NORMAL) != BMI160::RTN_NO_ERROR)
    {
        pc.printf("Failed to set accelerometer power mode\r\n");
        failures++;
    }
    wait_ms(100);


    BMI160::AccConfig accConfig;
    //example of using getSensorConfig
    if(imu.getSensorConfig(accConfig) == BMI160::RTN_NO_ERROR)
    {
        pc.printf("ACC Range = %d\r\n", accConfig.range);
        pc.printf("ACC UnderSampling = %d\r\n", accConfig.us);
        pc.printf("ACC BandWidthParam = %d\r\n", accConfig.bwp);
        pc.printf("ACC OutputDataRate = %d\r\n\r\n", accConfig.odr);
    }
    else
    {
        pc.printf("Failed to get accelerometer configuration\r\n");
        failures++;
    }

    //example of setting user defined configuration
    accConfig.range = BMI160::SENS_4G;
    accConfig.us = BMI160::ACC_US_OFF;
    accConfig.bwp = BMI160::ACC_BWP_2;
    accConfig.odr = BMI160::ACC_ODR_8;
    if(imu.setSensorConfig(accConfig) == BMI160::RTN_NO_ERROR)
    {
        pc.printf("ACC Range = %d\r\n", accConfig.range);
        pc.printf("ACC UnderSampling = %d\r\n", accConfig.us);
        pc.printf("ACC BandWidthParam = %d\r\n", accConfig.bwp);
        pc.printf("ACC OutputDataRate = %d\r\n\r\n", accConfig.odr);
    }
    else
    {
        pc.printf("Failed to set accelerometer configuration\r\n");
        failures++;
    }

    BMI160::GyroConfig gyroConfig;
    if(imu.getSensorConfig(gyroConfig) == BMI160::RTN_NO_ERROR)
    {
        pc.printf("GYRO Range = %d\r\n", gyroConfig.range);
        pc.printf("GYRO BandWidthParam = %d\r\n", gyroConfig.bwp);
        pc.printf("GYRO OutputDataRate = %d\r\n\r\n", gyroConfig.odr);
    }
    else
    {
        pc.printf("Failed to get gyroscope configuration\r\n");
        failures++;
    }

    wait(1.0);
    pc.printf("\033[H");  //home
    pc.printf("\033[0J");  //erase from cursor to end of screen

    if(failures == 0)
    {
        float imuTemperature;
        BMI160::SensorData accData;
        BMI160::SensorData gyroData;
        BMI160::SensorTime sensorTime;

        while(1)
        {
            imu.getGyroAccXYZandSensorTime(accData, gyroData, sensorTime, accConfig.range, gyroConfig.range);
            imu.getTemperature(&imuTemperature);

            pc.printf("ACC xAxis = %s%4.3f\r\n", "\033[K", accData.xAxis.scaled);
            pc.printf("ACC yAxis = %s%4.3f\r\n", "\033[K", accData.yAxis.scaled);
            pc.printf("ACC zAxis = %s%4.3f\r\n\r\n", "\033[K", accData.zAxis.scaled);

            pc.printf("GYRO xAxis = %s%5.1f\r\n", "\033[K", gyroData.xAxis.scaled);
            pc.printf("GYRO yAxis = %s%5.1f\r\n", "\033[K", gyroData.yAxis.scaled);
            pc.printf("GYRO zAxis = %s%5.1f\r\n\r\n", "\033[K", gyroData.zAxis.scaled);

            pc.printf("Sensor Time = %s%f\r\n", "\033[K", sensorTime.seconds);
            pc.printf("Sensor Temperature = %s%5.3f\r\n", "\033[K", imuTemperature);

            pc.printf("\033[H");  //home
            gLED = !gLED;
            wait(0.5);
        }
    }
    else
    {
        while(1)
        {
            rLED = !rLED;
            wait(0.25);
        }
    }
}


//*****************************************************************************
void dumpImuRegisters(BMI160 &imu)
{
    printRegister(imu, BMI160::CHIP_ID);
    printBlock(imu, BMI160::ERR_REG,BMI160::FIFO_DATA);
    printBlock(imu, BMI160::ACC_CONF, BMI160::FIFO_CONFIG_1);
    printBlock(imu, BMI160::MAG_IF_0, BMI160::SELF_TEST);
    printBlock(imu, BMI160::NV_CONF, BMI160::STEP_CONF_1);
    printRegister(imu, BMI160::CMD);
    pc.printf("\r\n");
}


//*****************************************************************************
void printRegister(BMI160 &imu, BMI160::Registers reg)
{
    uint8_t data;
    if(imu.readRegister(reg, &data) == BMI160::RTN_NO_ERROR)
    {
        pc.printf("IMU Register 0x%02x = 0x%02x\r\n", reg, data);
    }
    else
    {
        pc.printf("Failed to read register\r\n");
    }
}


//*****************************************************************************
void printBlock(BMI160 &imu, BMI160::Registers startReg, BMI160::Registers stopReg)
{
    uint8_t numBytes = ((stopReg - startReg) + 1);
    uint8_t buff[32];
    uint8_t offset = static_cast<uint8_t>(startReg);

    if(imu.readBlock(startReg, stopReg, buff) == BMI160::RTN_NO_ERROR)
    {
        for(uint8_t idx = offset; idx < (numBytes + offset); idx++)
        {
            pc.printf("IMU Register 0x%02x = 0x%02x\r\n", idx, buff[idx - offset]);
        }
    }
    else
    {
        pc.printf("Failed to read block\r\n");
    }
}
