#include "mbed.h"
#include "USBSerial.h"
#include "max32625pico.h"

USBSerial pc;

int readTemp(I2C &i2cbus);
void configTemp(I2C &i2cbus);

#define WRITE_ADDRESS   0x90
#define READ_ADDRESS    0x91
#define TEMP_REG        0x00
#define CONFIG_REG      0x01

int main()
{
    //initialize the MAX32625PICO board for +3.3V logic levels
    MAX32625PICO pico(MAX32625PICO::IOH_3V3, MAX32625PICO::VIO_IOH, MAX32625PICO::VIO_IOH);
    
    //initialize RGB LED channels and turn them off
    DigitalOut rLED(LED1, LED_OFF);
    DigitalOut gLED(LED2, LED_OFF);
    DigitalOut bLED(LED3, LED_OFF);
    
    //initialize the I2C master interface
    I2C i2c(P1_6, P1_7);    //SDA, SCL
    
    //delcare variables to store the raw temperature readings and the converted Celsius temperature
    int temperature_raw;
    float temperature_C;
    
    //write the configuration register
    configTemp(i2c);
    
    //the while loop will read the temperature reading, convert to Celsius, and then
    //print the result to a serial terminal. It will then toggle each on-board LED
    //for 11ms for a total of 33ms. A new temperature value is available every 12.5ms.
    while (true) {
        //the readTemp() function returns a raw 16-bit temperature reading
        //if successful and a zero if unsuccessful
        temperature_raw = readTemp(i2c);
        
        //print temperature only if valid data is received. Readings of exactly 0.00 degrees
        //will be thrown out but really what are the chances?
        if(temperature_raw != 0)
        {
            temperature_C = temperature_raw/256.0;
            
            //print a floating point value with 4 decimal places
            pc.printf("\r\n%.4f C",temperature_C);
        }
        else {
            pc.printf("\r\nTemp Read Error");
        }
        
        //toggle LEDs one at a time for a total of 33ms
        bLED = LED_OFF;
        rLED = LED_ON;
        wait(0.11);
        rLED = LED_OFF;
        gLED = LED_ON;
        wait(0.11);
        gLED = LED_OFF;
        bLED = LED_ON;
        wait(0.11);
    }
}

void configTemp(I2C &i2cbus)
{
    //creates an array to store the values to be written to the configuration register
    //values chosen will program the MAX31875 for 8 conversions/second and 12-bit resolution
    char data[3] = {CONFIG_REG, 0x00, 0x66};
    
    //this built-in I2C write function from mbed sends the configuration data to the MAX31875
    i2cbus.write(WRITE_ADDRESS, data, 3, false);
}

int readTemp(I2C &i2cbus)
{
    int temp_raw = 0;
    char data[2] = {TEMP_REG, 0};
    
    //the first I2C command sets the MAX31875 to point to the temperature register
    //it returns a zero on success or non-zero if unsuccessful
    int rtn_val = i2cbus.write(WRITE_ADDRESS, data, 1, true);
    
    //proceeds to read temperature only if previous command was successful
    if(rtn_val == 0)
    {
        //this I2C command reads the temperature and stores it in the 'data' array.
        //it returns a zero on success or non-zero if unsuccessful
        rtn_val = i2cbus.read(READ_ADDRESS, data, 2, false);
        
        //proceeds to format raw temperature data only if previous command was successful
        if(rtn_val == 0)
        {
            //combine both 8-bit register readings into one 16-bit variable
            temp_raw = ((data[0] << 8) | data[1]);
        }
    }
    //returns the 16-bit raw temperature reading
    return temp_raw;
}
    