/*******************************************************************************
* Copyright (C) 2019 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
*/
#include "max31723.h"
#include "USBSerial.h"
union max31723_raw_data {
    struct {
        uint8_t lsb;
        uint8_t msb;
    };
    struct {
        uint16_t magnitude_bits:15;
        uint16_t sign_bit:1;
    };
    uint16_t uwrd;
    int16_t swrd;
};
 
MAX31723::MAX31723(SPI &spi, DigitalOut &ce_pin) : m_spi(spi), m_chip_enable(ce_pin)
{
    m_chip_enable = 0;
}
 

int MAX31723::read_reg(uint8_t &val, uint8_t reg)
{
    if (reg <= MAX31723_REG_MAX) {
        m_chip_enable = 1;
        m_spi.write(reg);
        val = m_spi.write(0);
        m_chip_enable = 0;
        return MAX31723_NO_ERROR;
    } else {
        printf("Input read_reg is invalid, 0x%02X \r\n", reg);
        return MAX31723_ERROR;
    }
}

uint8_t MAX31723::read_cfg()
{
    uint8_t cfg;
    read_reg(cfg, MAX31723_REG_CFG);
    return cfg;
}

float MAX31723::read_reg_as_temperature(uint8_t reg)
{
    max31723_raw_data tmp;
    float temperature;
    if (reg >= MAX31723_REG_TEMP_LSB && reg <= MAX31723_REG_MAX) {
        read_reg(tmp.lsb, reg);
        read_reg(tmp.msb, reg+1);
        temperature = tmp.magnitude_bits * MAX31723_CF_LSB;
        if (tmp.sign_bit)
            temperature = -temperature;
        return temperature;
    } else {
        printf("Input read_registers_as_temperature is invalid, %d r\n",
            reg);
        return 0;
    }
}

float MAX31723::read_temperature(void)
{
    return read_reg_as_temperature(MAX31723_REG_TEMP_LSB);
}

float MAX31723::read_trip_low(void)
{
    return read_reg_as_temperature(MAX31723_REG_TRIP_LO_LSB);
}

float MAX31723::read_trip_high(void)
{
    return read_reg_as_temperature(MAX31723_REG_TRIP_HI_LSB);
}

int MAX31723::write_reg(uint8_t val, uint8_t reg)
{
    if (reg == (MAX31723_REG_CFG | MAX31723_WRITE_MASK) ||
            (reg >= (MAX31723_REG_TRIP_HI_LSB | MAX31723_WRITE_MASK) &&
            reg <= (MAX31723_REG_MAX | MAX31723_WRITE_MASK)) ) {
        m_chip_enable = 1;
        m_spi.write(reg);
        m_spi.write(val);
        m_chip_enable = 0;
        return MAX31723_NO_ERROR;
    } else {
        printf("Input write_reg is invalid, 0x%02X \r\n", reg);
        return MAX31723_ERROR;
    }
}
 
int MAX31723::write_cfg(uint8_t val)
{
        m_chip_enable = 1;
        m_spi.write(MAX31723_REG_CFG );
        m_spi.write(val);
        m_chip_enable = 0;
        return MAX31723_NO_ERROR;
}


int MAX31723::write_trip_low(float temperature)
{
    int ret;
    max31723_raw_data raw;
    raw.uwrd = 0;
    if (temperature < 0) {
        raw.sign_bit = 1;
        temperature = -temperature;
    }
    temperature /= MAX31723_CF_LSB;
    raw.magnitude_bits = uint16_t(temperature);
    ret = write_reg(raw.lsb, MAX31723_WRITE_MASK | MAX31723_REG_TRIP_LO_LSB);
    wait(.015);
    ret = write_reg(raw.msb, MAX31723_WRITE_MASK | MAX31723_REG_TRIP_LO_MSB);
    wait(.015);
    return ret;
}

int MAX31723::write_trip_high(float temperature)
{
    int ret;
    max31723_raw_data raw;
    raw.uwrd = 0;
    if (temperature < 0) {
        raw.sign_bit = 1;
        temperature = -temperature;
    }
    temperature /= MAX31723_CF_LSB;
    raw.magnitude_bits = uint16_t(temperature);
    ret = write_reg(raw.lsb, MAX31723_WRITE_MASK | MAX31723_REG_TRIP_HI_LSB);
    wait(.015);
    ret = write_reg(raw.msb, MAX31723_WRITE_MASK | MAX31723_REG_TRIP_HI_MSB);
    wait(.015);
    return ret;
}


int MAX31723::perform_one_shot(uint8_t resolution, uint8_t interrupt_mode)
{
    if (resolution <= MAX31723_CFG_RESOLUTION_12BIT) {
        write_reg(0, MAX31723_REG_CFG | MAX31723_WRITE_MASK);
        write_reg(interrupt_mode | MAX31723_CFG_1SHOT |
            resolution | MAX31723_CFG_STANDBY,
            MAX31723_REG_CFG | MAX31723_WRITE_MASK);
    } else
        return MAX31723_ERROR;
    return MAX31723_NO_ERROR;
}

int MAX31723::perform_one_shot_int(uint8_t resolution)
{
    int ret;
    ret = perform_one_shot(resolution, MAX31723_CFG_TM_MODE_INTERRUPT);
    return ret;
}

int MAX31723::perform_one_shot_comparator(uint8_t resolution)
{
    int ret;
    ret = perform_one_shot(resolution, 0);
    return ret;
}


float MAX31723::celsius_to_fahrenheit(float temp_c)
{
    float temp_f;
    temp_f = ((temp_c * 9)/5) + 32;
    return temp_f;
}

MAX31723::~MAX31723(void) 
{
  //empty block
}
 
