/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
* @file          MAX30208LibTest.cpp
* @brief         This is the testing file for the MAX30208 human body temperature sensor EV-Kit library. It uses the default configuration found with the MAX30208 EV-Kit. 
* @version       1.0
* @notes         This is a test program for the MAX30208 MBed Library. Both MAX30208.h and MAX30208.cpp need to be imported into the project. 
*****************************************************************************/
 
#include "mbed.h"
#include "max32630fthr.h"
#include "USBSerial.h"
#include "MAX30208.h"

MAX32630FTHR pegasus(MAX32630FTHR::VIO_3V3);

// Hardware serial port over DAPLink
Serial uart(P2_1, P2_0);

// Virtual serial port over USB
USBSerial microUSB; 
DigitalOut rLED(LED1);
DigitalOut gLED(LED2);
DigitalOut bLED(LED3);

I2C i2c(P3_4, P3_5);  //sda,scl

MAX30208 TempSensor(i2c, 0x50); //Constructor takes 7-bit slave adrs

int main()
{
    
    rLED = LED_ON;
    gLED = LED_ON;
    bLED = LED_OFF;
    
    //Initialize Configuration Registers
    MAX30208::Configuration_InterruptEnable InterruptEnable;
    MAX30208::Configuration_FIFOConfig2 FIFOConfig2;
    MAX30208::Configuration_GPIOSetup GPIOSetup;
    MAX30208::Configuration_GPIOControl GPIOControl;
    

    uint16_t val;
    
    
    while(1) {
    
    TempSensor.takeDataMeasurment();            //Send temperature record signal
    wait_ms(50);                                //Maximum temperature record time
    TempSensor.readData(val);                   //Read the data
    float temp = TempSensor.toCelsius(val);     //Convert raw data to celsius
    microUSB.printf("Temp C = %f\r\n",temp);
    temp = TempSensor.toFahrenheit(temp);       //Conver celsius to fahrenheit
    microUSB.printf("Temp F = %f\r\n",temp);
    
    
    
    //readInterruptRegister
    if (TempSensor.readInterruptRegister(InterruptEnable) == 0){    
        microUSB.printf("InterruptRegister intial config = %d\r\n", InterruptEnable.all);
    }
    
    //Set Interrupt Config Register Values
    InterruptEnable.config.TEMP_HI_EN = 1;
    InterruptEnable.config.TEMP_LO_EN = 1;
    
    if (TempSensor.writeInterruptRegister(InterruptEnable) != 0){
        microUSB.printf("Error writing Interrupt register\r\n");
    }
    if (TempSensor.readInterruptRegister(InterruptEnable) == 0){    
        microUSB.printf("InterruptRegister new config = %d\r\n", InterruptEnable.all);
    }
    microUSB.printf("\r\n\r\n");
    
    //readFIFOConfig2
    if (TempSensor.readFIFOConfig2(FIFOConfig2) == 0){
        microUSB.printf("FIFOConfig2 initial config = %d\r\n", FIFOConfig2.all);
    }
    
    //Set FIFO Config 2 Register Values
    FIFOConfig2.config.FIFO_RO = 1;         
    FIFOConfig2.config.A_FULL_TYPE = 1;
    if (TempSensor.writeFIFOConfig2(FIFOConfig2) != 0){
        microUSB.printf("Error writing FIFO Config2 register\r\n");
    }
    
    if (TempSensor.readFIFOConfig2(FIFOConfig2) == 0){
        microUSB.printf("FIFOConfig2 new config = %d\r\n", FIFOConfig2.all);
    }
    
    microUSB.printf("\r\n\r\n");
    
    //readGPIOSetup
    if (TempSensor.readGPIOSetup(GPIOSetup) == 0){
        microUSB.printf("GPIOSetup initial config = %d\r\n", GPIOSetup.all);
    }
    
    //Set GPIO Setup Register Values
    GPIOSetup.config.GPIO0_MODE = 2;
    GPIOSetup.config.GPIO1_MODE = 3;
    if (TempSensor.writeGPIOSetup(GPIOSetup) != 0){
        microUSB.printf("Error writing GPIO Setup register\r\n");
    }
    
    if (TempSensor.readGPIOSetup(GPIOSetup) == 0){
        microUSB.printf("GPIOSetup new config = %d\r\n", GPIOSetup.all);
    }
    
    microUSB.printf("\r\n\r\n");
    
    //readGPIOControl
    if (TempSensor.readGPIOControl(GPIOControl) == 0){
        microUSB.printf("GPIOControl initial config = %d\r\n", GPIOControl.all);
    }
    
    //Set GPIO Control Register Values
    GPIOControl.config.GPIO0_LL = 1;
    GPIOControl.config.GPIO1_LL = 1;
    if (TempSensor.writeGPIOControl(GPIOControl) != 0){
        microUSB.printf("Error writing register\r\n");
    }
    if (TempSensor.readGPIOControl(GPIOControl) == 0){
        microUSB.printf("GPIOControl new config = %d\r\n", GPIOControl.all);
    }
    
    TempSensor.resetDevice();       //Reset Device Registers
    

    microUSB.printf("\r\n\r\n");
    uint16_t PointerValue;
    
    //Write Data Pointer Test (Read Only)
    TempSensor.readWritePointer(PointerValue);
    microUSB.printf("Current Write Pointer Value: %d\r\n",PointerValue);
    microUSB.printf("New Value Measured\r\n");
    TempSensor.takeDataMeasurment();
    wait_ms(50);
    TempSensor.readWritePointer(PointerValue);
    microUSB.printf("Current Write Pointer Value: %d\r\n",PointerValue);

    microUSB.printf("\r\n\r\n");
    
    //Read Data Pointer Test (Read and Write)
    TempSensor.readReadPointer(PointerValue);    
    microUSB.printf("Current Read Pointer Value: %d\r\n",PointerValue);
    TempSensor.readData(val);
    microUSB.printf("New Value Read\r\n");
    TempSensor.readReadPointer(PointerValue);    
    microUSB.printf("Current Read Pointer Value: %d\r\n",PointerValue);
    microUSB.printf("Read Pointer Written to '15'\r\n");
    TempSensor.writeReadPointer(0x0F);
    TempSensor.readReadPointer(PointerValue);
    microUSB.printf("Current Read Pointer Value: %d\r\n",PointerValue);
    
    microUSB.printf("\r\n\r\n");
    
    
    TempSensor.resetDevice();       //Reset Device Registers
    
    uint16_t Counter;
    
    //Data Counter
    TempSensor.readDataCounter(Counter);
    microUSB.printf("Initial Data Read: %d\r\n",Counter);
    microUSB.printf("Reading 16 temperatures\r\n");
    for(int i=0;i<16;i++){
        TempSensor.takeDataMeasurment();
        wait_ms(50);
    }
    
    //Read Number of Data Points Collected
    TempSensor.readDataCounter(Counter);
    microUSB.printf("Curent Data Read: %d\r\n",Counter);
    microUSB.printf("\r\n\r\n");
    
    //Data Overflow
    microUSB.printf("Reading 30 temperatures\r\n");
    
    //Overflow data registers
    for (int i =0;i<30;i++){
           TempSensor.takeDataMeasurment();
           wait_ms(50);
    }
    
    //Read Overflow Register
    TempSensor.readOverflow(Counter);
    microUSB.printf("Overflow Counter: %d\r\n",Counter);
    
    microUSB.printf("\r\n\r\n");
    uint16_t config;
    
    TempSensor.resetDevice();   //Reset Device Registers
    
    
    //Config1
    TempSensor.readFIFOConfig1(config);    
    microUSB.printf("FIFO Config init Value: %d\r\n",config);
    microUSB.printf("Writing '3' to FIFO Config 1\r\n");
    TempSensor.writeFIFOConfig1(0x03);
    TempSensor.readFIFOConfig1(config);    
    microUSB.printf("FIFO Config Value: %d\r\n",config);
    
    microUSB.printf("\r\n\r\n");
    TempSensor.resetDevice();   //Reset Device Registers
    
    //Read Alarm Register
    uint16_t HI;
    uint16_t LO;
    TempSensor.readAlarmHigh(HI);
    TempSensor.readAlarmLow(LO);
    
    microUSB.printf("Inital Alarm High: %d\r\n",HI);
    microUSB.printf("Inital Alarm Low: %d\r\n",LO);
    microUSB.printf("Setting High to '4321'\r\n");
    microUSB.printf("Setting High to '1234'\r\n");
    
    //Write Alarm Registers
    TempSensor.writeAlarmHigh(0x10E1);
    TempSensor.writeAlarmLow(0x04D2);
    
    //Read Alarm Registers
    TempSensor.readAlarmHigh(HI);
    TempSensor.readAlarmLow(LO);
    
    microUSB.printf("Alarm High: %d\r\n",HI);
    microUSB.printf("Alarm Low: %d\r\n",LO);
    
    microUSB.printf("\r\n\r\n\r\n\r\n");
    TempSensor.resetDevice();       //Reset Device Registers
    wait(5);
    }
}

