/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
  
This is code for the MAX1473 RF Receiver circuit. This program will capture,
translate, and unencrypt messages received using the ForwardErrCorr.h format.
The example uses MAX30208 temperature data as an example.

Hardware Setup and connections:

    MAX32630FTHR->  MAX1473 Ev-Kit
    
    3.3V        ->  VDD
    GND         ->  GND
    P3_0        ->  DATA_OUT
    1.8V -> 200KOhm -> TP2
    
*******************************************************************************
*/

#include "mbed.h"
#include "max32630fthr.h"
#include "USBSerial.h"
#include "MAX30208.h"
#include "ForwardErrCorr.h"

MAX32630FTHR pegasus(MAX32630FTHR::VIO_3V3);

//Read UART Data
RawSerial uart(P3_1,P3_0);//tx,rx

I2C i2c(P3_4, P3_5);  //sda,scl

//Temperature Sensor Object to convert raw temp data to degress C
MAX30208 TempSensor(i2c, 0x50); //Constructor takes 7-bit slave adrs


char  TransTable[] = {0x1F,0x18,0x06,0x01}; //Used to translate data for FEC -- Make sure it is identical to transmitter
#define SymmetricKey "RfIsCoOl" //Set Symmetric Key here -- Make sure it is identical to transmitter
Translator transRx(SymmetricKey, TransTable);

// Virtual serial port over USB
USBSerial microUSB; 
DigitalOut rLED(LED1);
DigitalOut gLED(LED2); 
DigitalOut bLED(LED3);
DigitalIn TakeData(P5_6);
DigitalIn TakeData2(P5_5);
DigitalIn sw1(SW1);

//*****************************************************************************
int buildArray(int j, char *input,char *output){
    int i = j+1;
    int k = 0;
    while(input[i] != 'c' && i < 50){
        output[k] = input[i];
        i++;
        k++;
    }
    if(input[i] == 'c'){
        return(0);
    }
    return(1);
}
char dataIn[50];

//*****************************************************************************
void SerialCallback(){
    int i = strlen(dataIn);     //Make sure not to overflow the storage
    if (i >= 50){
        i = 0;
    }
    
    //Read Data while it is available on RX pin
    while(uart.readable()){
        dataIn[i] = uart.getc();
        i++;
    }
}

//*****************************************************************************

int main()
{
    //Turn on LEDS
    rLED = LED_ON;
    gLED = LED_ON;
    bLED = LED_OFF;
    rLED = LED_OFF;
    
    //Uart Settings
    uart.baud(9600);                //Set baud to 9600
    uart.attach(&SerialCallback);   //Set intterupt function to read uart data from reciever

    char dataOut[50];       //Hold incoming messages            
    uint16_t output[5];    //Hold packet information after decoding
    printf("Program Begin\r\n");
    
    //Runs continuously after initial setup
    while(1) {
        if (strlen(dataIn) >= 14){              //Wait until at least 1 packet has been recorded
            wait_ms(50);                        //Short pause to ensure interrupt function has completed data read
            
            //Find the start of the packet
            int start = 0;
            while(dataIn[j] != 'b' && j < 34){
                start++;
            }
            
            //Check that starting character was found
            if (dataIn[start] == 'b'){
                printf("Building Array\r\n");
                
                //Build array that holds only one data packet
                if (buildArray(start,dataIn,dataOut)==0){
                    microUSB.printf("\r\n");
                    
                    //Decrypt the packet
                    if (transRx.Decrypt(dataOut,output) == 1){
                        printf("Error reconstructing\r\n");
                    }//if
                    
                    //Print out the data value, Device ID, Device Type, and Temperature
                    printf("Decoded Data Value: %i\r\n",output[0]);
                    printf("Device Type: %c\r\n", dataIn[j+1]);
                    printf("Device ID: %i\r\n", dataIn[j+2]);
                    float clesius = TempSensor.toCelsius(output[0]);
                    float fare = TempSensor.toFahrenheit(clesius);
                    printf("Temperature: %f F\r\n",fare);
                    printf("Temperature: %f C",clesius);
                    printf("\r\n\r\n");
                }//if
                else{
                    
                    //If the end character cannot be found, clear the data and listen for new messages
                    printf("error building array\r\n");
                }
                
                //Clear the data array after the data has been decrypted
                for (int i = 0;i<sizeof(dataIn);i++){
                    dataIn[i] = 0;
                }//for
            }//if
            
            //If start character cannot be found, clear array before listening for more data
            else{
                for (int i = 0;i<sizeof(dataIn);i++){
                    dataIn[i] = 0;
                }//for
            }//else
        }//if
    }
}
