/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************

This is code for the MAX1472 RF Transmitter EV-Kit. This program will record a 
temperature fom the MAX30208 EV-Kit, encrypt the data using the user-generated
Symmetric key, apply forward error correction (FEC) encoding, and then send the
message via the MAX1472 transmitter.

  Hardware Setup and connections:

    MAX32630FTHR->  MAX1472 Ev-Kit
    
    3.3V        ->  VDD
    GND         ->  VSS
    P3_1        ->  Data_In
    P5_2        ->  ENABLE
    
*******************************************************************************
    MAX32630FTHR->  MAX30208 Ev-Kit
    
    1.8V        ->  VIN
    GND         ->  GND
    P3_4        ->  SDA
    P3_5        ->  SCL
    
*******************************************************************************
*/

#include "mbed.h"
#include "max32630fthr.h"
#include "USBSerial.h"
#include "MAX30208.h"
#include "ForwardErrCorr.h"

MAX32630FTHR pegasus(MAX32630FTHR::VIO_3V3);

//UART Port for Sending data
RawSerial uart(P3_1,P3_0);//tx,rx


I2C i2c(P3_4, P3_5);  //sda,scl
MAX30208 TempSensor(i2c, 0x50); //Temperature Sensor Object for reading temperature data

#define SymmetricKey "RfIsCoOl"   //Set Symmetric Key here -- Make sure it is identical to receiver
char  TransTable[] = {0x1F,0x18,0x06,0x01}; //Used to translate data for FEC -- Make sure it is identical to receiver
Translator transTx(SymmetricKey, TransTable); //Initialize Encoder

DigitalOut rLED(LED1);
DigitalOut gLED(LED2); 
DigitalOut bLED(LED3);      

DigitalIn sw1(SW1);         //Trigger for sending data
DigitalOut Enable(P5_2);    //Enable Transmitter

char DEVICETYPE = 'T';  //Set Device Type
char DEVICEID = 0x01;   //Set the Device ID

//******************************************************************************

/**
* @brief  Record and read temperature from MAX30208
* @param TempSensor - Refrence to MAX30208 temp sensor object
* @param &value[OUT]- Address to store read temperature at
* @return   0 on success, 1 on failure due to improper data record or read 
*/
int ReadData(MAX30208 TempSensor, uint16_t &value){
    if (TempSensor.takeDataMeasurment() != 0){
        printf("Error Taking data Meaurment\r\n");
        return(1);
    }
    wait_ms(50);    //max temp record time is 50ms
    if (TempSensor.readData(value) !=0){
        printf("Error reading temperature Data\r\n");    
        return(1);
    }
    return(0);
}

//******************************************************************************

/*
* @brief  Begin Communication with warm up bytes and device type/id
*/
void comInit(Translator trans){
    uart.putc(0xFF);        //Initializer bytes to "warm-up" the Peak Detector cicuitry of MAX1473 Receiver
    uart.putc(0xFF);
    uart.putc(0x00);
    uart.putc('b');         //Packet start character
    uart.putc(DEVICETYPE);  //Send Device Type
    uart.putc(DEVICEID);    //Send Device ID
}

//******************************************************************************

/**
* @brief  Send data and end transmission
* @param EncryptedData[IN]  - 8 bytes of encryted data to send via UART connection
*/
void comData(char *EncryptedData){
    for(int i=0;i<8;i++){
        uart.putc(EncryptedData[i]);    //Send all of the encrypted data
    }
    uart.putc('c'); //End of packet character
    uart.send_break();
}

//******************************************************************************

int main()
{
    wait(1);
    uart.baud(9600);
    printf("Starting Transmitter Program\r\n\r\n");
    
    rLED = LED_ON;
    gLED = LED_ON;
    bLED = LED_OFF;

    rLED = LED_OFF;
    
    //Start MAX30208 temperature sensor in factor default
    TempSensor.resetDevice();
    
    //Turn off Transmitter
    Enable = 0;

    while(1) {
        
        //Record a temperature and send the encrypted data each time the on-board button is pressed
        if(sw1==0){
            
            //Enable Transmitter
            Enable = 1;
            
            //Read a new temperature
            uint16_t tempData;
            if(ReadData(TempSensor,tempData) !=0){
                printf("Error reading data!\r\n");
            }
            
            //Encrypt Temperature Data
            char EncryptedData[8];
            printf("Original Data: %i\r\n", tempData);
            transTx.Encrypt(tempData,EncryptedData); 
            printf("Encrypted Data:\r\n");
            for(int i=0;i<8;i++){
                printf("%c  ",EncryptedData[i]);
            }
            printf("\r\n\r\n");
            
            //Initialize Communication
            comInit(transTx);
            
            //Send Encrypted Data
            comData(EncryptedData);
            
            //Turn off Transmitter
            Enable = 0;

            wait_ms(500);    //button debounce
        }//if
    }//while
}//main

