/*******************************************************************************
 * Copyright (C) 2015 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */

#ifndef PMIC_H_
#define PMIC_H_

#include "mbed.h"

/**
 * Example for the MAX14720 PMIC Driver
 *
 * @code
 * #include <stdio.h>
 * #include "mbed.h"
 * #include "MAX14720.h"
 *
 *  // I2C Master 2
 * I2C i2c2(I2C2_SDA, I2C2_SCL);
 *
 * #define  I2C_ADDR_PMIC   (0x54)                 
 * MAX14720 max14720(&i2c2,I2C_ADDR_PMIC);
 *
 * int main(void) {
 *     int result;
 * 
 *	   // initialize BoostVSet on the MAX14720 PMIC to 4.5 Volts
 *     printf("Init MAX14720\n");
 *     result = max14720.boostEn(MAX14720::BoostEn_Disable);
 *     if (result == MAX14720_ERROR) printf("Error communicating with MAX14720");
 *     result = max14720.boostVSet(MAX14720::BoostVSet_45);
 *     if (result == MAX14720_ERROR) printf("Error communicating with MAX14720");
 *     result = max14720.boostEn(MAX14720::BoostEn_Enable);
 *     if (result == MAX14720_ERROR) printf("Error communicating with MAX14720");
 * 
 *	   while (1) {
 *     }
 * }
 * @endcode
 */

#define MAX14720_NO_ERROR   0
#define MAX14720_ERROR      -1

class MAX14720
{
public:
    typedef enum {  // I2C Register Addresses
        CHIP_ID = 0x00,
        CHIP_REV = 0x01,
        BOOST_C_DIV = 0x03,
        BOOST_I_SET = 0x04,
        BOOST_V_SET = 0x05,
        BOOST_CFG = 0x06,
        BUCK_V_SET = 0x07,
        BUCK_CFG = 0x08,
        BUCK_I_SET = 0x09,
        LDOV_SET = 0x0A,
        LDO_CFG = 0x0B,
        SWITCH_CFG = 0x0C,
        BAT_TIME = 0x0D,
        BAT_CFG = 0x0E,
        BAT_BCV = 0x0F,
        BAT_OCV = 0x10,
        BAT_LCV = 0x11,
        MON_CFG = 0X19,
        BOOT_CFG = 0x1A,
        PIN_STAT = 0x1B,
        BBB_EXTRA = 0x1C,
        HAND_SHK = 0x1D,
        UVLO_CFG = 0x1E,
        PWR_OFF = 0x1F,
        OTP_MAP_0 = 0x20,
        OTP_MAP_1 = 0x21,
        OTP_MAP_2 = 0x22,
        OTP_MAP_3 = 0x23,
        OTP_MAP_4 = 0x24,
        OTP_MAP_5 = 0x25,
        OTP_MAP_6 = 0x26,
        OTP_MAP_7 = 0x27,
        OTP_MAP_8 = 0x28,
        OTP_MAP_9 = 0x29,
        OTP_MAP_A = 0x2A,
        OTP_MAP_b = 0x2B,
    } MAX14720_REG_map_t;

	/// @brief ChipId Register (0x00)
	typedef union ChipId_ {
	  char all;
	} max14720_ChipId;

	/// @brief ChipRev Register (0x01)
	typedef union ChipRev_ {
	  char all;
	} max14720_ChipRev;

	/// @brief BoostCDiv Register (0x03)
	typedef union BoostCDiv_ {
	  char all;
	  struct {
		char ClkDivSet : 7;
		char ClkDivEn : 1;
	  } bit;
	} max14720_BoostCDiv;

	/// @brief BoostISet Register (0x04)
	typedef union BoostISet_ {
	  char all;
	  struct {
		char BoostISet : 3;
	  } bit;
	} max14720_BoostISet;

	/// @brief BoostVSet Register (0x05)
	typedef union BoostVSet_ {
	  char all;
	  struct {
		char BoostVSet : 5;
	  } bit;
	} max14720_BoostVSet;

	/// @brief BoostCfg Register (0x06)
	typedef union BoostCfg_ {
	  char all;
	  struct {
		char BoostInd : 1;
		char BoostEMI : 1;
		char reserved : 1;
		char BoostEn : 2;
		char BoostSeq : 3;
	  } bit;
	} max14720_BoostCfg;

	/// @brief BuckVSet Register (0x07)
	typedef union BuckVSet_ {
	  char all;
	  struct {
		char BuckVSet : 6;
	  } bit;
	} max14720_BuckVSet;

	/// @brief BuckCfg Register (0x08)
	typedef union BuckCfg_ {
	  char all;
	  struct {
		char BuckFst : 1;
		char BuckMd : 2;
		char BoostEn : 2;
		char BuckSeq : 3;
	  } bit;
	} max14720_BuckCfg;

	/// @brief BuckISet Register (0x09)
	typedef union BuckISet_ {
	  char all;
	  struct {
		char BuckInteg : 1;
		char BuckMinOT : 1;
		char BuckHysOff : 1;
		char BuckInd : 1;
		char BuckCfg : 1;
		char BuckISet : 3;
	  } bit;
	} max14720_BuckISet;

	/// @brief LDOVSet Register (0x0A)
	typedef union LDOVSet_ {
	  char all;
	  struct {
		char LDOVSet : 5;
	  } bit;
	} max14720_LDOVSet;

	/// @brief LDOCfg Register (0x0B)
	typedef union LDOCfg_ {
	  char all;
	  struct {
		char LDOMode : 1;
		char LDOEn : 2;
		char LDO_ActDSC : 1;
		char LDO_PasDSC : 1;
		char LDOSeq : 3;
	  } bit;
	} max14720_LDOCfg;

	/// @brief SwitchCfg Register (0x0C)
	typedef union SwitchCfg_ {
	  char all;
	  struct {
		char SWSoftStart : 1;
		char SWEn : 2;
		char reserved : 2;
		char SWSeq : 3;
	  } bit;
	} max14720_SwitchCfg;

	/// @brief BatTime Register (0x0D)
	typedef union BatTime_ {
	  char all;
	  struct {
		char LCVTm : 2;
		char OCVTm : 2;
		char BCVTm : 2;
	  } bit;
	} max14720_BatTime;

	/// @brief BatCfg Register (0x0E)
	typedef union BatCfg_ {
	  char all;
	  struct {
		char BatImpCur : 3;
		char LcvDly2Skip : 1;
		char reserved : 2;
		char BIMAbort : 1;
		char BIA : 1;
	  } bit;
	} max14720_BatCfg;

	/// @brief BatV Register (0x0F)
	typedef union BatBCV_ {
	  char all;
	} max14720_BatBCV;

	/// @brief BatOCV Register (0x10)
	typedef union BatOCV_ {
	  char all;
	} max14720_BatOCV;

	/// @brief BatLCV Register (0x11)
	typedef union BatLCV_ {
	  char all;
	} max14720_BatLCV;

	/// @brief MONCfg Register (0x19)
	typedef union MONCfg_ {
	  char all;
	  struct {
		char MONCtr : 3;
		char MONHiZ : 1;
		char reserved : 3;
		char MONEn : 1;
	  } bit;
	} max14720_MONCfg;

	/// @brief BootCfg Register (0x1A)
	typedef union BootCfg_ {
	  char all;
	  struct {
		char BootDly : 2;
		char PFNPUDCfg : 1;
		char SftRstCfg : 1;
		char PwrRstCfg : 4;
	  } bit;
	} max14720_BootCfg;

	/// @brief PinStat Register (0x1B)
	typedef union PinStat_ {
	  char all;
	  struct {
		char RST_LEN : 1;
		char MPC_BEN : 1;
		char KOUT_HVEN : 1;
		char KIN_SWEN : 1;
	  } bit;
	} max14720_PinStat;

	/// @brief BBBExtra Register (0x1C)
	typedef union BBBExtra_ {
	  char all;
	  struct {
		char BuckFScl : 1;
		char BuckActDsc : 1;
		char BuckPasDsc : 1;
		char reserved : 2;
		char Boost_ActDsc : 1;
		char BoostPasDsc : 1;
		char Boost_HysOff : 1;
	  } bit;
	} max14720_BBBExtra;

	/// @brief HandShk Register (0x1D)
	typedef union HandShk_ {
	  char all;
	  struct {
		char StayOn : 1;
		char reserved : 5;
		char GlbPasDsc : 1;
		char StartOff : 1;
	  } bit;
	} max14720_HandShk;

	/// @brief UVLOCfg Register (0x1E)
	typedef union UVLOCfg_ {
	  char all;
	  struct {
		char LDO_UVLOsel : 1;
		char BBBUVLOsel : 1;
	  } bit;
	} max14720_UVLOCfg;

	/// @brief PWRCFG Register (0x1F)
	typedef union PWROFF_ {
	  char all;
	} max14720_PWROFF;

	/// All of the possible BoostVSet values
	typedef enum  {
		BoostVSet_25,
		BoostVSet_26,
		BoostVSet_27,
		BoostVSet_28,
		BoostVSet_29,
		BoostVSet_30,
		BoostVSet_31,
		BoostVSet_32,
		BoostVSet_33,
		BoostVSet_34,
		BoostVSet_35,
		BoostVSet_36,
		BoostVSet_37,
		BoostVSet_38,
		BoostVSet_39,
		BoostVSet_40,
		BoostVSet_41,
		BoostVSet_42,
		BoostVSet_43,
		BoostVSet_44,
		BoostVSet_45,
		BoostVSet_46,
		BoostVSet_47,
		BoostVSet_48,
		BoostVSet_49,
		BoostVSet_50,
	} BoostVSet_Values;

	typedef enum {
		BoostEn_Disable,
		BoostEn_Enable,
		BoostEn_Enable_MPC_High,
	} BoostEn_Values;

	typedef enum  {
		BoostISet_0,
		BoostISet_50mA,
		BoostISet_100mA,
		BoostISet_150mA,
		BoostISet_200mA,
		BoostISet_250mA,
		BoostISet_300mA,
		BoostISet_350mA,
	} BoostISet_Values;

	typedef enum  {
		BoostSeq_Disabled,		
		BoostSeq_Reserved,
		BoostSeq_Enabled_0,
		BoostSeq_Enabled_25,
		BoostSeq_Enabled_50,
		BoostSeq_Reserved2,
		BoostSeq_Controlled_HVEN,
		BoostSeq_Controlled_BoostEn,
	} BoostSeq_Values;

	typedef enum {
		BoostEMI_Active,
		BoostEMI_Disabled,
	} BoostEMI_Values;

	typedef enum {
		BoostInd_3_3,
		BoostInd_4_7,
	} BoostInd_Values;

	/**
		* MAX14720 constructor.
		*
		* @param sda mbed pin to use for SDA line of I2C interface.
		* @param scl mbed pin to use for SCL line of I2C interface.
		* @param slaveAddress Slave Address of the device.
		*/
	MAX14720(PinName sda, PinName scl, int slaveAddress);

	/**
		* MAX14720 constructor.
		*
		* @param i2c I2C object to use.
		* @param slaveAddress Slave Address of the device.
		*/
	MAX14720(I2C *i2c, int slaveAddress);

	/**
		* MAX14720 destructor.
		*/
	~MAX14720();

	/** 
	* Enable or disable Boost
	*/
	int boostEn(BoostEn_Values value);
	/**
	* Set the Boost Output Voltage from 2.5V to 5V with 100mV increments
	*/
	int boostVSet(BoostVSet_Values value);
	/**
	* Read the Boost Output Voltage
	*/
	int readBoostVSet(BoostVSet_Values *value);
	/**
	* Set the Buck-Boost Peak Current-Limit
	*/
	int boostISet(BoostISet_Values value);
	/**
	* Read the Buck-Boost Peak Current-Limit
	*/
	int readBoostISet(BoostISet_Values *value);

	/**
	* Write a device register
	*/
	int writeReg(MAX14720_REG_map_t reg, char value);
	/**
	* Read a device register
	*/
	int readReg(MAX14720_REG_map_t reg, char *value);
private:
	/// I2C pointer
	I2C *i2c;
	/// Is this object the owner of the I2C object
	bool isOwner;
	/// Device slave address
	int slaveAddress;
      
};

#endif /* PMIC_H_ */
