﻿// /*******************************************************************************
// * Copyright (C) 2021 Maxim Integrated Products, Inc., All Rights Reserved.
// *
// * Permission is hereby granted, free of charge, to any person obtaining a
// * copy of this software and associated documentation files (the "Software"),
// * to deal in the Software without restriction, including without limitation
// * the rights to use, copy, modify, merge, publish, distribute, sublicense,
// * and/or sell copies of the Software, and to permit persons to whom the
// * Software is furnished to do so, subject to the following conditions:
// *
// * The above copyright notice and this permission notice shall be included
// * in all copies or substantial portions of the Software.
// *
// * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
// * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
// * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
// * OTHER DEALINGS IN THE SOFTWARE.
// *
// * Except as contained in this notice, the name of Maxim Integrated
// * Products, Inc. shall not be used except as stated in the Maxim Integrated
// * Products, Inc. Branding Policy.
// *
// * The mere transfer of this software does not imply any licenses
// * of trade secrets, proprietary technology, copyrights, patents,
// * trademarks, maskwork rights, or any other form of intellectual
// * property whatsoever. Maxim Integrated Products, Inc. retains all
// * ownership rights.
// *******************************************************************************
// */
// example code includes
// standard include for target platform -- Platform_Include_Boilerplate
#include "mbed.h"
// Platforms:
//   - MAX32625MBED
//      - supports mbed-os-5.11, requires USBDevice library
//      - add https://developer.mbed.org/teams/MaximIntegrated/code/USBDevice/
//      - remove max32630fthr library (if present)
//      - remove MAX32620FTHR library (if present)
//   - MAX32600MBED
//      - Please note the last supported version is Mbed OS 6.3.
//      - remove max32630fthr library (if present)
//      - remove MAX32620FTHR library (if present)
//      - Windows 10 note:  Don't connect HDK until you are ready to load new firmware into the board.
//   - NUCLEO_F446RE
//      - remove USBDevice library
//      - remove max32630fthr library (if present)
//      - remove MAX32620FTHR library (if present)
//   - NUCLEO_F401RE
//      - remove USBDevice library
//      - remove max32630fthr library (if present)
//      - remove MAX32620FTHR library (if present)
//   - MAX32630FTHR
//      - #include "max32630fthr.h"
//      - add http://developer.mbed.org/teams/MaximIntegrated/code/max32630fthr/
//      - remove MAX32620FTHR library (if present)
//   - MAX32620FTHR
//      - #include "MAX32620FTHR.h"
//      - remove max32630fthr library (if present)
//      - add https://os.mbed.com/teams/MaximIntegrated/code/MAX32620FTHR/
//      - not tested yet
//   - MAX32625PICO
//      - #include "max32625pico.h"
//      - add https://os.mbed.com/users/switches/code/max32625pico/
//      - remove max32630fthr library (if present)
//      - remove MAX32620FTHR library (if present)
//      - not tested yet
//      - see https://os.mbed.com/users/switches/code/max32625pico/
//      - see https://os.mbed.com/users/switches/code/PICO_board_demo/
//      - see https://os.mbed.com/users/switches/code/PICO_USB_I2C_SPI/
//      - see https://os.mbed.com/users/switches/code/SerialInterface/
//      - Note: To load the MAX32625PICO firmware, hold the button while
//        connecting the USB cable, then copy firmware bin file 
//        to the MAINTENANCE drive.
//      - see https://os.mbed.com/platforms/MAX32625PICO/
//      - see https://os.mbed.com/teams/MaximIntegrated/wiki/MAX32625PICO-Firmware-Updates
//
// end Platform_Include_Boilerplate
#include "MAX11131.h"
#include "CmdLine.h"
#include "MaximTinyTester.h"

#include "MAX11131.h"
extern MAX11131 g_MAX11131_device; // defined in main.cpp



// CODE GENERATOR: MAX11131 needs print_value(CmdLine& cmdLine, int16_t value_u12, int channelId)
//----------------------------------------
void print_value(CmdLine& cmdLine, int16_t value_u12, int channelId)
{
    int channelPairIndex = channelId / 2;
    // format: 1 0 0 0 1 UCH0/1 UCH2/3 UCH4/5 UCH6/7 UCH8/9 UCH10/11 UCH12/13 UCH14/15 PDIFF_COM x x
    // unused variable: int UCHn = (g_MAX11131_device.UNIPOLAR >> (10 - channelPairIndex)) & 0x01;
    int BCHn = (g_MAX11131_device.BIPOLAR >> (10 - channelPairIndex)) & 0x01;
    // unused variable: int RANGEn = (g_MAX11131_device.RANGE >> (10 - channelPairIndex)) & 0x01;
    //
    cmdLine.serial().printf(" ch=");
    // TODO1: if CHANID=0 don't print ch=channelId
    if ((g_MAX11131_device.isExternalClock == 0) || (g_MAX11131_device.chan_id_0_1 == 1))
    {
        // Internal clock modes always use channel ID.
        // External clock modes use channel ID if ADC_MODE_CONTROL.CHAN_ID is 1.
        cmdLine.serial().printf("%d", channelId);
    } else {
        cmdLine.serial().printf("?");
    }
    if (BCHn)
    {
        cmdLine.serial().printf(" xb=");
        cmdLine.serial().printf("%d", g_MAX11131_device.TwosComplementValue(value_u12));
    }
    else
    {
        cmdLine.serial().printf(" xu=");
        cmdLine.serial().printf("%d", value_u12);
    }
    // dtostrf width and precision: 2.5v / 4096 LSB = 0.0006103515625 volts per LSB
    cmdLine.serial().printf(" = 0x%4.4x = %6.4fV", (value_u12 & 0xFFFF), g_MAX11131_device.VoltageOfCode(value_u12, channelId));
}

// CODE GENERATOR: MAX11131 needs AINcode_print_value_chanID(CmdLine& cmdLine, int nWords)
//----------------------------------------
// read data words
// @pre RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data
// @pre AINcode[NUM_CHANNELS] contains the latest readings in LSBs
// For internal clock modes, the data format always includes the channel address.
//     misoData16 = CH[3:0] DATA[11:0]
void AINcode_print_value_chanID(CmdLine& cmdLine, int nWords)
{
    cmdLine.serial().printf("\r\nScanRead_nWords_chanID nWords=");
    cmdLine.serial().printf("%d\r\n", nWords);
    for (int index = 0; index < nWords; index++) {
        //~ int16_t misoData16 = MAX11131_ScanRead();
        // For internal clock modes, the data format always includes the channel address.
        //     misoData16 = CH[3:0] DATA[11:0]
        int16_t value_u12 = (g_MAX11131_device.RAW_misoData16[index] & 0x0FFF);
        int channelId = ((g_MAX11131_device.RAW_misoData16[index] >> 12) & 0x000F);
        // diagnostic: print raw MISO data
        cmdLine.serial().printf("      MAX11131.MISO[");
        cmdLine.serial().printf("%d", index);
        cmdLine.serial().printf("]=0x");
        cmdLine.serial().printf("%0x4.4x", (g_MAX11131_device.RAW_misoData16[index] & 0xFFFF));
        cmdLine.serial().printf(":");
        print_value(cmdLine, value_u12, channelId);
        cmdLine.serial().printf("\r\n");
    }
}

// CODE GENERATOR: MAX11131 needs AINcode_print_value_externalClock(CmdLine& cmdLine, int nWords)
//----------------------------------------
// read data words
// @pre RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data
// @pre AINcode[NUM_CHANNELS] contains the latest readings in LSBs
// For external clock modes, the data format returned depends on the CHAN_ID bit.
//     when CHAN_ID = 0: misoData16 = 0 DATA[11:0] x x x
//     when CHAN_ID = 1: misoData16 = CH[3:0] DATA[11:0]
void AINcode_print_value_externalClock(CmdLine& cmdLine, int nWords)
{
    // For external clock modes, the data format returned depends on the CHAN_ID bit.
    //     when CHAN_ID = 0: misoData16 = 0 DATA[11:0] x x x
    //     when CHAN_ID = 1: misoData16 = CH[3:0] DATA[11:0]
    // For internal clock modes, the data format always includes the channel address.
    //     misoData16 = CH[3:0] DATA[11:0]
    if (g_MAX11131_device.chan_id_0_1 != 0) {
        AINcode_print_value_chanID(cmdLine, nWords);
        return;
    }
    cmdLine.serial().printf("\r\nScanRead_nWords_externalClock nWords=");
    cmdLine.serial().printf("%d\r\n", nWords);
    for (int index = 0; index < nWords; index++) {
        // int16_t misoData16 = MAX11131_ScanRead();
        int16_t value_u12 = ((g_MAX11131_device.RAW_misoData16[index] >> 3) & 0x0FFF);
        int channelId = g_MAX11131_device.channelNumber_0_15;
        // diagnostic: print raw MISO data
        cmdLine.serial().printf("      MAX11131.MISO[");
        cmdLine.serial().printf("%d", index);
        cmdLine.serial().printf("]=0x");
        cmdLine.serial().printf("0x%4.4x", (g_MAX11131_device.RAW_misoData16[index] & 0xFFFF));
        cmdLine.serial().printf(":");
        print_value(cmdLine, value_u12, channelId);
        cmdLine.serial().printf("\r\n");
    }
}

// CODE GENERATOR: MAX11131 needs AINcode_print_value_chanID_mean(CmdLine& cmdLine, int nWords)
//----------------------------------------
// read data words and calculate mean, stddev
// @pre RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data
// @pre AINcode[NUM_CHANNELS] contains the latest readings in LSBs
void AINcode_print_value_chanID_mean(CmdLine& cmdLine, int nWords)
{
  cmdLine.serial().printf("\r\nScanRead_nWords_chanID_mean nWords=");
  cmdLine.serial().printf("%d\r\n", nWords);
  double Sx = 0;
  double Sxx = 0;
  for (int index = 0; index < nWords; index++) {
    //~ int16_t misoData16 = MAX11131_ScanRead();
    // For internal clock modes, the data format always includes the channel address.
    //     misoData16 = CH[3:0] DATA[11:0]
    int16_t value_u12 = (g_MAX11131_device.RAW_misoData16[index] & 0x0FFF);
    int channelId = ((g_MAX11131_device.RAW_misoData16[index] >> 12) & 0x000F);
    // TODO: sign-extend value_s12 from value_u12
    //
    cmdLine.serial().printf("n=");
    cmdLine.serial().printf("%d", index);
    print_value(cmdLine, value_u12, channelId);
    //
    Sx = Sx + value_u12;
    Sxx = Sxx + ((double)value_u12 * value_u12);
    cmdLine.serial().printf(" Sx=%f", Sx);
    cmdLine.serial().printf(" Sxx=%f", Sxx);
    cmdLine.serial().printf("\r\n");
  }
  double mean = Sx / nWords;
  cmdLine.serial().printf("  mean=%f", mean);
  cmdLine.serial().printf("=0x%4.4x", (int)mean);
  // calculate standard deviation from N, Sx, Sxx
  if (nWords >= 2)
  {
    double variance = (Sxx - ( Sx * Sx / nWords) ) / (nWords - 1);
    cmdLine.serial().printf("  variance=%f", variance);
    // stddev = square root of variance
    double stddev = sqrt(variance);
    cmdLine.serial().printf("  stddev=%f", stddev);
  }
  cmdLine.serial().printf("\r\n");
}

void MAX11131_menu_help(CmdLine & cmdLine)
{
    cmdLine.serial().printf("\r\n ! -- Init");
    cmdLine.serial().printf("\r\n 0 NumWords=? -- ReadAINcode");
    cmdLine.serial().printf("\r\n 1 ch=? pm=? id=? -- ScanManual");
    cmdLine.serial().printf("\r\n 2 ch=? av=? n=? swcnv=? pm=? -- ScanRepeat");
    cmdLine.serial().printf("\r\n 3 ch=? av=? pm=? swcnv=? -- ScanStandardInternalClock");
    cmdLine.serial().printf("\r\n 4 ch=? pm=? id=? -- ScanStandardExternalClock");
    cmdLine.serial().printf("\r\n 5 ch=? av=? pm=? swcnv=? -- ScanUpperInternalClock");
    cmdLine.serial().printf("\r\n 6 ch=? pm=? id=? -- ScanUpperExternalClock");
    cmdLine.serial().printf("\r\n 7 enableMask=? av=? pm=? swcnv=? -- ScanCustomInternalClock");
    cmdLine.serial().printf("\r\n 8 enableMask=? pm=? id=? -- ScanCustomExternalClock");
    cmdLine.serial().printf("\r\n 9 channelsPattern...=? pm=? id=? -- ScanSampleSetExternalClock");
    cmdLine.serial().printf("\r\n IB ch=? -- Reconfigure_DifferentialBipolarFSVref");
    cmdLine.serial().printf("\r\n IR ch=? -- Reconfigure_DifferentialBipolarFS2Vref");
    cmdLine.serial().printf("\r\n IS ch=? -- Reconfigure_SingleEnded");
    cmdLine.serial().printf("\r\n IU ch=? -- Reconfigure_DifferentialUnipolar");
    //
    cmdLine.serial().printf("\r\n @ -- print MAX11131 configuration");

    //
        // case 'G'..'Z','g'..'z' are reserved for GPIO commands
        // case 'A'..'F','a'..'f' may be available if not claimed by bitstream commands
    cmdLine.serial().printf("\r\n C -- CNVST output PulseLow"); // TODO: ExternFunctionGPIOPinCommand testMenuGPIOItemsDict
    cmdLine.serial().printf("\r\n E -- EOC input value"); // TODO: ExternFunctionGPIOPinCommand testMenuGPIOItemsDict

    //
}

bool MAX11131_menu_onEOLcommandParser(CmdLine & cmdLine)
{


                    // parse argument int16_t ADC_CONFIGURATION
        int16_t ADC_CONFIGURATION = g_MAX11131_device.ADC_CONFIGURATION; // default to global property value
        if (cmdLine.parse_int16_dec("ADC_CONFIGURATION", ADC_CONFIGURATION))
        {
            g_MAX11131_device.ADC_CONFIGURATION = ADC_CONFIGURATION; // update global property value
        }

                    // parse argument int16_t ADC_MODE_CONTROL
        int16_t ADC_MODE_CONTROL = g_MAX11131_device.ADC_MODE_CONTROL; // default to global property value
        if (cmdLine.parse_int16_dec("ADC_MODE_CONTROL", ADC_MODE_CONTROL))
        {
            g_MAX11131_device.ADC_MODE_CONTROL = ADC_MODE_CONTROL; // update global property value
        }

                    // parse argument int16_t BIPOLAR
        int16_t BIPOLAR = g_MAX11131_device.BIPOLAR; // default to global property value
        if (cmdLine.parse_int16_dec("BIPOLAR", BIPOLAR))
        {
            g_MAX11131_device.BIPOLAR = BIPOLAR; // update global property value
        }

                    // parse argument int16_t CSCAN0
        int16_t CSCAN0 = g_MAX11131_device.CSCAN0; // default to global property value
        if (cmdLine.parse_int16_dec("CSCAN0", CSCAN0))
        {
            g_MAX11131_device.CSCAN0 = CSCAN0; // update global property value
        }

                    // parse argument int16_t CSCAN1
        int16_t CSCAN1 = g_MAX11131_device.CSCAN1; // default to global property value
        if (cmdLine.parse_int16_dec("CSCAN1", CSCAN1))
        {
            g_MAX11131_device.CSCAN1 = CSCAN1; // update global property value
        }

                    // parse argument uint16_t NumWords
        uint16_t NumWords = g_MAX11131_device.NumWords; // default to global property value
        if (cmdLine.parse_uint16_dec("NumWords", NumWords))
        {
            g_MAX11131_device.NumWords = NumWords; // update global property value
        }

                    // parse argument uint8_t PowerManagement_0_2
        uint8_t PowerManagement_0_2 = g_MAX11131_device.PowerManagement_0_2; // default to global property value
        if (cmdLine.parse_uint8_dec("PowerManagement_0_2", PowerManagement_0_2))
        {
            g_MAX11131_device.PowerManagement_0_2 = PowerManagement_0_2; // update global property value
        }
                    // "pm" is an alias for argument "PowerManagement_0_2"
        if (cmdLine.parse_uint8_dec("pm", PowerManagement_0_2))
        {
            g_MAX11131_device.PowerManagement_0_2 = PowerManagement_0_2; // update global property value
        }

                    // parse argument int16_t RANGE
        int16_t RANGE = g_MAX11131_device.RANGE; // default to global property value
        if (cmdLine.parse_int16_dec("RANGE", RANGE))
        {
            g_MAX11131_device.RANGE = RANGE; // update global property value
        }

                    // parse argument int16_t SAMPLESET
        int16_t SAMPLESET = g_MAX11131_device.SAMPLESET; // default to global property value
        if (cmdLine.parse_int16_dec("SAMPLESET", SAMPLESET))
        {
            g_MAX11131_device.SAMPLESET = SAMPLESET; // update global property value
        }

                    // parse argument uint8_t SPI_MOSI_Semantic
        uint8_t SPI_MOSI_Semantic = g_MAX11131_device.SPI_MOSI_Semantic; // default to global property value
        if (cmdLine.parse_uint8_dec("SPI_MOSI_Semantic", SPI_MOSI_Semantic))
        {
            g_MAX11131_device.SPI_MOSI_Semantic = SPI_MOSI_Semantic; // update global property value
        }

                    // parse argument uint8_t ScanMode
        uint8_t ScanMode = g_MAX11131_device.ScanMode; // default to global property value
        if (cmdLine.parse_uint8_dec("ScanMode", ScanMode))
        {
            g_MAX11131_device.ScanMode = ScanMode; // update global property value
        }

                    // parse argument int16_t UNIPOLAR
        int16_t UNIPOLAR = g_MAX11131_device.UNIPOLAR; // default to global property value
        if (cmdLine.parse_int16_dec("UNIPOLAR", UNIPOLAR))
        {
            g_MAX11131_device.UNIPOLAR = UNIPOLAR; // update global property value
        }

                    // parse argument double VRef
        double VRef = g_MAX11131_device.VRef; // default to global property value
        if (cmdLine.parse_double("VRef", VRef))
        {
            g_MAX11131_device.VRef = VRef; // update global property value
        }

                    // parse argument uint8_t average_0_4_8_16_32
        uint8_t average_0_4_8_16_32 = g_MAX11131_device.average_0_4_8_16_32; // default to global property value
        if (cmdLine.parse_uint8_dec("average_0_4_8_16_32", average_0_4_8_16_32))
        {
            g_MAX11131_device.average_0_4_8_16_32 = average_0_4_8_16_32; // update global property value
        }
                    // "av" is an alias for argument "average_0_4_8_16_32"
        if (cmdLine.parse_uint8_dec("av", average_0_4_8_16_32))
        {
            g_MAX11131_device.average_0_4_8_16_32 = average_0_4_8_16_32; // update global property value
        }

                    // parse argument uint8_t chan_id_0_1
        uint8_t chan_id_0_1 = g_MAX11131_device.chan_id_0_1; // default to global property value
        if (cmdLine.parse_uint8_dec("chan_id_0_1", chan_id_0_1))
        {
            g_MAX11131_device.chan_id_0_1 = chan_id_0_1; // update global property value
        }
                    // "id" is an alias for argument "chan_id_0_1"
        if (cmdLine.parse_uint8_dec("id", chan_id_0_1))
        {
            g_MAX11131_device.chan_id_0_1 = chan_id_0_1; // update global property value
        }

                    // parse argument uint8_t channelNumber_0_15
        uint8_t channelNumber_0_15 = g_MAX11131_device.channelNumber_0_15; // default to global property value
        if (cmdLine.parse_uint8_dec("channelNumber_0_15", channelNumber_0_15))
        {
            g_MAX11131_device.channelNumber_0_15 = channelNumber_0_15; // update global property value
        }
                    // "ch" is an alias for argument "channelNumber_0_15"
        if (cmdLine.parse_uint8_dec("ch", channelNumber_0_15))
        {
            g_MAX11131_device.channelNumber_0_15 = channelNumber_0_15; // update global property value
        }

                    // parse argument int16_t enabledChannelsMask
        int16_t enabledChannelsMask = g_MAX11131_device.enabledChannelsMask; // default to global property value
        if (cmdLine.parse_int16_dec("enabledChannelsMask", enabledChannelsMask))
        {
            g_MAX11131_device.enabledChannelsMask = enabledChannelsMask; // update global property value
        }
                    // "enableMask" is an alias for argument "enabledChannelsMask"
        if (cmdLine.parse_int16_dec("enableMask", enabledChannelsMask))
        {
            g_MAX11131_device.enabledChannelsMask = enabledChannelsMask; // update global property value
        }

                    // parse argument uint8_t enabledChannelsPatternLength_1_256
        uint8_t enabledChannelsPatternLength_1_256 = g_MAX11131_device.enabledChannelsPatternLength_1_256; // default to global property value
        if (cmdLine.parse_uint8_dec("enabledChannelsPatternLength_1_256", enabledChannelsPatternLength_1_256))
        {
            g_MAX11131_device.enabledChannelsPatternLength_1_256 = enabledChannelsPatternLength_1_256; // update global property value
        }

                    // parse argument uint8_t isExternalClock
        uint8_t isExternalClock = g_MAX11131_device.isExternalClock; // default to global property value
        if (cmdLine.parse_uint8_dec("isExternalClock", isExternalClock))
        {
            g_MAX11131_device.isExternalClock = isExternalClock; // update global property value
        }

                    // parse argument uint8_t nscan_4_8_12_16
        uint8_t nscan_4_8_12_16 = g_MAX11131_device.nscan_4_8_12_16; // default to global property value
        if (cmdLine.parse_uint8_dec("nscan_4_8_12_16", nscan_4_8_12_16))
        {
            g_MAX11131_device.nscan_4_8_12_16 = nscan_4_8_12_16; // update global property value
        }
                    // "n" is an alias for argument "nscan_4_8_12_16"
        if (cmdLine.parse_uint8_dec("n", nscan_4_8_12_16))
        {
            g_MAX11131_device.nscan_4_8_12_16 = nscan_4_8_12_16; // update global property value
        }

                    // parse argument uint8_t swcnv_0_1
        uint8_t swcnv_0_1 = g_MAX11131_device.swcnv_0_1; // default to global property value
        if (cmdLine.parse_uint8_dec("swcnv_0_1", swcnv_0_1))
        {
            g_MAX11131_device.swcnv_0_1 = swcnv_0_1; // update global property value
        }
                    // "swcnv" is an alias for argument "swcnv_0_1"
        if (cmdLine.parse_uint8_dec("swcnv", swcnv_0_1))
        {
            g_MAX11131_device.swcnv_0_1 = swcnv_0_1; // update global property value
        }

    switch (cmdLine[0])
    {
        case '@':
        {
                    cmdLine.serial().printf("ADC_MODE_CONTROL = ");
                    cmdLine.serial().printf("%d = 0x%4.4x\r\n", g_MAX11131_device.ADC_MODE_CONTROL, g_MAX11131_device.ADC_MODE_CONTROL);
                    cmdLine.serial().printf("ADC_CONFIGURATION = ");
                    cmdLine.serial().printf("%d = 0x%4.4x\r\n", g_MAX11131_device.ADC_CONFIGURATION, g_MAX11131_device.ADC_CONFIGURATION);
                    cmdLine.serial().printf("UNIPOLAR = ");
                    cmdLine.serial().printf("%d = 0x%4.4x\r\n", g_MAX11131_device.UNIPOLAR, g_MAX11131_device.UNIPOLAR);
                    cmdLine.serial().printf("BIPOLAR = ");
                    cmdLine.serial().printf("%d = 0x%4.4x\r\n", g_MAX11131_device.BIPOLAR, g_MAX11131_device.BIPOLAR);
                    cmdLine.serial().printf("RANGE = ");
                    cmdLine.serial().printf("%d = 0x%4.4x\r\n", g_MAX11131_device.RANGE, g_MAX11131_device.RANGE);
                    cmdLine.serial().printf("CSCAN0 = ");
                    cmdLine.serial().printf("%d = 0x%4.4x\r\n", g_MAX11131_device.CSCAN0, g_MAX11131_device.CSCAN0);
                    cmdLine.serial().printf("CSCAN1 = ");
                    cmdLine.serial().printf("%d = 0x%4.4x\r\n", g_MAX11131_device.CSCAN1, g_MAX11131_device.CSCAN1);
                    cmdLine.serial().printf("SAMPLESET = ");
                    cmdLine.serial().printf("%d = 0x%4.4x\r\n", g_MAX11131_device.SAMPLESET, g_MAX11131_device.SAMPLESET);
                    cmdLine.serial().printf("enabledChannelsPatternLength_1_256 = ");
                    cmdLine.serial().printf("%d = 0x%2.2x\r\n", g_MAX11131_device.enabledChannelsPatternLength_1_256, g_MAX11131_device.enabledChannelsPatternLength_1_256);
                for(int index = 0; (index < g_MAX11131_device.enabledChannelsPatternLength_1_256) && (index < 16); index++) {
                    cmdLine.serial().printf("enabledChannelsPattern[%d] = ", index);
                    cmdLine.serial().printf("%d = 0x%2.2x\r\n", g_MAX11131_device.enabledChannelsPattern[index], g_MAX11131_device.enabledChannelsPattern[index]);
                }
                    cmdLine.serial().printf("SPI_MOSI_Semantic = ");
                    cmdLine.serial().printf("%d = 0x%2.2x\r\n", g_MAX11131_device.SPI_MOSI_Semantic, g_MAX11131_device.SPI_MOSI_Semantic);
                    cmdLine.serial().printf("NumWords = ");
                    cmdLine.serial().printf("%d = 0x%4.4x\r\n", g_MAX11131_device.NumWords, g_MAX11131_device.NumWords);
                    cmdLine.serial().printf("isExternalClock = ");
                    cmdLine.serial().printf("%d = 0x%2.2x\r\n", g_MAX11131_device.isExternalClock, g_MAX11131_device.isExternalClock);
                    cmdLine.serial().printf("ScanMode = ");
                    cmdLine.serial().printf("%d = 0x%2.2x\r\n", g_MAX11131_device.ScanMode, g_MAX11131_device.ScanMode);
                    cmdLine.serial().printf("channelNumber_0_15 = ");
                    cmdLine.serial().printf("%d = 0x%2.2x\r\n", g_MAX11131_device.channelNumber_0_15, g_MAX11131_device.channelNumber_0_15);
                    cmdLine.serial().printf("PowerManagement_0_2 = ");
                    cmdLine.serial().printf("%d = 0x%2.2x\r\n", g_MAX11131_device.PowerManagement_0_2, g_MAX11131_device.PowerManagement_0_2);
                    cmdLine.serial().printf("chan_id_0_1 = ");
                    cmdLine.serial().printf("%d = 0x%2.2x\r\n", g_MAX11131_device.chan_id_0_1, g_MAX11131_device.chan_id_0_1);
                    cmdLine.serial().printf("average_0_4_8_16_32 = ");
                    cmdLine.serial().printf("%d = 0x%2.2x\r\n", g_MAX11131_device.average_0_4_8_16_32, g_MAX11131_device.average_0_4_8_16_32);
                    cmdLine.serial().printf("nscan_4_8_12_16 = ");
                    cmdLine.serial().printf("%d = 0x%2.2x\r\n", g_MAX11131_device.nscan_4_8_12_16, g_MAX11131_device.nscan_4_8_12_16);
                    cmdLine.serial().printf("swcnv_0_1 = ");
                    cmdLine.serial().printf("%d = 0x%2.2x\r\n", g_MAX11131_device.swcnv_0_1, g_MAX11131_device.swcnv_0_1);
                    cmdLine.serial().printf("enabledChannelsMask = ");
                    cmdLine.serial().printf("%d = 0x%4.4x\r\n", g_MAX11131_device.enabledChannelsMask, g_MAX11131_device.enabledChannelsMask);
                for(int index = 0; (index < 16) && (index < 16); index++) {
                    cmdLine.serial().printf("AINcode[%d] = ", index);
                    cmdLine.serial().printf("%d = 0x%4.4x\r\n", g_MAX11131_device.AINcode[index], g_MAX11131_device.AINcode[index]);
                }
                for(int index = 0; (index < 256) && (index < 16); index++) {
                    cmdLine.serial().printf("RAW_misoData16[%d] = ", index);
                    cmdLine.serial().printf("%d = 0x%4.4x\r\n", g_MAX11131_device.RAW_misoData16[index], g_MAX11131_device.RAW_misoData16[index]);
                }
                    cmdLine.serial().printf("VRef = ");
                    cmdLine.serial().printf("%1.6f\r\n", g_MAX11131_device.VRef);
                    return true; // command handled by MAX11131
            break;
        }
        // case 'G'..'Z','g'..'z' are reserved for GPIO commands
        // case 'A'..'F','a'..'f' may be available if not claimed by bitstream commands
        case 'C':
        {
                // TODO output PulseLow
                g_MAX11131_device.CNVSToutputPulseLow();
                    return true; // command handled by MAX11131
            break;
        }
        case 'E':
        {
                // TODO capture and print input Value
                cmdLine.serial().printf("%d", g_MAX11131_device.EOCinputValue());
                    return true; // command handled by MAX11131
            break;
        }
        // case '0'..'9','A'..'F','a'..'f' letters are reserved for bitstream commands
        case '!':
        {
                    // test menu command '!' handler:
                    // helpString='! -- Init'
                    // CMD_='None'
                    // CommandName='Init'
                    // CommandParamIn='void'
                    // CommandReturnType='void'
                    // @Pre='        // @pre one of the MAX11311_Scan functions was called, setting g_MAX11131_device.NumWords'
                    // @Pre='    // @pre one of the MAX11311_Scan functions was called, setting g_MAX11131_device.NumWords'
                    // @Pre='    // @pre one of the MAX11311_Scan functions was called, setting g_MAX11131_device.NumWords'
                    // @Param[in]=''
                    // @Param[out]=''
                    // @Post='    // @post RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data'
                    // @Post='    // @post AINcode[NUM_CHANNELS] contains the latest readings in LSBs'
                    // displayPost=''
                    // @Return=''
                    // @Test='@test group SPI48_3MSps // support 3MSps parts SCLK<=48MHz /*!Arduino:(enabled by default)*/'
                    // @Test='@test group SPI48_3MSps tinyTester.print("SPI 48MHz")'
                    // @Test='@test group SPI48_3MSps SPIfrequency(48000000); // support 3MSps parts SCLK<=48MHz'
                    // @Test='@test group SPI48_3MSps SPIgetFrequency() expect 48000000'
                    // @Test='@test group SPI48_3MSps tinyTester.settle_time_msec = 250 // default 250'
                    // @Test='@test group SPI48_3MSps tinyTester.Wait_Output_Settling()'
                    // @Test='@test group SPI48_3MSps SPIoutputCS(0)'
                    // @Test='@test group SPI48_3MSps SPIread16bits()'
                    // @Test='@test group SPI48_3MSps SPIoutputCS(1)'
                    // @Test='@test group SPI16MHz_1MSps // support 1MSps parts SCLK<=16MHz /*!Arduino:(enabled by default)*/'
                    // @Test='@test group SPI16MHz_1MSps tinyTester.print("SPI 16MHz")'
                    // @Test='@test group SPI16MHz_1MSps SPIfrequency(16000000); // support 1MSps parts SCLK<=16MHz'
                    // @Test='@test group SPI16MHz_1MSps SPIgetFrequency() expect 16000000'
                    // @Test='@test group SPI16MHz_1MSps tinyTester.settle_time_msec = 250 // default 250'
                    // @Test='@test group SPI16MHz_1MSps tinyTester.Wait_Output_Settling()'
                    // @Test='@test group SPI16MHz_1MSps SPIoutputCS(0)'
                    // @Test='@test group SPI16MHz_1MSps SPIread16bits()'
                    // @Test='@test group SPI16MHz_1MSps SPIoutputCS(1)'
                    // @Test='@test group SPI8MHz_500kSps // support 500kSps parts SCLK<=8MHz /*!Arduino:(enabled by default)*/'
                    // @Test='@test group SPI8MHz_500kSps tinyTester.print("SPI 8MHz")'
                    // @Test='@test group SPI8MHz_500kSps SPIfrequency(8000000); // support 500kSps parts SCLK<=8MHz'
                    // @Test='@test group SPI8MHz_500kSps SPIgetFrequency() expect 8000000'
                    // @Test='@test group SPI8MHz_500kSps tinyTester.settle_time_msec = 250 // default 250'
                    // @Test='@test group SPI8MHz_500kSps tinyTester.Wait_Output_Settling()'
                    // @Test='@test group SPI8MHz_500kSps SPIoutputCS(0)'
                    // @Test='@test group SPI8MHz_500kSps SPIread16bits()'
                    // @Test='@test group SPI8MHz_500kSps SPIoutputCS(1)'
                    // @Test='@test group SPI12MHz_1MSps // support 1MSps parts SCLK<=16MHz /*!Arduino:(enabled by default)*/'
                    // @Test='@test group SPI12MHz_1MSps tinyTester.print("SPI 12MHz")'
                    // @Test='@test group SPI12MHz_1MSps SPIfrequency(12000000); // support 1MSps parts SCLK<=16MHz'
                    // @Test='@test group SPI12MHz_1MSps SPIgetFrequency() expect 12000000'
                    // @Test='@test group SPI12MHz_1MSps tinyTester.settle_time_msec = 250 // default 250'
                    // @Test='@test group SPI12MHz_1MSps tinyTester.Wait_Output_Settling()'
                    // @Test='@test group SPI12MHz_1MSps SPIoutputCS(0)'
                    // @Test='@test group SPI12MHz_1MSps SPIread16bits()'
                    // @Test='@test group SPI12MHz_1MSps SPIoutputCS(1)'
                    // @Test='@test tinyTester.blink_time_msec = 75 // default 75 resume hardware self test'
                    // @Test='@test tinyTester.print("0.0: MAX11131.Init()")'
                    // @Test='@test Init()'
                    // @Test='@test VRef expect 2.500 // Nominal Full-Scale Voltage Reference'
                    // @Test='@test group TEST10_SCAN_0100 // Test SCAN_0100_StandardExt -- verify VDD,GND,SCLK,MOSI,MISO,CS /*!Arduino:(enabled by default)*/'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("1.0: Test SCAN_0100_StandardExt -- verify VDD,GND,SCLK,MOSI,MISO,CS")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("0000_0000_0100_0010  ADC_MODE_CONTROL SCAN_0000")'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("                                      CHSEL=0 RESET=2 CHANID=1")'
                    // @Test='@test group TEST10_SCAN_0100 SPIwrite16bits(0x0040)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits()'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits()'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits()'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("1000_0000_0000_0000  ADC_CONFIGURATION REFSEL=0 SPM[1:0]=0 ECHO=0")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIwrite16bits(0x8000)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("0010_0111_1010_0100  ADC_MODE_CONTROL SCAN_0100_StandardExt")'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("                                      CHSEL=15 RESET=1 CHANID=1")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIwrite16bits(0x27a4)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0x0xxx (channel ID 0)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0x0000 mask 0xF000 // expect 0x0xxx (channel ID 0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0x1xxx (channel ID 1)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0x1000 mask 0xF000 // expect 0x1xxx (channel ID 1)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0x2xxx (channel ID 2)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0x2000 mask 0xF000 // expect 0x2xxx (channel ID 2)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0x3xxx (channel ID 3)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0x3000 mask 0xF000 // expect 0x3xxx (channel ID 3)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0x4xxx (channel ID 4)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0x4000 mask 0xF000 // expect 0x4xxx (channel ID 4)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0x5xxx (channel ID 5)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0x5000 mask 0xF000 // expect 0x5xxx (channel ID 5)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0x6xxx (channel ID 6)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0x6000 mask 0xF000 // expect 0x6xxx (channel ID 6)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0x7xxx (channel ID 7)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0x7000 mask 0xF000 // expect 0x7xxx (channel ID 7)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0x8xxx (channel ID 8)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0x8000 mask 0xF000 // expect 0x8xxx (channel ID 8)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0x9xxx (channel ID 9)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0x9000 mask 0xF000 // expect 0x9xxx (channel ID 9)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0xaxxx (channel ID 10)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0xA000 mask 0xF000 // expect 0xaxxx (channel ID 10)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0xbxxx (channel ID 11)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0xB000 mask 0xF000 // expect 0xbxxx (channel ID 11)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0xcxxx (channel ID 12)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0xC000 mask 0xF000 // expect 0xcxxx (channel ID 12)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0xdxxx (channel ID 13)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0xD000 mask 0xF000 // expect 0xdxxx (channel ID 13)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0xexxx (channel ID 14)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0xE000 mask 0xF000 // expect 0xexxx (channel ID 14)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST10_SCAN_0100 tinyTester.print("MISO --> expect 0xfxxx (channel ID 15)")'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(0)'
                    // @Test='@test group TEST10_SCAN_0100 SPIread16bits() expect 0xF000 mask 0xF000 // expect 0xfxxx (channel ID 15)'
                    // @Test='@test group TEST10_SCAN_0100 SPIoutputCS(1)'
                    // @Test='@test group TEST4_SCAN_0100 // 4 ch=15 pm=0 id=1 -- ScanStandardExternalCloc /*!Arduino:(enabled by default)*/'
                    // @Test='@test group TEST4_SCAN_0100 tinyTester.print("4 ch=15 pm=0 id=1 -- ScanStandardExternalClock")'
                    // @Test='@test group TEST4_SCAN_0100 tinyTester.print("channelNumber_0_15 = 15")'
                    // @Test='@test group TEST4_SCAN_0100 channelNumber_0_15 = 15'
                    // @Test='@test group TEST4_SCAN_0100 tinyTester.print("PowerManagement_0_2 = 0")'
                    // @Test='@test group TEST4_SCAN_0100 PowerManagement_0_2 = 0 // 0=Normal'
                    // @Test='@test group TEST4_SCAN_0100 tinyTester.print("chan_id_0_1 = 1")'
                    // @Test='@test group TEST4_SCAN_0100 chan_id_0_1 = 1 // misoData16 = CH[3:0] DATA[11:0]'
                    // @Test='@test group TEST4_SCAN_0100 tinyTester.print("ScanStandardExternalClock() expect 16")'
                    // @Test='@test group TEST4_SCAN_0100 ScanStandardExternalClock() expect 16 // Scan_0100_StandardExt'
                    // @Test='@test group TEST4_SCAN_0100 tinyTester.print("NumWords expect 16")'
                    // @Test='@test group TEST4_SCAN_0100 NumWords expect 16'
                    // @Test='@test group TEST4_SCAN_0100 ReadAINcode()'
                    // @Test='@test group TEST4_SCAN_0100 tinyTester.print("Verify RAW_misoData16[0..15]>>12&0x000F == 0..15 channelId")'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[0]  expect 0x0000 mask 0xF000 // expect 0x0xxx (channel ID 0)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[1]  expect 0x1000 mask 0xF000 // expect 0x1xxx (channel ID 1)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[2]  expect 0x2000 mask 0xF000 // expect 0x2xxx (channel ID 2)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[3]  expect 0x3000 mask 0xF000 // expect 0x3xxx (channel ID 3)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[4]  expect 0x4000 mask 0xF000 // expect 0x4xxx (channel ID 4)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[5]  expect 0x5000 mask 0xF000 // expect 0x5xxx (channel ID 5)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[6]  expect 0x6000 mask 0xF000 // expect 0x6xxx (channel ID 6)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[7]  expect 0x7000 mask 0xF000 // expect 0x7xxx (channel ID 7)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[8]  expect 0x8000 mask 0xF000 // expect 0x8xxx (channel ID 8)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[9]  expect 0x9000 mask 0xF000 // expect 0x9xxx (channel ID 9)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[10] expect 0xA000 mask 0xF000 // expect 0xaxxx (channel ID 10)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[11] expect 0xB000 mask 0xF000 // expect 0xbxxx (channel ID 11)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[12] expect 0xC000 mask 0xF000 // expect 0xcxxx (channel ID 12)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[13] expect 0xD000 mask 0xF000 // expect 0xdxxx (channel ID 13)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[14] expect 0xE000 mask 0xF000 // expect 0xexxx (channel ID 14)'
                    // @Test='@test group TEST4_SCAN_0100 RAW_misoData16[15] expect 0xF000 mask 0xF000 // expect 0xfxxx (channel ID 15)'
                    // @Test='@test tinyTester.print("1000_0000_0000_0000  ADC_CONFIGURATION REFSEL=0 SPM[1:0]=0 ECHO=0")'
                    // @Test='@test SPIoutputCS(0)'
                    // @Test='@test SPIwrite16bits(0x8000)'
                    // @Test='@test SPIoutputCS(1)'
                    // @Test='@test tinyTester.print("0010_0111_1010_0100  ADC_MODE_CONTROL Scan_0100_StandardExt")'
                    // @Test='@test tinyTester.print("                                      CHSEL=15 RESET=1 CHANID=1")'
                    // @Test='@test SPIoutputCS(0)'
                    // @Test='@test SPIwrite16bits(0x27a4)'
                    // @Test='@test SPIoutputCS(1)'
                    // @Test='@test group TEST11_SCAN_0011 // 1.1: Test Scan_0011_StandardInt -- verify Internal Clock CNVST,EOC /*!Arduino:(enabled by default)*/'
                    // @Test='@test group TEST11_SCAN_0011 tinyTester.print("1.1: Test Scan_0011_StandardInt -- verify Internal Clock CNVST,EOC")'
                    // @Test='@test group TEST11_SCAN_0011 Init();'
                    // @Test='@test group TEST11_SCAN_0011 SPIoutputCS(0); // drive CS low'
                    // @Test='@test group TEST11_SCAN_0011 group TEST11_SCAN_0011 SPIoutputCS(1); // drive CS high'
                    // @Test='@test group TEST11_SCAN_0011 tinyTester.print("1000_0000_0000_0000  ADC_CONFIGURATION REFSEL=0 SPM[1:0]=0 ECHO=0")'
                    // @Test='@test group TEST11_SCAN_0011 SPIoutputCS(0); // drive CS low'
                    // @Test='@test group TEST11_SCAN_0011 SPIwrite16bits(0x8000);'
                    // @Test='@test group TEST11_SCAN_0011 SPIoutputCS(1); // drive CS high'
                    // @Test='@test group TEST11_SCAN_0011 tinyTester.print("0001_1001_1010_0000  ADC_MODE_CONTROL Scan_0011_StandardInt")'
                    // @Test='@test group TEST11_SCAN_0011 tinyTester.print("                                      CHSEL=3 RESET=1 SWCNV=0")'
                    // @Test='@test group TEST11_SCAN_0011 SPIoutputCS(0); // drive CS low'
                    // @Test='@test group TEST11_SCAN_0011 SPIwrite16bits(0x19a0);'
                    // @Test='@test group TEST11_SCAN_0011 SPIoutputCS(1); // drive CS high'
                    cmdLine.serial().printf("Init");
                    // call function Init
                    g_MAX11131_device.Init();
                    return true; // command handled by MAX11131
        } // end case '!'
        break;
        case '0':
        {
                    // test menu command '0' handler:
                    // helpString='0 NumWords=? -- ReadAINcode'
                    // CMD_='SCAN_0000_NOP'
                    // CommandName='ReadAINcode'
                    // CommandParamIn='void'
                    // CommandReturnType='void'
                    // @Pre='@pre one of the Scan functions was called, setting g_MAX11131_device.NumWords'
                    // @Param[in]='@param[in] g_MAX11131_device.NumWords: number of words to be read from the FIFO'
                    // @Param[out]=''
                    // @Post='@post g_MAX11131_device.RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data'
                    // @Post='@post g_MAX11131_device.AINcode[NUM_CHANNELS] contains the latest readings in LSBs'
                    // @Return=''
                    // exception MAX11131 Menu item '0' ReadAINcode logic flow -- omit ReadAINcode call here, will be handled in postprocessing
                    cmdLine.serial().printf("ReadAINcode");
                    //
                    // CODE GENERATOR: MAX11131 ReadAINCode and print data
                    if (g_MAX11131_device.isExternalClock)
                    {
                        cmdLine.serial().printf(" External Clock");
                        //
                        // Read raw ADC codes from device into AINcode[] and RAW_misoData16[]
                        // @pre one of the MAX11311_Scan functions was called, setting g_MAX11131_device.NumWords
                        g_MAX11131_device.ReadAINcode();
                        // @post RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data
                        // @post AINcode[NUM_CHANNELS] contains the latest readings in LSBs
                        //
                        AINcode_print_value_externalClock(cmdLine, g_MAX11131_device.NumWords);
                    }
                    else
                    {
                        cmdLine.serial().printf(" Internal Clock");
                        //
                        // Read raw ADC codes from device into AINcode[] and RAW_misoData16[]
                        // @pre one of the MAX11311_Scan functions was called, setting g_MAX11131_device.NumWords
                        g_MAX11131_device.ReadAINcode();
                        // @post RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data
                        // @post AINcode[NUM_CHANNELS] contains the latest readings in LSBs
                        //
                        AINcode_print_value_chanID(cmdLine, g_MAX11131_device.NumWords);
                    }
                    return true; // command handled by MAX11131
        } // end case '0'
        break;
        case '1':
        {
                    // test menu command '1' handler:
                    // helpString='1 ch=? pm=? id=? -- ScanManual'
                    // CMD_='SCAN_0001_Manual'
                    // CommandName='ScanManual'
                    // CommandParamIn='void'
                    // CommandReturnType='int'
                    // @Pre=''
                    // @Param[in]='@param[in] g_MAX11131_device.channelNumber_0_15: AIN Channel Number'
                    // @Param[in]='@param[in] g_MAX11131_device.PowerManagement_0_2: 0=Normal, 1=AutoShutdown, 2=AutoStandby'
                    // @Param[in]='@param[in] g_MAX11131_device.chan_id_0_1: ADC_MODE_CONTROL.CHAN_ID'
                    // @Param[out]=''
                    // @Post='@post NumWords = number of words to be read from the FIFO'
                    // @Return='@return number of ScanRead() words needed to retrieve the data.'
                    cmdLine.serial().printf("ScanManual");
                    // call function ScanManual
                    int result = g_MAX11131_device.ScanManual();
                    cmdLine.serial().printf(" =%d\r\n", result);
                    //
                    // CODE GENERATOR: MAX11131 post-ScanManual ReadAINCode and print data
                    // Read raw ADC codes from device into AINcode[] and RAW_misoData16[]
                    // @pre one of the MAX11311_Scan functions was called, setting g_MAX11131_device.NumWords
                    g_MAX11131_device.ReadAINcode();
                    // @post RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data
                    // @post AINcode[NUM_CHANNELS] contains the latest readings in LSBs
                    //
                    AINcode_print_value_externalClock(cmdLine, g_MAX11131_device.NumWords);
                    return true; // command handled by MAX11131
        } // end case '1'
        break;
        case '2':
        {
                    // test menu command '2' handler:
                    // helpString='2 ch=? av=? n=? swcnv=? pm=? -- ScanRepeat'
                    // CMD_='SCAN_0010_Repeat'
                    // CommandName='ScanRepeat'
                    // CommandParamIn='void'
                    // CommandReturnType='int'
                    // @Pre=''
                    // @Param[in]='@param[in] g_MAX11131_device.channelNumber_0_15: AIN Channel Number'
                    // @Param[in]='@param[in] g_MAX11131_device.average_0_4_8_16_32: Number of samples averaged per ScanRead() word.'
                    // @Param[in]='@param[in] g_MAX11131_device.nscan_4_8_12_16: Number of ScanRead() words to report.'
                    // @Param[in]='@param[in] g_MAX11131_device.swcnv_0_1: ADC_MODE_CONTROL.SWCNV'
                    // @Param[in]='@param[in] g_MAX11131_device.PowerManagement_0_2: 0=Normal, 1=AutoShutdown, 2=AutoStandby'
                    // @Param[out]=''
                    // @Post='@post NumWords = number of words to be read from the FIFO'
                    // @Return='@return number of ScanRead() words needed to retrieve the data.'
                    cmdLine.serial().printf("ScanRepeat");
                    // call function ScanRepeat
                    int result = g_MAX11131_device.ScanRepeat();
                    cmdLine.serial().printf(" =%d\r\n", result);
                    //
                    // CODE GENERATOR: MAX11131 post-ScanRepeat ReadAINCode and print data
                    // Read raw ADC codes from device into AINcode[] and RAW_misoData16[]
                    // @pre one of the MAX11311_Scan functions was called, setting g_MAX11131_device.NumWords
                    g_MAX11131_device.ReadAINcode();
                    // @post RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data
                    // @post AINcode[NUM_CHANNELS] contains the latest readings in LSBs
                    //
                    AINcode_print_value_chanID_mean(cmdLine, g_MAX11131_device.NumWords);
                    return true; // command handled by MAX11131
        } // end case '2'
        break;
        case '3':
        {
                    // test menu command '3' handler:
                    // helpString='3 ch=? av=? pm=? swcnv=? -- ScanStandardInternalClock'
                    // CMD_='SCAN_0011_StandardInternalClock'
                    // CommandName='ScanStandardInternalClock'
                    // CommandParamIn='void'
                    // CommandReturnType='int'
                    // @Pre=''
                    // @Param[in]='@param[in] g_MAX11131_device.channelNumber_0_15: AIN Channel Number'
                    // @Param[in]='@param[in] g_MAX11131_device.average_0_4_8_16_32: Number of samples averaged per ScanRead() word.'
                    // @Param[in]='@param[in] g_MAX11131_device.PowerManagement_0_2: 0=Normal, 1=AutoShutdown, 2=AutoStandby'
                    // @Param[in]='@param[in] g_MAX11131_device.swcnv_0_1: ADC_MODE_CONTROL.SWCNV'
                    // @Param[out]=''
                    // @Post='@post NumWords = number of words to be read from the FIFO'
                    // @Return='@return number of ScanRead() words needed to retrieve the data.'
                    cmdLine.serial().printf("ScanStandardInternalClock");
                    // call function ScanStandardInternalClock
                    int result = g_MAX11131_device.ScanStandardInternalClock();
                    cmdLine.serial().printf(" =%d\r\n", result);
                    //
                    // CODE GENERATOR: MAX11131 post-ScanStandardInternalClock ReadAINCode and print data
                    // Read raw ADC codes from device into AINcode[] and RAW_misoData16[]
                    // @pre one of the MAX11311_Scan functions was called, setting g_MAX11131_device.NumWords
                    g_MAX11131_device.ReadAINcode();
                    // @post RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data
                    // @post AINcode[NUM_CHANNELS] contains the latest readings in LSBs
                    //
                    AINcode_print_value_chanID(cmdLine, g_MAX11131_device.NumWords);
                    return true; // command handled by MAX11131
        } // end case '3'
        break;
        case '4':
        {
                    // test menu command '4' handler:
                    // helpString='4 ch=? pm=? id=? -- ScanStandardExternalClock'
                    // CMD_='SCAN_0100_StandardExternalClock'
                    // CommandName='ScanStandardExternalClock'
                    // CommandParamIn='void'
                    // CommandReturnType='int'
                    // @Pre=''
                    // @Param[in]='@param[in] g_MAX11131_device.channelNumber_0_15: AIN Channel Number'
                    // @Param[in]='@param[in] g_MAX11131_device.PowerManagement_0_2: 0=Normal, 1=AutoShutdown, 2=AutoStandby'
                    // @Param[in]='@param[in] g_MAX11131_device.chan_id_0_1: ADC_MODE_CONTROL.CHAN_ID'
                    // @Param[out]=''
                    // @Post='@post NumWords = number of words to be read from the FIFO'
                    // @Return='@return number of ScanRead() words needed to retrieve the data.'
                    cmdLine.serial().printf("ScanStandardExternalClock");
                    // call function ScanStandardExternalClock
                    int result = g_MAX11131_device.ScanStandardExternalClock();
                    cmdLine.serial().printf(" =%d\r\n", result);
                    //
                    // CODE GENERATOR: MAX11131 post-ScanStandardExternalClock ReadAINCode and print data
                    // Read raw ADC codes from device into AINcode[] and RAW_misoData16[]
                    // @pre one of the MAX11311_Scan functions was called, setting g_MAX11131_device.NumWords
                    g_MAX11131_device.ReadAINcode();
                    // @post RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data
                    // @post AINcode[NUM_CHANNELS] contains the latest readings in LSBs
                    //
                    AINcode_print_value_externalClock(cmdLine, g_MAX11131_device.NumWords);
                    return true; // command handled by MAX11131
        } // end case '4'
        break;
        case '5':
        {
                    // test menu command '5' handler:
                    // helpString='5 ch=? av=? pm=? swcnv=? -- ScanUpperInternalClock'
                    // CMD_='SCAN_0101_UpperInternalClock'
                    // CommandName='ScanUpperInternalClock'
                    // CommandParamIn='void'
                    // CommandReturnType='int'
                    // @Pre=''
                    // @Param[in]='@param[in] g_MAX11131_device.channelNumber_0_15: AIN Channel Number'
                    // @Param[in]='@param[in] g_MAX11131_device.average_0_4_8_16_32: Number of samples averaged per ScanRead() word.'
                    // @Param[in]='@param[in] g_MAX11131_device.PowerManagement_0_2: 0=Normal, 1=AutoShutdown, 2=AutoStandby'
                    // @Param[in]='@param[in] g_MAX11131_device.swcnv_0_1: ADC_MODE_CONTROL.SWCNV'
                    // @Param[out]=''
                    // @Post='@post NumWords = number of words to be read from the FIFO'
                    // @Return='@return number of ScanRead() words needed to retrieve the data.'
                    cmdLine.serial().printf("ScanUpperInternalClock");
                    // call function ScanUpperInternalClock
                    int result = g_MAX11131_device.ScanUpperInternalClock();
                    cmdLine.serial().printf(" =%d\r\n", result);
                    //
                    // CODE GENERATOR: MAX11131 post-ScanUpperInternalClock ReadAINCode and print data
                    // Read raw ADC codes from device into AINcode[] and RAW_misoData16[]
                    // @pre one of the MAX11311_Scan functions was called, setting g_MAX11131_device.NumWords
                    g_MAX11131_device.ReadAINcode();
                    // @post RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data
                    // @post AINcode[NUM_CHANNELS] contains the latest readings in LSBs
                    //
                    AINcode_print_value_chanID(cmdLine, g_MAX11131_device.NumWords);
                    return true; // command handled by MAX11131
        } // end case '5'
        break;
        case '6':
        {
                    // test menu command '6' handler:
                    // helpString='6 ch=? pm=? id=? -- ScanUpperExternalClock'
                    // CMD_='SCAN_0110_UpperExternalClock'
                    // CommandName='ScanUpperExternalClock'
                    // CommandParamIn='void'
                    // CommandReturnType='int'
                    // @Pre=''
                    // @Param[in]='@param[in] g_MAX11131_device.channelNumber_0_15: AIN Channel Number'
                    // @Param[in]='@param[in] g_MAX11131_device.PowerManagement_0_2: 0=Normal, 1=AutoShutdown, 2=AutoStandby'
                    // @Param[in]='@param[in] g_MAX11131_device.chan_id_0_1: ADC_MODE_CONTROL.CHAN_ID'
                    // @Param[out]=''
                    // @Post='@post NumWords = number of words to be read from the FIFO'
                    // @Return='@return number of ScanRead() words needed to retrieve the data.'
                    cmdLine.serial().printf("ScanUpperExternalClock");
                    // call function ScanUpperExternalClock
                    int result = g_MAX11131_device.ScanUpperExternalClock();
                    cmdLine.serial().printf(" =%d\r\n", result);
                    //
                    // CODE GENERATOR: MAX11131 post-ScanUpperExternalClock ReadAINCode and print data
                    // Read raw ADC codes from device into AINcode[] and RAW_misoData16[]
                    // @pre one of the MAX11311_Scan functions was called, setting g_MAX11131_device.NumWords
                    g_MAX11131_device.ReadAINcode();
                    // @post RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data
                    // @post AINcode[NUM_CHANNELS] contains the latest readings in LSBs
                    //
                    AINcode_print_value_externalClock(cmdLine, g_MAX11131_device.NumWords);
                    return true; // command handled by MAX11131
        } // end case '6'
        break;
        case '7':
        {
                    // test menu command '7' handler:
                    // helpString='7 enableMask=? av=? pm=? swcnv=? -- ScanCustomInternalClock'
                    // CMD_='SCAN_0111_CustomInternalClock'
                    // CommandName='ScanCustomInternalClock'
                    // CommandParamIn='void'
                    // CommandReturnType='int'
                    // @Pre=''
                    // @Param[in]='@param[in] g_MAX11131_device.enabledChannelsMask: Bitmap of AIN Channels to scan.'
                    // @Param[in]='@param[in] g_MAX11131_device.average_0_4_8_16_32: Number of samples averaged per ScanRead() word.'
                    // @Param[in]='@param[in] g_MAX11131_device.PowerManagement_0_2: 0=Normal, 1=AutoShutdown, 2=AutoStandby'
                    // @Param[in]='@param[in] g_MAX11131_device.swcnv_0_1: ADC_MODE_CONTROL.SWCNV'
                    // @Param[out]=''
                    // @Post='@post NumWords = number of words to be read from the FIFO'
                    // @Return='@return number of ScanRead() words needed to retrieve the data.'
                    cmdLine.serial().printf("ScanCustomInternalClock");
                    // call function ScanCustomInternalClock
                    int result = g_MAX11131_device.ScanCustomInternalClock();
                    cmdLine.serial().printf(" =%d\r\n", result);
                    //
                    // CODE GENERATOR: MAX11131 post-ScanCustomInternalClock ReadAINCode and print data
                    // Read raw ADC codes from device into AINcode[] and RAW_misoData16[]
                    // @pre one of the MAX11311_Scan functions was called, setting g_MAX11131_device.NumWords
                    g_MAX11131_device.ReadAINcode();
                    // @post RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data
                    // @post AINcode[NUM_CHANNELS] contains the latest readings in LSBs
                    //
                    AINcode_print_value_chanID(cmdLine, g_MAX11131_device.NumWords);
                    return true; // command handled by MAX11131
        } // end case '7'
        break;
        case '8':
        {
                    // test menu command '8' handler:
                    // helpString='8 enableMask=? pm=? id=? -- ScanCustomExternalClock'
                    // CMD_='SCAN_1000_CustomExternalClock'
                    // CommandName='ScanCustomExternalClock'
                    // CommandParamIn='void'
                    // CommandReturnType='int'
                    // @Pre=''
                    // @Param[in]='@param[in] g_MAX11131_device.enabledChannelsMask: Bitmap of AIN Channels to scan.'
                    // @Param[in]='@param[in] g_MAX11131_device.PowerManagement_0_2: 0=Normal, 1=AutoShutdown, 2=AutoStandby'
                    // @Param[in]='@param[in] g_MAX11131_device.chan_id_0_1: ADC_MODE_CONTROL.CHAN_ID'
                    // @Param[out]=''
                    // @Post='@post NumWords = number of words to be read from the FIFO'
                    // @Return='@return number of ScanRead() words needed to retrieve the data.'
                    cmdLine.serial().printf("ScanCustomExternalClock");
                    // call function ScanCustomExternalClock
                    int result = g_MAX11131_device.ScanCustomExternalClock();
                    cmdLine.serial().printf(" =%d\r\n", result);
                    //
                    // CODE GENERATOR: MAX11131 post-ScanCustomExternalClock ReadAINCode and print data
                    // Read raw ADC codes from device into AINcode[] and RAW_misoData16[]
                    // @pre one of the MAX11311_Scan functions was called, setting g_MAX11131_device.NumWords
                    g_MAX11131_device.ReadAINcode();
                    // @post RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data
                    // @post AINcode[NUM_CHANNELS] contains the latest readings in LSBs
                    //
                    AINcode_print_value_externalClock(cmdLine, g_MAX11131_device.NumWords);
                    return true; // command handled by MAX11131
        } // end case '8'
        break;
        case '9':
        {
                    // test menu command '9' handler:
                    // helpString='9 channelsPattern...=? pm=? id=? -- ScanSampleSetExternalClock'
                    // CMD_='SCAN_1001_SampleSetExternalClock'
                    // CommandName='ScanSampleSetExternalClock'
                    // CommandParamIn='void'
                    // CommandReturnType='int'
                    // @Pre='@pre g_MAX11131_device.enabledChannelsPatternLength_1_256: number of channel selections'
                    // @Pre='@pre g_MAX11131_device.enabledChannelsPattern: array containing channel selection pattern'
                    // @Param[in]='@param[in] g_MAX11131_device.enabledChannelsPattern: array of channel select, one channel per byte'
                    // @Param[in]='@param[in] g_MAX11131_device.PowerManagement_0_2: 0=Normal, 1=AutoShutdown, 2=AutoStandby'
                    // @Param[in]='@param[in] g_MAX11131_device.chan_id_0_1: ADC_MODE_CONTROL.CHAN_ID'
                    // @Param[out]=''
                    // @Post='@post NumWords = number of words to be read from the FIFO'
                    // @Return='@return number of ScanRead() words needed to retrieve the data.'
                    // exception MAX11131 Menu item '9' ScanSampleSetExternalClock
                    // parse channelsPattern using parse_byteCount_byteList_dec(size_t& byteCount, char *mosiDataBuf, size_t mosiDataBufSize)
                    size_t byteCount = g_MAX11131_device.enabledChannelsPatternLength_1_256;
                    char *mosiDataBuf = (char *)g_MAX11131_device.enabledChannelsPattern; // cast from uint8_t*
                    size_t mosiDataBufSize = sizeof(g_MAX11131_device.enabledChannelsPattern);
                    cmdLine.parse_byteCount_byteList_dec(byteCount, mosiDataBuf, mosiDataBufSize);
                    g_MAX11131_device.enabledChannelsPatternLength_1_256 = byteCount;
                    cmdLine.serial().printf("ScanSampleSetExternalClock");
                    // call function ScanSampleSetExternalClock
                    int result = g_MAX11131_device.ScanSampleSetExternalClock();
                    cmdLine.serial().printf(" =%d\r\n", result);
                    //
                    // CODE GENERATOR: MAX11131 post-ScanSampleSetExternalClock ReadAINCode and print data
                    // Read raw ADC codes from device into AINcode[] and RAW_misoData16[]
                    // @pre one of the MAX11311_Scan functions was called, setting g_MAX11131_device.NumWords
                    g_MAX11131_device.ReadAINcode();
                    // @post RAW_misoData16[index] contains the raw SPI Master-In,Slave-Out data
                    // @post AINcode[NUM_CHANNELS] contains the latest readings in LSBs
                    //
                    AINcode_print_value_externalClock(cmdLine, g_MAX11131_device.NumWords);
                    return true; // command handled by MAX11131
        } // end case '9'
        break;
        case 'I': // (multiple characters) (testMenuFirstCharHandler="I"):
        {
            switch (cmdLine[1])
            {
                case 'B': // (nested inside case 'I')
                {
                    // test menu command 'IB' handler:
                    // helpString='IB ch=? -- Reconfigure_DifferentialBipolarFSVref'
                    // CMD_='None'
                    // CommandName='Reconfigure_DifferentialBipolarFSVref'
                    // CommandParamIn='int channel_0_15'
                    // CommandReturnType='void'
                    // @Pre=''
                    // @Param[in]=''
                    // @Param[out]=''
                    // @Post=''
                    // displayPost=''
                    // @Return=''
                    // parse argument list
                    // parse argument int channel_0_15
                    int channel_0_15 = 0; // --- g_MAX11131_device.__WARNING_no_match_for_argname_channel_0_15_in_MAX11131_device_t__; // default to global property value
                    if (cmdLine.parse_int_dec("channel_0_15", channel_0_15))
                    {
                        // g_MAX11131_device.__WARNING_no_match_for_argname_channel_0_15_in_MAX11131_device_t__ = channel_0_15; // update global property value
                    }
                    // "ch" is an alias for argument "channel_0_15"
                    if (cmdLine.parse_int_dec("ch", channel_0_15))
                    {
                        // g_MAX11131_device.__WARNING_no_match_for_argname_channel_0_15_in_MAX11131_device_t__ = channel_0_15; // update global property value
                    }
                    // print arguments
                    cmdLine.serial().printf("Reconfigure_DifferentialBipolarFSVref");
                    cmdLine.serial().printf(" channel_0_15=%d", channel_0_15);
                    cmdLine.serial().printf("\r\n");
                    // call function Reconfigure_DifferentialBipolarFSVref(channel_0_15)
                    g_MAX11131_device.Reconfigure_DifferentialBipolarFSVref(channel_0_15);
                    return true; // command handled by MAX11131
                } // end nested case 'IB'
                break;
                case 'R': // (nested inside case 'I')
                {
                    // test menu command 'IR' handler:
                    // helpString='IR ch=? -- Reconfigure_DifferentialBipolarFS2Vref'
                    // CMD_='None'
                    // CommandName='Reconfigure_DifferentialBipolarFS2Vref'
                    // CommandParamIn='int channel_0_15'
                    // CommandReturnType='void'
                    // @Pre=''
                    // @Param[in]=''
                    // @Param[out]=''
                    // @Post=''
                    // displayPost=''
                    // @Return=''
                    // parse argument list
                    // parse argument int channel_0_15
                    int channel_0_15 = 0; // --- g_MAX11131_device.__WARNING_no_match_for_argname_channel_0_15_in_MAX11131_device_t__; // default to global property value
                    if (cmdLine.parse_int_dec("channel_0_15", channel_0_15))
                    {
                        // g_MAX11131_device.__WARNING_no_match_for_argname_channel_0_15_in_MAX11131_device_t__ = channel_0_15; // update global property value
                    }
                    // "ch" is an alias for argument "channel_0_15"
                    if (cmdLine.parse_int_dec("ch", channel_0_15))
                    {
                        // g_MAX11131_device.__WARNING_no_match_for_argname_channel_0_15_in_MAX11131_device_t__ = channel_0_15; // update global property value
                    }
                    // print arguments
                    cmdLine.serial().printf("Reconfigure_DifferentialBipolarFS2Vref");
                    cmdLine.serial().printf(" channel_0_15=%d", channel_0_15);
                    cmdLine.serial().printf("\r\n");
                    // call function Reconfigure_DifferentialBipolarFS2Vref(channel_0_15)
                    g_MAX11131_device.Reconfigure_DifferentialBipolarFS2Vref(channel_0_15);
                    return true; // command handled by MAX11131
                } // end nested case 'IR'
                break;
                case 'S': // (nested inside case 'I')
                {
                    // test menu command 'IS' handler:
                    // helpString='IS ch=? -- Reconfigure_SingleEnded'
                    // CMD_='None'
                    // CommandName='Reconfigure_SingleEnded'
                    // CommandParamIn='int channel_0_15'
                    // CommandReturnType='void'
                    // @Pre=''
                    // @Param[in]=''
                    // @Param[out]=''
                    // @Post=''
                    // displayPost=''
                    // @Return=''
                    // parse argument list
                    // parse argument int channel_0_15
                    int channel_0_15 = 0; // --- g_MAX11131_device.__WARNING_no_match_for_argname_channel_0_15_in_MAX11131_device_t__; // default to global property value
                    if (cmdLine.parse_int_dec("channel_0_15", channel_0_15))
                    {
                        // g_MAX11131_device.__WARNING_no_match_for_argname_channel_0_15_in_MAX11131_device_t__ = channel_0_15; // update global property value
                    }
                    // "ch" is an alias for argument "channel_0_15"
                    if (cmdLine.parse_int_dec("ch", channel_0_15))
                    {
                        // g_MAX11131_device.__WARNING_no_match_for_argname_channel_0_15_in_MAX11131_device_t__ = channel_0_15; // update global property value
                    }
                    // print arguments
                    cmdLine.serial().printf("Reconfigure_SingleEnded");
                    cmdLine.serial().printf(" channel_0_15=%d", channel_0_15);
                    cmdLine.serial().printf("\r\n");
                    // call function Reconfigure_SingleEnded(channel_0_15)
                    g_MAX11131_device.Reconfigure_SingleEnded(channel_0_15);
                    return true; // command handled by MAX11131
                } // end nested case 'IS'
                break;
                case 'U': // (nested inside case 'I')
                {
                    // test menu command 'IU' handler:
                    // helpString='IU ch=? -- Reconfigure_DifferentialUnipolar'
                    // CMD_='None'
                    // CommandName='Reconfigure_DifferentialUnipolar'
                    // CommandParamIn='int channel_0_15'
                    // CommandReturnType='void'
                    // @Pre=''
                    // @Param[in]=''
                    // @Param[out]=''
                    // @Post=''
                    // displayPost=''
                    // @Return=''
                    // parse argument list
                    // parse argument int channel_0_15
                    int channel_0_15 = 0; // --- g_MAX11131_device.__WARNING_no_match_for_argname_channel_0_15_in_MAX11131_device_t__; // default to global property value
                    if (cmdLine.parse_int_dec("channel_0_15", channel_0_15))
                    {
                        // g_MAX11131_device.__WARNING_no_match_for_argname_channel_0_15_in_MAX11131_device_t__ = channel_0_15; // update global property value
                    }
                    // "ch" is an alias for argument "channel_0_15"
                    if (cmdLine.parse_int_dec("ch", channel_0_15))
                    {
                        // g_MAX11131_device.__WARNING_no_match_for_argname_channel_0_15_in_MAX11131_device_t__ = channel_0_15; // update global property value
                    }
                    // print arguments
                    cmdLine.serial().printf("Reconfigure_DifferentialUnipolar");
                    cmdLine.serial().printf(" channel_0_15=%d", channel_0_15);
                    cmdLine.serial().printf("\r\n");
                    // call function Reconfigure_DifferentialUnipolar(channel_0_15)
                    g_MAX11131_device.Reconfigure_DifferentialUnipolar(channel_0_15);
                    return true; // command handled by MAX11131
                } // end nested case 'IU'
                break;
            } // end nested switch (cmdLine[1]) inside case 'I'
            break;
        } // end case 'I'
    } // end switch (cmdLine[0])
    return false; // command not handled by MAX11131
} // end bool MAX11131_menu_onEOLcommandParser(CmdLine & cmdLine)

