/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
*/

/**
 * @brief Library for the MAX30208\n
 *
 * @code
 * #include "mbed.h"
 * #include "max32630fthr.h"
 * #include "ForwardErrCorr.h"
 * 
 * MAX32630FTHR pegasus(MAX32630FTHR::VIO_3V3);
 *
 * #define SymmetricKey "PaSsWoRd"
 * char  TransTable[] = {0x1F,0x18,0x06,0x01};
 *
 * //Create translator instance
 * Translator transTx(SymmetricKey, TransTable); //Constructor takes 7-bit slave adrs
 *
 * int main(void) 
 * {
 *     //use Encoder
 * }
 * @endcode
 */

#ifndef __FORWARDERRCORR_H_
#define __FORWARDERRCORR_H_

#include "mbed.h"

class Translator{
    
public:

    /**
    * @brief  Constructor using reference to Symmetric key for encryption and FEC translation table
    * @param SymmetricKey - Symmetric Key used by tranmsitter and reciever to encrypt/decrypt transmitted messages
    * @param TransTable - Translation table used for Forward Error Correction code
    */
    Translator(char *SymmetricKey, char *TransTable);

    /**
    * @brief  De-constructor
    */
    ~Translator(void);

    /**
    * @brief  Takes 2 byte data packet, converts to FEC 8 byte packet, and encrypts each byte
    * @param tempData[IN] - 2 byte data that needs to be prepared for transmission
    * @param EncryptedData[OUT] - Pointer to array where encrypted data will be stored, ready to send
    * @return 0 on success, 1 on failure
    */
    uint32_t Encrypt(uint16_t tempData,char *EncryptedData);
    
    /**
    * @brief  Takes 1 byte data packet, converts to FEC 8 byte packet, and encrypts each byte
    * @param tempData[IN] - 2 byte data that needs to be prepared for transmission
    * @param EncryptedData[OUT] - Pointer to array where encrypted data will be stroed, ready to send
    * @return 0 on success, 1 on failure
    */
    uint32_t Encrypt(char tempData, char*EncryptedData);

    /**
    * @brief  Calculates the hamming disatnce between 2 bytes
    * @param ChkVal[IN] -   Byte to use in hamming distance check
    * @param TableVal[IN] - Byte to use in hamming distance check
    * @return Hamming distance between the two provided bytes (range of 0-8)
    */
    int HamDist(char ChkVal, char TableVal);

    /**
    * @brief  Converts 1 byte of FEC code back to original 2 bit data
    * @param ChkVal - 1 byte of encoded data that needs to be decoded
    * @return 0,1,2,3 to indicate correctly decoded table index, 4 on transmission error
    */
    int ChkHam(char ChkVal);

    /**
    * @brief Function that takes encrypted FEC data and converts it back to 2 byte data
    * @param[IN] EncryptedData - Array of the encrypted data
    * @param[OUT] Output - 16 bit data returned after decryption
    * @return   0 on sucess, 1 on failure
    */
    uint32_t Decrypt(char *EncryptedData, uint16_t *Output);
    
private:
    
    char *m_TransTable, *m_SymmetricKey;
};

#endif /* __ForwardErrCorr_H_*/