/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************

This is code for the MAX1472 RF Transmitter EV-Kit. This program will record a 
temperature fom the MAX30208 EV-Kit, encrypt the data using the user-generated
Symmetric key, apply forward error correction (FEC) encoding, and then send the
message via the MAX1472 transmitter. When the MAX1472 is not sending data, it will
go into a power saving sleep mode.

  Hardware Setup and connections:

    MAX32630FTHR->  MAX1472 Ev-Kit
    
    3.3V        ->  VDD
    GND         ->  VSS
    P3_1        ->  Data_In
    P5_1        ->  ENABLE
    
*******************************************************************************
    MAX32630FTHR->  MAX30208 Ev-Kit
    
    1.8V        ->  VIN
    GND         ->  GND
    P3_4        ->  SDA
    P3_5        ->  SCL
    
*******************************************************************************
*/

#include "mbed.h"
#include "max32630fthr.h"
#include "mxc_config.h"
#include "lp.h"
#include "gpio.h"
#include "rtc.h"
#include "MAX14690.h"
#include "USBSerial.h"
#include "MAX30208.h"
#include "ForwardErrCorr.h"

/*User Defined Paramaters*/
#define SymmetricKey "RfIsCoOl"   //Set Private Key here -- Make sure it is identical to receiver
#define SendDelay   5   //Number of seconds between sending data
#define TransmissionLength 1000 //How long to continuosuly send the message (ms)

char DEVICETYPE = 'T';  //'T' for Temperature Sensor
char DEVICEID = 0x01;   //Set the Device ID

/*End of User Defined Variables */


MAX32630FTHR pegasus(MAX32630FTHR::VIO_3V3);
I2C i2c(P3_4, P3_5);  //sda,scl
RawSerial uart(P3_1,P3_0);//tx,rx
MAX30208 TempSensor(i2c, 0x50); //Constructor takes 7-bit slave adrs

char  TransTable[] = {0x1F,0x18,0x06,0x01}; //Used to translate data for FEC -- Make sure it is identical to receiver
Translator transTx(SymmetricKey, TransTable); //Initialize Encoder

DigitalOut rLED(LED1);
DigitalOut gLED(LED2);
DigitalOut bLED(LED3);

DigitalIn sw2(SW1);             //Used on start-up to stay in active mode for re-programming
DigitalOut TXEnable(P5_1);      //Used to Enable Transmitter

// *****************************************************************************
/*
* @brief  Configures the RTC to tick every second so it can be compared to during sleep intterupt
*/
void RTC_Setup()
{
    rtc_cfg_t RTCconfig;
    RTCconfig.prescaler = RTC_PRESCALE_DIV_2_12; //Set tick frequency to 1 second
    RTCconfig.prescalerMask = RTC_PRESCALE_DIV_2_12; //1 second frequency
    RTCconfig.snoozeCount = 0;
    RTCconfig.snoozeMode = RTC_SNOOZE_DISABLE;
    RTC_Init(&RTCconfig);
    RTC_Start();
}

//****************************************************************************
/**
* @brief  Record and read temperature from MAX30208
* @param TempSensor - Refrence to MAX30208 temp sensor object
* @param &value[OUT]- Address to store read temperature at
* @return   0 on success, 1 on failure due to improper data record or read 
*/
uint16_t ReadData(MAX30208 TempSensor, uint16_t &value){
    if (TempSensor.takeDataMeasurment() != 0){
        printf("Error Taking data Meaurment\r\n");
        return(1);
    }
    wait_ms(50);    //max temp read is 50ms
    if (TempSensor.readData(value) !=0){
        printf("Error reading temperature Data\r\n");    
        return(1);
    }
    return(0);
}

//*****************************************************************************
/*
* @brief  Begin Communication with warm up bytes and device type/id
*/
void comInit(){
    uart.putc(0xFF);
    uart.putc(0xFF);
    uart.putc(0x00);
    uart.putc('b');
    uart.putc(DEVICETYPE);
    uart.putc(DEVICEID);
}

//*****************************************************************************
/**
* @brief  Send data and end transmission
* @param EncryptedData[IN]  - 8 bytes of encryted data to send via UART connection
*/
void comData(char *EncryptedData){
    for(int i=0;i<8;i++){
        uart.putc(EncryptedData[i]);    //Send all of the encrypted data
    }
    uart.putc('c'); //End of packet character
    uart.send_break();
}

//*****************************************************************************

int main(void)
{
    //initialize PMIC
    MAX32630FTHR pegasus(MAX32630FTHR::VIO_3V3);
    uart.baud(9600);    //Set baud rate for 9600
    
    //check if starting at main because of LP0 wake-up
    if(LP_IsLP0WakeUp()) {
        //empty
    }
    
    else {
        //We did not wake up from sleep and this is first power-on
        //Only configure RTC the first time around 
        RTC_Setup();
    }
    
    //Set LEDs on to show device is awake
    gLED = LED_ON;
    rLED = LED_ON;
    bLED = LED_ON;
    
    //Main loop
    while(1) {
        
        //hold down switch 1 to prevent the microcontroller from going into LP0
        while(sw2 == 0);
        
        uint16_t tempData;
        if(ReadData(TempSensor,tempData) !=0){
                printf("Error reading data!\r\n");
            }
            
            //Encrypt Temperature Data
            char EncryptedData[8];
            transTx.Encrypt(tempData,EncryptedData); 

        TXEnable = 1;       //Turn on Transmitter
        wait_us(450);       //Minimum Wake-up time for transmitter per data sheet -- Do not lower
        
        //Start a timer for sending data
        Timer timer;
        timer.start();
        
        //Send the data for 1 sec
        while(timer.read_ms() < TransmissionLength) {
        
        //Send Data continuously
        comInit();
        comData(EncryptedData);
        }//while
        
        TXEnable = 0;   //Turn off Transmitter
        timer.stop();   //Stop send timer
        
        //Clear existing wake-up config
        LP_ClearWakeUpConfig();

        //Clear any event flags
        LP_ClearWakeUpFlags();

        //configure wake-up on RTC compare 0
        //LP_ConfigRTCWakeUp(enable compare0, enable compare1, set prescale, set rollover)
        LP_ConfigRTCWakeUp(1, 0, 0, 0);
        
        //disable unused PMIC rails to minimize power consumption
        pegasus.max14690.ldo2SetMode(MAX14690::LDO_DISABLED);
        pegasus.max14690.ldo3SetMode(MAX14690::LDO_DISABLED);
        
        //Turn off LEDs to show low power mode
        gLED = LED_OFF;
        rLED = LED_OFF;
        bLED = LED_OFF;
        
        //Reset the RTC back to 0 ticks
        RTC_SetCount(0);
        
        //set RTC to generate an interrupt every SendDelay seconds (User defined value at top of program)
        RTC_SetCompare(0,SendDelay);      //(Compare index, Compare Value in ticks)
        
        //clear comparison flag in the RTC registers
        RTC_ClearFlags(MXC_F_RTC_FLAGS_COMP0);

        //Enter Deep Sleep Mode
        LP_EnterLP0();

        //firmware will reset with no prior knowledge on wake-up
    }
}
