/*******************************************************************************
 * Copyright (C) 2018 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */
#ifndef SH_Max8614x_BareMetal_H_
#define SH_Max8614x_BareMetal_H_

#include <stdint.h>
#include <stdio.h>
#include "demoDefinitions.h"


#define COMM_SUCCESS        0
#define COMM_GENERAL_ERROR  -1
#define COMM_INVALID_PARAM  -254
#define COMM_NOT_RECOGNIZED -255

enum enAlgoMode{
	kAlgoModeHeartRate = 0,
	kAlgoModeSPO2      = 1,
	kAlgoModeContSPO2  = 2,
};

/**
 * @brief	Get the value of register at addr
 *
 * @param[in]	addr - adress of the register
 * @param[out]	val - value at the register

 */
void SH_Max8614x_get_reg(uint8_t addr, uint32_t *val);

/**
* @brief	Write a value to the register at address addr
*
* @param[in]	addr - adress of the register to write
* @param[in]	val - value of the register to write
*/
void SH_Max8614x_set_reg(uint8_t addr, uint32_t val);

/**
* @brief	Initialize Max86140 with default configuration
*
* @param[in] paramAlgoMode - the mode of the algorithm to initialize
*
* @return 0 on SUCCESS
*/
int SH_Max8614x_default_init(enum enAlgoMode paramAlgoMode);

/**
* @brief	Check the data stored in the Sensor Hub. Reads and prints
* 			the data if available
*
*/
int SH_Max8614x_data_report_execute(void);

/**
* @brief	Stops the active sensors and algorithms
*
*/
void SH_Max8614x_stop(void);




/*FOR THIS EXAMPLE DEMO PURPOSES*/
int Max8614x_Set_WSPO2Mode(int mode);

extern uint16_t HrmResult;
extern uint16_t SPO2Result;
extern uint8_t  HrmConfidence;
extern uint8_t  SPo2Confidence;

/* define to see debug messages*/
#define DEBUG_INFO


int SH_Max8614x_get_whrm_dataformat(const char *null_arg);
int SH_Max8614x_get_wspo2_dataformat(const char *null_arg);
int SH_Max8614x_set_ppgreg(const char *addr_value_args);
int SH_Max8614x_get_ppgreg(const char *addr_arg);
int SH_Max8614x_measure_whrm(const char *null_arg);
int SH_Max8614x_measure_wspo2(const char *null_arg);
int SH_Max8614x_self_test_ppg(const char *null_arg);
int SH_Max8614x_self_test_acc(const char *null_arg);


int SH_Max8614x_set_spo2_calibration(const char *calib_args);
int SH_Max8614x_get_spo2_calibration(const char *null_arg);
int SH_Max8614x_set_spo2_algomode(const char *mode_arg);
int SH_Max8614x_get_spo2_algomode(const char *null_arg);
int SH_Max8614x_set_spo2_samplerate(const char *srate_arg);
int SH_Max8614x_get_spo2_samplerate(const char *null_arg);
int SH_Max8614x_set_spo2_agcusage(const char *onoff_arg);
int SH_Max8614x_get_spo2_agcusage(const char *null_arg);
int SH_Max8614x_set_spo2_agctimeout(const char *timeout_arg);
int SH_Max8614x_get_spo2_agctimeout(const char *null_arg);
int SH_Max8614x_set_spo2_algotimeout(const char *timeout_arg);
int SH_Max8614x_get_spo2_algotimeout(const char *null_arg);
int SH_Max8614x_set_spo2_motionusage(const char *onoff_arg);
int SH_Max8614x_get_spo2_motionusage(const char *null_arg);
int SH_Max8614x_set_spo2_motionperiod(const char *period_arg);
int SH_Max8614x_get_spo2_motionperiod(const char *null_arg);
int SH_Max8614x_set_spo2_motionthresh(const char *thresh_arg);
int SH_Max8614x_get_spo2_motionthresh(const char *null_arg);
int SH_Max8614x_set_spo2_pdiodeconfig(const char *pdcfg_arg);
int SH_Max8614x_get_spo2_pdiodeconfig(const char *null_arg);

int SH_Max8614x_set_whrm_aecusage(const char *onoff_arg);
int SH_Max8614x_get_whrm_aecusage(const char *null_arg);
int SH_Max8614x_set_whrm_scdusage(const char *onoff_arg);
int SH_Max8614x_get_whrm_scdusage(const char *null_arg);
int SH_Max8614x_set_whrm_scdadjperiod(const char *period_arg);
int SH_Max8614x_get_whrm_scdadjperiod(const char *null_arg);
int SH_Max8614x_set_whrm_scddebouncewin(const char *dwindow_arg);
int SH_Max8614x_get_whrm_scddebouncewin(const char *null_arg);
int SH_Max8614x_set_whrm_motionthresh(const char *motion_arg);
int SH_Max8614x_get_whrm_motionthresh(const char *null_arg);
int SH_Max8614x_set_whrm_minpdiodecurr(const char *curr_arg);
int SH_Max8614x_get_whrm_minpdiodecurr(const char *null_arg);
int SH_Max8614x_set_whrm_pdiodeconfig(const char *cfg_arg);
int SH_Max8614x_get_whrm_pdiodeconfig(const char *null_arg);
int SH_Max8614x_send_raw(const char *raw_data);
int SH_Max8614x_stop_acquisition(const char *null_arg);


#define NUMCMDS8614X (43)
#define CONSOLE_STR_BUF_SZ  ((int)1024);


typedef int (*cmdExecFunc)( const char*); // typedef void (*cmdExecFunc)( const void*);

typedef struct {
	char const* cmdStr;
	cmdExecFunc execute;
	char const *help;
}cmd_interface_t;


static int command_help(const char *str);


const cmd_interface_t CMDTABLE8614x[] = {

		{  "get_format whrmdata"     		, SH_Max8614x_get_whrm_dataformat		,  "returns whrm algo sample format in terms of bit fileds"												},
		{  "get_format wspo2data"    		, SH_Max8614x_get_wspo2_dataformat 		,  "returns wspo2 algo sample format in terms of bit fileds"											},
		{  "enable_measurement whrm" 		, SH_Max8614x_measure_whrm 				,  "start whrm measurement with default settings"  														},
		{  "enable_measurement wspo2"		, SH_Max8614x_measure_wspo2 			,  "start wspo2 measurement with default settings" 														},
		{  "get_reg ppgsensor"       		, SH_Max8614x_get_ppgreg 				,  "get register value of 8614x sensor, usage:  get_reg ppgsensor rAddr(1byte)" 						},
		{  "set_reg ppgsensor"       		, SH_Max8614x_set_ppgreg 				,  "set register value of 8614x sensor, usage :  set_reg ppgsensor rAddr(1byte) rval(1byte)" 			},
		{  "self_test ppgsensor"     		, SH_Max8614x_self_test_ppg 			,  "starts self test of onboard ppg sensor" 															},
		{  "self_test accel"         		, SH_Max8614x_self_test_acc 			,  "starts self test of onboard accel sensor" 															},
		{  "set_cfg spo2 cal"        		, SH_Max8614x_set_spo2_calibration 		,  "sets spo2 calibration, usage: set_cfg spo2 cal 0xAAAAAAAA 0xBBBBBBBB 0xCCCCCCCC" 			        },
		{  "get_cfg spo2 cal"        		, SH_Max8614x_get_spo2_calibration 		,  "gets spo2 calibration" 																				},
		{  "set_cfg spo2 algomode"   		, SH_Max8614x_set_spo2_algomode 		,  "sets the spo2 algo mode to oneshot(0) or continious(1), usage: set_cfg spo2 algomode mode(1byte)" 	},
		{  "get_cfg spo2 algomode"			, SH_Max8614x_get_spo2_algomode 		,  "check if spo2 is oneshot/continious" 																},
	    {  "set_cfg spo2 samplerate" 		, SH_Max8614x_set_spo2_samplerate		,  "sets spo2 sample rate, uasge: set_cfg spo2 samplerate sampleRate(numeric value)" 					},
		{  "get_cfg spo2 samplerate" 		, SH_Max8614x_get_spo2_samplerate 		,  "gets spo2 sample rate" 																				},
		{  "set_cfg sop2 agc"		 		, SH_Max8614x_set_spo2_agcusage 		,  "on/off  automatic gain control in wspo, usage: set_cfg wsop2 agc X(1byte)  X: 0 off 1 on" 			},
		{  "get_cfg sop2 agc" 				, SH_Max8614x_get_spo2_agcusage 		,  "check if agc is on/off" 																			},
		{  "set_cfg spo2 agctimeout"  	    , SH_Max8614x_set_spo2_agctimeout 		,  "sets agc timeout for spo2, usage:  set_cfg spo2 agctimeout timeout(numeric)" 						},
		{  "get_cfg spo2 agctimeout"        , SH_Max8614x_get_spo2_agctimeout 		,  "gets agctimeout for spo2" 																			},
		{  "set_cfg spo2 timeout"  	        , SH_Max8614x_set_spo2_algotimeout 		,  "sets timeout for spo2 incase of no valid result, usage:  set_cfg spo2 timeout timeout(numeric)" 	},
		{  "get_cfg spo2 timeout"           , SH_Max8614x_get_spo2_algotimeout 		,  "gets valid result timeout for spo2" 																},
		{  "set_cfg spo2 motion"  	        , SH_Max8614x_set_spo2_motionusage 		,  "on/off motion data usage for spo2, usage: set_cfg spo2 motionenable X(1byte)  X: 0 off 1 on " 		},
		{  "get_cfg spo2 motion"  	        , SH_Max8614x_get_spo2_motionusage		,  "checks if spo2 uses motion data" 																	},
		{  "set_cfg spo2 motionperiod"  	, SH_Max8614x_set_spo2_motionperiod 	,  "sets motion period for spo2, usage:  set_cfg spo2 motionperiod period(numeric)" 					},
		{  "get_cfg spo2 motionperiod"      , SH_Max8614x_get_spo2_motionperiod		,  "gets motion period for spo2" 																		},
		{  "set_cfg spo2 motionthreshold"   , SH_Max8614x_set_spo2_motionthresh		,  "sets motion threshold for spo2, usage:  set_cfg spo2 motionthreshold thres(numeric)" 				},
		{  "get_cfg spo2 motionthreshold"  	, SH_Max8614x_get_spo2_motionthresh 	,  "gets motion threshold for spo2" 																	},
		{  "set_cfg spo2 pdconfig"          , SH_Max8614x_set_spo2_pdiodeconfig     ,  "sets spo2 photodiode config, usage set_cfg spo2 pdconfig X , X: 1-pd1, 2-pd2, 3-both, 4-avaraged" 	},
		{  "get_cfg spo2 pdconfig"          , SH_Max8614x_get_spo2_pdiodeconfig     ,  "gets spo2 photodiode config" },
		{  "set_cfg whrm aecenable"         , SH_Max8614x_set_whrm_aecusage       	,  "on/off whrm aec algo,  usage: set_cfg whrm aec X , X: 0 off 1 on" 									},
		{  "get_cfg whrm aecenable"         , SH_Max8614x_get_whrm_aecusage       	,  "check if aec algo for whrm is on/off" 																},
        {  "set_cfg whrm scdenable"         , SH_Max8614x_set_whrm_scdusage      	,  "on/off whrm skin contact detection,  usage: set_cfg whrm scd X , X: 0 off 1 on" 					},
		{  "get_cfg whrm scdenable"         , SH_Max8614x_get_whrm_scdusage       	,  "check if whrm skin contact detection is on/off" 													},
        {  "set_cfg whrm scdadjperiod"      , SH_Max8614x_set_whrm_scdadjperiod     ,  "sets photo diode adj peroiod for whrm scd algo, usage:  set_cfg whrm scdadjperiod period(numeric)" 	},
		{  "get_cfg whrm scdadjperiod"      , SH_Max8614x_get_whrm_scdadjperiod     ,  "gets photo diode adj peroiod for whrm scd algo" 													},
		{  "set_cfg whrm scddebouncewin"    , SH_Max8614x_set_whrm_scddebouncewin   ,  "sets debounce window for whrm scd algo, usage:  set_cfg whrm scddebouncewin window(numeric)" 		},
		{  "get_cfg whrm scddebouncewin"    , SH_Max8614x_get_whrm_scddebouncewin   ,  "gets debounce window for whrm scd algo" 															},
		{  "set_cfg whrm motionthreshold"   , SH_Max8614x_set_whrm_motionthresh 	,  "sets motion threshold for whrm, usage:  set_cfg whrm motionthreshold threshold(numeric)" 			},
		{  "get_cfg whrm motionthreshold"  	, SH_Max8614x_get_whrm_motionthresh 	,  "gets motion threshold for whrm" 																	},
		{  "set_cfg whrm minpdcurrent"      , SH_Max8614x_set_whrm_minpdiodecurr 	,  "sets photodiode current range for whrm, usage:  set_cfg whrm minpdcurrent current(numeric)" 		},
		{  "get_cfg whrm minpdcurrent"  	, SH_Max8614x_get_whrm_minpdiodecurr 	,  "gets photodiode current rangefor whrm" 																},
        {  "set_cfg whrm pdconfig"          , SH_Max8614x_set_whrm_pdiodeconfig 	,  "sets photdiode config for whrm, usage:  set_cfg whrmaecscd pdconfig configstring" 					},
		{  "get_cfg whrm pdconfig"          , SH_Max8614x_get_whrm_pdiodeconfig 	,  "gets photdiode config for whrm" 																	},
		{  "stop"                           , SH_Max8614x_stop_acquisition          ,  "stops raw&algorithm data acquisition within 8614x instance of hub"                            },
		{  "help"                           , command_help          		        ,  "commnand info" 	}
		//		{  "send_raw"                       , SH_Max8614x_send_raw 					,  "sends raw data to sensor hub, usage: send_raw datastring" 											},
};


static int command_help(const char *str){
	int cmdIdx = 0;

	SERIALOUT("AVAILABLE COMMANDS: \r\n\r\n");
	while( cmdIdx != NUMCMDS8614X){
		SERIALOUT(" %s : \r\n  %s \r\n\r\n", CMDTABLE8614x[cmdIdx].cmdStr , CMDTABLE8614x[cmdIdx].help );
        cmdIdx++;
	};

}


#endif



