/*******************************************************************************
 * Copyright (C) 2018 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */
#ifndef SH_Max8614x_BareMetal_H_
#define SH_Max8614x_BareMetal_H_

#include <stdint.h>
#include <stdio.h>
#include "demoDefinitions.h"


#define COMM_SUCCESS        0
#define COMM_GENERAL_ERROR  -1
#define COMM_INVALID_PARAM  -254
#define COMM_NOT_RECOGNIZED -255

enum enAlgoMode{
	kAlgoModeHeartRate = 0,
	reinitHeartRate    = 1,
};

/**
 * @brief	Get the value of register at addr
 *
 * @param[in]	addr - adress of the register
 * @param[out]	val - value at the register

 */
void SH_Max8614x_get_reg(uint8_t addr, uint32_t *val);

/**
* @brief	Write a value to the register at address addr
*
* @param[in]	addr - adress of the register to write
* @param[in]	val - value of the register to write
*/
void SH_Max8614x_set_reg(uint8_t addr, uint32_t val);

/**
* @brief	Initialize Max86140 with default configuration
*
* @param[in] paramAlgoMode - the mode of the algorithm to initialize
*
* @return 0 on SUCCESS
*/
int SH_Max8614x_default_init(enum enAlgoMode paramAlgoMode);

/**
* @brief	Check the data stored in the Sensor Hub. Reads and prints
* 			the data if available
*
*/
int SH_Max8614x_data_report_execute(void);

/**
* @brief	Stops the active sensors and algorithms
*
*/
void SH_Max8614x_stop(void);


extern uint16_t HrmResult;
extern uint8_t  HrmConfidence;


/* define to see debug messages*/
//#define DEBUG_INFO

/**
* @brief	gets the data format of reported whrm algo result sample , in form of struct fields with number of bits they are represented.
*
* @param[in] null_arg : NULL string, just to match the form of command table function pointer type
*
* @return 0
*/
int SH_Max8614x_get_whrm_dataformat(const char *null_arg);


/**
* @brief	sets the register value of ppg sensor (8614x is connected to ME11 SENSOR HUB for this demo but can be used for all sensors physically
*           connected to ME11 SENSOR HUB).
*
* @param[in] addr_value_args : byte string including command followed by register address in hex and register value in hex
*                              "set_reg ppgsensor 0xAB 0xCD" or "set_reg ppgsensor AB CD"
*
* @return 0x00 on success and prints on command console
*/
int SH_Max8614x_set_ppgreg(const char *addr_value_args);

/**
* @brief	gets the register value of ppg sensor (8614x is connected to ME11 SENSOR HUB for this demo but can be used for all sensors physically
*           connected to ME11 SENSOR HUB).
*
* @param[in] addr_arg: byte string including command followed by register address in hex
*                      "set_reg ppgsensor 0xAB" or "set_reg ppgsensor AB"
*
* @return 0x00 on success and prints register value on command console
*/
int SH_Max8614x_get_ppgreg(const char *addr_arg);

/**
* @brief	  initalizes and starts whrm measurement from ME11 SENSOR HUB.
*             It default initilizes datatype(to BOTH) input fifo lenght(to 5 , this is FIFO used by accel data from host)
*
* @param[in]  null_arg : NULL string, just to match the form of command table function pointer type
*
** @return 0x00 on success and prints status on command console
*/
int SH_Max8614x_measure_whrm(const char *null_arg);


/**
* @brief	  starts self test of ppg sensor
*
* @param[in]  null_arg : NULL string, just to match the form of command table function pointer type
*
** @return 0x00 on success and prints status on command console
*/
int SH_Max8614x_self_test_ppg(const char *null_arg);

/**
* @brief	  starts self test of accelerometer sensor physically connected to ME11 SENSOR HUB
*             NOTE:Not supported within this demo as Customer is assumed to feed accel data from HOST.
*
* @param[in]  null_arg : NULL string, just to match the form of command table function pointer type
*
** @return 0x00 on success and prints status on command console
*/
int SH_Max8614x_self_test_acc(const char *null_arg);

/**
* @brief	  sets the aec algo usage of SPO2 algorithm
*
* @param[in]  onoff_arg : byte string including command followed by mode value in HEX uo to 1 hexadecimal digits.
*                         "set_cfg whrm aecenable 0/1"  0:off 1: On
*
*@return  status info, 0x00 on success
*
**/
int SH_Max8614x_set_whrm_aecusage(const char *onoff_arg);

/**
* @brief	  gets the aec algo usage of SPO2 algorithm
*             prints aec algo usage usage on/off value commnad console. 0 : Off 1: On
*
* @param[in]  null_arg : NULL string, just to match the form of command table function pointer type
*
* @return status info, 0x00 on success.
**/
int SH_Max8614x_get_whrm_aecusage(const char *null_arg);

/**
* @brief	  sets the skin contact detection usage of SPO2 algorithm
*
* @param[in]  onoff_arg : byte string including command followed by mode value in HEX uo to 1 hexadecimal digits.
*                         "set_cfg whrm aecenable 0/1"  0:off 1: On
*
*@return  status info, 0x00 on success
*
**/
int SH_Max8614x_set_whrm_scdusage(const char *onoff_arg);

/**
* @brief	  gets the skin contact detection usage  of SPO2 algorithm
*             prints skin contact detection usage on/off value commnad console. 0 : Off 1: On
*
* @param[in]  null_arg : NULL string, just to match the form of command table function pointer type
*
* @return status info, 0x00 on success.
**/
int SH_Max8614x_get_whrm_scdusage(const char *null_arg);

/**
* @brief	  sets the adjustment period parameter for skin contact detection for SPO2 algorithm.
*
* @param[in]  period_arg : byte string including command followed by period value in HEX up to 4 hexadecimal digits.
*                          "set_cfg whrm scdadjperiod 0xABCD"
*
* @return status info, 0x00 on success.
**/
int SH_Max8614x_set_whrm_scdadjperiod(const char *period_arg);

/**
* @brief	  Gets the adjustment period parameter for skin contact detection for SPO2 algorithm.
*             prints SPo2 period value on command console
*
* @param[in]  null_arg : NULL string, just to match the form of command table function pointer type
*
* @return status info, 0x00 on success.
*
**/
int SH_Max8614x_get_whrm_scdadjperiod(const char *null_arg);

/**
* @brief	  sets the  debounce window parameter for skin contact detection for SPO2 algorithm.
*
* @param[in]  dwindow_arg : byte string including command followed by debounce window value in HEX up to 4 hexadecimal digits.
*                          "set_cfg whrm scddebouncewin 0xABCD"
*
* @return status info, 0x00 on success.
**/
int SH_Max8614x_set_whrm_scddebouncewin(const char *dwindow_arg);

/**
* @brief	  gets the  debounce window parameter for skin contact detection for SPO2 algorithm.
*
* @param[in]  null_arg : NULL string, just to match the form of command table function pointer type
*
* @return status info, 0x00 on success.
**/
int SH_Max8614x_get_whrm_scddebouncewin(const char *null_arg);


/**
* @brief	  sets the motion threshold parameter for WHRM algorithm.
*
* @param[in]  timeout_arg : byte string including command followed by motion threshold value in HEX up to 4 hexadecimal digits.
*                           "set_cfg whrm motionthreshold 0xABCD"
*
* @return status info, 0x00 on success.
*
**/
int SH_Max8614x_set_whrm_motionthresh(const char *motion_arg);

/**
* @brief	  Gets the motion threshold for WHRM algorithm. prints WHRM algorith motion threshold value on command console
*
* @param[in]  null_arg : NULL string, just to match the form of command table function pointer type
*
* @return status info, 0x00 on success.
*
**/
int SH_Max8614x_get_whrm_motionthresh(const char *null_arg);

/**
* @brief	  sets the minimum photodiode current parameter for WHRM algorithm.
*
* @param[in]  curr_arg : byte string including command followed by diode current value in HEX up to 4 hexadecimal digits.
*                           "set_cfg whrm motionthreshold 0xABCD"
*
* @return status info, 0x00 on success.
*
**/
int SH_Max8614x_set_whrm_minpdiodecurr(const char *curr_arg);

/**
* @brief	  Gets the minimum photodiode current parameter for WHRM algorithm. prints WHRM algorithm photodiode current parameter value on command console
*
* @param[in]  null_arg : NULL string, just to match the form of command table function pointer type
*
* @return status info, 0x00 on success.
*
**/
int SH_Max8614x_get_whrm_minpdiodecurr(const char *null_arg);

/**
* @brief	  sets the sensor photodiode configuration for WHRM algorithm.
*
* @param[in]  pdcfg_arg : byte string including command followed by photodiode configuration value in HEX up to 1 hexadecimal digits.
                          "set_cfg whrm pdconfig 0xA" or "set_cfg whrm pdconfig A"
*
* @return status info, 0x00 on success.
**/
int SH_Max8614x_set_whrm_pdiodeconfig(const char *cfg_arg);

/**
* @brief	  Gets the sensor photodiode configuration for WHRM algorithm. prints WHRM algorithm photodiode configuration  on command console
*
* @param[in]  null_arg : NULL string, just to match the form of command table function pointer type
*
* @return status info, 0x00 on success.
*
**/
int SH_Max8614x_get_whrm_pdiodeconfig(const char *null_arg);

/**
* @brief	  Sends raw byte stream to ME11 SENSOR HUB. For debug purposes mainly.
*
* @param[in]  raw_data : byte stream to be send
*
* @return status info, 0x00 on success.
*
**/
//int SH_Max8614x_send_raw(const char *raw_data);



/**
* @brief	  Stops sensor+algorithm data acquisition and operation for all algorithms of ME11 sensor HUB .
*             Mainly for use in this demo code!!
*
* @param[in]  null_arg : NULL string, just to match the form of command table function pointer type
*
* @return status info, 0x00 on success.
*
**/
int SH_Max8614x_stop_acquisition(const char *null_arg);


#if defined(RAW_DATA_ONLY)
	int SH_Max8614x_disable_whrm(const char *onoff_arg);
	int SH_Max8614x_enable_pureraw(const char *onoff_arg);
    #define NUMCMDS8614X (24)
#else
	#define NUMCMDS8614X (22)
#endif

/*Number of commands accosicated with 8614x sesnor command table*/

#define CONSOLE_STR_BUF_SZ  ((int)1024);


typedef int (*cmdExecFunc)( const char*); // typedef void (*cmdExecFunc)( const void*);

typedef struct {
	char const* cmdStr;
	cmdExecFunc execute;
	char const *help;
}cmd_interface_t;


static int command_help(const char *str);


const cmd_interface_t CMDTABLE8614x[] = {

		{  "get_format whrmdata"     		, SH_Max8614x_get_whrm_dataformat		,  "returns whrm algo sample format in terms of bit fileds"												},
		{  "enable_measurement whrm" 		, SH_Max8614x_measure_whrm 				,  "start whrm measurement with default settings"  														},
		{  "get_reg ppgsensor"       		, SH_Max8614x_get_ppgreg 				,  "get register value of 8614x sensor, usage:  get_reg ppgsensor rAddr(1byte)" 						},
		{  "set_reg ppgsensor"       		, SH_Max8614x_set_ppgreg 				,  "set register value of 8614x sensor, usage :  set_reg ppgsensor rAddr(1byte) rval(1byte)" 			},
		{  "self_test ppgsensor"     		, SH_Max8614x_self_test_ppg 			,  "starts self test of onboard ppg sensor" 															},
		{  "self_test accel"         		, SH_Max8614x_self_test_acc 			,  "starts self test of onboard accel sensor" 															},
		{  "set_cfg whrm aecenable"         , SH_Max8614x_set_whrm_aecusage       	,  "on/off whrm aec algo,  usage: set_cfg whrm aec X , X: 0 off 1 on" 									},
		{  "get_cfg whrm aecenable"         , SH_Max8614x_get_whrm_aecusage       	,  "check if aec algo for whrm is on/off" 																},
        {  "set_cfg whrm scdenable"         , SH_Max8614x_set_whrm_scdusage      	,  "on/off whrm skin contact detection,  usage: set_cfg whrm scd X , X: 0 off 1 on" 					},
		{  "get_cfg whrm scdenable"         , SH_Max8614x_get_whrm_scdusage       	,  "check if whrm skin contact detection is on/off" 													},
        {  "set_cfg whrm scdadjperiod"      , SH_Max8614x_set_whrm_scdadjperiod     ,  "sets photo diode adj peroiod for whrm scd algo, usage:  set_cfg whrm scdadjperiod period(numeric)" 	},
		{  "get_cfg whrm scdadjperiod"      , SH_Max8614x_get_whrm_scdadjperiod     ,  "gets photo diode adj peroiod for whrm scd algo" 													},
		{  "set_cfg whrm scddebouncewin"    , SH_Max8614x_set_whrm_scddebouncewin   ,  "sets debounce window for whrm scd algo, usage:  set_cfg whrm scddebouncewin window(numeric)" 		},
		{  "get_cfg whrm scddebouncewin"    , SH_Max8614x_get_whrm_scddebouncewin   ,  "gets debounce window for whrm scd algo" 															},
		{  "set_cfg whrm motionthreshold"   , SH_Max8614x_set_whrm_motionthresh 	,  "sets motion threshold for whrm, usage:  set_cfg whrm motionthreshold threshold(numeric)" 			},
		{  "get_cfg whrm motionthreshold"  	, SH_Max8614x_get_whrm_motionthresh 	,  "gets motion threshold for whrm" 																	},
		{  "set_cfg whrm minpdcurrent"      , SH_Max8614x_set_whrm_minpdiodecurr 	,  "sets photodiode current range for whrm, usage:  set_cfg whrm minpdcurrent current(numeric)" 		},
		{  "get_cfg whrm minpdcurrent"  	, SH_Max8614x_get_whrm_minpdiodecurr 	,  "gets photodiode current rangefor whrm" 																},
        {  "set_cfg whrm pdconfig"          , SH_Max8614x_set_whrm_pdiodeconfig 	,  "sets photdiode config for whrm, usage:  set_cfg whrmaecscd pdconfig configstring" 					},
		{  "get_cfg whrm pdconfig"          , SH_Max8614x_get_whrm_pdiodeconfig 	,  "gets photdiode config for whrm" 																	},
		{  "stop"                           , SH_Max8614x_stop_acquisition          ,  "stops raw&algorithm data acquisition within 8614x instance of hub"                            		},
		{  "help"                           , command_help          		        ,  "commnand info" 																						},
#if defined(RAW_DATA_ONLY)
		{  "disable_whrm"                  , SH_Max8614x_disable_whrm               ,  "disable algorithms" 																				},
		{  "enable_raw_only"               , SH_Max8614x_enable_pureraw             ,  "enable just raw ppg data" 																				},
#endif
		//		{  "send_raw"                       , SH_Max8614x_send_raw 					,  "sends raw data to sensor hub, usage: send_raw datastring" 									},
};


static int command_help(const char *str){
	int cmdIdx = 0;

	SERIALOUT("AVAILABLE COMMANDS: \r\n\r\n");
	while( cmdIdx != NUMCMDS8614X){
		SERIALOUT(" %s : \r\n  %s \r\n\r\n", CMDTABLE8614x[cmdIdx].cmdStr , CMDTABLE8614x[cmdIdx].help );
        cmdIdx++;
	};

}


#endif



