/*******************************************************************************
 * Copyright (C) 2018 Maxim Integrated Products, Inc., All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Maxim Integrated
 * Products, Inc. shall not be used except as stated in the Maxim Integrated
 * Products, Inc. Branding Policy.
 *
 * The mere transfer of this software does not imply any licenses
 * of trade secrets, proprietary technology, copyrights, patents,
 * trademarks, maskwork rights, or any other form of intellectual
 * property whatsoever. Maxim Integrated Products, Inc. retains all
 * ownership rights.
 *******************************************************************************
 */


/**********************************************************************************
 *
 *  Desc: Example Code to get algorithm estimation results of Heart rate( HRM) from sensor hub and display it on screen.
 *        Example starts by monitoring heart rate and reinits WHRM algorithm via a button press (switch button)
 *
 *        Example,
 *
 *        1. Initializes user interface
 *                   initialize display screen and switch button for algo selection.
 *
 *        2. Initializes underlying hardware port for sensor hub communication:
 *                   setup i2c comm. inits reset pin and mfio event pin and connects interrupt to mfio pin.
 *
 *        3. When switched to a minitoring mode
 *                   disables previous algorithm, clears mfio event
 *                   calls default init function for the sensor whose data is used by enabled algorithms. Algorithms are
 *                   registered under sesnor instance for this example. Fcunction
 *                   	        1. initialize algorithm config struct enabled
 *                              2. enable data type to both raw sensor and algorithm data
 *                              3. get input fifo size to learn fifo capacity
 *                              4. set fifo threshold for mfio event frequency
 *                              5. reaenable sensor to acquire ppg data
 *                              6. enable accompanying accel sensor
 *                              7. enable switched algorithm
 *
 *     4. Sensor Hub now starts to write raw sensor/algorithm data to its data report FIFO which
 *        reports mfio event when data size determined by fifo threshold is written to report fifo.
 *
 *     5. Example calls SH_Max8614x_data_report_execute() which
 *                 1. calls SH API's sh_ss_execute_once() function which:
 *                            writes sensor hub's report fifo content (sensor/algorithm data samples) to a buffer(1).
 *                 2. calls CSTMR_SH_FeedAccDataIntoSH() to send accelerometer data to sensor hub which os required for heart rate
 *                 3. Parses buffer(1) data to extract numeric sensor and algorithm samples according to enabled algorithms.
 *                    look: whrm_data_rx() , max8614x_data_rx() and sample structs whrm_mode1_data and max8614x_mode1_data
 *
 *     6. numeric values  are written to HrmResult  within MAX8614x  whrm_data_rx() ... and included as extern in main.cpp
 *
 *     7. Example calls demoUI_display_algo_estimations() to display result on watch screen
 *
 *
 ***********************************************************************************/

#include <events/mbed_events.h>
#include <mbed.h>
#include "max32630hsp.h"
#include "SHComm.h"
#include "SH_Max8614x_BareMetal.h"
#include "bmi160.h"
#include "cmdInterface.h"
#include "demoUI.h"
#include "demoDefinitions.h"

extern uint16_t HrmResult;
extern uint8_t  HrmConfidence;

DigitalOut debugled(LED1, 1);
// Hardware serial port over DAPLink
Serial daplink(USBTX, USBRX, 115200);

#include "USBSerial.h"
USBSerial microUSB(0x1f00, 0x2012, 0x0001, false);



// ICARUS Board initialization
InterruptIn interruptIn_PowerButton(P7_6);
MAX32630HSP icarus(MAX32630HSP::VIO_1V8, &interruptIn_PowerButton);

#define WAIT_SENSORHUB_STABLE_BOOTUP_MS  ((uint32_t)2000)


static bool isWhrmInitialized     = false;

int main() {

	wait_ms(WAIT_SENSORHUB_STABLE_BOOTUP_MS);

	int hostMode = HOSTMODEAPPLICATION;

	demoUI_init();

	sh_init_hwcomm_interface();
	sh_disable_irq_mfioevent();
	sh_clear_mfio_event_flag();
	sh_enable_irq_mfioevent();
	int i = 0;

	int displayMode;
	uint16_t resultToDisplay;
	uint8_t  confidenceToDisplay;

	while(1) {

		char ch;
		while ( SERIAL_AVAILABLE()) {

			ch = SERIALIN();
			cmdIntf_build_command(ch);
		}

        hostMode  = get_internal_operating_mode();
		if( hostMode  == HOSTMODEAPPLICATION) {

				displayMode = demoUI_display_get_mode();

				if( displayMode == kAlgoModeHeartRate ){
					 if( !isWhrmInitialized){
#if defined(DEBUG_INFO)
						 SERIALOUT(" WHRM inititalized \r\n");
#endif
						 SH_Max8614x_stop();
						 SH_Max8614x_default_init(kAlgoModeHeartRate);
						 isWhrmInitialized = true;
						 wait_ms(100); /* for display screen*/
					 }
					 resultToDisplay     = HrmResult;
					 confidenceToDisplay = HrmConfidence;

				}else if( displayMode == reinitHeartRate ){
					isWhrmInitialized = false;
              	}

				int cumSampleCNt = SH_Max8614x_data_report_execute();

				if(cumSampleCNt){ /* If data samples ara avaliable display on screen*/
#if defined(DEBUG_INFO)
					//SERIALOUT("estimate: =%d conf: %d dispMode: %d \r\n", resultToDisplay , confidenceToDisplay, displayMode);
#endif					if( displayMode == kAlgoModeHeartRate)
						     demoUI_display_algo_estimations(resultToDisplay, -1);

				}


		}else {

				demoUI_display_bootldr_screen();
               //wait_ms(10);
		}

	}


}

