/*******************************************************************************
* @file DS4424.h
* Copyright (C) 2018 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************
*/
 #ifndef DS4424_H
#define DS4424_H
 
#include "mbed.h"

#define DS4424_NO_ERROR   0
#define DS4424_ERROR      -1

#define DS4424_MAX_DAC_CHANNELS 4
#define DS4422_MAX_DAC_CHANNELS 2

#define DS4424_DAC_ADDR(chan)   ((chan) + 0xf8)
#define DS4424_SOURCE_I     1
#define DS4424_SINK_I       0

/**
 * @brief Four/two Channel I2C Sink/Source Current DAC
 * @version 1.0000.3
 *
 * @details The DS4424/DS422 contains four/two I2C
 * programmable current DACs. This driver enables
 * the writing of source/since values to the channel
 * registers.
 *
 * @code 
 * #include "mbed.h"
 * #include "DS4424.h"
 * #include "max32630fthr.h"
 * MAX32630FTHR pegasus(MAX32630FTHR::VIO_3V3);
 * I2C i2cBus(P3_4, P3_5);
 * DS4424 ds4424_dac(i2cBus, DS4424::DS4424_I2C_SLAVE_ADRS0, DS4424::DS4424_IC);
    
 * int main()
 * {
 *    int32_t value[4]={0,0,0,0};
 *    int ret;
 *    
 *    i2cBus.frequency(400000);
 *
 *    // read channel 0 register
 *    ret = ds4424_dac.read_raw(value[0], DS4424::REG_OUT0); 
 *
 *      //... rest of application
 * }
 * @endcode
 */

class DS4424
{
    public:
    

    /**
     * @brief IC's supported with this driver
     * @details DS4424, DS4422
     */ 
    typedef enum
    {
        DS4424_IC = 0,
        DS4422_IC
    }DS4424_ic_t;
    
    
    /**
     * @brief Valid 7-bit I2C addresses 
     * @details The 8 valid I2C addresses set via A0, and A1 pins
     * of the DS4424
     */ 
    typedef enum
    {
        DS4424_I2C_SLAVE_ADRS0 = (int32_t)(0x20 >> 1), // code uses the 7 bit address
        DS4424_I2C_SLAVE_ADRS1 = (int32_t)(0x60 >> 1),
        DS4424_I2C_SLAVE_ADRS2 = (int32_t)(0xA0 >> 1),
        DS4424_I2C_SLAVE_ADRS3 = (int32_t)(0xe0 >> 1) 
    }DS4424_i2c_adrs_t;

    
    /**
     * @brief   Register Addresses
     * @details Enumerated MAX20303 register addresses
     */
    enum ChannelRegAddr_e {
        REG_OUT0            = 0xF8,  // channel 0
        REG_OUT1            = 0xF9,
        REG_OUT2            = 0xFA,
        REG_OUT3            = 0xFB
    };

    
    /**********************************************************//**
     * @brief Constructor for DS4424 Class.  
     * 
     * @details Allows user to use existing I2C object
     *
     * On Entry:
     *     @param[in] i2c_bus - pointer to existing I2C object
     *     @param[in] i2c_adrs - 7-bit slave address of DS4424
     *     @param[in] ic_variant - which type of DS44xx is used
     *
     * On Exit:
     *
     * @return None
     **************************************************************/
    DS4424(I2C &i2c_bus, DS4424_i2c_adrs_t slaveAddress, DS4424_ic_t ic_variant);
    
    
 
    /**********************************************************//**
     * @brief Default destructor for DS4424 Class.  
     *
     * @details Destroys I2C object if owner 
     *
     * On Entry:
     *
     * On Exit:
     *
     * @return None
     **************************************************************/
    ~DS4424();
    
    /** 
     * @brief Read raw value of a channel. Negative values are sink; Positive values are source.
     * @param result - Value that is read from the register
     * @param chan_addr - Channel 0 to 3
     * @return 0 on success, negative number on failure
     */
    int read_raw(int32_t &result, ChannelRegAddr_e chan_addr);
    
    /** 
     * @brief Write raw value to a channel.
     * @param value - Value that is stored to the register
     * @param chan_addr - Channel 0 to 3
     * @return 0 on success, negative number on failure
     */
    int write_raw(int32_t value, ChannelRegAddr_e chan_addr);

protected:

    /**
     * @brief  Read register of device at slave address
     * @param reg - Register address
     * @param[out] value - Read data on success
     * @return 0 on success, negative number on failure
     */
    int read_register(ChannelRegAddr_e reg, uint8_t &value);

    /** 
     * @brief Write register of device at slave address
     * @param reg - Register address
     * @param value - Value to write
     * @return 0 on success, negative number on failure
     */
    int write_register(ChannelRegAddr_e reg, uint8_t value);
 
private:
    /** I2C object */
    I2C &m_i2c;
    /** Device slave addresses */
    uint8_t m_writeAddress, m_readAddress;
    /** Address of the Maximum Register for this device */
    ChannelRegAddr_e m_max_ch_reg_addr;
};
 
#endif/* DS4424_H */