/******************************************************************//**
* @file max14661.h
*
* @author Justin Jordan
*
* @version 1.0
*
* Started: 11NOV14
*
* Updated: 
*
* @brief Header file for MAX14661 class
*
***********************************************************************
*
* @copyright 
* Copyright (C) 2015 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/


#ifndef MAX14661_H
#define MAX14661_H


#include "mbed.h"


/******************************************************************//**
* Max14661 Class
**********************************************************************/
class Max14661 : public I2C
{
    uint8_t w_adrs, r_adrs;
    
    public:
    
        /**
        * max14661_i2c_adrs_t - enumerated MAX14661 I2C Addresses
        */
        typedef enum
        {
            MAX14661_I2C_ADRS0 = 0x4C,
            MAX14661_I2C_ADRS1,
            MAX14661_I2C_ADRS2,
            MAX14661_I2C_ADRS3
        }max14661_i2c_adrs_t;
        
        
        /**
        * max14661_regs_t - enumerated MAX14661 register addresses
        */
        typedef enum
        {
            DIR0,         //Switches 8A–1A direct read/write access
            DIR1,         //Switches 16A–9A direct read/write access
            DIR2,         //Switches 8B–1B direct read/write access
            DIR3,         //Switches 16B–9B direct read/write access
            SHDW0 = 0x10, //Switches 8A–1A shadow read/write access
            SHDW1,        //Switches 16A–9A shadow read/write access
            SHDW2,        //Switches 8B–1B shadow read/write access
            SHDW3,        //Switches 16B–9B shadow read/write access
            CMD_A,        //Set mux A command (reads 0x00)
            CMD_B         //Set mux B command (reads 0x00)
        }max14661_regs_t;
        
        
        /**
        * max14661_cmds_t - enumerated MAX14661 commands
        */
        typedef enum
        {
            ENABLE_SW01,     //enables sw1 on bank only
            ENABLE_SW02,     //enables sw2 on bank only
            ENABLE_SW03,     //enables sw3 on bank only
            ENABLE_SW04,     //enables sw4 on bank only
            ENABLE_SW05,     //enables sw5 on bank only
            ENABLE_SW06,     //enables sw6 on bank only
            ENABLE_SW07,     //enables sw7 on bank only
            ENABLE_SW08,     //enables sw8 on bank only
            ENABLE_SW09,     //enables sw9 on bank only
            ENABLE_SW10,     //enables sw10 on bank only
            ENABLE_SW11,     //enables sw11 on bank only
            ENABLE_SW12,     //enables sw12 on bank only
            ENABLE_SW13,     //enables sw13 on bank only
            ENABLE_SW14,     //enables sw14 on bank only
            ENABLE_SW15,     //enables sw15 on bank only
            ENABLE_SW16,     //enables sw16 on bank only
            DISABLE_BANK,    //opens all switches on bank
            COPY_SHADOW,     //copies both shadow registers for bank
            NO_CHANGE,
        }max14661_cmds_t;
        
        
        /**
        * max14661_sw_t - enumerated MAX14661 switch bitmasks
        */
        typedef enum
        {
            SW01 = (1 << 0),
            SW02 = (1 << 1),
            SW03 = (1 << 2),
            SW04 = (1 << 3),
            SW05 = (1 << 4),
            SW06 = (1 << 5),
            SW07 = (1 << 6),
            SW08 = (1 << 7),
            SW09 = (1 << 8),
            SW10 = (1 << 9),
            SW11 = (1 << 10),
            SW12 = (1 << 11),
            SW13 = (1 << 12),
            SW14 = (1 << 13),
            SW15 = (1 << 14),
            SW16 = (1 << 15)
        }max14661_sw_t;
        
        
        /**********************************************************//**
        * Constructor for Max14661 Class
        *
        * On Entry:
        *     @param[in] sda - sda pin of I2C bus
        *     @param[in] scl - scl pin of I2C bus
        *     @param[in] i2c_adrs - 7-bit slave address of MAX14661
        *
        * On Exit:
        *    @return none
        *
        * Example:
        * @code
        * 
        * //declare mux object
        * Max14661 mux(D14, D15, MAX14661_I2C_ADRS0); 
        *
        * @endcode
        **************************************************************/
        Max14661(PinName sda, PinName scl, max14661_i2c_adrs_t i2c_adrs);
        

        /******************************************************************//**
        * Writes given commands to CMD_A and CMD_B
        *
        * On Entry:
        *     @param[in] cmdA - command for CMD_A
        *     @param[in] cmdB - command for CMD_B
        *
        * On Exit:
        *     @return return value = 0 on success, non-0 on failure
        *
        * Example:
        * @code
        *
        * //declare mux object
        * Max14661 mux(D14, D15, MAX14661_I2C_ADRS0); 
        *
        * uint16_t rtn_val;  
        *
        * rtn_val = mux.wrt_cmd_registers(DISABLE_BANK, DISABLE_BANK);
        * 
        * @endcode
        **********************************************************************/
        uint16_t wrt_cmd_registers(max14661_cmds_t cmdA, max14661_cmds_t cmdB);


        /******************************************************************//**
        * Writes bankA and bankB to coresponding shadow registers
        *
        * On Entry:
        *     @param[in] bankA - binary representation of switch states
        *     @param[in] bankB - binary representation of switch states
        *
        * On Exit:
        *     @return return value = 0 on success, non-0 on failure
        *
        * Example:
        * @code
        *
        * //declare mux object
        * Max14661 mux(D14, D15, MAX14661_I2C_ADRS0); 
        *
        * uint16_t bankA = (SW12 | SW02); //example only
        * uint16_t bankB = (SW11 | SW01);
        * uint16_t rtn_val;  
        *  
        * //wite shadow registers
        * rtn_val = mux.wrt_shadow_registers(bankA, bankB);
        *
        * @endcode
        **********************************************************************/
        uint16_t wrt_shadow_registers(uint16_t bankA, uint16_t bankB);


        /******************************************************************//**
        * Writes bankA and bankB to coresponding direct access registers 
        *
        * On Entry:
        *     @param[in] bankA - binary representation of switch states
        *     @param[in] bankB - binary representation of switch states
        *
        * On Exit:
        *     @return return value = 0 on success, non-0 on failure
        *
        * Example:
        * @code  
        *
        * //declare mux object
        * Max14661 mux(D14, D15, MAX14661_I2C_ADRS0);   
        *
        * uint16_t bankA = (SW12 | SW02); //example only
        * uint16_t bankB = (SW11 | SW01);
        * uint16_t rtn_val;  
        *  
        * //wite shadow registers
        * rtn_val = mux.wrt_dir_registers(bankA, bankB);
        *
        * @endcode
        **********************************************************************/
        uint16_t wrt_dir_registers(uint16_t bankA, uint16_t bankB);


        /******************************************************************//**
        * Writes bankA and bankB to coresponding shadow register and then 
        * issues copy command for both banks
        *
        * On Entry:
        *     @param[in] bankA - binary representation of switch states
        *     @param[in] bankB - binary representation of switch states
        *
        * On Exit:
        *     @return return value = 0 on success, non-0 on failure
        *
        * Example:
        * @code  
        *
        * //declare mux object
        * Max14661 mux(D14, D15, MAX14661_I2C_ADRS0);   
        *
        * uint16_t bankA = (SW12 | SW02); //example only
        * uint16_t bankB = (SW11 | SW01);
        * uint16_t rtn_val;  
        *  
        * //wite shadow registers
        * rtn_val = mux.set_switches(bankA, bankB);
        *
        * @endcode
        **********************************************************************/
        uint16_t set_switches(uint16_t bankA, uint16_t bankB);


        /**********************************************************//**
        * Reads data from direct access registers starting at DIR0 and 
        * stores it in byte array pointed at by 'data'
        *
        * On Entry:
        *     @param[in] data - pointer to byte array for storing data
        *
        * On Exit:
        *     @param[out] data - data buffer now contains data read 
        *                        from dir registers
        *     @return return value = 0 on success, non-0 on failure
        *
        * Example:
        * @code
        *
        * //declare mux object
        * Max14661 mux(D14, D15, MAX14661_I2C_ADRS0); 
        *
        * uint16_t rtn_val;  
        * uint8_t data[4];
        * 
        * //read direct access registers
        * rtn_val = mux.rd_dir_registers(data);
        *
        * @endcode
        **********************************************************************/
        uint16_t rd_dir_registers(uint8_t* data);


        /**********************************************************//**
        * Reads data from shadow registers starting at SHDW0 and stores 
        * it in byte array pointed at by 'data'
        *
        * On Entry:
        *     @param[in] data - pointer to byte array for storing data
        *
        * On Exit:
        *     @param[out] data - data buffer now contains data read 
        *                        from shadow registers
        *     @return return value = 0 on success, non-0 on failure
        *
        * Example:
        * @code
        *
        * //declare mux object
        * Max14661 mux(D14, D15, MAX14661_I2C_ADRS0); 
        *
        * uint16_t rtn_val;  
        * uint8_t data[4];
        * 
        * //read shadow registers
        * rtn_val = mux.rd_shadow_registers(data);
        *
        * @endcode
        **************************************************************/      
        uint16_t rd_shadow_registers(uint8_t* data);  
};
                                  
                                          
#endif /* MAX14661_H*/
