/**********************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/


#include "mbed.h"
#include "OneWire.h"
#include "Terminal.h"

using namespace OneWire;
using namespace RomCommands;

//Uncomment the master for your setup

//#define I2C_TO_OW_MASTER
//#define SERIAL_TO_OW_MASTER
#define BIT_BANG_MASTER

void print_rom_id(Terminal & term, const RomId & romId);
void DS18B20Menu(Terminal & term, const RomId & romId, MultidropRomIterator & selector);

int main(void)
{
    Terminal term(USBTX, USBRX);
    char user_entry = 'y';
    
    #if defined(I2C_TO_OW_MASTER)
        //Instantiates a 1-wire master using the given I2C bus SDA/SCL pins
        //Requires a DS248X series of masters on I2C bus
        //https://para.maximintegrated.com/en/results.mvp?fam=1wire&tree=master
        
        I2C i2cBus(D14, D15);
        DS2484 owm(i2cBus);
        term.printf("\nStarting I2C to 1-wire Demo\n");
        
    #elif defined(SERIAL_TO_OW_MASTER)
        //Instantiates a 1-wire master using the given UART tx/rx pins
        //Requires a DS2480B serial to 1-wire master
        //https://www.maximintegrated.com/en/products/interface/controllers-expanders/DS2480B.html
        
        Serial ow_uart(D1, D0);
        Ds2480b owm(ow_uart);
        term.printf("\nStarting Serial to 1-wire Demo\n");
        
    #elif defined(BIT_BANG_MASTER)
        //Instantiates a 1-wire master on the given GPIO pin
        //Requires MAX32600MBED platform
        //https://www.maximintegrated.com/en/products/digital/microcontrollers/MAX32600MBED.html
        
        OwGpio owm(D2);
        term.printf("\nStarting Bit-Bang 1-wire Demo\n");
        
    #else
        #error One Wire Master Not Defined  
    #endif
    
    OneWireMaster::CmdResult result = owm.OWInitMaster();
    
    if(result == OneWireMaster::Success)
    {
        MultidropRomIterator selector(owm);
        
        //Search state var for main loop
        SearchState searchState;
        
        while(user_entry != 'n')
        {
            result = owm.OWReset();
            if(result == OneWireMaster::Success)
            {
                term.printf("\nOWReset success, starting search\n");
                
                result = OWFirst(owm, searchState);
                if(result == OneWireMaster::Success)
                {
                    do
                    {
                        //print current devices rom id
                        print_rom_id(term, searchState.romId);
                        
                        
                        //This could be a switch statement based on familyCodes 
                        //if your app has more than one device on the bus
                        if(searchState.romId.familyCode() == 0x28)
                        {
                            DS18B20Menu(term, searchState.romId, selector);
                        }
                        
                        //find the next device if any
                        result = OWNext(owm, searchState);
                    }
                    while(result == OneWireMaster::Success);
                }
                else
                {
                    term.printf("\nSearch failed\n");
                    term.printf("\nError code = %d\n", result);
                }
            }
            else 
            {
                term.printf("\nFailed to find any 1-wire devices on bus\n");
                term.printf("\nError code = %d\n", result);
            }
            
            user_entry = term.get_char("\nEnter 'y' to continue, or 'n' to quit: ", 'n', 'y');
            term.cls();
            term.home();
        }
    }
    else
    {
        term.printf("\nFailed to initialize the 1-wire Master\n");
        term.printf("\nError code = %d\n", result);
    }
    
    term.printf("\nEnding Program\n");
    
    return 0;
}
//***** end main ******************************************************


//*********************************************************************
void print_rom_id(Terminal & term, const RomId & romId)
{
    //print the rom number
    term.printf("\n");
    int8_t idx = 8;
    do
    {
        term.printf("0x%02x ", romId.buffer[idx--]);
    }
    while(idx >= 0);
    term.printf("\n");
}


//*********************************************************************
void DS18B20Menu(Terminal & term, const RomId & romId, MultidropRomIterator & selector)
{
    char userEntry = '0';
    uint8_t th, tl, idx, intResolution;
    DS18B20::Resolution resolution = DS18B20::TwelveBit;
    bool localPower = false;
    uint8_t scratchPadBuff[9];
    float temperature;
    
    DS18B20 tempSensor(selector);
    OneWireSlave::CmdResult result = OneWireSlave::OperationFailure;
    tempSensor.setRomId(romId);
    
    while(userEntry != '8')
    {
        term.printf("\nDS18B20 Menu Options\n");
        term.printf("\n%t1. Write ScratchPad");
        term.printf("\n%t2. Read Scratchpad");
        term.printf("\n%t3. Copy Scratchpad");
        term.printf("\n%t4. Convert Temperature");
        term.printf("\n%t5. Recall EEPROM");
        term.printf("\n%t6. Read Power Supply");
        term.printf("\n%t7. Clear Screen");
        term.printf("\n%t8. Continuously Convert Temperature");
        term.printf("\n%t9. Quit");
        
        userEntry = term.get_char("\nPlease select an option above: ", '1', '9');
        
        switch(userEntry)
        {
            case '1':
                th = term.get_int32("\nPlease enter upper temperature threshold, integer values only: ", 0, 100);
                tl = term.get_int32("\nPlease enter lower temperature threshold, integer values only: ", -55, 0);
                intResolution = term.get_int32("\nPlease enter desired reolution (9 to 12): ", 9, 12);
                
                switch(intResolution)
                {
                    case 9:
                        resolution = DS18B20::NineBit;
                    case 10:
                        resolution = DS18B20::TenBit;
                    case 11:
                        resolution = DS18B20::ElevenBit;
                    case 12:
                    default:
                        resolution = DS18B20::TwelveBit;
                    break;
                }
                
                result = tempSensor.writeScratchPad(th, tl, resolution);
                if(result == OneWireSlave::Success)
                {
                    term.printf("\nWrite Scratchpad Success\n");
                }
                else
                {
                    term.printf("\nWrite Scratchpad Fail\n");
                }
                
            break;
            
            case '2':
            
                result = tempSensor.readScratchPad(scratchPadBuff);
                if(result == OneWireSlave::Success)
                {
                    term.printf("\nRead Scratchpad Success\n");
                    term.printf("\nScratchpad = ");
                    for(idx = 0; idx < 8; idx++)
                    {
                        if(scratchPadBuff[idx] < 16)
                        {
                            term.printf("0x0%x ", scratchPadBuff[idx]);
                        }
                        else
                        {
                            term.printf("0x%2x ", scratchPadBuff[idx]);
                        }
                    }
                    term.printf("\n");
                    
                }
                else
                {
                    term.printf("\nRead Scratchpad Fail\n");
                }
                
            break;
            
            case '3':
                
                result = tempSensor.copyScratchPad();
                if(result == OneWireSlave::Success)
                {
                    term.printf("\nCopy Scratchpad Success\n");
                }
                else
                {
                    term.printf("\nCopy Scratchpad Fail\n");
                }
                
            break;
            
            case '4':
                
                result = tempSensor.convertTemperature(temperature);
                if(result == OneWireSlave::Success)
                {
                    term.printf("\nConvert Temperature Success\n");
                    term.printf("\nTemperature = %.4f", temperature);
                }
                else
                {
                    term.printf("\nConvert Temperature Fail\n");
                }
                
            break;
            
            case '5':
                
                result = tempSensor.recallEEPROM();
                if(result == OneWireSlave::Success)
                {
                    term.printf("\nRecall EEPROM Success\n");
                }
                else
                {
                    term.printf("\nRecall EEPROM Fail\n");
                }
                
            break;
            
            case '6':
                result = tempSensor.readPowerSupply(localPower);
                if(result == OneWireSlave::Success)
                {
                    term.printf("Read Power Supply Success\n");
                    term.printf("DS18B20 %s\n", localPower ? "Has Local Power" : "Is Parasite Powered");
                }
                else
                {
                    term.printf("Read Power Supply Failed\n");
                }
            break;
            
            case '7':
                term.cls();
                term.home();
            break;
            
            case '8':
                do
                {
                    result = tempSensor.convertTemperature(temperature);
                    if(result == OneWireSlave::Success)
                    {
                        term.printf("\nTemperature = %.4f", temperature);
                    }
                    else
                    {
                        term.printf("\nConvert Temperature Fail\n");
                    }
                }
                while(result == OneWireSlave::Success);
            break;
            
            case '9':
                term.printf("\nLeaving DS18B20 Menu Options\n");
            break;
            
            default:
                mbed_die();
            break;
        }
    }
}
