/******************************************************************//**
* @file maxrefdes99.h
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/

#ifndef MAXREFDES99_H
#define MAXREFDES99_H


#include "mbed.h"
#include "max7219.h"
    

/**********************************************************//**
* @brief Gets character bitmap fom memory
* 
* @details 
*
* On Entry:
*     @param[in] c - character to get bitmap for
*     @param[in] char_buff - pointer to buffer to store bitmap in
*                            must be at least 6 bytes
*
* On Exit:
*    @return None
**************************************************************/
void get_5x7_character(char c, uint8_t *char_buff);


/**********************************************************//**
* @brief Prints character to MAXREFDES99 at given position
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*     @param[in] position - position to print char to
*                posiion 1 is top left column, position 32
*                is bottom right column
*     @param[in] c - char to print
*
* On Exit:
*    @return None
**************************************************************/
void print_char(Max7219 *p_display, uint8_t position, char c);


/**********************************************************//**
* @brief Prints given string to MAXREFDES99 at given position
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*     @param[in] position - position to print char to
*                posiion 1 is top left column, position 32
*                is bottom right column
*     @param[in] s - pointer to string to print
*
* On Exit:
*    @return None
**************************************************************/
void print_string(Max7219 *p_display, uint8_t position, const char *s);


/**********************************************************//**
* @brief Shifts display right 'count' positions with given
*        delay between shifts 
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*     @param[in] count - number of positions to shift
*     @param[in] delay - delay between shifts
*
* On Exit:
*    @return None
**************************************************************/
void shift_display_right(Max7219 *p_display, uint8_t count, uint8_t delay);


/**********************************************************//**
* @brief Shifts display left 'count' positions with given
*        delay between shifts
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*     @param[in] count - number of positions to shift
*     @param[in] delay - delay between shifts
*
* On Exit:
*    @return None
**************************************************************/
void shift_display_left(Max7219 *p_display, uint8_t count, uint8_t delay);


/**********************************************************//**
* @brief Turns on all leds for given quadrant, with quad 1 being
*        top left and quad 4 being bottom right
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*     @param[in] quad - see brief
*
* On Exit:
*    @return None
**************************************************************/
void quad_all_on(Max7219 *p_display, uint8_t quad);


/**********************************************************//**
* @brief Turns off all leds for given quadrant, with quad 1 being
*        top left and quad 4 being bottom right
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*     @param[in] quad - see brief
*
* On Exit:
*    @return None
**************************************************************/
void quad_all_off(Max7219 *p_display, uint8_t quad);


/**********************************************************//**
* @brief Turns all leds for whole display on
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*
* On Exit:
*    @return None
**************************************************************/
void all_on(Max7219 *p_display);


/**********************************************************//**
* @brief Turns all leds for whole display off
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*
* On Exit:
*    @return None
**************************************************************/
void all_off(Max7219 *p_display);


/**********************************************************//**
* @brief Demo loop for MAXREFDES99
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*     @param[in] display_config - structure holding configuration data
*     @param[in] endless_loop - if true run demo in endless loop
*
* On Exit:
*    @return None
**************************************************************/
void demo(Max7219 *display, max7219_configuration_t display_config, bool endless_loop);


/**********************************************************//**
* @brief Shift display in given direction forever
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*     @param[in] scroll_right - if true shift right, else shift left
*
* On Exit:
*    @return None
**************************************************************/
void endless_scroll_display(Max7219 *display, uint32_t scroll_right);

/**********************************************************//**
* @brief Gets 16x16 character bitmap fom memory
* 
* @details 
*
* On Entry:
*     @param[in] c - character to get bitmap for
*     @param[in] char_buff - pointer to buffer to store bitmap 
*                            Buffer should be 32 bytes
*     @param[in] font_type - 16x16 font type used for display 
* On Exit:
*    @return None
**************************************************************/
void get_16x16_character(char c, uint8_t *char_buff, uint8_t font_type);


/**********************************************************//**
* @brief Prints character to MAXREFDES99 16x16 LED display
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
      @param[in] position -  for single char print this value is zero,
                 for string case it is multiple of 32 bytes as each char data is 32 bytes, 
*     @param[in] c - char to print
*     @param[in] font_type - 16x16 font type used for display 
*
* On Exit:
*    @return None
**************************************************************/
void print_char_16x16(Max7219 *p_display, uint16_t position, char c, uint8_t font_type);


/**********************************************************//**
* @brief Prints given string to MAXREFDES99 16x16 LED display
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*     @param[in] s - pointer to string to print
*     @param[in] font_type - 16x16 font type used for display 
*
* On Exit:
*    @return None
**************************************************************/
void print_string_16x16(Max7219 *p_display, char *s, uint8_t font_type);


/**********************************************************//**
* @brief Shifts 16x16 font display right 'count' positions with given
*        delay between shifts 
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*     @param[in] count - number of positions to shift
*     @param[in] delay - delay between shifts in milliseconds
*
* On Exit:
*    @return None
**************************************************************/
void shift_display_right_16x16(Max7219 *p_display, uint8_t count, uint8_t delay);


/**********************************************************//**
* @brief Shifts 16x16 font display left 'count' positions with given
*        delay between shifts
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*     @param[in] count - number of positions to shift
*     @param[in] delay - delay between shifts in milliseconds
*
* On Exit:
*    @return None
**************************************************************/
void shift_display_left_16x16(Max7219 *p_display, uint8_t count, uint8_t delay);

/**********************************************************//**
* @brief Demo loop for MAXREFDES99 with 16x16 font display
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*     @param[in] display_config - structure holding configuration data
*     @param[in] font_type - 16x16 font type used for display 
*     @param[in] endless_loop - if true run demo in endless loop
*
* On Exit:
*    @return None
**************************************************************/
void demo_16x16(Max7219 *display, max7219_configuration_t display_config, uint8_t font_type, bool endless_loop);


/**********************************************************//**
* @brief Shift 16x16 font display in given direction forever
* 
* @details 
*
* On Entry:
*     @param[in] p_display - pointer to Max7219 object
*     @param[in] scroll_right - if true shift right, else shift left
*
* On Exit:
*    @return None
**************************************************************/
void endless_scroll_display_16x16(Max7219 *display, uint32_t scroll_right);


/**********************************************************//**
* @brief Print demo menu
* 
* @details 
*
* On Entry:
*
* On Exit:
*    @return User entry
**************************************************************/
uint32_t print_menu(void);


/**********************************************************//**
* @brief Get integer value from user
* 
* @details 
*
* On Entry:
*     @param[in] msg - prompt for user
*     @param[in] max_val - maximum allowable input
*
* On Exit:
*    @return User entry
**************************************************************/
uint32_t get_user_input(char *msg, uint32_t max_val);


/**********************************************************//**
* @brief Get char from user
* 
* @details 
*
* On Entry:
*     @param[in] msg - prompt for user
*
* On Exit:
*    @return User entry
**************************************************************/
char get_user_char(char *msg);


/**********************************************************//**
* @brief Get string from user
* 
* @details 
*
* On Entry:
*     @param[in] msg - prompt for user
*
* On Exit:
*    @return User entry
**************************************************************/
char * get_user_string(char *msg);


void clear_buffer(void);


#endif /*MAXREFDES99_H*/

