/**********************************************************************
* Copyright (C) 2015 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/


#include "mbed.h"
#include "max14871_shield.h"


//helper functions for demo
uint32_t menu_select_motor_driver(void);

void menu_set_duty_cycle_(uint8_t md, Max14871_Shield *shld);

void menu_set_op_mode(uint8_t md, Max14871_Shield *shld);

void menu_set_current_reg(uint8_t md, Max14871_Shield *shld);

void menu_print_motor_data(uint8_t md, Max14871_Shield *shld);

void menu_print_maxim_banner(void);

void clear_screen(void);


int main(void)
{
    uint8_t idx = 0;
    uint32_t user_entry = 0;
    const float PWM_PERIOD = 40e-6f; //~25KHz
    
    I2C i2c_bus(D14, D15);

    Max14871_Shield shld(&i2c_bus, true);
    
    //set pwm switching frequency
    for(idx = 0; idx < 4; idx++)
    {
        shld.set_pwm_period((Max14871_Shield::max14871_motor_driver_t) (idx + 1), PWM_PERIOD);
    }
    
    idx = 1;
    
    //main menu                    
    while(user_entry != 5)
    {
        menu_print_maxim_banner();
        
        //print currently selected motor's data
        menu_print_motor_data(idx, &shld);
        
        printf("Please select an option below:\n");
        printf("%t1. Select Active Motor Driver\n");
        printf("%t2. Set PWM Duty Cycle\n");
        printf("%t3. Set Operating Mode\n");
        printf("%t4. Set Current Regulation Mode\n");
        printf("%t5. Quit\n");
        
        scanf("%d", &user_entry);
        
        switch(user_entry)
        {
            case 1:
                idx = menu_select_motor_driver();
                clear_screen();
            break;
            
            case 2:
                menu_set_duty_cycle_(idx, &shld);
                clear_screen();
            break;
            
            case 3:
                menu_set_op_mode(idx, &shld);
                clear_screen();
            break;
            
            case 4:
                menu_set_current_reg(idx, &shld);
                clear_screen();
            break;
            
            case 5:
                printf("Ending Program\n");
            break;
            
            default:
                printf("Invalid entry, please try again\n");
            break;
        }
    }
    
    return 0;
}


uint32_t menu_select_motor_driver(void)
{
    uint32_t md;
    
    do
    {
        printf("\nPlease enter motor driver number, 1-4\n");
        scanf("%d", &md);
        
        if((md < 1) || (md > 4))
        {
            printf("Value out of range\n");
        }
    }
    while((md < 1) || (md > 4));
    
    return md;
}


void menu_set_duty_cycle_(uint8_t md, Max14871_Shield *shld)
{
    float dc = 0.0f;
    
    do
    {
        printf("\nPlease enter new duty cycle, 0.0 - 1.0\n");
        scanf("%f", &dc);
        
        if((dc < 0.0f) || (dc > 1.0f))
        {
            printf("Value out of range\n");
        }
    }
    while((dc < 0.0f) || (dc > 1.0f));
    
    shld->set_pwm_duty_cycle((Max14871_Shield::max14871_motor_driver_t) md, dc);
}


void menu_set_op_mode(uint8_t md, Max14871_Shield *shld)
{
    uint32_t mode;
    
    do
    {
        printf("\nPlease enter one of the following modes:\n");
        printf("%t1. COAST\n");
        printf("%t2. BRAKE\n");
        printf("%t3. REVERSE\n");
        printf("%t4. FORWARD\n");
        
        scanf("%d", &mode);
        
        if((mode < 1) || (mode > 4))
        {
            printf("Value out of range\n");
        }
    }
    while((mode < 1) || (mode > 4));
    
    shld->set_operating_mode((Max14871_Shield::max14871_motor_driver_t) md,(Max14871_Shield::max14871_operating_mode_t) mode);
}


void menu_set_current_reg(uint8_t md, Max14871_Shield *shld)
{
    uint32_t mode;
    float v_ref = 0.0f;
    
    do
    {
        printf("\nPlease enter one of the following modes:\n");
        printf("%t1. 25%% ripple, internal reference\n");
        printf("%t2. 25%% ripple, external reference\n");
        printf("%t3. TCOFF fast decay, internall reference\n");
        printf("%t4. TCOFF slow decay, internall reference\n");
        printf("%t5. TCOFF fast decay, external reference\n");
        printf("%t6. TCOFF slow decay, external reference\n");
        
        scanf("%d", &mode);
        
        if((mode < 1) || (mode > 6))
        {
            printf("Value out of range\n");
        }
    }
    while((mode < 1) || (mode > 6));
    
    if((mode == 2) || (mode >= 5))
    {
        do
        {
            printf("Please enter external Vref, 0.0V - 2.0V\n");
            scanf("%f", &v_ref);
            
            if((v_ref < 0.0f) || (v_ref > 2.0f))
            {
                printf("Value out of range\n");
            }
        }
        while((v_ref < 0.0f) || (v_ref > 2.0f));
    }
    
    shld->set_current_regulation_mode((Max14871_Shield::max14871_motor_driver_t) md,(Max14871_Shield::max14871_current_regulation_mode_t) mode, v_ref);
}


void menu_print_motor_data(uint8_t md, Max14871_Shield *shld)
{
    Max14871_Shield::max14871_operating_mode_t op_state;
    Max14871_Shield::max14871_current_regulation_mode_t reg_state;
    float dc;
    float v_ref;
    
    op_state = shld->get_operating_mode((Max14871_Shield::max14871_motor_driver_t) md);
        
    reg_state = shld->get_current_regulation_mode((Max14871_Shield::max14871_motor_driver_t) md);
        
    dc = shld->get_pwm_duty_cycle((Max14871_Shield::max14871_motor_driver_t) md);
        
    v_ref = shld->get_external_voltage_ref((Max14871_Shield::max14871_motor_driver_t) md);
    
    printf("Active Motor Driver = %d\n", md);
    
    switch(op_state)
    {
        case 1:
            printf("Operating Mode = COAST\n");
        break;
        
        case 2:
            printf("Operating Mode = BRAKE\n");
        break;
        
        case 3:
            printf("Operating Mode = REVERSE\n");
        break;
        
        case 4:
            printf("Operating Mode = FORWARD\n");
        break;
        
        default:
            printf("Operating Mode = UNKNOWN\n");
        break;
    }
    
    
    switch(reg_state)
    {
        case 1:
            printf("Current Regulation Mode = 25%% ripple, internal reference\n");
        break;
        
        case 2:
            printf("Current Regulation Mode = 25%% ripple, external reference\n");
        break;
         
        case 3:
            printf("Current Regulation Mode = TCOFF fast decay, internall reference\n");
        break;
        
        case 4:
            printf("Current Regulation Mode = TCOFF slow decay, internall reference\n");
        break;
        
        case 5:
            printf("Current Regulation Mode = TCOFF fast decay, external reference\n");
        break;
        
        case 6:
            printf("Current Regulation Mode = TCOFF slow decay, external reference\n");
        break;
        
        default:
            printf("Current Regulation Mode = UNKNOWN\n");
        break;
    }
    
    printf("Duty Cycle = %0.2f\n", dc);
    printf("Voltage Reference = %0.2f\n", v_ref);
}


void menu_print_maxim_banner(void)
{
   printf( "\r\n\r\n");
   printf( "///////////////////////////////////////////////////////////////////\r\n");
   printf( "//                        _____    _____                         //\r\n");
   printf( "//                       /  __ \\  / __  \\                        //\r\n");
   printf( "//                       | |  \\/ / /  | |                        //\r\n");
   printf( "//                       | |    / /   | |                        //\r\n");
   printf( "//                       | |   /  \\   | |                        //\r\n");
   printf( "//                       | |  / /\\ \\  | |                        //\r\n");
   printf( "//                       |_| /_/  \\_\\ |_|                        //\r\n");
   printf( "//                                                               //\r\n");
   printf( "///////////////////////////////////////////////////////////////////\r\n");
   printf("\r\n");
}


void clear_screen(void)
{
    printf("%c[2J", 0x1B); //clear screen
    printf("%c[H", 0x1B); //move cursor to Home
}

