/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#include <algorithm>
#include <functional>
#include "Bitmap.hpp"
#include "WindowManager.hpp"

static void deleteWindow(Window * window) { delete window; }

WindowManager::~WindowManager() {
  // Free all windows.
  std::for_each(windowStack.begin(), windowStack.end(), deleteWindow);
  std::for_each(actionQueue.begin(), actionQueue.end(), deleteWindow);
}

void WindowManager::processAction(Window * window) {
  if (window) {
    windowStack.push_back(window);
  } else if (!windowStack.empty()) {
    delete windowStack.back();
    windowStack.pop_back();
  }
}

void WindowManager::push(std::auto_ptr<Window> & window) {
  if (window.get()) {
    window->resize(canvas.width(), canvas.height());
    window->move(0, 0);
    window->setWindowManager(NULL);
    actionQueue.push_back(window.release());
  }
}

void WindowManager::pop() { actionQueue.push_back(NULL); }

bool WindowManager::update() {
  bool redraw = false;

  // Perform all queued push / pop actions.
  if (!actionQueue.empty()) {
    if (!windowStack.empty()) {
      windowStack.back()->setWindowManager(NULL);
    }
    std::for_each(
        actionQueue.begin(), actionQueue.end(),
        std::bind1st(std::mem_fun(&WindowManager::processAction), this));
    actionQueue.clear();
    if (windowStack.empty()) {
      canvas.clear();
      redraw = true;
    } else {
      windowStack.back()->setWindowManager(this);
    }
  }

  if (!windowStack.empty()) {
    // Update all windows. Only allow redraw of the top window.
    for (std::vector<Window *>::reverse_iterator it = windowStack.rbegin();
         it != windowStack.rend(); ++it) {
      if ((*it)->update(it == windowStack.rbegin() ? &canvas : NULL)) {
        redraw = true;
      }
    }
  }

  return redraw;
}

void WindowManager::processKey(Key key) {
  if (!windowStack.empty()) {
    windowStack.back()->processKey(key);
  }
}
