/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#include <memory>
#include <mbed-os/features/netsocket/TCPSocket.h>
#include <mbed-os/platform/mbed_wait_api.h>
#include "CC3100.hpp"
#include "DisplayIdWindow.hpp"
#include "ErrorWindow.hpp"
#include "MakeFunction.hpp"
#include "NormalOperationWindow.hpp"
#include "WifiConnectWindow.hpp"
#include "WindowManager.hpp"

static const char wifiSsid[] = "MAXREFDES155";
static const char wifiPassword[] = "maxim1234";
static const char serverAddress[] = "www.maxim-security.com";
static const uint16_t serverPort = 3900;

WifiConnectWindow::WifiConnectWindow() : state(NotStarted) {
  description.setParent(this);
  description.setText("WiFi SSID: " + std::string(wifiSsid) +
                      "\nWiFi Password: " + std::string(wifiPassword));
  description.setWordWrap(true);
  backButton.setParent(this);
  backButton.setText("Back");
  backButton.setClickedHandler(
      makeFunction(this, &WifiConnectWindow::backButtonClicked));
  continueButton.setParent(this);
  continueButton.setText("Connect to WiFi");
  continueButton.setClickedHandler(
      makeFunction(this, &WifiConnectWindow::continueButtonClicked));
  continueButton.setFocused();
}

void WifiConnectWindow::updated() {
  switch (state) {
  case NotStarted:
    break;

  case PreConnect:
    backButton.setParent(NULL);
    continueButton.setParent(NULL);
    description.setText("Connecting to WiFi network...");
    state = WiFiConnect;
    break;

  case WiFiConnect:
    // Connect to AP.
    if (CC3100::instance().connect(wifiSsid, wifiPassword,
                                   NSAPI_SECURITY_WPA2) != 0) {
      if (windowManager()) {
        std::auto_ptr<Window> window(new ErrorWindow("WiFi connect failed"));
        windowManager()->push(window);
      }
      break;
    }

    description.setText("Connecting to server...");
    state = ServerConnect;
    break;

  case ServerConnect: {
    // Get IP address from DNS lookup.
    SocketAddress socketAddress;
    if (CC3100::instance().gethostbyname(serverAddress, &socketAddress) != 0) {
      if (windowManager()) {
        std::auto_ptr<Window> window(new ErrorWindow("Hostname lookup failed"));
        windowManager()->push(window);
      }
      break;
    }

    // Open socket connection.
    std::auto_ptr<TCPSocket> socket(new TCPSocket(&CC3100::instance()));
    socket->set_blocking(false);
    socketAddress.set_port(serverPort);
    if (socket->connect(socketAddress) != 0) {
      if (windowManager()) {
        std::auto_ptr<Window> window(new ErrorWindow("Socket connect failed"));
        windowManager()->push(window);
      }
      break;
    }

    if (windowManager()) {
      windowManager()->pop();
      std::auto_ptr<Window> window(new NormalOperationWindow(socket));
      windowManager()->push(window);
    }
    break;
  }
  }
}

void WifiConnectWindow::resized() {
  backButton.resize(backButton.preferredWidth(),
                    backButton.preferredHeight());
  backButton.move(0, height() - backButton.height());
  continueButton.resize(continueButton.preferredWidth(),
                        continueButton.preferredHeight());
  continueButton.move(width() - continueButton.width(),
                      height() - continueButton.height());
  description.resize(width(),
                     std::min(backButton.y(), continueButton.y()));
}

bool WifiConnectWindow::doProcessKey(Key key) {
  bool handled;
  switch (key) {
  case LeftKey:
    backButton.setFocused();
    handled = true;
    break;

  case RightKey:
    continueButton.setFocused();
    handled = true;
    break;

  default:
    handled = false;
    break;
  }
  return handled;
}

void WifiConnectWindow::continueButtonClicked(Button *) {
  if (state == NotStarted) {
    state = PreConnect;
  }
}

void WifiConnectWindow::backButtonClicked(Button *) {
  if (windowManager()) {
    windowManager()->pop();
    std::auto_ptr<Window> window(
        new DisplayIdWindow(DisplayIdWindow::PreConnectMode));
    windowManager()->push(window);
  }
}
