/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#ifndef GRAPHIC_HPP
#define GRAPHIC_HPP

#include <stddef.h>
#include <vector>
#include "Keys.hpp"

class Bitmap;

/// @brief Base class for all graphical elements.
/// @details
/// Includes unique parent-child relationships for creating trees of graphical
/// objects.
class Graphic {
public:
  typedef std::vector<Graphic *> ChildContainer;

  Graphic();
  virtual ~Graphic();

  /// @name Parent
  /// @brief
  /// Get or set the parent graphic of this graphic. Set to NULL if this graphic
  /// has no parent.
  /// @{

  Graphic * parent() { return parent_; }

  const Graphic * parent() const { return parent_; }

  /// @note Adds this graphic to the parent's list of children.
  /// @sa childrenChanged
  void setParent(Graphic * parent);

  /// @}

  /// @brief List of child graphics for this parent.
  /// @note
  /// Children should be added and removed by calling setParent. Children will
  /// be removed automatically when they are destroyed.
  const ChildContainer & children() const { return children_; }

  /// @brief Check if this graphic is focused.
  /// @returns True if focused.
  bool focused() const;

  /// @brief Set this graphic as the focused graphic.
  /// @details
  /// The focused graphic is the first to receive input events such as key
  /// presses.
  /// @sa focusChanged
  void setFocused();

  /// @brief Coordinates of this graphic in pixels.
  /// @details
  /// Coordinates are relative to the top-left corner of the parent graphic.
  /// @{

  int x() const { return x_; }
  int y() const { return y_; }

  /// @}

  /// @brief Displayed dimensions of this graphic in pixels.
  /// @{

  int width() const { return width_; }
  int height() const { return height_; }

  /// @}

  /// @brief Move graphic to a new location measured in pixels.
  /// @details
  /// Coordinates are relative to the top-left corner of the parent graphic.
  /// @sa moved
  void move(int x, int y);

  /// @brief
  /// Resize graphic to a new size measure in pixels. Minimum width and height
  /// is 1.
  /// @sa resized
  void resize(int width, int height);

  /// @brief Render this graphic onto a bitmap.
  /// @{

  void render(Bitmap & bitmap, int xOffset, int yOffset) const;
  void render(Bitmap & bitmap) const;

  /// @}

  /// @brief
  /// Update this graphic and all child graphics. Checks if graphic has been
  /// invalidated and should be redrawn.
  /// @param canvas Canvas used for rendering. May be set to NULL to defer redraw.
  /// @returns True if the canvas has been updated.
  /// @sa updated
  bool update(Bitmap * canvas);

  /// @brief Process a key-press input event.
  /// @details
  /// The event will first be directed to the focused graphic. Processing will
  /// proceed to each parent graphic until it has been handled.
  /// @sa doProcessKey
  /// @returns True if the key event was handled.
  bool processKey(Key key);

protected:
  /// @brief Mark the visual region as invalid.
  /// @note Indicates a redraw is necessary during next update.
  void invalidate() { valid_ = false; }

  /// Event handler for when a child is added or removed.
  virtual void childrenChanged();

  /// @brief Event handler for when this graphic has been focused or unfocused.
  /// @param focused True if focused or false if unfocused.
  virtual void focusChanged(bool focused);

  /// Event handler for when this graphic has been moved.
  virtual void moved();

  /// Event handler for when this graphic has been resized.
  virtual void resized();

  /// Event handler for when this graphic has been updated.
  virtual void updated();

  /// @brief Render this graphic onto a bitmap.
  /// @details The default implementation renders each child in order.
  virtual void doRender(Bitmap & bitmap, int xOffset, int yOffset) const;

  /// @brief Process a key-press input event.
  /// @returns
  /// True if the key event was handled. False if the key event should be
  /// propagated.
  virtual bool doProcessKey(Key);

private:
  Graphic * parent_;
  ChildContainer children_;
  Graphic * focusedChild_;
  int x_;
  int y_;
  int width_;
  int height_;
  bool valid_;

  void updateAll();
  bool redrawInvalid(Bitmap & canvas);
  void setAllValid();

  // Uncopyable
  Graphic(const Graphic &);
  const Graphic & operator=(const Graphic &);
};

#endif
