/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#include <algorithm>
#include "Bitmap.hpp"

static const int minWidthHeight = 1;
static const int pixelsPerSegment = 8;

template <typename T>
static T divideRoundUp(const T dividend, const T divisor) {
  return (dividend / divisor) + (((dividend % divisor) == 0) ? 0 : 1);
}

static int calculateSegmentsPerLine(int width) {
  return divideRoundUp(width, pixelsPerSegment);
}

static int calculateHeight(size_t size, int width) {
  return divideRoundUp<size_t>(size, calculateSegmentsPerLine(width));
}

static int calculateSegmentIndex(int x, int y, int width) {
  return y * calculateSegmentsPerLine(width) + x / pixelsPerSegment;
}

static unsigned int calculatePixelMask(int x) {
  return 1 << (pixelsPerSegment - 1 - (x % pixelsPerSegment));
}

Bitmap::Bitmap(int width, int height)
    : width_(std::max(width, minWidthHeight)),
      height_(std::max(height, minWidthHeight)),
      data_(calculateSegmentsPerLine(width_) * height_, 0x00) {}

Bitmap::Bitmap(const uint8_t * data, size_t size, int width)
    : width_(std::max(width, minWidthHeight)),
      height_(std::max(calculateHeight(size, width), minWidthHeight)),
      data_(calculateSegmentsPerLine(width_) * height_, 0x00) {
  std::copy(data, data + size, data_.begin());
}

bool Bitmap::pixelEnabled(int x, int y) const {
  bool enabled = false;
  if ((x >= 0) && (x < width_) && (y >= 0) && (y < height_)) {
    enabled =
        data_[calculateSegmentIndex(x, y, width_)] & calculatePixelMask(x);
  }
  return enabled;
}

void Bitmap::setPixelEnabled(int x, int y, bool enabled) {
  if ((x >= 0) && (x < width_) && (y >= 0) && (y < height_)) {
    uint8_t & dataSegment = data_[calculateSegmentIndex(x, y, width_)];
    const unsigned int dataMask = calculatePixelMask(x);
    if (enabled) {
      dataSegment |= dataMask;
    } else {
      dataSegment &= ~dataMask;
    }
  }
}

void Bitmap::overlay(int x, int y, const uint8_t * data, size_t size,
                     int width) {
  if (width < minWidthHeight) {
    return;
  }

  const int segmentsPerLine = calculateSegmentsPerLine(width);
  for (size_t segment = 0; segment < size; ++segment) {
    const int curY = segment / segmentsPerLine;
    if (!((y + curY) < height_)) {
      break;
    }
    for (int pixel = 0; pixel < pixelsPerSegment; ++pixel) {
      const int curX = (segment % segmentsPerLine) * pixelsPerSegment + pixel;
      if (!(((x + curX) < width_) && (curX < width))) {
        break;
      }
      setPixelEnabled(x + curX, y + curY,
                      data[segment] & (1 << (pixelsPerSegment - 1 - pixel)));
    }
  }
}

void Bitmap::overlay(int x, int y, const Bitmap & src) {
  overlay(x, y, &src.data_[0], src.data_.size(), src.width_);
}

void Bitmap::clear() { std::fill(data_.begin(), data_.end(), 0); }

void Bitmap::clear(int x, int y, int width, int height) {
  if (!((x >= 0) && (x < width_) && (y >= 0) && (y < height_))) {
    return;
  }
  if ((x + width) > width_) {
    width = width_ - x;
  }
  if ((y + height) > height_) {
    height = height_ - y;
  }

  const int startX = x;
  const int startY = y;
  while (y < (startY + height)) {
    x = startX;
    while (x < (startX + width)) {
      if (((x % pixelsPerSegment) == 0) &&
          ((x + pixelsPerSegment) < (startX + width))) {
        data_[calculateSegmentIndex(x, y, width_)] = 0;
        x += pixelsPerSegment;
      } else {
        setPixelEnabled(x, y, false);
        ++x;
      }
    }
    ++y;
  }
}
