/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#ifndef SENSORNODE_HPP
#define SENSORNODE_HPP

#include <stdint.h>
#include <MaximInterfaceCore/Function.hpp>
#include <MaximInterfaceCore/ManId.hpp>
#include <MaximInterfaceCore/Result.hpp>
#include <MaximInterfaceCore/RomId.hpp>
#include <MaximInterfaceDevices/DS28C36_DS2476.hpp>

namespace MaximInterfaceCore {

class I2CMaster;
class Sleep;

} // namespace MaximInterfaceCore
namespace MaximInterfaceDevices {

class DS2476;

} // namespace MaximInterfaceDevices

/// Interface to the authenticated sensor node peripheral board.
class SensorNode {
public:
  /// Prints a null-terminated char string.
  typedef MaximInterfaceCore::Function<void(const char *)> PrintHandler;

  /// Style of temperature to measure.
  enum TempStyle { AmbientTemp, ObjectTemp };

  enum State {
    Disconnected,       ///< No sensor node is connected.
    Invalid,            ///< Sensor node is not valid.
    ValidLaserDisabled, ///< Sensor node is valid, and laser is disabled.
    ValidLaserEnabled,  ///< Sensor node is valid, and laser is enabled.
  };

  /// @param i2c
  /// I2C bus connected to the sensor node that communicates with DS28C36 and
  /// MLX90614.
  /// @param ds2476 Coprocessor used for authentication computations.
  SensorNode(MaximInterfaceCore::Sleep & sleep,
             MaximInterfaceCore::I2CMaster & i2c,
             MaximInterfaceDevices::DS2476 & ds2476);

  /// Detects if a potential sensor node is connected.
  State detect();

  /// @brief Enable or disable the laser.
  /// @param enabled True to enable the laser or false to disable.
  /// @param print
  /// Optional callback for displaying informational messages to the user.
  /// @returns True if the operation was successful.
  MaximInterfaceCore::Result<void>
  setLaserEnabled(bool enabled, const PrintHandler & print = PrintHandler());

  /// @brief Get a temperature measurement from the MLX90614.
  /// @param style Temperature style to read.
  /// @returns Temperature in Celsius.
  MaximInterfaceCore::Result<double> readTemp(TempStyle style);

private:
  /// Checks if the sensor node is authentic.
  MaximInterfaceCore::Result<void> authenticate();

  /// @brief Checks if the laser is enabled.
  /// @returns True if the laser is enabled.
  MaximInterfaceCore::Result<bool> getLaserEnabled();

  MaximInterfaceCore::I2CMaster * i2c;
  MaximInterfaceDevices::DS28C36 ds28c36;
  MaximInterfaceDevices::DS2476 * ds2476;
  MaximInterfaceCore::RomId::array romId;
  MaximInterfaceCore::ManId::array manId;

  friend class HardwareTestWindow;
};

#endif
