/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#include <MaximInterfaceDevices/DS28C36_DS2476.hpp>
#include "Factory.hpp"

#define TRY MaximInterfaceCore_TRY
#define TRY_VALUE MaximInterfaceCore_TRY_VALUE

using MaximInterfaceCore::none;
using MaximInterfaceCore::Result;
using MaximInterfaceDevices::DS28C36;
using MaximInterfaceDevices::DS2476;

// Authority (web server) public key x-component.
static const DS28C36::Page::array authPublicKeyX = {
    0x7A, 0xB9, 0xCD, 0x00, 0x3F, 0x42, 0xF3, 0x30, 0x76, 0x25, 0x9B,
    0x6B, 0xFD, 0xC2, 0x6D, 0xE2, 0xDB, 0x59, 0xA8, 0xD9, 0xE0, 0x68,
    0x3E, 0x1B, 0xFF, 0x50, 0xCB, 0x6C, 0x18, 0xB6, 0xF2, 0xEB};

// Authority (web server) public key y-component.
static const DS28C36::Page::array authPublicKeyY = {
    0x7F, 0xFC, 0xEE, 0xDD, 0x77, 0xE9, 0x63, 0x07, 0x62, 0x37, 0x33,
    0x81, 0x17, 0x16, 0x58, 0x75, 0x12, 0x88, 0x85, 0x58, 0x57, 0xC0,
    0x15, 0xB8, 0x08, 0xDE, 0xB2, 0x3B, 0xD7, 0x8A, 0x9D, 0x2C};

// Authority (web server) private key.
static const DS28C36::Page::array authPrivateKey = {
    0xC5, 0x45, 0x5F, 0xFB, 0x45, 0xEA, 0x77, 0x0B, 0xF1, 0x1B, 0xE5,
    0xD2, 0x21, 0xAD, 0x35, 0xF5, 0x0B, 0x61, 0x7F, 0x66, 0xDB, 0xA0,
    0xBD, 0xB6, 0x64, 0x75, 0x21, 0x4E, 0xB0, 0x98, 0x2D, 0x8E};

// Master secret for SHA-256 HMAC authentication.
static const DS28C36::Page::array masterSecret = {
    0x6D, 0x52, 0xB6, 0x15, 0xDC, 0x80, 0xCF, 0xB1, 0x25, 0xB0, 0x76,
    0xB7, 0x7C, 0xAC, 0x00, 0xF2, 0xBC, 0x19, 0xBE, 0xD3, 0x2F, 0x9D,
    0xC1, 0x42, 0x2A, 0xA5, 0xF6, 0xAE, 0x71, 0xF2, 0x25, 0xB6};

Result<void> provisionAuthenticator(DS28C36 & ds28c36) {
  // Check if already provisioned;
  if (const Result<DS28C36::PageProtection> protection =
          ds28c36.readPageProtection(DS28C36::gpioControlPage)) {
    if (protection.value().test(DS28C36::APH)) {
      return none;
    }
  } else {
    return protection.error();
  }

  const DS28C36::Page::array zeroPage = {};
  // Page 0 - 15
  for (int pageNum = 0; pageNum <= 15; ++pageNum) {
    TRY(ds28c36.writeMemory(pageNum, zeroPage));
  }
  // Page 16, 17, 22
  DS28C36::Page::array page;
  TRY(ds28c36.readRng(page));
  TRY(ds28c36.writeMemory(DS28C36::publicKeyAxPage, page));
  TRY(ds28c36.readRng(page));
  TRY(ds28c36.writeMemory(DS28C36::publicKeyAyPage, page));
  TRY(ds28c36.readRng(page));
  TRY(ds28c36.writeMemory(DS28C36::privateKeyAPage, page));
  // Page 18, 19, 23
  TRY(ds28c36.readRng(page));
  TRY(ds28c36.writeMemory(DS28C36::publicKeyBxPage, page));
  TRY(ds28c36.readRng(page));
  TRY(ds28c36.writeMemory(DS28C36::publicKeyByPage, page));
  TRY(ds28c36.readRng(page));
  TRY(ds28c36.writeMemory(DS28C36::privateKeyBPage, page));
  // Page 20, 21, 24
  TRY(ds28c36.readRng(page));
  TRY(ds28c36.writeMemory(DS28C36::publicKeyCxPage, page));
  TRY(ds28c36.readRng(page));
  TRY(ds28c36.writeMemory(DS28C36::publicKeyCyPage, page));
  TRY(ds28c36.readRng(page));
  TRY(ds28c36.writeMemory(DS28C36::privateKeyCPage, page));
  // Page 25
  TRY(ds28c36.writeMemory(DS28C36::secretAPage, masterSecret));
  TRY(ds28c36.writeBuffer(zeroPage));
  TRY(ds28c36.computeAndLockSha2Secret(0, DS28C36::SecretNumA,
                                       DS28C36::SecretNumA, false));
  // Page 26
  TRY(ds28c36.readRng(page));
  TRY(ds28c36.writeMemory(DS28C36::secretBPage, page));
  // Page 28
  TRY(ds28c36.setPageProtection(DS28C36::romOptionsPage, DS28C36::APH));
  // Page 29
  TRY(ds28c36.setPageProtection(DS28C36::gpioControlPage, DS28C36::APH));
  return none;
}

Result<void> provisionCoprocessor(DS2476 & ds2476) {
  // Check if already provisioned;
  if (const Result<DS2476::Page::array> page =
          ds2476.readMemory(DS2476::publicKeyCxPage)) {
    if (page.value() == authPublicKeyX) {
      return none;
    }
  } else {
    return page.error();
  }

  // Page 0, 1
  const DS2476::Page::array zeroPage = {};
  TRY(ds2476.writeMemory(0, zeroPage));
  TRY(ds2476.writeMemory(1, zeroPage));
  // Page 2 - 13
  DS2476::Page::array page;
  for (int pageNum = 2; pageNum <= 13; ++pageNum) {
    TRY(ds2476.readRng(page));
    TRY(ds2476.writeMemory(pageNum, page));
  }
  // Page 16, 17, 22
  TRY(ds2476.generateEcc256KeyPair(DS2476::KeyNumA, false));
  // Page 18, 19, 23
  TRY(ds2476.readRng(page));
  TRY(ds2476.writeMemory(DS2476::publicKeyBxPage, page));
  TRY(ds2476.readRng(page));
  TRY(ds2476.writeMemory(DS2476::publicKeyByPage, page));
  TRY(ds2476.readRng(page));
  TRY(ds2476.writeMemory(DS2476::privateKeyBPage, page));
  // Page 20, 21, 24
  TRY(ds2476.writeMemory(DS2476::publicKeyCxPage, authPublicKeyX));
  TRY(ds2476.writeMemory(DS2476::publicKeyCyPage, authPublicKeyY));
  TRY(ds2476.writeMemory(DS2476::privateKeyCPage, authPrivateKey));
  // Page 25
  TRY(ds2476.writeMemory(DS2476::secretAPage, masterSecret));
  // Page 26
  TRY(ds2476.readRng(page));
  TRY(ds2476.writeMemory(DS2476::secretBPage, page));
  // Page 29
  // TRY(ds2476.setPageProtection(DS2476::gpioControlPage, DS2476::APH));
  // Page 14, 15
  std::vector<uint8_t> publicKeyA;
  publicKeyA.reserve(page.size() * 2 + 1);
  TRY_VALUE(page, ds2476.readMemory(DS2476::publicKeyAxPage));
  publicKeyA.assign(page.begin(), page.end());
  TRY_VALUE(page, ds2476.readMemory(DS2476::publicKeyAyPage));
  publicKeyA.insert(publicKeyA.end(), page.begin(), page.end());
  publicKeyA.insert(publicKeyA.end(), 0x00); // Customization
  TRY(ds2476.writeBuffer(publicKeyA));
  MaximInterfaceCore::Ecc256::Signature::array writeCertificate;
  TRY_VALUE(writeCertificate, ds2476.generateEcdsaSignature(DS2476::KeyNumC));
  TRY(ds2476.writeMemory(14, writeCertificate.r));
  TRY(ds2476.writeMemory(15, writeCertificate.s));
  // Remove Private Key C and set protection
  TRY(ds2476.readRng(page));
  TRY(ds2476.writeMemory(DS2476::privateKeyCPage, page));
  // TRY(ds2476.setPageProtection(DS2476::publicKeyCxPage,
  //                              DS2476::PageProtection::AUTH));
  
  return none;
}
