/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#include <string>
#include <MaximInterfaceCore/HexString.hpp>
#include <mbed-os/drivers/I2C.h>
#include "SensorNode.hpp"

#define TRY MaximInterfaceCore_TRY
#define TRY_VALUE MaximInterfaceCore_TRY_VALUE

using namespace MaximInterfaceCore;
using MaximInterfaceDevices::DS2476;
using MaximInterfaceDevices::DS28C36;

// I2C address of the MLX90614.
static const uint8_t mlx90614Addr = 0xB4;

SensorNode::SensorNode(Sleep & sleep, I2CMaster & i2c, DS2476 & ds2476)
    : i2c(&i2c), ds28c36(sleep, i2c), ds2476(&ds2476) {}

SensorNode::State SensorNode::detect() {
  if (Result<DS28C36::Page::array> page =
          ds28c36.readMemory(DS28C36::romOptionsPage)) {
    const DS28C36::RomOptions romOptions(page.value());
    copy(romOptions.romId(), make_span(romId));
    copy(romOptions.manId(), make_span(manId));
  } else {
    return Disconnected;
  }

  if (!valid(romId)) {
    return Disconnected;
  }

  {
    const Result<void> result = authenticate();
    if (!result) {
      return (result.error() == DS28C36::AuthenticationError) ? Invalid
                                                              : Disconnected;
    }
  }

  const Result<bool> laserEnabled = getLaserEnabled();
  return laserEnabled
             ? (laserEnabled.value() ? ValidLaserEnabled : ValidLaserDisabled)
             : Disconnected;
}

Result<double> SensorNode::readTemp(TempStyle style) {
  uint8_t data[2];
  switch (style) {
  case ObjectTemp:
  default:
    data[0] = 0x07;
    break;

  case AmbientTemp:
    data[0] = 0x06;
    break;
  }
  TRY(i2c->writePacket(mlx90614Addr, make_span(data, 1),
                       I2CMaster::StopOnError));
  TRY(i2c->readPacket(mlx90614Addr, data, I2CMaster::Stop));
  return ((static_cast<unsigned int>(data[1]) << 8) | data[0]) * 0.02 - 273.15;
}

Result<void> SensorNode::authenticate() {
  // Compute slave secret on coprocessor.
  DS28C36::PageAuthenticationData message;
  message.setPageNum(0);
  if (const Result<DS28C36::Page::array> page =
          ds28c36.readMemory(message.pageNum())) {
    message.setPage(page.value());
  } else {
    return page.error();
  }
  message.setRomId(romId);
  message.setManId(manId);
  TRY(ds2476->writeBuffer(message.result()));
  TRY(ds2476->computeSha2UniqueSecret(DS2476::SecretNumA));

  // Compute HMAC on coprocessor.
  // ROM ID, Page 0, and MAN ID are already in message.
  TRY(ds2476->readRng(message.challenge()));
  TRY(ds2476->writeBuffer(message.result()));
  DS2476::Page::array computedHmac;
  TRY_VALUE(computedHmac, ds2476->computeSha2Hmac());

  // Compute HMAC on device.
  TRY(ds28c36.writeBuffer(message.challenge()));
  DS28C36::Page::array deviceHmac;
  TRY_VALUE(deviceHmac, ds28c36.computeAndReadPageAuthentication(
                            message.pageNum(), DS28C36::SecretNumA));

  return (computedHmac == deviceHmac) ? makeResult(none)
                                      : DS28C36::AuthenticationError;
}

Result<void> SensorNode::setLaserEnabled(bool enabled,
                                         const PrintHandler & print) {
  if (print) {
    print((std::string(enabled ? "Enabling" : "Disabling") + " laser").c_str());
  }

  // Compute write HMAC.
  // ROM ID and MAN ID are already in message.
  if (print) {
    print("Reading GPIO Control page");
    print((std::string(
               "Modify copy of GPIO Control page to set GPIO B state to ") +
           (enabled ? "conducting" : "high-impedance"))
              .c_str());
  }
  DS28C36::WriteAuthenticationData message;
  message.setPageNum(DS28C36::gpioControlPage);
  if (const Result<DS28C36::Page::array> oldPage =
          ds28c36.readMemory(message.pageNum())) {
    message.setOldPage(oldPage.value());
  } else {
    return oldPage.error();
  }
  message.setNewPage(message.oldPage());
  DS28C36::GpioControl(message.newPage()).setPiobConducting(enabled);
  message.setRomId(romId);
  message.setManId(manId);
  if (print) {
    print(("Creating HMAC message (ROM ID | Old Page Data | New Page Data | "
           "Page # | MAN ID): " +
           toHexString(message.result()))
              .c_str());
    print("Writing HMAC message to DS2476 buffer");
  }
  TRY(ds2476->writeBuffer(message.result()));
  DS2476::Page::array hmac;
  TRY_VALUE(hmac, ds2476->computeSha2Hmac());

  if (print) {
    print(("DS2476 computes write authentication HMAC from slave secret "
           "(Secret S) and message in buffer: " +
           toHexString(hmac))
              .c_str());
    print("Write authentication HMAC is written to DS28C36 buffer");
    print("DS28C36 computes an HMAC to compare to the write authentication "
          "HMAC after receiving Page # and New Page Data");
  }

  // Write page data.
  TRY(ds28c36.writeBuffer(hmac));
  TRY(ds28c36.authenticatedSha2WriteMemory(
      message.pageNum(), DS28C36::SecretNumA, message.newPage()));
  if (print) {
    print("DS28C36 updates page data which changes GPIO B state");
    print((std::string("Laser is now ") + (enabled ? "enabled" : "disabled"))
              .c_str());
  }
  return none;
}

Result<bool> SensorNode::getLaserEnabled() {
  DS28C36::Page::array page;
  TRY_VALUE(page, ds28c36.readMemory(DS28C36::gpioControlPage));
  return DS28C36::GpioControl(page).piobConducting();
}
