/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#ifndef NORMALOPERATIONWINDOW_HPP
#define NORMALOPERATIONWINDOW_HPP

#include <memory>
#include <vector>
#include <MaximInterfaceCore/array.hpp>
#include <mbed-os/features/netsocket/TCPSocket.h>
#include <rapidjson/document.h>
#include "Button.hpp"
#include "SensorNode.hpp"
#include "Window.hpp"

/// @brief
/// Handles normal operation of the demo including displaying status information
/// and processing commands from the server.
class NormalOperationWindow : public Window {
public:
  /// @param socket
  /// Socket that has been initialized by connecting to the web server. This
  /// must be a heap allocated object that NormalOperationWindow will take
  /// ownership of.
  explicit NormalOperationWindow(std::auto_ptr<TCPSocket> & socket);

protected:
  virtual void resized();
  virtual void updated();
  virtual void doRender(Bitmap & bitmap, int xOffset, int yOffset) const;
  virtual bool doProcessKey(Key key);

private:
  enum Result {
    NoChange,      // Windows not changed; No redraw required.
    WindowsChanged // Windows have changed; Redraw is required.
  };

  std::auto_ptr<TCPSocket> socket;

  // Socket receive buffer. Must be large enough to hold the largest command.
  char recvBuf[1280];

  // Challenge sent to the server with a command response to sign the next
  // command against.
  MaximInterfaceCore::array<uint8_t, 32> commandChallenge;
  bool sendChallenge;

  // Device status information.
  bool validSignature;
  SensorNode::State lastSensorNodeState;
  double lastObjectTemp;
  double lastAmbientTemp;

  Button validSignatureButton;
  Button showWebIdButton;

  // Button event handlers.
  void showWebId(Button *);
  void toggleValidSignature(Button *);

  // Creates a new command challenge, and adds it to an existing JSON document.
  MaximInterfaceCore::Result<void>
  addCommandChallenge(rapidjson::Document & document);

  // Adds signature information to an existing JSON document.
  static MaximInterfaceCore::Result<void>
  signData(rapidjson::Document & document, bool validSignature,
           const std::vector<uint8_t> & challenge);

  // Finalizes a command response to the server by adding the next command
  // challenge and signing the data.
  MaximInterfaceCore::Result<void>
  finalizeResponse(rapidjson::Document & document, bool validSignature,
                   const std::vector<uint8_t> & responseChallenge);

  // Parse and verify a signed JSON string.
  MaximInterfaceCore::Result<void>
  verifySignedData(rapidjson::Document & signedData,
                   MaximInterfaceCore::span<const char> verifyDataIn);

  // Send a JSON document to the server.
  void sendJson(const rapidjson::Value & document);

  // Send a message to the server message log.
  void sendMessage(const char * message);

  Result processReceivedData(size_t recvBufSize);
  Result sendStatus(const std::vector<uint8_t> & responseChallenge);
  Result sendObjectTemp(const std::vector<uint8_t> & responseChallenge);
  Result sendAmbientTemp(const std::vector<uint8_t> & responseChallenge);
  void displayImage(const std::vector<uint8_t> & imageData);
};

#endif
