/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#include <string>
#include <MaximInterfaceCore/HexString.hpp>
#include <MaximInterfaceCore/RomId.hpp>
#include <MaximInterfaceDevices/DS28C36_DS2476.hpp>
#include <mbed-os/platform/mbed_wait_api.h>
#include "Bitmap.hpp"
#include "DisplayIdWindow.hpp"
#include "CC3100.hpp"
#include "ErrorWindow.hpp"
#include "HardwareTestWindow.hpp"
#include "InitWindow.hpp"
#include "MakeFunction.hpp"
#include "SensorNode.hpp"
#include "Text.hpp"
#include "WindowManager.hpp"

using MaximInterfaceCore::Result;
using MaximInterfaceDevices::DS2476;

extern DS2476 coproc;
extern SensorNode sensorNode;
extern std::string webId;

static const int maximLogoWidth = 128;
static const int maximLogoHeight = 39;
static const uint8_t maximLogoData[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xff, 0xe0,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x1f, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xff, 0xfc, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xfe,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0xff, 0xff, 0xff, 0x00, 0x1b, 0x9c, 0x1f, 0x18, 0xcc, 0x6e, 0x70,
    0x00, 0x00, 0x00, 0x00, 0x03, 0xff, 0xff, 0xff, 0x80, 0x1f, 0xfe, 0x3f,
    0x98, 0xcc, 0x7f, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x07, 0xff, 0xff, 0xff,
    0xc0, 0x1c, 0xe6, 0x31, 0x8d, 0x8c, 0x73, 0x98, 0x00, 0x00, 0x00, 0x00,
    0x0f, 0xff, 0xff, 0xff, 0xe0, 0x18, 0xc6, 0x01, 0x8f, 0x8c, 0x63, 0x18,
    0x00, 0x00, 0x00, 0x00, 0x0f, 0x80, 0xfe, 0x03, 0xe0, 0x18, 0xc6, 0x1f,
    0x87, 0x0c, 0x63, 0x18, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x7c, 0x01,
    0xf0, 0x18, 0xc6, 0x39, 0x8f, 0x8c, 0x63, 0x18, 0x00, 0x00, 0x00, 0x00,
    0x1f, 0x00, 0x3c, 0x01, 0xf0, 0x18, 0xc6, 0x31, 0x8d, 0x8c, 0x63, 0x18,
    0x00, 0x00, 0x00, 0x00, 0x3f, 0x00, 0x38, 0x01, 0xf8, 0x18, 0xc6, 0x3f,
    0x98, 0xcc, 0x63, 0x18, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x04, 0x10, 0x41,
    0xf8, 0x18, 0xc6, 0x1d, 0x98, 0xcc, 0x63, 0x18, 0x00, 0x00, 0x00, 0x00,
    0x3f, 0x06, 0x30, 0xe1, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x7f, 0x06, 0x20, 0xe1, 0xfc, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0x07, 0x61, 0xe1,
    0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18,
    0x7f, 0x07, 0xc1, 0xe1, 0xfc, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x18, 0x7f, 0x07, 0xc3, 0xe1, 0xfc, 0x18, 0x00, 0x18,
    0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x18, 0x7f, 0x07, 0x83, 0xe1,
    0xfc, 0x00, 0x00, 0x18, 0x00, 0x00, 0x40, 0x00, 0x06, 0x00, 0x00, 0x18,
    0x7f, 0x07, 0x81, 0xe1, 0xfc, 0x18, 0xde, 0x3e, 0x1e, 0x0f, 0xcd, 0x8f,
    0x8f, 0x87, 0x81, 0xd8, 0x3f, 0x07, 0x01, 0xe1, 0xf8, 0x18, 0xff, 0x3e,
    0x3f, 0x1f, 0xcf, 0x9f, 0xcf, 0x8f, 0xc3, 0xf8, 0x3f, 0x07, 0x00, 0xe1,
    0xf8, 0x18, 0xe3, 0x18, 0x73, 0x99, 0x8e, 0x18, 0xc6, 0x1c, 0xe7, 0x38,
    0x3f, 0x06, 0x00, 0xe1, 0xf8, 0x18, 0xc3, 0x18, 0x61, 0x99, 0x8c, 0x00,
    0xc6, 0x18, 0x66, 0x18, 0x1f, 0x06, 0x10, 0x61, 0xf0, 0x18, 0xc3, 0x18,
    0x7f, 0x9f, 0x8c, 0x0f, 0xc6, 0x1f, 0xe6, 0x18, 0x1f, 0x04, 0x38, 0x61,
    0xf0, 0x18, 0xc3, 0x18, 0x60, 0x0f, 0x0c, 0x1c, 0xc6, 0x18, 0x06, 0x18,
    0x1f, 0x04, 0x38, 0x21, 0xf0, 0x18, 0xc3, 0x18, 0x71, 0x98, 0x0c, 0x18,
    0xc6, 0x1c, 0x67, 0x38, 0x0f, 0xff, 0xff, 0xff, 0xe0, 0x18, 0xc3, 0x1e,
    0x3f, 0x9f, 0x8c, 0x1f, 0xc7, 0x8f, 0xe3, 0xf8, 0x07, 0xff, 0xff, 0xff,
    0xe0, 0x18, 0xc3, 0x0e, 0x1f, 0x3f, 0xcc, 0x0e, 0xc3, 0x87, 0xc1, 0xd8,
    0x07, 0xff, 0xff, 0xff, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x30, 0xc0, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0xff, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00,
    0x00, 0x3f, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xff, 0xff, 0xff,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0xff, 0xff, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xff, 0xf8, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0xff, 0xf0,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x0f, 0xff, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

InitWindow::InitWindow(Mode mode)
    : state((mode == InitMode) ? NotStarted : Completed) {
  startButton.setParent(this);
  startButton.setText("Start");
  startButton.setClickedHandler(
      makeFunction(this, &InitWindow::startButtonClicked));
  hardwareTestButton.setParent(this);
  hardwareTestButton.setText("Hardware test");
  hardwareTestButton.setClickedHandler(
      makeFunction(this, &InitWindow::hardwareTestButtonClicked));
  startButton.setFocused();
}

void InitWindow::resized() {
  startButton.resize(startButton.preferredWidth(),
                     startButton.preferredHeight());
  startButton.move(0, height() - startButton.height());
  hardwareTestButton.resize(hardwareTestButton.preferredWidth(),
                            hardwareTestButton.preferredHeight());
  hardwareTestButton.move(width() - hardwareTestButton.width(),
                          height() - hardwareTestButton.height());
}

void InitWindow::doRender(Bitmap & bitmap, int xOffset, int yOffset) const {
  bitmap.overlay(xOffset + x(), yOffset + y(), maximLogoData,
                 sizeof(maximLogoData) / sizeof(maximLogoData[0]),
                 maximLogoWidth);
  Text description;
  description.setText("MAXREFDES155#");
  if (state == Completed) {
    startButton.render(bitmap, xOffset + x(), yOffset + y());
    hardwareTestButton.render(bitmap, xOffset + x(), yOffset + y());
  } else {
    description.setText(description.text() + "\nInitializing...");
    description.setLineSpacing(3);
  }
  description.resize(description.preferredWidth(),
                     description.preferredHeight());
  description.move(0, maximLogoHeight + 2);
  description.render(bitmap, xOffset + x(), yOffset + y());
}

void InitWindow::updated() {
  switch (state) {
  case NotStarted:
    state = Running;
    invalidate();
    break;

  case Running: {
    // Enable coprocessor.
    if (!enableCoprocessor(coproc)) {
      if (windowManager()) {
        std::auto_ptr<Window> window(
            new ErrorWindow("Failed to enable coprocessor"));
        windowManager()->push(window);
      }
      break;
    }

    // Enable ROM ID.
    if (!enableRomId(coproc)) {
      if (windowManager()) {
        std::auto_ptr<Window> window(
            new ErrorWindow("Failed to enable ROM ID"));
        windowManager()->push(window);
      }
      break;
    }

    // Read ROM ID.
    if (Result<DS2476::Page::array> page =
            coproc.readMemory(DS2476::romOptionsPage)) {
      const MaximInterfaceCore::RomId::const_span romId =
          DS2476::RomOptions(page.value()).romId();
      if (!valid(romId)) {
        if (windowManager()) {
          std::auto_ptr<Window> window(new ErrorWindow("ROM ID is not valid"));
          windowManager()->push(window);
        }
        break;
      }
      webId = MaximInterfaceCore::toHexString(romId);
    } else {
      if (windowManager()) {
        std::auto_ptr<Window> window(new ErrorWindow("Failed to read ROM ID"));
        windowManager()->push(window);
      }
      break;
    }

    // Always start with laser disabled.
    sensorNode.detect();
    sensorNode.setLaserEnabled(false);

    if (CC3100::instance().start() != 0) {
      if (windowManager()) {
        std::auto_ptr<Window> window(
            new ErrorWindow("Failed to start WiFi interface"));
        windowManager()->push(window);
      }
      break;
    }
    wait_ms(1500);
    invalidate();
    state = Completed;
    break;
  }

  case Completed:
    break;
  }
}

bool InitWindow::doProcessKey(Key key) {
  bool handled = false;
  if (state == Completed) {
    switch (key) {
    case LeftKey:
      startButton.setFocused();
      handled = true;
      break;

    case RightKey:
      hardwareTestButton.setFocused();
      handled = true;
      break;

    default:
      break;
    }
  }
  return handled;
}

void InitWindow::startButtonClicked(Button *) {
  if (windowManager()) {
    std::auto_ptr<Window> window(
        new DisplayIdWindow(DisplayIdWindow::PreConnectMode));
    windowManager()->push(window);
  }
}

void InitWindow::hardwareTestButtonClicked(Button *) {
  if (windowManager()) {
    windowManager()->pop();
    std::auto_ptr<Window> window(new HardwareTestWindow);
    windowManager()->push(window);
  }
}
