/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#include <algorithm>
#include <functional>
#include "Bitmap.hpp"
#include "Graphic.hpp"

static const int minWidthHeight = 1;

Graphic::Graphic()
    : parent_(NULL), children_(), focusedChild_(NULL), x_(0), y_(0),
      width_(minWidthHeight), height_(minWidthHeight), valid_(false) {}

static void setParentNull(Graphic * node) { node->setParent(NULL); }

Graphic::~Graphic() {
  // Set children's parent to NULL.
  std::for_each(children_.begin(), children_.end(), setParentNull);
  // Remove from parent.
  setParent(NULL);
}

void Graphic::setParent(Graphic * parent) {
  if ((parent_ == parent) || (parent == this)) {
    return;
  }

  if (parent_) {
    // Remove this from the old parent's list of children.
    parent_->children_.erase(std::remove(parent_->children_.begin(),
                                         parent_->children_.end(), this));
    // Ensure that the old parent's focused child is not this.
    if (parent_->focusedChild_ == this) {
      parent_->focusedChild_ = NULL;
    }
    // Signal children changed event on old parent.
    parent_->childrenChanged();
  }
  if (parent) {
    // Add this to new parent's list of children.
    parent->children_.push_back(this);
    // Signal children changed event on new parent.
    parent->childrenChanged();
  }
  parent_ = parent;
}

bool Graphic::focused() const {
  // First check if a focused child has not been set on this graphic.
  bool focused = !focusedChild_;
  // Then check if each parent has the correct focused child set.
  for (const Graphic * node = this; focused && node->parent_;
       node = node->parent_) {
    focused = (node->parent_->focusedChild_ == node);
  }
  return focused;
}

void Graphic::setFocused() {
  // Locate top-level parent.
  Graphic * node = this;
  while (node->parent_) {
    node = node->parent_;
  }
  // Locate currently focused item.
  while (node->focusedChild_) {
    node = node->focusedChild_;
  }
  // Do nothing if this is already focused.
  if (node == this) {
    return;
  }

  // Create new focus chain by setting the focused child of each parent.
  focusedChild_ = NULL;
  for (Graphic * node = this; node->parent_; node = node->parent_) {
    node->parent_->focusedChild_ = node;
  }
  // Raise focus changed events on both nodes.
  node->focusChanged(false);
  focusChanged(true);
}

void Graphic::move(int x, int y) {
  if (x_ != x || y_ != y) {
    x_ = x;
    y_ = y;
    invalidate();
    moved();
  }
}

void Graphic::resize(int width, int height) {
  if (width_ != width || height_ != height) {
    width_ = std::max(width, minWidthHeight);
    height_ = std::max(height, minWidthHeight);
    invalidate();
    resized();
  }
}

// Functor overlays rendered graphics onto a bitmap.
class RenderOverlay {
public:
  RenderOverlay(Bitmap & bitmap, int xOffset, int yOffset)
      : bitmap(bitmap), xOffset(xOffset), yOffset(yOffset) {}

  void operator()(const Graphic * graphic) {
    if (graphic) {
      graphic->render(bitmap, xOffset, yOffset);
    }
  }

private:
  Bitmap & bitmap;
  const int xOffset;
  const int yOffset;
};

void Graphic::doRender(Bitmap & bitmap, int xOffset, int yOffset) const {
  std::for_each(children_.begin(), children_.end(),
                RenderOverlay(bitmap, xOffset + x(), yOffset + y()));
}

void Graphic::render(Bitmap & bitmap, int xOffset, int yOffset) const {
  // Clear region.
  bitmap.clear(xOffset + x(), yOffset + y(), width(), height());
  // Do actual rendering.
  doRender(bitmap, xOffset, yOffset);
}

void Graphic::render(Bitmap & bitmap) const {
  int xOffset = 0;
  int yOffset = 0;
  for (const Graphic * graphic = parent(); graphic;
       graphic = graphic->parent()) {
    xOffset += graphic->x();
    yOffset += graphic->y();
  }
  render(bitmap, xOffset, yOffset);
}

void Graphic::updateAll() {
  // Perform updated event on this graphic.
  updated();
  // Call recursively on each child.
  std::for_each(children_.begin(), children_.end(),
                std::mem_fun(&Graphic::updateAll));
}

bool Graphic::redrawInvalid(Bitmap & canvas) {
  bool redraw = !valid_;
  if (redraw) {
    // Redraw if invalid.
    render(canvas);
    // Set all children to valid since they were incorporated in the redraw.
    setAllValid();
  } else {
    // Call recursively on each child.
    for (ChildContainer::iterator it = children_.begin(); it != children_.end();
         ++it) {
      if ((*it)->redrawInvalid(canvas)) {
        redraw = true;
      }
    }
  }
  return redraw;
}

void Graphic::setAllValid() {
  valid_ = true;
  // Call recursively on each child.
  std::for_each(children_.begin(), children_.end(),
                std::mem_fun(&Graphic::setAllValid));
}

bool Graphic::update(Bitmap * canvas) {
  bool redraw = false;
  updateAll();
  if (canvas) {
    redraw = redrawInvalid(*canvas);
  }
  return redraw;
}

bool Graphic::processKey(Key key) {
  // Find focused child.
  Graphic * receiver = this;
  while (receiver->focusedChild_) {
    receiver = receiver->focusedChild_;
  }
  // Pass event to focused child and then to each parent until handled.
  bool handled = false;
  do {
    handled = receiver->doProcessKey(key);
    receiver = receiver->parent_;
  } while (!handled && receiver);
  return handled;
}

void Graphic::childrenChanged() { invalidate(); }

void Graphic::focusChanged(bool) {}

void Graphic::moved() {}

void Graphic::resized() {}

void Graphic::updated() {}

bool Graphic::doProcessKey(Key) { return false; }
