/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#ifndef CC3100_HPP
#define CC3100_HPP

#include <stdint.h>
#include <stddef.h>
#include <ctime>
#include <string>
#include <mbed-os/drivers/DigitalOut.h>
#include <mbed-os/drivers/SPI.h>
#include <mbed-os/features/netsocket/NetworkStack.h>
#include <mbed-os/features/netsocket/WiFiInterface.h>

/// CC3100 Wi-Fi station interface for mbed using SPI.
class CC3100 : public NetworkStack, public WiFiInterface {
public:
  /// SPI bus interface.
  class SPI {
  public:
    SPI(PinName mosi, PinName miso, PinName sclk, PinName ssel);

    void transfer(const uint8_t * txData, size_t dataSize, uint8_t * rxData);

  private:
    mbed::SPI spi;
    mbed::DigitalOut cs;
  };

  /// CC3100 device state.
  enum State {
    Stopped,
    Disconnected,
    ConnectedWithoutIp,
    Connected,
    Pinging
  };

  /// @brief
  /// Result code occurring when the CC3100 is in an invalid state to perform
  /// an operation.
  static const int invalidState = -1;

  /// Retrieve the singleton instance.
  static CC3100 & instance();

  /// Current state of the CC3100.
  State state() const;

  /// Start the CC3100 so that nework functionality is available.
  int start();

  /// Stop the CC3100 and enter a low power state.
  int stop();

  /// Update the networking stack. Should be called regularly to dispatch events.
  void update();

  /// Set the current date and time.
  int setDateTime(const std::tm & dateTime);

  // WiFi station functionality

  virtual int set_credentials(const char * ssid, const char * pass,
                              nsapi_security_t security = NSAPI_SECURITY_NONE);

  virtual int set_channel(uint8_t channel) {
    this->channel = channel;
    return 0;
  }

  virtual int8_t get_rssi() { return 0; } // Not implemented

  virtual int connect(const char * ssid, const char * pass,
                      nsapi_security_t security = NSAPI_SECURITY_NONE,
                      uint8_t channel = 0);

  virtual int connect() {
    return connect(ssid.c_str(), pass.c_str(), security, channel);
  }

  int connect(const char * ssid, const char * username, const char * password);

  virtual int disconnect();

  virtual int scan(WiFiAccessPoint * res, unsigned count) {
    // Not implemented
    return invalidState;
  }

  // Network functionality
  virtual const char * get_ip_address() { return NULL; } // Not implemented

  virtual int gethostbyname(const char * host, SocketAddress * address,
                            nsapi_version_t version = NSAPI_UNSPEC);

  int ping(const SocketAddress & address);
  
  virtual NetworkStack * get_stack() { return this; }

  // Socket functionality

  virtual int socket_open(nsapi_socket_t * handle, nsapi_protocol_t proto);

  int socket_set_cert_path(nsapi_socket_t handle, const char * certPath);

  virtual int socket_close(nsapi_socket_t handle);

  virtual int socket_bind(nsapi_socket_t handle, const SocketAddress & address);

  virtual int socket_listen(nsapi_socket_t handle, int backlog);

  virtual int socket_connect(nsapi_socket_t handle,
                             const SocketAddress & address);

  virtual int socket_accept(nsapi_socket_t server, nsapi_socket_t * handle,
                            SocketAddress * address = 0) {
    // Not implemented
    return invalidState;
  }

  virtual int socket_send(nsapi_socket_t handle, const void * data,
                          unsigned size);

  virtual int socket_recv(nsapi_socket_t handle, void * data, unsigned size);

  virtual int socket_sendto(nsapi_socket_t handle,
                            const SocketAddress & address, const void * data,
                            unsigned size) {
    // Not implemented
    return invalidState;
  }

  virtual int socket_recvfrom(nsapi_socket_t handle, SocketAddress * address,
                              void * buffer, unsigned size) {
    // Not implemented
    return invalidState;
  }

  virtual void socket_attach(nsapi_socket_t handle, void (*callback)(void *),
                             void * data) {
    // Not implemented
  }

private:
  SPI spi;

  std::string ssid;
  std::string pass;
  nsapi_security_t security;
  uint8_t channel;

  CC3100();

  // Uncopyable
  CC3100(const CC3100 &);
  const CC3100 & operator=(const CC3100 &);
};

#endif
