/*******************************************************************************
* Copyright (C) Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
*******************************************************************************/

#ifndef BITMAP_HPP
#define BITMAP_HPP

#include <stddef.h>
#include <stdint.h>
#include <vector>

/// Stores a black and white bitmap image.
class Bitmap {
public:
  /// @brief Construct a zero-initialized bitmap with specified dimensions.
  /// @param width Width in pixels that is >= 1.
  /// @param height Height in pixels that is >= 1.
  Bitmap(int width, int height);

  /// @brief Load bitmap from a byte array.
  /// @param data Array of scanline bitmap data.
  /// @param size Size of data array.
  /// @param width Width of loaded image in pixels that is >= 1.
  Bitmap(const uint8_t * data, size_t size, int width);

  /// Width in pixels.
  int width() const { return width_; }

  /// Height in pixels.
  int height() const { return height_; }

  /// @brief Check if a pixel is enabled (black).
  /// @returns
  /// True if the pixel is enabled or false if the coordinate is out of range.
  bool pixelEnabled(int x, int y) const;

  /// @brief Enable or disable a pixel.
  /// @param True to set to black. False to set to white.
  void setPixelEnabled(int x, int y, bool enabled);

  /// @brief Overlay another bitmap on top of this bitmap.
  /// @param x x-coordinate location to overlay.
  /// @param y y-coordinate location to overlay.
  /// @param src Bitmap to overlay.
  void overlay(int x, int y, const Bitmap & src);

  /// @brief Overlay bitmap data from a byte array on top of this bitmap.
  /// @param x x-coordinate location to overlay.
  /// @param y y-coordinate location to overlay.
  /// @param data Array of scanline bitmap data.
  /// @param size Size of data array.
  /// @param width Width of overlayed image in pixels that is >= 1.
  void overlay(int x, int y, const uint8_t * data, size_t size, int width);

  /// Reset to initial state.
  void clear();

  /// @brief Reset region to initial state.
  /// @param x,y Coordinates to begin clearing at.
  /// @param width,height Dimensions of the cleared region.
  void clear(int x, int y, int width, int height);

private:
  int width_;
  int height_;
  std::vector<uint8_t> data_;
};

#endif
