/**********************************************************************
* Copyright (C) 2016 Maxim Integrated Products, Inc., All Rights Reserved.
*
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
* OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
* IN NO EVENT SHALL MAXIM INTEGRATED BE LIABLE FOR ANY CLAIM, DAMAGES
* OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
* Except as contained in this notice, the name of Maxim Integrated
* Products, Inc. shall not be used except as stated in the Maxim Integrated
* Products, Inc. Branding Policy.
*
* The mere transfer of this software does not imply any licenses
* of trade secrets, proprietary technology, copyrights, patents,
* trademarks, maskwork rights, or any other form of intellectual
* property whatsoever. Maxim Integrated Products, Inc. retains all
* ownership rights.
**********************************************************************/


#include <string>
#include "maxrefdes131_demo.h"

enum DemoState
{
    STATE_POR,
    STATE_ESTABLISH_COMMS,
    STATE_ENUMERATE_SENSORS,
    STATE_SELECT_SENSOR,
    STATE_SEND_DATA,
    STATE_STOP_DATA,
    STATE_IDLE,
    STATE_INVALID
};

volatile DemoState demoState = STATE_POR;
volatile uint8_t selectedSensor = 0;

Serial comms(USBTX, USBRX);
DigitalOut frameRecvdPulse(D4, 1);

void rx_callback(void)
{
    std::string rx_str;
    //size to 32 chars
    rx_str.resize(32);
    
    size_t str_idx = 0;
    do
    {
        rx_str[str_idx++] = comms.getc();
    }
    while(rx_str[str_idx - 1] != '\n');
    
    //Just get first 5 chars that represent command
    std::string rx_cmd = rx_str.substr(0,5);
    
    
    if(rx_cmd.compare(0, 5, "CMD_1") == 0)
    {
        demoState = STATE_ESTABLISH_COMMS;
    }
    else if(rx_cmd.compare(0, 5, "CMD_2") == 0)
    {
        demoState = STATE_ENUMERATE_SENSORS;
    }
    else if(rx_cmd.compare(0, 5, "CMD_3") == 0)
    {
        selectedSensor = (rx_str[6] - 48);
        demoState = STATE_SELECT_SENSOR;
    }
    else if(rx_cmd.compare(0, 5, "CMD_4") == 0)
    {
        demoState = STATE_SEND_DATA;
    }
    else if(rx_cmd.compare(0, 5, "CMD_5") == 0)
    {
        demoState = STATE_STOP_DATA;
    }
    else if(rx_cmd.compare(0, 5, "CMD_6") == 0)
    {
        frameRecvdPulse = 0;
        frameRecvdPulse = 1;
        demoState = STATE_SEND_DATA;
    }
    else
    {
        demoState = STATE_INVALID;
    }
}

int main(void)
{
    comms.baud(115200);
    comms.attach(&rx_callback, SerialBase::RxIrq);
    
    I2C i2cBus(D14, D15);
    i2cBus.frequency(400000);
    DS2484 owm(i2cBus);
    
    OneWireMaster::CmdResult ow_result;
    MultidropRomIterator selector(owm);
    
    //look up table for sensors, element 0 is location 1
    OWGridEye * ow_grid_eye_array[10]; 
    uint8_t num_sensors = 0; 
    OWGridEye * p_sensor;
    OWGridEye::CmdResult sensor_result;
    
    uint8_t idx;
    int16_t recvBuff[64];
    int16_t imageBuff[64];
    float temperatureBuff[64];
    float thermistor;
    
    bool sensorsEnumerated = false;
    bool sensorSelected = false;
    
    while(demoState == STATE_POR);
    
    while(1)
    {   
        switch(demoState)
        {
            case STATE_ESTABLISH_COMMS:
            
                ow_result = owm.OWInitMaster();
                if(ow_result == OneWireMaster::Success)
                {
                    comms.printf("MBED: 1-wire Master Initialized\n");
                    demoState = STATE_IDLE;
                }
                else
                {
                    comms.printf("MBED: Failed to initialize the 1-wire Master\n");
                    wait_ms(500);
                }
                
            break;
            
            case STATE_ENUMERATE_SENSORS:
            
                if(sensorsEnumerated)
                {
                    for(uint8_t reset_idx = 0; reset_idx < num_sensors; reset_idx++)
                    {
                        delete ow_grid_eye_array[reset_idx];
                        ow_grid_eye_array[reset_idx] = 0;
                    }
                    sensorsEnumerated = false;
                }
                
                if(!sensorsEnumerated)
                {
                    //ensure default state of DS2413 PIO
                    num_sensors = init_sensor_state(owm, comms);
                    comms.printf("MBED: %d sensors initialized\n", num_sensors);
                
                    num_sensors = enumerate_sensors(comms, owm, selector, ow_grid_eye_array);
                    comms.printf("MBED: %d sensors enumerated\n", num_sensors);
                    
                    comms.printf("MBED: NUM_Sensors %d\n", num_sensors);
                }
                
                demoState = STATE_IDLE;
                
            break;
            
            case STATE_SELECT_SENSOR:
                
                if(selectedSensor > 0)
                {
                    if(sensorSelected)
                    {
                        sensor_result = p_sensor->disconnectGridEye();
                        if(sensor_result != OWGridEye::Success)
                        {
                            comms.printf("MBED: Failed to disconnect sensor\n");
                        }
                        else
                        {
                            sensorSelected = false;
                        }
                    }
                    
                    if(!sensorSelected)
                    {
                        p_sensor = ow_grid_eye_array[selectedSensor - 1];
                        sensor_result = p_sensor->connectGridEye();
                        if(sensor_result == OWGridEye::Success)
                        {
                            //wait for sensor to init
                            wait_ms(50);
                            sensorSelected = true;
                        }
                        else
                        {
                            comms.printf("MBED: Failed to connect sensor\n");
                            sensorSelected = false;
                        }
                    }
                }
                else
                {
                    comms.printf("MBED: Invalid Sensor Index\n");
                }
                
                demoState = STATE_IDLE;
                
            break;
            
            case STATE_SEND_DATA:
                
                if(sensorSelected)
                {
                    //get raw data from GridEye
                    sensor_result = p_sensor->gridEyeGetFrameTemperature(recvBuff); 
                    if(sensor_result == OWGridEye::Success)
                    {
                        //rotate it 180 degress
                        vAMG_PUB_IMG_ConvertRotate180(8, 8, recvBuff, imageBuff);
                        
                        //move rotated image to recvBuff so we can flip it and put it back
                        //in imageBuff
                        std::memcpy(recvBuff, imageBuff, 128);
                        
                        //flip on X axis
                        vAMG_PUB_IMG_ConvertFlipX(8, 8, recvBuff, imageBuff);
                        
                        //convert to float
                        for(idx = 0; idx < 64; idx++)
                        {
                            temperatureBuff[idx] = fAMG_PUB_CMN_ConvStoF(imageBuff[idx]);
                        }
                        
                        //get thermistor data
                        p_sensor->gridEyeGetThermistor(recvBuff[0]);
                        
                        //convert to float
                        thermistor = fAMG_PUB_CMN_ConvStoF(recvBuff[0]);
                        
                        //send to GUI
                        send_data(comms, temperatureBuff, thermistor);
                        
                        demoState = STATE_IDLE;
                    }
                    else
                    {
                        comms.printf("MBED: Failed to Get Frame\n");
                    }
                }
                else
                {
                    comms.printf("MBED: Please Select Sensor\n");
                }
                 
            break;
            
            case STATE_STOP_DATA:
                demoState = STATE_IDLE;
            break;
            
            case STATE_IDLE:
                //do nothing
            break;
            
            default:
                mbed_die();  
            break;
        }
    }
}
